/* JavaScript for Deals Table Order module */
jQuery(document).ready(function() {
    console.log('Deals Table Order: Script loaded, jQuery version:', jQuery.fn.jquery);
    console.log('Deals Table Order: DataTables version:', jQuery.fn.DataTable.version || 'Not loaded');

    if (jQuery('#deals.table-deals').length) {
        console.log('Deals Table Order: Deals table detected');
        waitForUtilities(function() {
            injectConfigureColumnsButton();
            waitForDataTable(function() {
                applyCustomColumnOrder();
            });
        }, 10000);
    } else {
        console.log('Deals Table Order: Deals table not found');
    }

    jQuery(document).on('click', '.deals-table-order-btn', function(e) {
        e.preventDefault();
        console.log('Deals Table Order: Configure Columns button clicked');
        loadColumnOrderModal();
    });
});

function waitForUtilities(callback, timeoutMs) {
    let startTime = Date.now();
    function checkUtilities() {
        if (typeof _l !== 'undefined' && typeof get_option !== 'undefined') {
            console.log('Deals Table Order: Utilities defined');
            callback();
        } else if (Date.now() - startTime > timeoutMs) {
            console.warn('Deals Table Order: Timeout waiting for utilities');
            callback();
        } else {
            setTimeout(checkUtilities, 100);
        }
    }
    checkUtilities();
}

function waitForDataTable(callback) {
    let attempts = 0;
    const maxAttempts = 200;
    function checkDataTable() {
        const $table = jQuery('#deals.table-deals');
        if ($table.length && jQuery.fn.DataTable.isDataTable($table)) {
            console.log('Deals Table Order: DataTable initialized');
            callback();
        } else if (attempts >= maxAttempts) {
            console.warn('Deals Table Order: Timeout waiting for DataTable');
            callback();
        } else {
            attempts++;
            setTimeout(checkDataTable, 100);
        }
    }
    checkDataTable();
}

function injectConfigureColumnsButton() {
    if (jQuery('.deals-table-order-btn').length) {
        console.log('Deals Table Order: Button already exists');
        return;
    }

    let $buttonsContainer = jQuery('._buttons .tw-flex.tw-justify-between');
    let $searchContainer = $buttonsContainer.find('.tw-inline');

    if (!$buttonsContainer.length || !$searchContainer.length) {
        $buttonsContainer = jQuery('._buttons');
        $searchContainer = null;
    }

    if ($buttonsContainer.length) {
        var configureText = typeof _l !== 'undefined' ? _l('deals_table_order_configure') : 'Configure Columns';
        var buttonHtml = `
            <div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
                <a href="#" class="btn btn-default deals-table-order-btn" data-toggle="modal" data-target="#deals-table-order-modal">
                    <i class="fa fa-columns tw-mr-1"></i> ${configureText}
                </a>
            </div>
        `;
        if ($searchContainer) {
            $searchContainer.before(buttonHtml);
        } else {
            $buttonsContainer.append(buttonHtml);
        }
        console.log('Deals Table Order: Button injected');
    } else {
        console.error('Deals Table Order: Could not find _buttons container');
    }
}

function applyCustomColumnOrder() {
    console.log('Deals Table Order: Applying custom column order');
    jQuery.get(admin_url + 'deals/get_column_order')
        .done(function(response) {
            let parsedResponse;
            try {
                parsedResponse = JSON.parse(response);
            } catch (e) {
                console.error('Deals Table Order: Invalid JSON response:', response);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            if (parsedResponse.success && parsedResponse.column_order) {
                const $table = jQuery('#deals.table-deals');
                const $thead = $table.find('thead tr:last');
                const $tbody = $table.find('tbody');
                const $rows = $tbody.find('tr');

                console.log('Deals Table Order: Found', $rows.length, 'rows in tbody');

                // Map column IDs to indices
                const columnIdToIndex = {};
                const headers = jQuery('th', $thead).map(function(index) {
                    const $th = jQuery(this);
                    let colId;
                    if ($th.find('input[type="checkbox"][id="mass_select_all"]').length) {
                        colId = 'checkbox';
                    } else {
                        colId = $th.attr('id') ? $th.attr('id').replace('th-', '') : 'column_' + index;
                    }
                    columnIdToIndex[colId] = index;
                    return { id: colId, index: index };
                }).get();
                console.log('Deals Table Order: Current headers:', headers);
                console.log('Deals Table Order: Checkbox index:', columnIdToIndex['checkbox'] || 'not found');

                // Create new column order
                let columnOrder = [];
                parsedResponse.column_order.forEach(function(colId) {
                    if (columnIdToIndex[colId] !== undefined) {
                        columnOrder.push(columnIdToIndex[colId]);
                    }
                });
                if (columnIdToIndex['checkbox'] !== undefined && !parsedResponse.column_order.includes('checkbox')) {
                    columnOrder.unshift(columnIdToIndex['checkbox']);
                }

                console.log('Deals Table Order: New column order:', columnOrder);

                // Apply column order to DataTable
                const table = $table.DataTable();
                table.order(columnOrder.map(function(idx) { return [idx, 'asc']; }));
                table.columns().every(function(index) {
                    const colId = headers.find(h => h.index === index).id;
                    const isVisible = parsedResponse.column_order.includes(colId);
                    this.visible(isVisible);
                });
                table.draw();

                // Reorder DOM elements
                const $newTheadRow = jQuery('<tr></tr>');
                columnOrder.forEach(function(index) {
                    const $th = $thead.find('th').eq(index).clone(true);
                    $newTheadRow.append($th);
                });
                $thead.empty().append($newTheadRow);

                $rows.each(function() {
                    const $row = jQuery(this);
                    const $newRow = jQuery('<tr></tr>').addClass($row.attr('class'));
                    columnOrder.forEach(function(index) {
                        const $td = $row.find('td').eq(index).clone(true);
                        $newRow.append($td);
                    });
                    $row.replaceWith($newRow);
                });

                console.log('Deals Table Order: Columns reordered and visibility applied');
            } else {
                console.warn('Deals Table Order: No column order data received');
            }
        })
        .fail(function(xhr, status, error) {
            console.error('Deals Table Order: Failed to fetch column order:', status, error);
            alert_float('danger', 'Failed to load column order.');
        });
}

function loadColumnOrderModal() {
    console.log('Deals Table Order: Loading column order modal');
    jQuery.get(admin_url + 'deals/get_column_order')
        .done(function(response) {
            let parsedResponse;
            try {
                parsedResponse = JSON.parse(response);
            } catch (e) {
                console.error('Deals Table Order: Invalid JSON response:', response);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            const $table = jQuery('#deals.table-deals');
            const $thead = $table.find('thead tr:last');
            const headers = $thead.find('th').map(function(index) {
                const $th = jQuery(this);
                let colId, colText;
                if ($th.find('input[type="checkbox"][id="mass_select_all"]').length) {
                    colId = 'checkbox';
                    colText = 'Checkbox';
                } else {
                    colId = $th.attr('id') ? $th.attr('id').replace('th-', '') : 'column_' + index;
                    colText = $th.text().trim() || 'Column ' + index;
                }
                return { id: colId, text: colText };
            }).get();

            let modalHtml = `
                <div class="modal fade" id="deals-table-order-modal" tabindex="-1" role="dialog">
                    <div class="modal-dialog" role="document">
                        <div class="modal-content">
                            <div class="modal-header">
                                <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
                                <h4 class="modal-title">${_l('deals_table_order_configure') || 'Configure Columns'}</h4>
                            </div>
                            <div class="modal-body">
                                <ul class="list-group deals-column-order-list">
                                    ${headers.map(function(header, index) {
                                        const isChecked = parsedResponse.column_order.includes(header.id) ? 'checked' : '';
                                        return `
                                            <li class="list-group-item" data-column-id="${header.id}">
                                                <div class="checkbox checkbox-primary">
                                                    <input type="checkbox" id="col_${header.id}" ${isChecked}>
                                                    <label for="col_${header.id}">${header.text}</label>
                                                </div>
                                            </li>
                                        `;
                                    }).join('')}
                                </ul>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-default" data-dismiss="modal">${_l('close') || 'Close'}</button>
                                <button type="button" class="btn btn-primary" onclick="saveDealsColumnOrder()">${_l('save') || 'Save'}</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            jQuery('body').append(modalHtml);
            jQuery('#deals-table-order-modal .deals-column-order-list').sortable({
                placeholder: 'ui-state-highlight',
                update: function() {
                    console.log('Deals Table Order: Columns reordered in modal');
                }
            });
            jQuery('#deals-table-order-modal').modal('show');
        })
        .fail(function(xhr, status, error) {
            console.error('Deals Table Order: Failed to load modal data:', status, error);
            alert_float('danger', 'Failed to load column configuration.');
        });
}

function saveDealsColumnOrder() {
    console.log('Deals Table Order: Saving column order');
    const $modal = jQuery('#deals-table-order-modal');
    const columnOrder = [];
    $modal.find('.deals-column-order-list li').each(function() {
        const $li = jQuery(this);
        const colId = $li.data('column-id');
        const isVisible = $li.find('input[type="checkbox"]').is(':checked');
        if (isVisible && colId !== 'checkbox') {
            columnOrder.push(colId);
        }
    });

    jQuery.post(admin_url + 'deals/save_column_order', {
        column_order: JSON.stringify(columnOrder)
    }).done(function(response) {
        let parsedResponse;
        try {
            parsedResponse = JSON.parse(response);
        } catch (e) {
            console.error('Deals Table Order: Invalid JSON response:', response);
            alert_float('danger', 'Error parsing server response.');
            return;
        }

        if (parsedResponse.success) {
            console.log('Deals Table Order: Column order saved successfully');
            alert_float('success', parsedResponse.message || 'Column order saved.');
            $modal.modal('hide');
            applyCustomColumnOrder();
        } else {
            console.error('Deals Table Order: Failed to save column order:', parsedResponse.message);
            alert_float('danger', parsedResponse.message || 'Failed to save column order.');
        }
    }).fail(function(xhr, status, error) {
        console.error('Deals Table Order: Failed to save column order:', status, error);
        alert_float('danger', 'Failed to save column order.');
    });
}