<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('deals/deals_model');
    }

    public function index()
    {
        if ($this->input->get('view') === 'kanban') {
            return $this->kanban();
        }

        $data['title'] = _l('deals');
        $data['deals'] = $this->deals_model->get_all();
        // dd($data);
        $this->load->view('list', $data);
    }

    public function kanban()
    {
        $data['title'] = _l('deals_kanban');
        $data['stages'] = $this->deals_model->get_stages_with_deals();
        $this->load->view('kanban', $data);
    }

    public function create()
    {
        if ($this->input->post()) {
            $post = $this->input->post();
            // Create base deal
            $dealId = $this->deals_model->create($post);
            // Save multi-pipelines
            if ($dealId && isset($post['pipelines']) && is_array($post['pipelines'])) {
                foreach ($post['pipelines'] as $pipe) {
                    $pipelineId = (int)($pipe['pipeline_id'] ?? 0);
                    if ($pipelineId <= 0) { continue; }
                    $stageId = isset($pipe['stage_id']) ? (int)$pipe['stage_id'] : null;
                    $score = isset($pipe['score']) && $pipe['score'] !== '' ? (int)$pipe['score'] : null;
                    $this->deals_model->add_pipeline_to_deal($dealId, $pipelineId, $stageId, null, $score);
                }
            }
            set_alert('success', _l('added_successfully', _l('deal')));

            return redirect(admin_url('deals'));
        }
        $data['title'] = _l('new_deal');
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $data['sources'] = $this->deals_model->get_sources();
        $data['statuses'] = $this->deals_model->get_statuses();
        $data['staff'] = $this->deals_model->get_staff(); // New: Add staff for assigned
        $this->load->view('create', $data);
    }

    public function edit($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }
        if ($this->input->post()) {
            $post = $this->input->post();
            $this->deals_model->update_deal($id, $post);
            // Update pipelines
            if (isset($post['pipelines']) && is_array($post['pipelines'])) {
                // Remove all existing relations first for simplicity
                $this->db->where('deal_id', (int)$id)->delete(db_prefix() . 'deals_pipeline_relations');
                foreach ($post['pipelines'] as $pipe) {
                    $pipelineId = (int)($pipe['pipeline_id'] ?? 0);
                    if ($pipelineId <= 0) { continue; }
                    $stageId = isset($pipe['stage_id']) ? (int)$pipe['stage_id'] : null;
                    $score = isset($pipe['score']) && $pipe['score'] !== '' ? (int)$pipe['score'] : null;
                    $this->deals_model->add_pipeline_to_deal($id, $pipelineId, $stageId, null, $score);
                }
            }
            set_alert('success', _l('updated_successfully', _l('deal')));
            return redirect(admin_url('deals/view/' . $id));
        }
        $data['title'] = _l('edit');
        $data['deal'] = $deal;
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $data['stages'] = $this->deals_model->get_stages($deal['pipeline_id']);
        $data['sources'] = $this->deals_model->get_sources();
        $data['statuses'] = $this->deals_model->get_statuses();
        $data['staff'] = $this->deals_model->get_staff(); // New
        $this->load->view('create', $data);
    }



    public function validate_deal_import()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        if (staff_can('create', 'deals') || staff_can('edit', 'deals')) {
            if (!class_exists('XLSXReader_fin')) {
                require_once module_dir_path('deals') . 'libraries/XLSXReader/XLSXReader.php';
            }
            require_once module_dir_path('deals') . 'libraries/XLSXWriter/xlsxwriter.class.php';

            $simulate = $this->input->post('simulate');
            $formats = ['xlsx', 'xls', 'ods', 'csv'];
            $file = $this->input->post('file') ? $this->input->post('file') : null;

            if (is_null($file) || !in_array(pathinfo($file['name'], PATHINFO_EXTENSION), $formats)) {
                set_alert('warning', _l('import_upload_failed'));
                redirect(admin_url('deals/import'));
            }

            $path = $file['tmp_name'];
            $total_imported = 0;
            $total_duplicate = 0;
            $total_failed = 0;
            $total_invalid = 0;
            $rows = [];

            $reader = new XLSXReader($path);
            $sheetNames = $reader->getSheetNames();
            $sheet = $reader->getSheet($sheetNames[1]);
            $total_rows = 0;

            $data = $sheet->getData();
            foreach ($data as $key => $row) {
                $tmp_total_rows++;
                if ($tmp_total_rows == 1) {
                    continue;
                }
                $record = [];

                $record['name'] = isset($row[0]) ? trim($row[0]) : '';
                $record['company'] = isset($row[1]) ? trim($row[1]) : '';
                $record['description'] = isset($row[2]) ? trim($row[2]) : '';
                $record['deal_value'] = isset($row[3]) ? trim($row[3]) : 0;
                $record['expected_close_date'] = isset($row[4]) ? trim($row[4]) : null;
                $record['source'] = isset($row[5]) ? trim($row[5]) : null;
                $record['status'] = isset($row[6]) ? trim($row[6]) : null;
                $record['assigned'] = isset($row[7]) ? trim($row[7]) : null;

                // Validation
                if (empty($record['name'])) {
                    $total_invalid++;
                    continue;
                }

                $rows[] = $record;
                $total_rows++;
            }

            if ($this->input->post('simulate') || $simulate) {
                if ($total_rows > 0) {
                    $heading = ['Name', 'Company', 'Description', 'Value', 'Expected Close', 'Source', 'Status', 'Assigned'];
                    echo '<table class="table table-import-preview"><thead><tr>';
                    foreach ($heading as $column_heading) {
                        echo '<th>' . $column_heading . '</th>';
                    }
                    echo '</tr></thead><tbody>';
                    foreach ($rows as $row) {
                        echo '<tr>';
                        foreach ($row as $val) {
                            echo '<td>' . e($val) . '</td>';
                        }
                        echo '</tr>';
                    }
                    echo '</tbody></table>';
                } else {
                    echo '<div class="alert alert-info">' . _l('import_no_data_found') . '</div>';
                }
                exit;
            }

            // Actual import
            foreach ($rows as $row) {
                $insert = [
                    'name' => $row['name'],
                    'company' => $row['company'],
                    'description' => $row['description'],
                    'deal_value' => $row['deal_value'],
                    'expected_close_date' => $row['expected_close_date'],
                    'source' => $this->deals_model->get_source_id_by_name($row['source']),
                    'status' => $this->deals_model->get_status_id_by_name($row['status']),
                    'assigned' => $this->deals_model->get_staff_id_by_name($row['assigned']),
                    'dateadded' => date('Y-m-d H:i:s'),
                    'addedfrom' => get_staff_user_id(),
                    'pipeline_id' => $this->deals_model->get_default_pipeline_id(),
                    'stage_id' => $this->deals_model->get_default_open_stage_id(),
                    'status_final' => 'open',
                ];

                $id = $this->deals_model->create($insert);
                if ($id) {
                    $total_imported++;
                } else {
                    $total_failed++;
                }
            }

            set_alert('success', _l('import_total_imported', $total_imported));
            redirect(admin_url('deals'));
        } else {
            access_denied('deals');
        }
    }

    public function update_amount($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $amount = $this->input->post('deal_value');
        if (!is_numeric($amount) || $amount < 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid amount']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['deal_value' => $amount]);
        echo json_encode(['success' => (bool) $updated]);
    }

    public function update_close_date($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $date = $this->input->post('expected_close_date');
        if (!$date || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            echo json_encode(['success' => false, 'message' => 'Invalid date']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['expected_close_date' => $date]);
        echo json_encode(['success' => (bool) $updated]);
    }


    public function delete($id)
    {
        if ($this->deals_model->delete($id)) {
            set_alert('success', _l('deleted', _l('deal')));
        }
        redirect(admin_url('deals'));
    }

    public function table()
    {
        $this->app->get_table_data('deals', [
            'select' => [
                'd.id',
                'd.name',
                'd.company',
                'd.deal_value',
                'd.dateadded',
                'd.status_final',
                'dp.name as pipeline_name',
                'ds.name as stage_name',
                'CONCAT(s.firstname, " ", s.lastname) as assigned_staff'
            ],
            'from' => db_prefix() . 'deals d',
            'join' => [
                'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = d.pipeline_id',
                'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = d.stage_id',
                'LEFT JOIN ' . db_prefix() . 'staff s ON s.staffid = d.addedfrom'
            ]
        ]);
    }

    // Deals table column order (per staff)
    public function get_column_order()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $result = $this->deals_model->get_column_order();
        echo json_encode($result);
    }

    public function save_column_order()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $order = $this->input->post('column_order');
        if (!is_array($order)) {
            echo json_encode(['success' => false, 'message' => _l('deals_table_order_invalid_data')]);
            return;
        }

        $response = $this->deals_model->save_column_order($order);
        echo json_encode($response);
    }

    public function get_stages($pipeline_id)
    {
        $stages = $this->deals_model->get_stages($pipeline_id);
        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => true,
            'stages' => $stages
        ]));
    }

    public function bulk_action()
    {
        $action = $this->input->post('bulk_action');
        $ids = $this->input->post('ids');

        if (!$ids || !$action) {
            $this->output->set_content_type('application/json')->set_output(json_encode([
                'success' => false,
                'message' => _l('no_action_selected')
            ]));
            return;
        }

        $success = false;
        $message = '';

        switch ($action) {
            case 'delete':
                foreach ($ids as $id) {
                    $this->deals_model->delete($id);
                }
                $success = true;
                $message = _l('deals_deleted_successfully');
                break;

            case 'move_pipeline':
                $pipeline_id = $this->input->post('pipeline_id');
                $stage_id = $this->input->post('stage_id');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, [
                        'pipeline_id' => $pipeline_id,
                        'stage_id' => $stage_id
                    ]);
                }
                $success = true;
                $message = _l('deals_moved_successfully');
                break;

            case 'assign':
                $assigned_to = $this->input->post('assigned_to');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, ['addedfrom' => $assigned_to]);
                }
                $success = true;
                $message = _l('deals_assigned_successfully');
                break;
        }

        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => $success,
            'message' => $message
        ]));
    }

    public function view($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }
        $data['deal'] = $deal;

        // Multi-pipeline support: get all pipeline relations for this deal
        $deal_pipelines = $this->deals_model->get_deal_pipelines($id);

        // For each pipeline relation, ensure 'pipeline_stages' is set to avoid undefined array key error
        foreach ($deal_pipelines as &$rel) {
            // If pipeline info is present, get its stages
            if (isset($rel['pipeline']['id'])) {
                $rel['pipeline_stages'] = $this->deals_model->get_stages($rel['pipeline']['id']);
            } else {
                $rel['pipeline_stages'] = [];
            }
        }
        unset($rel); // break reference

        $data['deal_pipelines'] = $deal_pipelines;

        // For backward compatibility, also provide the stages for the main pipeline (first pipeline or deal's pipeline_id)
        $pipelineId = $deal['pipeline_id'] ?: $this->deals_model->get_default_pipeline_id();
        $data['stages'] = $this->deals_model->get_stages($pipelineId);

        $data['lost_reasons'] = $this->deals_model->get_lost_reasons();
        $data['title'] = $deal['name'];

        // Load counts for tabs
        $data['total_proposals'] = $this->get_deal_proposals_count($id);
        $data['total_tasks'] = $this->get_deal_tasks_count($id);
        $data['total_reminders'] = $this->get_deal_reminders_count($id);
        $data['total_notes'] = $this->get_deal_notes_count($id);
        $data['total_attachments'] = $this->get_deal_attachments_count($id);
        $data['total_activities'] = $this->get_deal_activities_count($id);
        $data['total_emails'] = $this->get_deal_emails_count($id);
        $data['total_calls'] = $this->get_deal_calls_count($id);

        // Load data for modals
        $data['deal_id'] = $id;
        $data['deal_name'] = $deal['name'];
        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        $data['priorities'] = [
            ['id' => 1, 'name' => _l('low')],
            ['id' => 2, 'name' => _l('medium')],
            ['id' => 3, 'name' => _l('high')],
            ['id' => 4, 'name' => _l('urgent')]
        ];
        $data['repeat_every'] = [
            ['id' => 1, 'name' => _l('day')],
            ['id' => 7, 'name' => _l('week')],
            ['id' => 30, 'name' => _l('month')],
            ['id' => 365, 'name' => _l('year')]
        ];

        $this->load->view('view', $data);
    }

    public function set_stage($id)
    {
        $stageId = (int) $this->input->post('stage_id');
        $pipelineId = (int) $this->input->post('pipeline_id');
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }
        if ($deal['status_final'] === 'won' || $deal['status_final'] === 'lost') {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Cannot change stage for won/lost deals']);
                return;
            }
            return redirect(admin_url('deals/view/' . $id));
        }

        // If pipeline_id is not provided, fallback to main deal pipeline
        if (!$pipelineId) {
            $pipelineId = $deal['pipeline_id'];
        }

        // Get old stage for this pipeline
        $old_stage_id = $this->deals_model->get_deal_pipeline_stage($id, $pipelineId);
        $old_stage = $this->deals_model->get_stage_name($old_stage_id);
        $new_stage = $this->deals_model->get_stage_name($stageId);

        // Update the stage in the respected pipeline only
        $this->deals_model->update_deal_pipeline_stage($id, $pipelineId, $stageId);

        // If the main pipeline is being updated, also update the main deal stage_id for backward compatibility
        if ($pipelineId == $deal['pipeline_id']) {
            $this->deals_model->update_deal($id, ['stage_id' => $stageId]);
        }

        // Log activity
        $activity_message = 'Stage changed from ' . $old_stage . ' to ' . $new_stage . ' in pipeline.';
        $this->deals_model->add_deal_activity($id, ['description' => $activity_message, 'type' => 'activities']);

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            redirect(admin_url('deals/view/' . $id));
        }
    }

    public function update_status($id)
    {
        $status = $this->input->post('status'); // open|won|lost
        $old_status = $this->deals_model->get_one($id)['status_final'];
        $this->deals_model->update_status($id, $status);

        // Log activity
        $this->deals_model->add_deal_activity($id, ['description' => 'Status changed from ' . $old_status . ' to ' . $status, 'type' => 'activities']);

        set_alert('success', _l('updated_successfully', _l('deal')));
        redirect($_SERVER['HTTP_REFERER']);
    }

    public function mark_won($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }

        $this->deals_model->mark_won($id);

        // Log activity
        $this->deals_model->add_deal_activity($id, ['description' => 'Deal marked as won', 'type' => 'activities']);

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    public function mark_lost($id)
    {
        $reasonId = $this->input->post('lost_reason_id') ? (int) $this->input->post('lost_reason_id') : null;
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }

        $reason_name = '';
        if ($reasonId) {
            $reason = $this->deals_model->get_lost_reason($reasonId);
            $reason_name = $reason ? $reason['name'] : '';
        }

        $this->deals_model->mark_lost($id, $reasonId);

        // Log activity
        $activity_message = 'Deal marked as lost';
        if ($reason_name) {
            $activity_message .= ' - Reason: ' . $reason_name;
        }
        $this->deals_model->add_deal_activity($id, ['description' => $activity_message, 'type' => 'activities']);

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    // Deal Contacts Methods
    public function add_contact_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }



        $contact_id = (int) $this->input->post('contact_id');
        if ($contact_id) {
            // Add existing contact to deal
            $this->deals_model->add_contact_to_deal($deal_id, $contact_id);
            $this->deals_model->add_deal_activity($deal_id, ['description' => 'Contact added to deal', 'type' => 'activities']);
            echo json_encode(['success' => true, 'message' => 'Contact added']);
        } else {
            // Create new contact and add to deal
            $contact_data = [
                'firstname' => $this->input->post('firstname'),
                'lastname' => $this->input->post('lastname'),
                'email' => $this->input->post('email'),
                'phonenumber' => $this->input->post('phonenumber'),
            ];

            $new_contact_id = $this->deals_model->create_module_contact($contact_data);

            if ($new_contact_id) {
                $this->deals_model->add_contact_to_deal($deal_id, $new_contact_id);
                $this->deals_model->add_deal_activity($deal_id, ['description' => 'New contact created and added to deal', 'type' => 'activities']);
                echo json_encode(['success' => true, 'message' => 'Contact created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
            }
        }
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_contact_from_deal($deal_id, $contact_id);
        $this->deals_model->add_deal_activity($deal_id, ['description' => 'Contact removed from deal', 'type' => 'activities']);
        echo json_encode(['success' => true, 'message' => 'Contact removed']);
    }

    public function get_deal_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_deal_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    public function get_available_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_available_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    // Deal Companies Methods
    public function add_company_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $company_id = (int) $this->input->post('company_id');
        if ($company_id) {
            // Add existing company to deal
            $this->deals_model->add_company_to_deal($deal_id, $company_id);
            $this->deals_model->add_deal_activity($deal_id, ['description' => 'Company added to deal', 'type' => 'activities']);
            echo json_encode(['success' => true, 'message' => 'Company added']);
        } else {
            // Create new company and add to deal
            $company_data = [
                'name' => $this->input->post('company_name'),
                'email' => $this->input->post('company_email'),
                'phone' => $this->input->post('company_phone'),
                'website' => $this->input->post('company_website'),
                'address' => $this->input->post('company_address')
            ];

            $new_company_id = $this->deals_model->create_company($company_data);

            if ($new_company_id) {
                $this->deals_model->add_company_to_deal($deal_id, $new_company_id);
                $this->deals_model->add_deal_activity($deal_id, ['description' => 'New company created and added to deal', 'type' => 'activities']);
                echo json_encode(['success' => true, 'message' => 'Company created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create company']);
            }
        }
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_company_from_deal($deal_id, $company_id);
        $this->deals_model->add_deal_activity($deal_id, ['description' => 'Company removed from deal', 'type' => 'activities']);
        echo json_encode(['success' => true, 'message' => 'Company removed']);
    }

    public function get_deal_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_deal_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function get_available_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_available_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function upload_attachment($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            show_404();
        }

        if ($this->input->post()) {
            $config['upload_path'] = './uploads/deals/';
            $config['allowed_types'] = 'gif|jpg|png|pdf|doc|docx';
            $config['max_size'] = 2048;

            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0755, true);
            }

            $this->load->library('upload', $config);

            if ($this->upload->do_upload('attachment')) {
                $upload_data = $this->upload->data();
                $attachment_data = [
                    'deal_id' => $id,
                    'file_name' => $upload_data['file_name'],
                    'description' => $this->input->post('description'),
                    'dateadded' => date('Y-m-d H:i:s')
                ];
                $this->deals_model->add_attachment($attachment_data);
                set_alert('success', _l('file_uploaded_successfully'));
            } else {
                set_alert('danger', $this->upload->display_errors());
            }
        }
        redirect(admin_url('deals/view/' . $id));
    }

    public function add_activity($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $payload = $this->input->post();
        $id = $this->deals_model->add_deal_activity($deal_id, $payload);
        echo json_encode(['success' => (bool) $id, 'id' => $id]);
    }

    // Deal Attachments Methods
    public function add_attachment_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Upload new attachment and add to deal
        $config['upload_path'] = './uploads/deals/';
        $config['allowed_types'] = 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx|txt';
        $config['max_size'] = 2048;

        if (!is_dir($config['upload_path'])) {
            mkdir($config['upload_path'], 0755, true);
        }

        $this->load->library('upload', $config);

        if ($this->upload->do_upload('attachment')) {
            $upload_data = $this->upload->data();
            $attachment_data = [
                'file_name' => $upload_data['file_name'],
                'filetype' => $upload_data['file_type'],
                'description' => $this->input->post('description'),
                'staffid' => get_staff_user_id(),
                'dateadded' => date('Y-m-d H:i:s')
            ];

            $new_attachment_id = $this->deals_model->add_attachment($attachment_data);

            if ($new_attachment_id) {
                $this->deals_model->add_attachment_to_deal($deal_id, $new_attachment_id);
                $this->deals_model->add_deal_activity($deal_id, ['description' => 'New attachment uploaded and added to deal', 'type' => 'documents']);
                echo json_encode(['success' => true, 'message' => 'Attachment added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to save attachment']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => $this->upload->display_errors()]);
        }
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_attachment_from_deal($deal_id, $attachment_id);
        $this->deals_model->add_deal_activity($deal_id, ['description' => 'Attachment removed from deal', 'type' => 'documents']);
        echo json_encode(['success' => true, 'message' => 'Attachment removed']);
    }


    public function get_available_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $attachments = $this->deals_model->get_available_attachments($deal_id);
        echo json_encode(['success' => true, 'attachments' => $attachments]);
    }

    public function download_attachment($attachment_id)
    {
        $attachment = $this->deals_model->get_attachment($attachment_id);
        if (!$attachment) {
            show_404();
        }
        $file_path = FCPATH . 'uploads/deals/' . $attachment['file_name'];
        if (!file_exists($file_path)) {
            show_404();
        }
        $this->load->helper('download');
        force_download($file_path, null);
    }

    public function update_activity_status($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $status = $this->input->post('status');
        $success = $this->deals_model->update_activity_status($id, $status);
        echo json_encode(['success' => $success]);
    }

    public function get_staff_ajax()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        echo json_encode(['success' => true, 'staff' => $this->deals_model->get_staff()]);
    }
    public function add_note($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $description = trim((string) $this->input->post('description'));
        if ($description === '') {
            echo json_encode(['success' => false, 'message' => 'Description required']);
            return;
        }
        $id = $this->deals_model->add_note($deal_id, $description);
        echo json_encode(['success' => (bool) $id, 'id' => $id]);
    }

    public function get_notes($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $notes = $this->deals_model->get_notes($deal_id);
        echo json_encode(['success' => true, 'notes' => $notes]);
    }

    public function add_call($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $payload = $this->input->post();
        $id = $this->deals_model->add_call($deal_id, $payload);
        echo json_encode(['success' => (bool) $id, 'id' => $id]);
    }
    public function get_calls($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $calls = $this->deals_model->get_calls($deal_id);
        echo json_encode(['success' => true, 'calls' => $calls]);
    }

    public function test_db($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Test creating a contact
        $contact_data = [
            'firstname' => 'Test',
            'lastname' => 'Contact',
            'email' => 'test' . time() . '@example.com',
            'phonenumber' => '+1234567890'
        ];

        $contact_id = $this->deals_model->create_module_contact($contact_data);

        if ($contact_id) {
            // Test linking contact to deal
            $link_id = $this->deals_model->add_contact_to_deal($deal_id, $contact_id);

            if ($link_id) {
                // Test activity logging
                $activity_id = $this->deals_model->add_deal_activity($deal_id, ['description' => 'Test contact added', 'type' => 'activities']);

                echo json_encode([
                    'success' => true,
                    'message' => 'Database test successful',
                    'contact_id' => $contact_id,
                    'link_id' => $link_id,
                    'activity_id' => $activity_id
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to link contact to deal']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
        }
    }

    // Get sources for dropdown
    public function get_sources()
    {
        return $this->db->get(db_prefix() . 'deals_sources')->result_array();
    }

    // Get statuses for dropdown
    public function get_statuses()
    {
        return $this->db->get(db_prefix() . 'deals_statuses')->result_array();
    }

    // Get staff for assigned dropdown
    public function get_staff()
    {
        return $this->db->where('active', 1)->get(db_prefix() . 'staff')->result_array();
    }

    // New methods for tabs functionality
    public function get_deal_proposals_count($deal_id)
    {
        $this->db->where('rel_type', 'deal');
        $this->db->where('rel_id', $deal_id);
        return $this->db->count_all_results(db_prefix() . 'proposals');
    }

    public function get_deal_tasks_count($deal_id)
    {
        $this->db->where('rel_type', 'deal');
        $this->db->where('rel_id', $deal_id);
        return $this->db->count_all_results(db_prefix() . 'tasks');
    }

    public function get_deal_reminders_count($deal_id)
    {
        $this->db->where('deal_id', $deal_id);
        return $this->db->count_all_results(db_prefix() . 'deals_reminders');
    }

    public function get_deal_notes_count($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        return $this->db->count_all_results(db_prefix() . 'deals_notes');
    }

    public function get_deal_attachments_count($deal_id)
    {
        $this->db->where('deal_id', $deal_id);
        return $this->db->count_all_results(db_prefix() . 'deals_attachments');
    }

    public function get_deal_activities_count($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        $this->db->where('type', 'activities');
        return $this->db->count_all_results(db_prefix() . 'deals_activity');
    }

    public function get_deal_emails_count($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        $this->db->where('type', 'emails');
        return $this->db->count_all_results(db_prefix() . 'deals_activity');
    }

    public function get_deal_calls_count($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        $this->db->where('type', 'calls');
        return $this->db->count_all_results(db_prefix() . 'deals_activity');
    }

    // AJAX methods for tab content
    public function get_deal_proposals($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_type', 'deal');
        $this->db->where('rel_id', $deal_id);
        $proposals = $this->db->get(db_prefix() . 'proposals')->result_array();

        echo json_encode($proposals);
    }

    public function get_deal_tasks($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_type', 'deal');
        $this->db->where('rel_id', $deal_id);
        $tasks = $this->db->get(db_prefix() . 'tasks')->result_array();

        echo json_encode($tasks);
    }

    public function get_deal_reminders($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('deal_id', $deal_id);
        $reminders = $this->db->get(db_prefix() . 'deals_reminders')->result_array();

        echo json_encode($reminders);
    }

    public function get_deal_notes($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_id', $deal_id);
        $notes = $this->db->get(db_prefix() . 'deals_notes')->result_array();

        echo json_encode($notes);
    }

    public function get_deal_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('deal_id', $deal_id);
        $attachments = $this->db->get(db_prefix() . 'deals_attachments')->result_array();

        echo json_encode($attachments);
    }

    public function get_deal_activities($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_id', $deal_id);
        $activities = $this->db->get(db_prefix() . 'deals_activity')->result_array();

        echo json_encode($activities);
    }

    public function get_deal_emails($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_id', $deal_id);
        $this->db->where('type', 'emails');
        $emails = $this->db->get(db_prefix() . 'deals_activity')->result_array();

        echo json_encode($emails);
    }

    public function get_deal_calls($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $this->db->where('rel_id', $deal_id);
        $this->db->where('type', 'calls');
        $calls = $this->db->get(db_prefix() . 'deals_activity')->result_array();

        echo json_encode($calls);
    }

    // Add new items methods
    public function add_deal_note($deal_id)
    {
        // Handle both AJAX and regular form submissions
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Deal not found']);
                return;
            }
            show_404();
        }

        $data = $this->input->post();
        $data['rel_id'] = $deal_id;
        $data['dateadded'] = date('Y-m-d H:i:s');
        $data['addedfrom'] = get_staff_user_id();

        $this->db->insert(db_prefix() . 'deals_notes', $data);

        // Log activity
        $this->log_deal_activity($deal_id, 'not_deal_activity_note_added', false, serialize([
            get_staff_full_name(get_staff_user_id()),
            substr($data['description'], 0, 50) . '...'
        ]));

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'message' => _l('note_added_successfully')]);
        } else {
            set_alert('success', _l('note_added_successfully'));
            redirect(admin_url('deals/view/' . $deal_id));
        }
    }


    public function add_deal_call($deal_id)
    {
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $data = $this->input->post();
        $data['deal_id'] = $deal_id;
        $data['staffid'] = get_staff_user_id();
        $data['dateadded'] = date('Y-m-d H:i:s');
        $data['call_date'] = !empty($data['call_date']) ? $data['call_date'] : date('Y-m-d H:i:s');

        $this->db->insert(db_prefix() . 'deals_calls', $data);

        // Log activity
        $this->deals_model->add_deal_activity($deal_id, [
            'description' => 'Call logged: ' . $data['call_type'] . ' - ' . $data['outcome'],
            'type' => 'calls',
            'additional_data' => json_encode($data)
        ]);

        echo json_encode(['success' => true, 'message' => 'Call logged successfully!']);
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'message' => 'Call logged successfully!']);
        } else {
            set_alert('success', 'Call logged successfully!');
            redirect(admin_url('deals/view/' . $deal_id));
        }
    }

    public function add_deal_email($deal_id)
    {
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $staff_id = get_staff_user_id();
        $subject = $this->input->post('subject', true);
        $message = $this->input->post('message', false);
        $to_email = $this->input->post('to_email', true);

        // Compose data array as expected by Mailbox_model->add
        $mail_data = [
            'to'         => $to_email,
            'cc'         => '', // Extend if needed
            'subject'    => $subject,
            'body'       => $message,
            'tagid'      => null,
            'templateid' => null,
            'scheduled_at' => null,
        ];

        // Load the mailbox model
        $this->load->model('mailbox/mailbox_model');

        // Send and store email using mailbox model
        $result = $this->mailbox_model->add($mail_data, $staff_id);

        // Store the email in deals_emails table as well
        $deal_email_data = [
            'deal_id'    => $deal_id,
            'from_email' => get_staff_email_by_id($staff_id),
            'from_name'  => get_staff_full_name($staff_id),
            'to_email'   => $to_email,
            'subject'    => $subject,
            'message'    => $message,
            'sent'       => 1,
            'sent_date'  => date('Y-m-d H:i:s'),
            'dateadded'  => date('Y-m-d H:i:s'),
            'staffid'    => $staff_id,
        ];

        // Handle attachments: store file names if any were uploaded
        if (!empty($_FILES['attachments']['name'][0])) {
            $uploaded_files = [];
            foreach ($_FILES['attachments']['name'] as $i => $filename) {
                $_FILES['single_attachment']['name'] = $_FILES['attachments']['name'][$i];
                $_FILES['single_attachment']['type'] = $_FILES['attachments']['type'][$i];
                $_FILES['single_attachment']['tmp_name'] = $_FILES['attachments']['tmp_name'][$i];
                $_FILES['single_attachment']['error'] = $_FILES['attachments']['error'][$i];
                $_FILES['single_attachment']['size'] = $_FILES['attachments']['size'][$i];
                $config['upload_path'] = './uploads/deal_emails/';
                $config['allowed_types'] = get_option('allowed_files');
                $config['max_size'] = get_option('max_file_size') * 1024; // MB to KB
                if (!is_dir($config['upload_path'])) {
                    mkdir($config['upload_path'], 0755, true);
                }
                $this->load->library('upload', $config);
                if ($this->upload->do_upload('single_attachment')) {
                    $uploaded_data = $this->upload->data();
                    $uploaded_files[] = $uploaded_data['file_name'];
                }
            }
            $deal_email_data['attachments'] = json_encode($uploaded_files);
        }

        $this->db->insert(db_prefix() . 'deals_emails', $deal_email_data);

        // Log activity in deals
        $this->deals_model->add_deal_activity($deal_id, [
            'description' => 'Email sent: ' . $subject,
            'type' => 'emails',
            'additional_data' => json_encode([
                'to' => $to_email,
                'subject' => $subject,
                'body' => $message
            ])
        ]);

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'message' => _l('Email sent successfully!')]);
        } else {
            set_alert('success', _l('Email sent successfully!'));
            redirect(admin_url('deals/view/' . $deal_id));
        }

        echo json_encode(['success' => true, 'message' => 'Email sent successfully!']);
    }

    public function add_deal_reminder($deal_id)
    {
        // Handle both AJAX and regular form submissions
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Deal not found']);
                return;
            }
            show_404();
        }

        $data = $this->input->post();
        $data['deal_id'] = $deal_id;
        $data['date'] = date('Y-m-d H:i:s');
        $data['creator'] = get_staff_user_id();
        $data['datecreated'] = date('Y-m-d H:i:s');

        $this->db->insert(db_prefix() . 'deals_reminders', $data);

        // Log activity
        $this->log_deal_activity($deal_id, 'not_deal_activity_reminder_added', false, serialize([
            get_staff_full_name(get_staff_user_id()),
            _dt($data['date'])
        ]));

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'message' => _l('reminder_added_successfully')]);
        } else {
            set_alert('success', _l('reminder_added_successfully'));
            redirect(admin_url('deals/view/' . $deal_id));
        }
    }

    public function add_deal_activity($deal_id)
    {
        // Handle both AJAX and regular form submissions
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Deal not found']);
                return;
            }
            show_404();
        }

        $data = $this->input->post();
        $data['rel_id'] = $deal_id;
        $data['date'] = date('Y-m-d H:i:s');
        $data['staffid'] = get_staff_user_id();
        $data['full_name'] = get_staff_full_name(get_staff_user_id());
        $data['full_name'] = get_staff_full_name(get_staff_user_id());
        $data['custom_activity'] = 1;

        $this->db->insert(db_prefix() . 'deals_activity', $data);

        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'message' => _l('activity_added_successfully')]);
        } else {
            set_alert('success', _l('activity_added_successfully'));
            redirect(admin_url('deals/view/' . $deal_id));
        }
    }

    public function add_task($deal_id, $id = '')
    {
        // Handle both AJAX and regular form submissions
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }
    
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Deal not found']);
                return;
            }
            show_404();
        }
    
        if ($id !== '') {
            // For edit, ensure the task belongs to this deal
            $task = $this->tasks_model->get($id);
            if (!$task || $task->rel_type !== 'deal' || $task->rel_id != $deal_id) {
                if ($this->input->is_ajax_request()) {
                    echo json_encode(['success' => false, 'message' => 'Task not found or not associated with this deal']);
                    return;
                }
                show_404();
            }
        }
    
        if (staff_cant('edit', 'tasks') && staff_cant('create', 'tasks')) {
            ajax_access_denied();
        }
    
        $data = [];
        // For new task add directly from the projects milestones (adapted, but deals may not have milestones)
        if ($this->input->get('milestone_id')) {
            $this->db->where('id', $this->input->get('milestone_id'));
            $milestone = $this->db->get(db_prefix() . 'milestones')->row();
            if ($milestone) {
                $data['_milestone_selected_data'] = [
                    'id'       => $milestone->id,
                    'due_date' => _d($milestone->due_date),
                ];
            }
        }
        if ($this->input->get('start_date')) {
            $data['start_date'] = $this->input->get('start_date');
        }
        if ($this->input->post()) {
            $data                = $this->input->post();
            $data['description'] = html_purify($this->input->post('description', false));
    
            // Force rel_type and rel_id for deals when adding new
            if ($id == '') {
                $data['rel_type'] = 'deal';
                $data['rel_id'] = $deal_id;
                $data['dateadded'] = date('Y-m-d H:i:s');
                $data['addedfrom'] = get_staff_user_id();
                $data['is_added_from_contact'] = 0;
            }
    
            if ($id == '') {
                if (staff_cant('create', 'tasks')) {
                    header('HTTP/1.0 400 Bad error');
                    echo json_encode([
                        'success' => false,
                        'message' => _l('access_denied'),
                    ]);
                    die;
                }
                $new_task_id = $this->tasks_model->add($data);
                $_id     = false;
                $success = false;
                $message = '';
                if ($new_task_id) {
                    $success       = true;
                    $_id           = $new_task_id;
                    $message       = _l('added_successfully', _l('task'));
                    $uploadedFiles = handle_task_attachments_array($new_task_id);
                    if ($uploadedFiles && is_array($uploadedFiles)) {
                        foreach ($uploadedFiles as $file) {
                            $this->misc_model->add_attachment_to_database($new_task_id, 'task', [$file]);
                        }
                    }
    
                    // Log activity for new task
                    $this->log_deal_activity($deal_id, 'not_deal_activity_task_created', false, serialize([
                        get_staff_full_name(get_staff_user_id()),
                        $data['name']
                    ]));
                }
                if ($this->input->is_ajax_request()) {
                    echo json_encode([
                        'success' => $success,
                        'id'      => $_id,
                        'message' => $message,
                    ]);
                } else {
                    if ($success) {
                        set_alert('success', $message);
                    } else {
                        set_alert('warning', _l('task_add_failed'));
                    }
                    redirect(admin_url('deals/view/' . $deal_id));
                }
                die;
            } else {
                if (staff_cant('edit', 'tasks')) {
                    header('HTTP/1.0 400 Bad error');
                    echo json_encode([
                        'success' => false,
                        'message' => _l('access_denied'),
                    ]);
                    die;
                }
                $success = $this->tasks_model->update($data, $id);
                $message = '';
                if ($success) {
                    $message = _l('updated_successfully', _l('task'));
                    // Handle new attachments for update
                    $uploadedFiles = handle_task_attachments_array($id);
                    if ($uploadedFiles && is_array($uploadedFiles)) {
                        foreach ($uploadedFiles as $file) {
                            $this->misc_model->add_attachment_to_database($id, 'task', [$file]);
                        }
                    }
                    // Optionally log update activity if needed
                    // $this->log_deal_activity($deal_id, 'not_deal_activity_task_updated', false, serialize([...]));
                }
                if ($this->input->is_ajax_request()) {
                    echo json_encode([
                        'success' => $success,
                        'message' => $message,
                        'id'      => $id,
                    ]);
                } else {
                    if ($success) {
                        set_alert('success', $message);
                    } else {
                        set_alert('warning', _l('task_update_failed'));
                    }
                    redirect(admin_url('deals/view/' . $deal_id));
                }
                die;
            }
        } else {
            show_404();
        }
    }


    function handle_deal_attachments($deal_id, $index_name = 'attachment', $form_activity = false)
    {
        $hookData = hooks()->apply_filters('before_handle_deal_attachment', [
            'deal_id' => $deal_id,
            'index_name' => $index_name,
            'handled_externally' => false,
            'form_activity' => $form_activity,
            'handled_externally_successfully' => false,
            'files' => $_FILES
        ]);
    
        if ($hookData['handled_externally']) {
            return $hookData['handled_externally_successfully'];
        }
    
        $path = get_upload_path_by_type('deal') . $deal_id . '/';
        $CI = &get_instance();
        $CI->load->model('deals_model');
    
        if (isset($_FILES[$index_name]['name']) 
            && ($_FILES[$index_name]['name'] != '' 
                || is_array($_FILES[$index_name]['name']) && count($_FILES[$index_name]['name']) > 0)) {
            if (!is_array($_FILES[$index_name]['name'])) {
                $_FILES[$index_name]['name'] = [$_FILES[$index_name]['name']];
                $_FILES[$index_name]['type'] = [$_FILES[$index_name]['type']];
                $_FILES[$index_name]['tmp_name'] = [$_FILES[$index_name]['tmp_name']];
                $_FILES[$index_name]['error'] = [$_FILES[$index_name]['error']];
                $_FILES[$index_name]['size'] = [$_FILES[$index_name]['size']];
            }
    
            _file_attachments_index_fix($index_name);
    
            for ($i = 0; $i < count($_FILES[$index_name]['name']); $i++) {
                $tmpFilePath = $_FILES[$index_name]['tmp_name'][$i];
    
                if (!empty($tmpFilePath) && $tmpFilePath != '') {
                    if (_perfex_upload_error($_FILES[$index_name]['error'][$i]) 
                        || !_upload_extension_allowed($_FILES[$index_name]['name'][$i])) {
                        continue;
                    }
    
                    _maybe_create_upload_path($path);
                    $filename = unique_filename($path, $_FILES[$index_name]['name'][$i]);
                    $newFilePath = $path . $filename;
    
                    if (move_uploaded_file($tmpFilePath, $newFilePath)) {
                        $attachment = [
                            'file_name' => $filename,
                            'filetype' => $_FILES[$index_name]['type'][$i],
                            'original_name' => $_FILES[$index_name]['name'][$i],
                            'file_size' => $_FILES[$index_name]['size'][$i] / 1024, // Convert to KB
                            'dateadded' => date('Y-m-d H:i:s'),
                            'staffid' => get_staff_user_id()
                        ];
    
                        if ($CI->input->post('description')) {
                            $attachment['description'] = $CI->input->post('description');
                        }
    
                        if ($CI->input->post('external')) {
                            $attachment['external'] = $CI->input->post('external');
                        }
    
                        $CI->deals_model->add_attachment_to_database($deal_id, [$attachment], false, $form_activity);
                    }
                }
            }
        }
    
        return true;
    }
    public function add_attachment($deal_id)
    {
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Deal not found']);
                return;
            }
            show_404();
        }

        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $lastFile = $this->input->post('last_file');
        $success = $this->handle_deal_attachments($deal_id);

        if ($this->input->is_ajax_request()) {
            echo json_encode([
                'success' => $success,
                'message' => $success ? _l('attachment_added_successfully') : _l('attachment_add_failed'),
                'dealView' => $lastFile ? $this->deals_model->get_one($deal_id) : [],
                'id' => $deal_id
            ]);
        } else {
            if ($success) {
                set_alert('success', _l('attachment_added_successfully'));
            } else {
                set_alert('danger', _l('attachment_add_failed'));
            }
            redirect(admin_url('deals/view/' . $deal_id));
        }
    }



    // public function add_attachment($deal_id)
    // {
    //     // Handle both AJAX and regular form submissions
    //     if (!$this->input->is_ajax_request() && !$this->input->post()) {
    //         show_404();
    //     }

    //     $deal = $this->deals_model->get_one($deal_id);
    //     if (!$deal) {
    //         if ($this->input->is_ajax_request()) {
    //             echo json_encode(['success' => false, 'message' => 'Deal not found']);
    //             return;
    //         }
    //         show_404();
    //     }

    //     $this->load->library('upload');

    //     $config['upload_path'] = get_upload_path_by_type('deal') . $deal_id . '/';
    //     $config['allowed_types'] = get_option('allowed_files');
    //     $config['max_size'] = (int)get_option('max_file_size') * 1024;
    //     $config['encrypt_name'] = true;

    //     if (!file_exists($config['upload_path'])) {
    //         mkdir($config['upload_path'], 0755, true);
    //     }

    //     $this->upload->initialize($config);

    //     if ($this->upload->do_upload('attachment')) {
    //         $upload_data = $this->upload->data();

    //         $data = [
    //             'deal_id' => $deal_id,
    //             'file_name' => $upload_data['file_name'],
    //             'original_name' => $upload_data['orig_name'],
    //             'file_size' => $upload_data['file_size'],
    //             'file_type' => $upload_data['file_type'],
    //             'dateadded' => date('Y-m-d H:i:s'),
    //             'staffid' => get_staff_user_id()
    //         ];

    //         if ($this->input->post('description')) {
    //             $data['description'] = $this->input->post('description');
    //         }

    //         if ($this->input->post('external')) {
    //             $data['external'] = $this->input->post('external');
    //         }

    //         $this->db->insert(db_prefix() . 'deals_attachments', $data);

    //         // Log activity
    //         $this->log_deal_activity($deal_id, 'not_deal_activity_attachment_added', false, serialize([
    //             get_staff_full_name(get_staff_user_id()),
    //             $upload_data['orig_name']
    //         ]));

    //         if ($this->input->is_ajax_request()) {
    //             echo json_encode(['success' => true, 'message' => _l('attachment_added_successfully')]);
    //         } else {
    //             set_alert('success', _l('attachment_added_successfully'));
    //             redirect(admin_url('deals/view/' . $deal_id));
    //         }
    //     } else {
    //         if ($this->input->is_ajax_request()) {
    //             echo json_encode(['success' => false, 'message' => $this->upload->display_errors()]);
    //         } else {
    //             set_alert('danger', $this->upload->display_errors());
    //             redirect(admin_url('deals/view/' . $deal_id));
    //         }
    //     }
    // }

    // Helper method to log deal activities
    private function log_deal_activity($deal_id, $description_key, $additional_data = false, $additional_data_serialized = false)
    {
        $data = [
            'rel_id' => $deal_id,
            'description' => $description_key,
            'additional_data' => $additional_data_serialized,
            'staffid' => get_staff_user_id(),
            'full_name' => get_staff_full_name(get_staff_user_id()),
            'date' => date('Y-m-d H:i:s'),
            'type' => 'activity',
            'custom_activity' => 0
        ];

        $this->db->insert(db_prefix() . 'deals_activity', $data);
    }

    // Sources Management
    public function sources()
    {
        if (!is_admin()) {
            access_denied('Deals Sources');
        }
        $data['sources'] = $this->deals_model->get_source();
        $data['title'] = 'Deals sources';
        $this->load->view('manage_sources', $data);
    }

    public function source()
    {
        if (!is_admin() && get_option('staff_members_create_inline_deal_source') == '0') {
            access_denied('Deals Sources');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }

                $id = $this->deals_model->add_source($data);

                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('deal_source')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->deals_model->update_source($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('deal_source')));
                }
            }
        }
    }

    public function delete_source($id)
    {
        if (!is_admin()) {
            access_denied('Delete Deal Source');
        }
        if (!$id) {
            redirect(admin_url('deals/sources'));
        }
        $response = $this->deals_model->delete_source($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('deal_source_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('deal_source')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('deal_source_lowercase')));
        }
        redirect(admin_url('deals/sources'));
    }

    // Statuses Management
    public function statuses()
    {
        if (!is_admin()) {
            access_denied('Deals Statuses');
        }
        $data['statuses'] = $this->deals_model->get_status();
        $data['title'] = 'Deals statuses';
        $this->load->view('manage_statuses', $data);
    }

    public function status()
    {
        if (!is_admin() && get_option('staff_members_create_inline_deal_status') == '0') {
            access_denied('Deals Statuses');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }
                $id = $this->deals_model->add_status($data);
                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('deal_status')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->deals_model->update_status($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('deal_status')));
                }
            }
        }
    }

    public function delete_status($id)
    {
        if (!is_admin()) {
            access_denied('Deals Statuses');
        }
        if (!$id) {
            redirect(admin_url('deals/statuses'));
        }
        $response = $this->deals_model->delete_status($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('deal_status_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('deal_status')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('deal_status_lowercase')));
        }
        redirect(admin_url('deals/statuses'));
    }

    // Convert to Customer
    public function get_convert_data($id)
    {
        if (!is_staff_member() || !$this->deals_model->staff_can_access_deal($id)) {
            ajax_access_denied();
        }
        if (is_gdpr() && get_option('gdpr_enable_consent_for_contacts') == '1') {
            $this->load->model('gdpr_model');
            $data['purposes'] = $this->gdpr_model->get_consent_purposes($id, 'deal');
        }
        $data['deal'] = $this->deals_model->get_one($id);
        $this->load->view('convert_to_customer', $data);
    }

    public function convert_to_customer()
    {
        if (!is_staff_member()) {
            access_denied('Deal Convert to Customer');
        }

        if ($this->input->post()) {
            $default_country = get_option('customer_default_country');
            $data = $this->input->post();
            $data['password'] = $this->input->post('password', false);

            $original_deal_email = $data['original_deal_email'];
            unset($data['original_deal_email']);

            if (isset($data['transfer_notes'])) {
                $notes = $this->misc_model->get_notes($data['dealid'], 'deal');
                unset($data['transfer_notes']);
            }

            if (isset($data['transfer_consent'])) {
                $this->load->model('gdpr_model');
                $consents = $this->gdpr_model->get_consents(['deal_id' => $data['dealid']]);
                unset($data['transfer_consent']);
            }

            if (isset($data['merge_db_fields'])) {
                $merge_db_fields = $data['merge_db_fields'];
                unset($data['merge_db_fields']);
            }

            if (isset($data['merge_db_contact_fields'])) {
                $merge_db_contact_fields = $data['merge_db_contact_fields'];
                unset($data['merge_db_contact_fields']);
            }

            if (isset($data['include_deals_custom_fields'])) {
                $include_deals_custom_fields = $data['include_deals_custom_fields'];
                unset($data['include_deals_custom_fields']);
            }

            if ($data['country'] == '' && $default_country != '') {
                $data['country'] = $default_country;
            }

            $data['billing_street'] = $data['address'];
            $data['billing_city'] = $data['city'];
            $data['billing_state'] = $data['state'];
            $data['billing_zip'] = $data['zip'];
            $data['billing_country'] = $data['country'];

            $data['is_primary'] = 1;
            $id = $this->clients_model->add($data, true);
            if ($id) {
                $primary_contact_id = get_primary_contact_user_id($id);

                if (isset($notes)) {
                    foreach ($notes as $note) {
                        $this->db->insert(db_prefix() . 'notes', [
                            'rel_id' => $id,
                            'rel_type' => 'customer',
                            'dateadded' => $note['dateadded'],
                            'addedfrom' => $note['addedfrom'],
                            'description' => $note['description'],
                            'date_contacted' => $note['date_contacted'],
                        ]);
                    }
                }
                if (isset($consents)) {
                    foreach ($consents as $consent) {
                        unset($consent['id']);
                        unset($consent['purpose_name']);
                        $consent['deal_id'] = 0;
                        $consent['contact_id'] = $primary_contact_id;
                        $this->gdpr_model->add_consent($consent);
                    }
                }
                if (staff_cant('view', 'customers') && get_option('auto_assign_customer_admin_after_deal_convert') == 1) {
                    $this->db->insert(db_prefix() . 'customer_admins', [
                        'date_assigned' => date('Y-m-d H:i:s'),
                        'customer_id' => $id,
                        'staff_id' => get_staff_user_id(),
                    ]);
                }
                $this->log_deal_activity($data['dealid'], 'not_deal_activity_converted', false, serialize([
                    get_staff_full_name(),
                ]));
                $default_status = $this->deals_model->get_status('', [
                    'isdefault' => 1,
                ]);
                $this->db->where('id', $data['dealid']);
                $this->db->update(db_prefix() . 'deals', [
                    'date_converted' => date('Y-m-d H:i:s'),
                    'status' => $default_status[0]['id'],
                ]);
                // Check if deal email is different then client email
                $contact = $this->clients_model->get_contact(get_primary_contact_user_id($id));
                if ($contact->email != $original_deal_email) {
                    if ($original_deal_email != '') {
                        $this->log_deal_activity($data['dealid'], 'not_deal_activity_converted_email', false, serialize([
                            $original_deal_email,
                            $contact->email,
                        ]));
                    }
                }
                if (isset($include_deals_custom_fields)) {
                    foreach ($include_deals_custom_fields as $fieldid => $value) {
                        // checked don't merge
                        if ($value == 5) {
                            continue;
                        }
                        // get the value of this deals custom fiel
                        $this->db->where('relid', $data['dealid']);
                        $this->db->where('fieldto', 'deals');
                        $this->db->where('fieldid', $fieldid);
                        $deal_custom_field_value = $this->db->get(db_prefix() . 'customfieldsvalues')->row()->value;
                        // Is custom field for contact ot customer
                        if ($value == 1 || $value == 4) {
                            if ($value == 4) {
                                $field_to = 'contacts';
                            } else {
                                $field_to = 'customers';
                            }
                            $this->db->where('id', $fieldid);
                            $field = $this->db->get(db_prefix() . 'customfields')->row();
                            // check if this field exists for custom fields
                            $this->db->where('fieldto', $field_to);
                            $this->db->where('name', $field->name);
                            $exists = $this->db->get(db_prefix() . 'customfields')->row();
                            $copy_custom_field_id = null;
                            if ($exists) {
                                $copy_custom_field_id = $exists->id;
                            } else {
                                // there is no name with the same custom field for deals at the custom side create the custom field now
                                $this->db->insert(db_prefix() . 'customfields', [
                                    'fieldto' => $field_to,
                                    'name' => $field->name,
                                    'required' => $field->required,
                                    'type' => $field->type,
                                    'options' => $field->options,
                                    'display_inline' => $field->display_inline,
                                    'field_order' => $field->field_order,
                                    'slug' => slug_it($field_to . '_' . $field->name, [
                                        'separator' => '_',
                                    ]),
                                    'active' => $field->active,
                                    'only_admin' => $field->only_admin,
                                    'show_on_table' => $field->show_on_table,
                                    'bs_column' => $field->bs_column,
                                ]);
                                $new_customer_field_id = $this->db->insert_id();
                                if ($new_customer_field_id) {
                                    $copy_custom_field_id = $new_customer_field_id;
                                }
                            }
                            if ($copy_custom_field_id != null) {
                                $insert_to_custom_field_id = $id;
                                if ($value == 4) {
                                    $insert_to_custom_field_id = get_primary_contact_user_id($id);
                                }
                                $this->db->insert(db_prefix() . 'customfieldsvalues', [
                                    'relid' => $insert_to_custom_field_id,
                                    'fieldid' => $copy_custom_field_id,
                                    'fieldto' => $field_to,
                                    'value' => $deal_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 2) {
                            if (isset($merge_db_fields)) {
                                $db_field = $merge_db_fields[$fieldid];
                                // in case user don't select anything from the db fields
                                if ($db_field == '') {
                                    continue;
                                }
                                if ($db_field == 'country' || $db_field == 'shipping_country' || $db_field == 'billing_country') {
                                    $this->db->where('iso2', $deal_custom_field_value);
                                    $this->db->or_where('short_name', $deal_custom_field_value);
                                    $this->db->or_like('long_name', $deal_custom_field_value);
                                    $country = $this->db->get(db_prefix() . 'countries')->row();
                                    if ($country) {
                                        $deal_custom_field_value = $country->country_id;
                                    } else {
                                        $deal_custom_field_value = 0;
                                    }
                                }
                                $this->db->where('userid', $id);
                                $this->db->update(db_prefix() . 'clients', [
                                    $db_field => $deal_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 3) {
                            if (isset($merge_db_contact_fields)) {
                                $db_field = $merge_db_contact_fields[$fieldid];
                                if ($db_field == '') {
                                    continue;
                                }
                                $this->db->where('id', $primary_contact_id);
                                $this->db->update(db_prefix() . 'contacts', [
                                    $db_field => $deal_custom_field_value,
                                ]);
                            }
                        }
                    }
                }
                // set the deal to status client in case is not status client
                $this->db->where('isdefault', 1);
                $status_client_id = $this->db->get(db_prefix() . 'deals_status')->row()->id;
                $this->db->where('id', $data['dealid']);
                $this->db->update(db_prefix() . 'deals', [
                    'status' => $status_client_id,
                ]);

                set_alert('success', _l('deal_to_client_base_converted_success'));

                if (is_gdpr() && get_option('gdpr_after_deal_converted_delete') == '1') {
                    // When deal is deleted
                    // move all proposals to the actual customer record
                    $this->db->where('rel_id', $data['dealid']);
                    $this->db->where('rel_type', 'deal');
                    $this->db->update('proposals', [
                        'rel_id' => $id,
                        'rel_type' => 'customer',
                    ]);

                    $this->deals_model->delete($data['dealid']);

                    $this->db->where('userid', $id);
                    $this->db->update(db_prefix() . 'clients', ['dealid' => null]);
                }

                log_activity('Created Deal Client Profile [DealID: ' . $data['dealid'] . ', ClientID: ' . $id . ']');
                hooks()->do_action('deal_converted_to_customer', ['deal_id' => $data['dealid'], 'customer_id' => $id]);
                redirect(admin_url('clients/client/' . $id));
            }
        }
    }

    // Import functionality
    public function import()
    {
        if (!is_admin() && get_option('allow_non_admin_members_to_import_deals') != '1') {
            access_denied('Deals Import');
        }

        $dbFields = $this->db->list_fields(db_prefix() . 'deals');
        array_push($dbFields, 'tags');

        $this->load->library('import/import_deals', [], 'import');
        $this->import->setDatabaseFields($dbFields)
            ->setCustomFields(get_custom_fields('deals'));

        if ($this->input->post('download_sample') === 'true') {
            $this->import->downloadSample();
        }

        if (
            $this->input->post()
            && isset($_FILES['file_csv']['name']) && $_FILES['file_csv']['name'] != ''
        ) {
            $this->import->setSimulation($this->input->post('simulate'))
                ->setTemporaryFileLocation($_FILES['file_csv']['tmp_name'])
                ->setFilename($_FILES['file_csv']['name'])
                ->perform();

            $data['total_rows_post'] = $this->import->totalRows();

            if (!$this->import->isSimulation()) {
                set_alert('success', _l('import_total_imported', $this->import->totalImported()));
            }
        }

        $data['statuses'] = $this->deals_model->get_status();
        $data['sources'] = $this->deals_model->get_source();
        $data['members'] = $this->staff_model->get('', ['is_not_staff' => 0, 'active' => 1]);

        $data['title'] = _l('import');
        $this->load->view('import', $data);
    }
}