<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals_model extends App_Model
{

    public function get_column_order()
    {
        $this->db->select('column_order');
        $this->db->where('staff_id', get_staff_user_id());
        $result = $this->db->get(db_prefix() . 'deals_column_order')->row();

        if ($result && !empty($result->column_order)) {
            return [
                'success' => true,
                'column_order' => json_decode($result->column_order, true)
            ];
        }

        // Default column order
        return [
            'success' => true,
            'column_order' => [
                'deal-number',
                'deal-name',
                'deal-company',
                'deal-pipeline',
                'deal-stage',
                'deal-value',
                'deal-assigned',
                'deal-status',
                'deal-created'
            ]
        ];
    }

    /**
     * Save column order for the deals table
     * @param array $column_order Array of column IDs
     * @return array
     */
    public function save_column_order($column_order)
    {
        $staff_id = get_staff_user_id();
        $data = [
            'staff_id' => $staff_id,
            'column_order' => json_encode($column_order),
            'dateupdated' => date('Y-m-d H:i:s')
        ];

        $this->db->where('staff_id', $staff_id);
        $existing = $this->db->get(db_prefix() . 'deals_column_order')->row();

        if ($existing) {
            $this->db->where('staff_id', $staff_id);
            $this->db->update(db_prefix() . 'deals_column_order', $data);
            $affected = $this->db->affected_rows();
        } else {
            $this->db->insert(db_prefix() . 'deals_column_order', $data);
            $affected = $this->db->affected_rows();
        }

        if ($affected > 0) {
            return ['success' => true, 'message' => _l('deals_column_order_updated')];
        }

        return ['success' => false, 'message' => _l('deals_column_order_update_failed')];
    }

    /**
     * Get deals with applied filters
     * @param array $filters Filter parameters
     * @return array
     */
    public function get_deals($filters = [])
    {
        $this->db->select(db_prefix() . 'deals.*, ' . db_prefix() . 'deals_pipelines.name as pipeline_name, ' . db_prefix() . 'deals_stages.name as stage_name, ' . db_prefix() . 'deals_statuses.name as status_name, ' . get_sql_select_staff_full_names('assigned') . ' as assigned_fullname');
        $this->db->from(db_prefix() . 'deals');
        $this->db->join(db_prefix() . 'deals_pipelines', db_prefix() . 'deals.pipeline_id = ' . db_prefix() . 'deals_pipelines.id', 'left');
        $this->db->join(db_prefix() . 'deals_stages', db_prefix() . 'deals.stage_id = ' . db_prefix() . 'deals_stages.id', 'left');
        $this->db->join(db_prefix() . 'deals_statuses', db_prefix() . 'deals.status = ' . db_prefix() . 'deals_statuses.id', 'left');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'deals.assigned = ' . db_prefix() . 'staff.staffid', 'left');

        // Apply filters
        if (!empty($filters)) {
            foreach ($filters as $field => $value) {
                if (!empty($value)) {
                    switch ($field) {
                        case 'status_final':
                            $this->db->where_in(db_prefix() . 'deals.status_final', $value);
                            break;
                        case 'pipeline_id':
                            $this->db->where_in(db_prefix() . 'deals.pipeline_id', $value);
                            break;
                        case 'stage_id':
                            $this->db->where_in(db_prefix() . 'deals.stage_id', $value);
                            break;
                        case 'assigned':
                            $this->db->where_in(db_prefix() . 'deals.assigned', $value);
                            break;
                        case 'tags':
                            $this->db->where('FIND_IN_SET(' . $this->db->escape($value) . ',' . db_prefix() . 'deals.tags)');
                            break;
                        case 'dateadded':
                            $this->db->where('DATE(' . db_prefix() . 'deals.dateadded) >=', $value);
                            break;
                        case 'lastcontact':
                            $this->db->where('DATE(' . db_prefix() . 'deals.lastcontact) >=', $value);
                            break;
                        case 'deal_value':
                            $this->db->where(db_prefix() . 'deals.deal_value >=', $value);
                            break;
                    }
                }
            }
        }

        // Apply permissions
        if (!has_permission('deals', '', 'view')) {
            $this->db->where(db_prefix() . 'deals.assigned', get_staff_user_id());
        }

        $this->db->order_by(db_prefix() . 'deals.id', 'DESC');
        return $this->db->get()->result_array();
    }

    /**
     * Get summary of deals by status
     * @return array
     */
    public function get_deals_summary()
    {
        $this->db->select('status_final, COUNT(*) as total, color');
        $this->db->from(db_prefix() . 'deals');
        $this->db->join(db_prefix() . 'deals_statuses', db_prefix() . 'deals.status = ' . db_prefix() . 'deals_statuses.id', 'left');
        $this->db->group_by('status_final');
        $statuses = $this->db->get()->result_array();

        $summary = [];
        foreach ($statuses as $status) {
            $summary[] = [
                'name' => $status['status_final'],
                'total' => $status['total'],
                'color' => $status['color'] ?? '#28B8DA',
                'percent' => round(($status['total'] / max(array_sum(array_column($statuses, 'total')), 1)) * 100, 2)
            ];
        }

        return $summary;
    }
    public function get_all()
    {
        // Get all deals
        $deals = $this->db->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();

        // Get all pipelines, stages, and staff for mapping
        $pipelines = $this->db->get(db_prefix() . 'deals_pipelines')->result_array();
        $stages = $this->db->get(db_prefix() . 'deals_stages')->result_array();
        $staff = $this->db->select('staffid, firstname, lastname')->get(db_prefix() . 'staff')->result_array();

        // Build id => label maps
        $pipelineMap = [];
        foreach ($pipelines as $pipeline) {
            $pipelineMap[$pipeline['id']] = $pipeline['name'];
        }
        $stageMap = [];
        foreach ($stages as $stage) {
            $stageMap[$stage['id']] = $stage['name'];
        }
        $staffMap = [];
        foreach ($staff as $member) {
            $staffMap[$member['staffid']] = $member['firstname'] . ' ' . $member['lastname'];
        }

        // Replace pipeline_id, stage_id, and assigned with array of id and label
        foreach ($deals as &$deal) {
            $pipeline_id = $deal['pipeline_id'];
            $stage_id = $deal['stage_id'];
            $assigned_id = isset($deal['assigned']) ? $deal['assigned'] : null;

            $deal['pipeline'] = [
                'id' => $pipeline_id,
                'label' => isset($pipelineMap[$pipeline_id]) ? $pipelineMap[$pipeline_id] : null,
            ];
            $deal['stage'] = [
                'id' => $stage_id,
                'label' => isset($stageMap[$stage_id]) ? $stageMap[$stage_id] : null,
            ];
            if ($assigned_id !== null) {
                $deal['assigned'] = [
                    'id' => $assigned_id,
                    'label' => isset($staffMap[$assigned_id]) ? $staffMap[$assigned_id] : null,
                ];
            }

            // Multi-pipelines for this deal
            $deal['pipelines'] = $this->get_deal_pipelines($deal['id']);

            unset($deal['pipeline_id'], $deal['stage_id']);
        }
        unset($deal);

        return $deals;
    }

    /**
     * Return all pipelines assigned to a deal with current stage and score
     */
    public function get_deal_pipelines($deal_id)
    {
        $this->db->select('r.id as relation_id, r.pipeline_id, r.stage_id, r.rating_id, r.score, dp.name as pipeline_name, ds.name as stage_name, dr.name as rating_name')
            ->from(db_prefix() . 'deals_pipeline_relations r')
            ->join(db_prefix() . 'deals_pipelines dp', 'dp.id = r.pipeline_id', 'left')
            ->join(db_prefix() . 'deals_stages ds', 'ds.id = r.stage_id', 'left')
            ->join(db_prefix() . 'deals_ratings dr', 'dr.id = r.rating_id', 'left')
            ->where('r.deal_id', (int) $deal_id)
            ->order_by('dp.`order`', 'ASC');

        $rows = $this->db->get()->result_array();
        return array_map(function ($row) {
            return [
                'relation_id' => (int) $row['relation_id'],
                'pipeline' => ['id' => (int) $row['pipeline_id'], 'label' => $row['pipeline_name']],
                'stage' => ['id' => (int) $row['stage_id'], 'label' => $row['stage_name']],
                'rating' => ['id' => (int) $row['rating_id'], 'label' => $row['rating_name']],
                'score' => $row['score'] !== null ? (int) $row['score'] : null,
            ];
        }, $rows);
    }

    /** Assign pipeline to deal (no duplicates) */
    public function add_pipeline_to_deal($deal_id, $pipeline_id, $stage_id = null, $rating_id = null, $score = null)
    {
        // Check if exists
        $exists = $this->db->where(['deal_id' => (int)$deal_id, 'pipeline_id' => (int)$pipeline_id])
            ->get(db_prefix() . 'deals_pipeline_relations')->row();
        if ($exists) {
            return (int) $exists->id;
        }
        $payload = [
            'deal_id' => (int)$deal_id,
            'pipeline_id' => (int)$pipeline_id,
            'stage_id' => $stage_id ? (int)$stage_id : null,
            'rating_id' => $rating_id ? (int)$rating_id : null,
            'score' => $score === null ? null : (int)$score,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
        ];
        $this->db->insert(db_prefix() . 'deals_pipeline_relations', $payload);
        return $this->db->insert_id();
    }

    public function update_pipeline_relation($relation_id, $data)
    {
        $update = [];
        foreach (['stage_id','rating_id','score'] as $key) {
            if (array_key_exists($key, $data)) {
                $update[$key] = $data[$key] === null ? null : (int)$data[$key];
            }
        }
        if (empty($update)) {
            return true;
        }
        $this->db->where('id', (int)$relation_id)->update(db_prefix() . 'deals_pipeline_relations', $update);
        return $this->db->affected_rows() > 0;
    }

    public function remove_pipeline_relation($relation_id)
    {
        $this->db->where('id', (int)$relation_id)->delete(db_prefix() . 'deals_pipeline_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages_with_deals($pipelineId = null)
    {
        if (!$pipelineId) {
            $pipelineId = $this->get_default_pipeline_id();
        }
        $stages = $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
        foreach ($stages as &$stage) {
            $stage['deals'] = $this->db->where('stage_id', (int) $stage['id'])->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
        }
        return $stages;
    }

    public function get_one($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals')->row_array();
    }

    public function update_deal($id, array $data)
    {
        // Remove any non-scalar fields (e.g. multi-pipelines array)
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                unset($data[$key]);
            }
        }
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals');
        return $this->db->affected_rows() > 0;
    }

    public function add_attachment($data)
    {
        $this->db->insert(db_prefix() . 'deals_attachments', $data);
        return $this->db->insert_id();
    }

    public function get_attachments($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)->get(db_prefix() . 'deals_attachments')->result_array();
    }



    public function create($data)
    {
        $insert = [
            'name' => $data['name'] ?? null,
            'company' => $data['company'] ?? null,
            'description' => $data['description'] ?? null,
            'status' => $data['status'] ?? null,
            'pipeline_id' => $data['pipeline_id'] ?? $this->get_default_pipeline_id(),
            'stage_id' => $data['stage_id'] ?? $this->get_default_open_stage_id(),
            'deal_value' => $data['deal_value'] ?? 0,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'status_final' => 'open',
            'source' => $data['source'] ?? null, // New field
            'status' => $data['status'] ?? null, // New field (status ID)
            'assigned' => $data['assigned'] ?? 0, // New field
        ];
        $this->db->insert(db_prefix() . 'deals', $insert);
        return $this->db->insert_id();
    }

    // public function update_status($id, $status)
    // {
    //     $statusId = $this->map_status_to_stage($status);
    //     $payload = [
    //         'last_status_change' => date('Y-m-d H:i:s'),
    //         'status' => $statusId,
    //     ];
    //     if ($status === 'won' || $status === 'lost') {
    //         $payload['status_final'] = $status;
    //     }
    //     $this->db->where('id', $id)->update(db_prefix() . 'deals', $payload);
    //     return $this->db->affected_rows() > 0;
    // }


    public function mark_lost($id, $reasonId = null)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'lost',
            'lost_reason_id' => $reasonId,
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }
    public function mark_won($id)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'won',
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }

    private function map_status_to_stage($status)
    {
        $statusName = null;
        if ($status === 'won') {
            $statusName = 'Won';
        } elseif ($status === 'lost') {
            $statusName = 'Lost';
        } else {
            $statusName = 'Open';
        }

        $row = $this->db->where('name', $statusName)->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : $this->get_default_open_stage_id();
    }

    // Pipelines helpers
    public function get_default_pipeline_id()
    {
        $row = $this->db->order_by('`order`', 'ASC')->limit(1)->get(db_prefix() . 'deals_pipelines')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_default_open_stage_id()
    {
        $pipelineId = $this->get_default_pipeline_id();
        if (!$pipelineId) {
            return null;
        }
        $row = $this->db->where(['pipeline_id' => $pipelineId, 'is_won' => 0, 'is_lost' => 0])->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_pipelines()
    {
        return $this->db->order_by('`order`', 'ASC')->get(db_prefix() . 'deals_pipelines')->result_array();
    }

    public function create_pipeline($name)
    {
        $this->db->insert(db_prefix() . 'deals_pipelines', ['name' => $name, 'order' => (int) ($this->db->count_all(db_prefix() . 'deals_pipelines') + 1)]);
        return $this->db->insert_id();
    }

    public function update_pipeline($id, $data)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals_pipelines', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_pipeline($id)
    {
        $this->db->where('id', $id)->delete(db_prefix() . 'deals_pipelines');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages($pipelineId)
    {
        return $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
    }

    public function create_stage($pipelineId, $data)
    {
        $data['pipeline_id'] = (int) $pipelineId;
        if (!isset($data['position'])) {
            $data['position'] = (int) $this->db->where('pipeline_id', (int) $pipelineId)->count_all_results(db_prefix() . 'deals_stages') + 1;
        }
        $this->db->insert(db_prefix() . 'deals_stages', $data);
        return $this->db->insert_id();
    }

    public function update_stage($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_stages', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_stage($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_stages');
        return $this->db->affected_rows() > 0;
    }

    public function reorder_stages($pipelineId, $orderedIds)
    {
        $position = 1;
        foreach ((array) $orderedIds as $stageId) {
            $this->db->where('id', (int) $stageId)->where('pipeline_id', (int) $pipelineId)->update(db_prefix() . 'deals_stages', ['position' => $position]);
            $position++;
        }
        return true;
    }

    // Lost reasons CRUD
    public function get_lost_reasons()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_lost_reasons')->result_array();
    }

    public function get_lost_reason($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_lost_reasons')->row_array();
    }

    public function get_staff()
    {
        return $this->db->select('staffid, firstname, lastname')
            ->where('active', 1)
            ->order_by('firstname', 'ASC')
            ->get(db_prefix() . 'staff')
            ->result_array();
    }



    /**
     * Get the stage_id for a deal in a specific pipeline.
     * If not found, fallback to the main deal stage_id.
     *
     * @param int $deal_id
     * @param int $pipeline_id
     * @return int|null
     */
    public function get_deal_pipeline_stage($deal_id, $pipeline_id)
    {
        $row = $this->db->where('deal_id', (int)$deal_id)
                        ->where('pipeline_id', (int)$pipeline_id)
                        ->get(db_prefix() . 'deals_pipeline_relations')
                        ->row_array();
        if ($row && isset($row['stage_id'])) {
            return (int)$row['stage_id'];
        }
        // Fallback to main deal stage_id
        $deal = $this->db->where('id', (int)$deal_id)->get(db_prefix() . 'deals')->row_array();
        return $deal ? (int)$deal['stage_id'] : null;
    }

    /**
     * Update the stage_id for a deal in a specific pipeline.
     * If the relation does not exist, create it.
     *
     * @param int $deal_id
     * @param int $pipeline_id
     * @param int $stage_id
     * @return bool
     */
    public function update_deal_pipeline_stage($deal_id, $pipeline_id, $stage_id)
    {
        $exists = $this->db->where('deal_id', (int)$deal_id)
                           ->where('pipeline_id', (int)$pipeline_id)
                           ->get(db_prefix() . 'deals_pipeline_relations')
                           ->row_array();
        if ($exists) {
            $this->db->where('deal_id', (int)$deal_id)
                     ->where('pipeline_id', (int)$pipeline_id)
                     ->update(db_prefix() . 'deals_pipeline_relations', ['stage_id' => (int)$stage_id]);
        } else {
            $this->db->insert(db_prefix() . 'deals_pipeline_relations', [
                'deal_id' => (int)$deal_id,
                'pipeline_id' => (int)$pipeline_id,
                'stage_id' => (int)$stage_id
            ]);
        }
        return true;
    }

    public function get_stage_name($stage_id)
    {
        $stage = $this->db->where('id', (int) $stage_id)->get(db_prefix() . 'deals_stages')->row_array();
        return $stage ? $stage['name'] : 'Unknown Stage';
    }

    public function create_lost_reason($name)
    {
        $this->db->insert(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->insert_id();
    }
    public function update_lost_reason($id, $name)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->affected_rows() > 0;
    }
    public function delete_lost_reason($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_lost_reasons');
        return $this->db->affected_rows() > 0;
    }

    // Deal Contacts Methods
    public function get_deal_contacts($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_contact_links l')
            ->join(db_prefix() . 'deals_contacts_book cb', 'cb.id = l.contact_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_contact_to_deal($deal_id, $contact_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'contact_id' => (int) $contact_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_contact_links', $data);
        return $this->db->insert_id();
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('contact_id', (int) $contact_id)
            ->delete(db_prefix() . 'deal_contact_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_contacts($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_contacts_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT contact_id FROM ' . db_prefix() . 'deal_contact_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.firstname', 'ASC')->get()->result_array();
    }

    public function create_module_contact($data)
    {
        $insert = [
            'firstname' => $data['firstname'] ?? '',
            'lastname' => $data['lastname'] ?? '',
            'email' => $data['email'] ?? '',
            'phonenumber' => $data['phonenumber'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1,
        ];
        $this->db->insert(db_prefix() . 'deals_contacts_book', $insert);
        return $this->db->insert_id();
    }

    // Deal Companies Methods
    public function get_deal_companies($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_company_links l')
            ->join(db_prefix() . 'deals_companies_book cb', 'cb.id = l.company_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_company_to_deal($deal_id, $company_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'company_id' => (int) $company_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_company_links', $data);
        return $this->db->insert_id();
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('company_id', (int) $company_id)
            ->delete(db_prefix() . 'deal_company_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_companies($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_companies_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT company_id FROM ' . db_prefix() . 'deal_company_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.name', 'ASC')->get()->result_array();
    }

    // Company CRUD Methods (module-specific)
    public function create_company($data)
    {
        // Check if email or phone already exists
        if (!empty($data['email'])) {
            $existing = $this->db->where('email', $data['email'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Email already exists', 'existing_id' => $existing['id']];
            }
        }

        if (!empty($data['phone'])) {
            $existing = $this->db->where('phone', $data['phone'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Phone number already exists', 'existing_id' => $existing['id']];
            }
        }

        $insert = [
            'name' => $data['name'] ?? '',
            'email' => $data['email'] ?? '',
            'phone' => $data['phone'] ?? '',
            'website' => $data['website'] ?? '',
            'address' => $data['address'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1
        ];

        $this->db->insert(db_prefix() . 'deals_companies_book', $insert);
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function get_company($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_companies_book')->row_array();
    }

    public function update_company($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_companies_book', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_company($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_companies_book');
        return $this->db->affected_rows() > 0;
    }

    // Deal Attachments Relations Methods
    public function get_deal_attachments($deal_id)
    {
        $this->db->select('da.*, dar.dateadded as linked_date')
            ->from(db_prefix() . 'deal_attachments_relations dar')
            ->join(db_prefix() . 'deals_attachments da', 'da.id = dar.attachment_id', 'left')
            ->where('dar.deal_id', (int) $deal_id)
            ->order_by('dar.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function get_attachment($attachment_id)
    {
        return $this->db->where('id', (int) $attachment_id)->get(db_prefix() . 'deals_attachments')->row_array();
    }

    public function add_attachment_to_deal($deal_id, $attachment_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'attachment_id' => (int) $attachment_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];

        $this->db->insert(db_prefix() . 'deal_attachments_relations', $data);
        return $this->db->insert_id();
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('attachment_id', (int) $attachment_id)
            ->delete(db_prefix() . 'deal_attachments_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_attachments($deal_id = null)
    {
        $this->db->select('da.*')
            ->from(db_prefix() . 'deals_attachments da');

        if ($deal_id) {
            $this->db->where('da.id NOT IN (SELECT attachment_id FROM ' . db_prefix() . 'deal_attachments_relations WHERE deal_id = ' . (int) $deal_id . ')');
        }

        return $this->db->order_by('da.dateadded', 'DESC')->get()->result_array();
    }

    // Deal Activities Methods with type
    public function get_deal_activities($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('date', 'DESC')
            ->get(db_prefix() . 'deals_activity')
            ->result_array();
    }

    public function add_deal_activity($deal_id, $payload)
    {
        $data = [
            'title' => $payload['title'] ?? '',
            'rel_id' => (int) $deal_id,
            'description' => $payload['description'] ?? '',
            'additional_data' => $payload['additional_data'] ?? null,
            'staffid' => get_staff_user_id(),
            'full_name' => get_staff_full_name(get_staff_user_id()),
            'date' => date('Y-m-d H:i:s'),
            'type' => $payload['type'] ?? 'activity',
            'activity_date' => $payload['activity_date'] ?? null,
            'reminder' => $payload['reminder'] ?? 0,
            'owner_id' => $payload['owner_id'] ?? get_staff_user_id(),
            'guests' => $payload['guests'] ?? null,
            'status' => $payload['status'] ?? 'pending',
        ];
        $this->db->insert(db_prefix() . 'deals_activity', $data);
        return $this->db->insert_id();
    }

    // Notes
    public function add_note($deal_id, $description)
    {
        $noteData = [
            'rel_id' => (int) $deal_id,
            'description' => (string) $description,
            'addedfrom' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
            'date_contacted' => null,
        ];
        $this->db->insert(db_prefix() . 'deals_notes', $noteData);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Note added',
                'additional_data' => $description,
                'type' => 'notes'
            ]);
        }
        return $insertId;
    }
    public function update_activity_status($id, $status)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_activity', ['status' => $status]);
        return $this->db->affected_rows() > 0;
    }

    public function get_notes($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('dateadded', 'DESC')
            ->get(db_prefix() . 'deals_notes')
            ->result_array();
    }

    // Calls
    public function add_call($deal_id, $payload)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'outcome' => $payload['outcome'] ?? '',
            'call_notes' => $payload['call_notes'] ?? null,
            'call_date' => $payload['call_date'] ?? date('Y-m-d H:i:s'),
            'staffid' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        // dd($data);
        $this->db->insert(db_prefix() . 'deals_calls', $data);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Call logged',
                'additional_data' => json_encode([
                    'outcome' => $data['outcome'],
                    'date' => $data['call_date'],
                    'notes' => $data['call_notes']
                ]),
                'type' => 'calls'
            ]);
        }
        return $insertId;
    }

    public function get_calls($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)
            ->order_by('call_date', 'DESC')
            ->get(db_prefix() . 'deals_calls')
            ->result_array();
    }

    // Import support (adapted from Leads_model)
    public function import($data)
    {
        $this->db->insert(db_prefix() . 'deals', $data);
        return $this->db->insert_id();
    }

    public function get_source_id_by_name($name)
    {
        $row = $this->db->where('name', $name)->get(db_prefix() . 'deals_sources')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_status_id_by_name($name)
    {
        $row = $this->db->where('name', $name)->get(db_prefix() . 'deals_statuses')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_staff_id_by_name($name)
    {
        $parts = explode(' ', trim($name));
        $firstname = $parts[0] ?? '';
        $lastname = $parts[1] ?? '';
        $row = $this->db->where('firstname', $firstname)
            ->where('lastname', $lastname)
            ->get(db_prefix() . 'staff')
            ->row();
        return $row ? (int) $row->staffid : null;
    }

    public function get_sources()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_sources')->result_array();
    }

    public function get_statuses()
    {
        return $this->db->order_by('statusorder', 'ASC')->get(db_prefix() . 'deals_statuses')->result_array();
    }

    // Sources Management
    public function get_source($id = '')
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);
            return $this->db->get(db_prefix() . 'deals_sources')->row();
        }
        return $this->db->order_by('name','ASC')->get(db_prefix() . 'deals_sources')->result_array();
    }

    public function add_source($data)
    {
        $this->db->insert(db_prefix() . 'deals_sources', $data);
        return $this->db->insert_id();
    }

    public function update_source($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals_sources', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_source($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'deals_sources');
        return $this->db->affected_rows() > 0;
    }

    // Statuses Management
    public function get_status($id = '')
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);
            return $this->db->get(db_prefix() . 'deals_statuses')->row();
        }
        $this->db->order_by('statusorder', 'asc');
        return $this->db->get(db_prefix() . 'deals_statuses')->result_array();
    }

    public function add_status($data)
    {
        $this->db->insert(db_prefix() . 'deals_statuses', $data);
        return $this->db->insert_id();
    }

    public function update_status($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals_statuses', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_status($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'deals_statuses');
        return $this->db->affected_rows() > 0;
    }

    // Staff access check
    public function staff_can_access_deal($id)
    {
        if (staff_can('view', 'deals')) {
            return true;
        }
        
        $this->db->where('id', $id);
        $this->db->where('(assigned = ' . get_staff_user_id() . ' OR addedfrom=' . get_staff_user_id() . ' OR is_public=1)');
        $deal = $this->db->get(db_prefix() . 'deals')->row();
        
        return $deal ? true : false;
    }

    // Get deal attachments
    // public function get_deal_attachments($deal_id)
    // {
    //     $this->db->where('deal_id', $deal_id);
    //     return $this->db->get(db_prefix() . 'deals_attachments')->result_array();
    // }

    // Get deal activities
    // public function get_deal_activities($deal_id)
    // {
    //     $this->db->where('rel_id', $deal_id);
    //     $this->db->order_by('date', 'desc');
    //     return $this->db->get(db_prefix() . 'deals_activity')->result_array();
    // }

    // Get deal notes
    public function get_deal_notes($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        $this->db->order_by('dateadded', 'desc');
        return $this->db->get(db_prefix() . 'deals_notes')->result_array();
    }

    // Get deal reminders
    public function get_deal_reminders($deal_id)
    {
        $this->db->where('deal_id', $deal_id);
        $this->db->order_by('date', 'asc');
        return $this->db->get(db_prefix() . 'deals_reminders')->result_array();
    }
}
