<?php

use app\services\imap\Imap;
use app\services\LeadProfileBadges;
use app\services\leads\LeadsKanban;
use app\services\imap\ConnectionErrorException;
use Ddeboer\Imap\Exception\MailboxDoesNotExistException;

header('Content-Type: text/html; charset=utf-8');
defined('BASEPATH') or exit('No direct script access allowed');

class Leads extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('leads_model');
    }

    /* List all leads */
    public function index($id = '')
    {
        close_setup_menu();

        if (!is_staff_member()) {
            access_denied('Leads');
        }

        $data['switch_kanban'] = true;

        if ($this->session->userdata('leads_kanban_view') == 'true') {
            $data['switch_kanban'] = false;
            $data['bodyclass']     = 'kan-ban-body';
        }

        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
            $this->load->model('gdpr_model');
            $data['consent_purposes'] = $this->gdpr_model->get_consent_purposes();
        }
        $data['summary']  = get_leads_summary();
        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();
        $data['title']    = _l('leads');
        $data['table'] = App_table::find('leads');
        // in case accesed the url leads/index/ directly with id - used in search
        $data['leadid']   = $id;
        $data['isKanBan'] = $this->session->has_userdata('leads_kanban_view') &&
            $this->session->userdata('leads_kanban_view') == 'true';

        $this->load->view('admin/leads/manage_leads', $data);
    }

    public function table()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        App_table::find('leads')->output();
    }

    public function kanban()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $data['statuses']      = $this->leads_model->get_status();
        $data['base_currency'] = get_base_currency();
        $data['summary']       = get_leads_summary();

        echo $this->load->view('admin/leads/kan-ban', $data, true);
    }

    /* Add or update lead */
    public function lead($id = '')
    {
        if (!is_staff_member() || ($id != '' && !$this->leads_model->staff_can_access_lead($id))) {
            ajax_access_denied();
        }

        if ($this->input->post()) {
            if ($id == '') {
                $id      = $this->leads_model->add($this->input->post());
                $message = $id ? _l('added_successfully', _l('lead')) : '';

                echo json_encode([
                    'success'  => $id ? true : false,
                    'id'       => $id,
                    'message'  => $message,
                    'leadView' => $id ? $this->_get_lead_data($id) : [],
                ]);
            } else {
                $emailOriginal   = $this->db->select('email')->where('id', $id)->get(db_prefix() . 'leads')->row()->email;
                $proposalWarning = false;
                $message         = '';
                $success         = $this->leads_model->update($this->input->post(), $id);

                if ($success) {
                    $emailNow = $this->db->select('email')->where('id', $id)->get(db_prefix() . 'leads')->row()->email;

                    $proposalWarning = (total_rows(db_prefix() . 'proposals', [
                        'rel_type' => 'lead',
                        'rel_id'   => $id, ]) > 0 && ($emailOriginal != $emailNow) && $emailNow != '') ? true : false;

                    $message = _l('updated_successfully', _l('lead'));
                }
                echo json_encode([
                    'success'          => $success,
                    'message'          => $message,
                    'id'               => $id,
                    'proposal_warning' => $proposalWarning,
                    'leadView'         => $this->_get_lead_data($id),
                ]);
            }
            die;
        }

        echo json_encode([
            'leadView' => $this->_get_lead_data($id),
        ]);
    }

    private function _get_lead_data($id = '')
    {
        $reminder_data         = '';
        $data['lead_locked']   = false;
        $data['openEdit']      = $this->input->get('edit') ? true : false;
        $data['members']       = $this->staff_model->get('', ['is_not_staff' => 0, 'active' => 1]);
        $data['status_id']     = $this->input->get('status_id') ? $this->input->get('status_id') : get_option('leads_default_status');
        $data['base_currency'] = get_base_currency();

        if (is_numeric($id)) {
            $leadWhere = (staff_can('view',  'leads') ? [] : '(assigned = ' . get_staff_user_id() . ' OR addedfrom=' . get_staff_user_id() . ' OR is_public=1)');

            $lead = $this->leads_model->get($id, $leadWhere);

            if (!$lead) {
                header('HTTP/1.0 404 Not Found');
                echo _l('lead_not_found');
                die;
            }

            if (total_rows(db_prefix() . 'clients', ['leadid' => $id ]) > 0) {
                $data['lead_locked'] = ((!is_admin() && get_option('lead_lock_after_convert_to_customer') == 1) ? true : false);
            }

            $reminder_data = $this->load->view('admin/includes/modals/reminder', [
                    'id'             => $lead->id,
                    'name'           => 'lead',
                    'members'        => $data['members'],
                    'reminder_title' => _l('lead_set_reminder_title'),
                ], true);

            $data['lead']          = $lead;
            $data['mail_activity'] = $this->leads_model->get_mail_activity($id);
            $data['notes']         = $this->misc_model->get_notes($id, 'lead');
            $data['activity_log']  = $this->leads_model->get_lead_activity_log($id);

            if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
                $this->load->model('gdpr_model');
                $data['purposes'] = $this->gdpr_model->get_consent_purposes($lead->id, 'lead');
                $data['consents'] = $this->gdpr_model->get_consents(['lead_id' => $lead->id]);
            }

            $leadProfileBadges         = new LeadProfileBadges($id);
            $data['total_reminders']   = $leadProfileBadges->getCount('reminders');
            $data['total_notes']       = $leadProfileBadges->getCount('notes');
            $data['total_attachments'] = $leadProfileBadges->getCount('attachments');
            $data['total_tasks']       = $leadProfileBadges->getCount('tasks');
            $data['total_proposals']   = $leadProfileBadges->getCount('proposals');
        }


        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();

        $data = hooks()->apply_filters('lead_view_data', $data);

        return [
            'data'          => $this->load->view('admin/leads/lead', $data, true),
            'reminder_data' => $reminder_data,
        ];
    }

    public function leads_kanban_load_more()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $status = $this->input->get('status');
        $page   = $this->input->get('page');

        $this->db->where('id', $status);
        $status = $this->db->get(db_prefix() . 'leads_status')->row_array();

        $leads = (new LeadsKanban($status['id']))
        ->search($this->input->get('search'))
        ->sortBy(
            $this->input->get('sort_by'),
            $this->input->get('sort')
        )
        ->page($page)->get();

        foreach ($leads as $lead) {
            $this->load->view('admin/leads/_kan_ban_card', [
                'lead'   => $lead,
                'status' => $status,
            ]);
        }
    }

    public function switch_kanban($set = 0)
    {
        if ($set == 1) {
            $set = 'true';
        } else {
            $set = 'false';
        }
        $this->session->set_userdata([
            'leads_kanban_view' => $set,
        ]);
        redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
    }

    public function export($id)
    {
        if (is_admin()) {
            $this->load->library('gdpr/gdpr_lead');
            $this->gdpr_lead->export($id);
        }
    }

    /* Delete lead from database */
    public function delete($id)
    {
        if (!$id) {
            redirect(admin_url('leads'));
        }

        if (staff_cant('delete', 'leads')) {
            access_denied('Delete Lead');
        }

        $response = $this->leads_model->delete($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_lowercase')));
        } elseif ($response === true) {
            set_alert('success', _l('deleted', _l('lead')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_lowercase')));
        }

        $ref = $_SERVER['HTTP_REFERER'];

        // if user access leads/inded/ID to prevent redirecting on the same url because will throw 404
        if (!$ref || strpos($ref, 'index/' . $id) !== false) {
            redirect(admin_url('leads'));
        }

        redirect($ref);
    }

    public function mark_as_lost($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->mark_as_lost($id);
        if ($success) {
            $message = _l('lead_marked_as_lost');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function unmark_as_lost($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->unmark_as_lost($id);
        if ($success) {
            $message = _l('lead_unmarked_as_lost');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function mark_as_junk($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->mark_as_junk($id);
        if ($success) {
            $message = _l('lead_marked_as_junk');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function unmark_as_junk($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->unmark_as_junk($id);
        if ($success) {
            $message = _l('lead_unmarked_as_junk');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function add_activity()
    {
        $leadid = $this->input->post('leadid');
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($leadid)) {
            ajax_access_denied();
        }
        if ($this->input->post()) {
            $message = $this->input->post('activity');
            $aId     = $this->leads_model->log_lead_activity($leadid, $message);
            if ($aId) {
                $this->db->where('id', $aId);
                $this->db->update(db_prefix() . 'lead_activity_log', ['custom_activity' => 1]);
            }
            echo json_encode(['leadView' => $this->_get_lead_data($leadid), 'id' => $leadid]);
        }
    }

    public function get_convert_data($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        if (is_gdpr() && get_option('gdpr_enable_consent_for_contacts') == '1') {
            $this->load->model('gdpr_model');
            $data['purposes'] = $this->gdpr_model->get_consent_purposes($id, 'lead');
        }
        $data['lead'] = $this->leads_model->get($id);
        $this->load->view('admin/leads/convert_to_customer', $data);
    }

    /**
     * Convert lead to client
     * @since  version 1.0.1
     * @return mixed
     */
    public function convert_to_customer()
    {
        if (!is_staff_member()) {
            access_denied('Lead Convert to Customer');
        }

        if ($this->input->post()) {
            $default_country  = get_option('customer_default_country');
            $data             = $this->input->post();
            $data['password'] = $this->input->post('password', false);

            $original_lead_email = $data['original_lead_email'];
            unset($data['original_lead_email']);

            if (isset($data['transfer_notes'])) {
                $notes = $this->misc_model->get_notes($data['leadid'], 'lead');
                unset($data['transfer_notes']);
            }

            if (isset($data['transfer_consent'])) {
                $this->load->model('gdpr_model');
                $consents = $this->gdpr_model->get_consents(['lead_id' => $data['leadid']]);
                unset($data['transfer_consent']);
            }

            if (isset($data['merge_db_fields'])) {
                $merge_db_fields = $data['merge_db_fields'];
                unset($data['merge_db_fields']);
            }

            if (isset($data['merge_db_contact_fields'])) {
                $merge_db_contact_fields = $data['merge_db_contact_fields'];
                unset($data['merge_db_contact_fields']);
            }

            if (isset($data['include_leads_custom_fields'])) {
                $include_leads_custom_fields = $data['include_leads_custom_fields'];
                unset($data['include_leads_custom_fields']);
            }

            if ($data['country'] == '' && $default_country != '') {
                $data['country'] = $default_country;
            }

            $data['billing_street']  = $data['address'];
            $data['billing_city']    = $data['city'];
            $data['billing_state']   = $data['state'];
            $data['billing_zip']     = $data['zip'];
            $data['billing_country'] = $data['country'];

            $data['is_primary'] = 1;
            $id                 = $this->clients_model->add($data, true);
            if ($id) {
                $primary_contact_id = get_primary_contact_user_id($id);

                if (isset($notes)) {
                    foreach ($notes as $note) {
                        $this->db->insert(db_prefix() . 'notes', [
                            'rel_id'         => $id,
                            'rel_type'       => 'customer',
                            'dateadded'      => $note['dateadded'],
                            'addedfrom'      => $note['addedfrom'],
                            'description'    => $note['description'],
                            'date_contacted' => $note['date_contacted'],
                            ]);
                    }
                }
                if (isset($consents)) {
                    foreach ($consents as $consent) {
                        unset($consent['id']);
                        unset($consent['purpose_name']);
                        $consent['lead_id']    = 0;
                        $consent['contact_id'] = $primary_contact_id;
                        $this->gdpr_model->add_consent($consent);
                    }
                }
                if (staff_cant('view', 'customers') && get_option('auto_assign_customer_admin_after_lead_convert') == 1) {
                    $this->db->insert(db_prefix() . 'customer_admins', [
                        'date_assigned' => date('Y-m-d H:i:s'),
                        'customer_id'   => $id,
                        'staff_id'      => get_staff_user_id(),
                    ]);
                }
                $this->leads_model->log_lead_activity($data['leadid'], 'not_lead_activity_converted', false, serialize([
                    get_staff_full_name(),
                ]));
                $default_status = $this->leads_model->get_status('', [
                    'isdefault' => 1,
                ]);
                $this->db->where('id', $data['leadid']);
                $this->db->update(db_prefix() . 'leads', [
                    'date_converted' => date('Y-m-d H:i:s'),
                    'status'         => $default_status[0]['id'],
                    'junk'           => 0,
                    'lost'           => 0,
                ]);
                // Check if lead email is different then client email
                $contact = $this->clients_model->get_contact(get_primary_contact_user_id($id));
                if ($contact->email != $original_lead_email) {
                    if ($original_lead_email != '') {
                        $this->leads_model->log_lead_activity($data['leadid'], 'not_lead_activity_converted_email', false, serialize([
                            $original_lead_email,
                            $contact->email,
                        ]));
                    }
                }
                if (isset($include_leads_custom_fields)) {
                    foreach ($include_leads_custom_fields as $fieldid => $value) {
                        // checked don't merge
                        if ($value == 5) {
                            continue;
                        }
                        // get the value of this leads custom fiel
                        $this->db->where('relid', $data['leadid']);
                        $this->db->where('fieldto', 'leads');
                        $this->db->where('fieldid', $fieldid);
                        $lead_custom_field_value = $this->db->get(db_prefix() . 'customfieldsvalues')->row()->value;
                        // Is custom field for contact ot customer
                        if ($value == 1 || $value == 4) {
                            if ($value == 4) {
                                $field_to = 'contacts';
                            } else {
                                $field_to = 'customers';
                            }
                            $this->db->where('id', $fieldid);
                            $field = $this->db->get(db_prefix() . 'customfields')->row();
                            // check if this field exists for custom fields
                            $this->db->where('fieldto', $field_to);
                            $this->db->where('name', $field->name);
                            $exists               = $this->db->get(db_prefix() . 'customfields')->row();
                            $copy_custom_field_id = null;
                            if ($exists) {
                                $copy_custom_field_id = $exists->id;
                            } else {
                                // there is no name with the same custom field for leads at the custom side create the custom field now
                                $this->db->insert(db_prefix() . 'customfields', [
                                    'fieldto'        => $field_to,
                                    'name'           => $field->name,
                                    'required'       => $field->required,
                                    'type'           => $field->type,
                                    'options'        => $field->options,
                                    'display_inline' => $field->display_inline,
                                    'field_order'    => $field->field_order,
                                    'slug'           => slug_it($field_to . '_' . $field->name, [
                                        'separator' => '_',
                                    ]),
                                    'active'        => $field->active,
                                    'only_admin'    => $field->only_admin,
                                    'show_on_table' => $field->show_on_table,
                                    'bs_column'     => $field->bs_column,
                                ]);
                                $new_customer_field_id = $this->db->insert_id();
                                if ($new_customer_field_id) {
                                    $copy_custom_field_id = $new_customer_field_id;
                                }
                            }
                            if ($copy_custom_field_id != null) {
                                $insert_to_custom_field_id = $id;
                                if ($value == 4) {
                                    $insert_to_custom_field_id = get_primary_contact_user_id($id);
                                }
                                $this->db->insert(db_prefix() . 'customfieldsvalues', [
                                    'relid'   => $insert_to_custom_field_id,
                                    'fieldid' => $copy_custom_field_id,
                                    'fieldto' => $field_to,
                                    'value'   => $lead_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 2) {
                            if (isset($merge_db_fields)) {
                                $db_field = $merge_db_fields[$fieldid];
                                // in case user don't select anything from the db fields
                                if ($db_field == '') {
                                    continue;
                                }
                                if ($db_field == 'country' || $db_field == 'shipping_country' || $db_field == 'billing_country') {
                                    $this->db->where('iso2', $lead_custom_field_value);
                                    $this->db->or_where('short_name', $lead_custom_field_value);
                                    $this->db->or_like('long_name', $lead_custom_field_value);
                                    $country = $this->db->get(db_prefix() . 'countries')->row();
                                    if ($country) {
                                        $lead_custom_field_value = $country->country_id;
                                    } else {
                                        $lead_custom_field_value = 0;
                                    }
                                }
                                $this->db->where('userid', $id);
                                $this->db->update(db_prefix() . 'clients', [
                                    $db_field => $lead_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 3) {
                            if (isset($merge_db_contact_fields)) {
                                $db_field = $merge_db_contact_fields[$fieldid];
                                if ($db_field == '') {
                                    continue;
                                }
                                $this->db->where('id', $primary_contact_id);
                                $this->db->update(db_prefix() . 'contacts', [
                                    $db_field => $lead_custom_field_value,
                                ]);
                            }
                        }
                    }
                }
                // set the lead to status client in case is not status client
                $this->db->where('isdefault', 1);
                $status_client_id = $this->db->get(db_prefix() . 'leads_status')->row()->id;
                $this->db->where('id', $data['leadid']);
                $this->db->update(db_prefix() . 'leads', [
                    'status' => $status_client_id,
                ]);

                set_alert('success', _l('lead_to_client_base_converted_success'));

                if (is_gdpr() && get_option('gdpr_after_lead_converted_delete') == '1') {
                    // When lead is deleted
                    // move all proposals to the actual customer record
                    $this->db->where('rel_id', $data['leadid']);
                    $this->db->where('rel_type', 'lead');
                    $this->db->update('proposals', [
                        'rel_id'   => $id,
                        'rel_type' => 'customer',
                    ]);

                    $this->leads_model->delete($data['leadid']);

                    $this->db->where('userid', $id);
                    $this->db->update(db_prefix() . 'clients', ['leadid' => null]);
                }

                log_activity('Created Lead Client Profile [LeadID: ' . $data['leadid'] . ', ClientID: ' . $id . ']');
                hooks()->do_action('lead_converted_to_customer', ['lead_id' => $data['leadid'], 'customer_id' => $id]);
                redirect(admin_url('clients/client/' . $id));
            }
        }
    }

    /* Used in kanban when dragging and mark as */
    public function update_lead_status()
    {
        if ($this->input->post() && $this->input->is_ajax_request()) {
            $this->leads_model->update_lead_status($this->input->post());
        }
    }

    public function update_status_order()
    {
        if ($post_data = $this->input->post()) {
            $this->leads_model->update_status_order($post_data);
        }
    }

    public function add_lead_attachment()
    {
        $id       = $this->input->post('id');
        $lastFile = $this->input->post('last_file');

        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }

        handle_lead_attachments($id);
        echo json_encode(['leadView' => $lastFile ? $this->_get_lead_data($id) : [], 'id' => $id]);
    }

    public function add_external_attachment()
    {
        if ($this->input->post()) {
            $this->leads_model->add_attachment_to_database(
                $this->input->post('lead_id'),
                $this->input->post('files'),
                $this->input->post('external')
            );
        }
    }

    public function delete_attachment($id, $lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }
        echo json_encode([
            'success'  => $this->leads_model->delete_lead_attachment($id),
            'leadView' => $this->_get_lead_data($lead_id),
            'id'       => $lead_id,
        ]);
    }

    public function delete_note($id, $lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }
        echo json_encode([
            'success'  => $this->misc_model->delete_note($id),
            'leadView' => $this->_get_lead_data($lead_id),
            'id'       => $lead_id,
        ]);
    }

    public function update_all_proposal_emails_linked_to_lead($id)
    {
        $success = false;
        $email   = '';
        if ($this->input->post('update')) {
            $this->load->model('proposals_model');

            $this->db->select('email');
            $this->db->where('id', $id);
            $email = $this->db->get(db_prefix() . 'leads')->row()->email;

            $proposals = $this->proposals_model->get('', [
                'rel_type' => 'lead',
                'rel_id'   => $id,
            ]);
            $affected_rows = 0;

            foreach ($proposals as $proposal) {
                $this->db->where('id', $proposal['id']);
                $this->db->update(db_prefix() . 'proposals', [
                    'email' => $email,
                ]);
                if ($this->db->affected_rows() > 0) {
                    $affected_rows++;
                }
            }

            if ($affected_rows > 0) {
                $success = true;
            }
        }

        echo json_encode([
            'success' => $success,
            'message' => _l('proposals_emails_updated', [
                _l('lead_lowercase'),
                $email,
            ]),
        ]);
    }

    public function save_form_data()
    {
        $data = $this->input->post();

        // form data should be always sent to the request and never should be empty
        // this code is added to prevent losing the old form in case any errors
        if (!isset($data['formData']) || isset($data['formData']) && !$data['formData']) {
            echo json_encode([
                'success' => false,
            ]);
            die;
        }

        // If user paste with styling eq from some editor word and the Codeigniter XSS feature remove and apply xss=remove, may break the json.
        $data['formData'] = preg_replace('/=\\\\/m', "=''", $data['formData']);

        $this->db->where('id', $data['id']);
        $this->db->update(db_prefix() . 'web_to_lead', [
            'form_data' => $data['formData'],
        ]);
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'success' => true,
                'message' => _l('updated_successfully', _l('web_to_lead_form')),
            ]);
        } else {
            echo json_encode([
                'success' => false,
            ]);
        }
    }

    public function form($id = '')
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }
        if ($this->input->post()) {
            if ($id == '') {
                $data = $this->input->post();
                $id   = $this->leads_model->add_form($data);
                if ($id) {
                    set_alert('success', _l('added_successfully', _l('web_to_lead_form')));
                    redirect(admin_url('leads/form/' . $id));
                }
            } else {
                $success = $this->leads_model->update_form($id, $this->input->post());
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('web_to_lead_form')));
                }
                redirect(admin_url('leads/form/' . $id));
            }
        }

        $data['formData'] = [];
        $custom_fields    = get_custom_fields('leads', 'type != "link"');

        $cfields       = format_external_form_custom_fields($custom_fields);
        $data['title'] = _l('web_to_lead');

        if ($id != '') {
            $data['form'] = $this->leads_model->get_form([
                'id' => $id,
            ]);
            $data['title']    = $data['form']->name . ' - ' . _l('web_to_lead_form');
            $data['formData'] = $data['form']->form_data;
        }

        $this->load->model('roles_model');
        $data['roles']    = $this->roles_model->get();
        $data['sources']  = $this->leads_model->get_source();
        $data['statuses'] = $this->leads_model->get_status();

        $data['members'] = $this->staff_model->get('', [
            'active'       => 1,
            'is_not_staff' => 0,
        ]);

		$this->db->select('userid, company');
		$this->db->where('active', 1);
		$data['clients'] = $this->db->get(db_prefix() . 'clients')->result_array();
		
        $data['languages'] = $this->app->get_available_languages();
        $data['cfields']   = $cfields;

        $db_fields = [];
        $fields    = [
            'name',
            'title',
            'email',
            'phonenumber',
            'lead_value',
            'company',
            'address',
            'city',
            'state',
            'country',
            'zip',
            'description',
            'website',
        ];

        $fields = hooks()->apply_filters('lead_form_available_database_fields', $fields);

        $className = 'form-control';

        foreach ($fields as $f) {
            $_field_object = new stdClass();
            $type          = 'text';
            $subtype       = '';
            if ($f == 'email') {
                $subtype = 'email';
            } elseif ($f == 'description' || $f == 'address') {
                $type = 'textarea';
            } elseif ($f == 'country') {
                $type = 'select';
            }

            if ($f == 'name') {
                $label = _l('lead_add_edit_name');
            } elseif ($f == 'email') {
                $label = _l('lead_add_edit_email');
            } elseif ($f == 'phonenumber') {
                $label = _l('lead_add_edit_phonenumber');
            } elseif ($f == 'lead_value') {
                $label = _l('lead_add_edit_lead_value');
                $type  = 'number';
            } else {
                $label = _l('lead_' . $f);
            }

            $field_array = [
                'subtype'   => $subtype,
                'type'      => $type,
                'label'     => $label,
                'className' => $className,
                'name'      => $f,
            ];

            if ($f == 'country') {
                $field_array['values'] = [];

                $field_array['values'][] = [
                    'label'    => '',
                    'value'    => '',
                    'selected' => false,
                ];

                $countries = get_all_countries();
                foreach ($countries as $country) {
                    $selected = false;
                    if (get_option('customer_default_country') == $country['country_id']) {
                        $selected = true;
                    }
                    array_push($field_array['values'], [
                        'label'    => $country['short_name'],
                        'value'    => (int) $country['country_id'],
                        'selected' => $selected,
                    ]);
                }
            }

            if ($f == 'name') {
                $field_array['required'] = true;
            }

            $_field_object->label    = $label;
            $_field_object->name     = $f;
            $_field_object->fields   = [];
            $_field_object->fields[] = $field_array;
            $db_fields[]             = $_field_object;
        }
        $data['bodyclass'] = 'web-to-lead-form';
        $data['db_fields'] = $db_fields;
        $this->load->view('admin/leads/formbuilder', $data);
    }

    public function forms($id = '')
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }

        if ($this->input->is_ajax_request()) {
            $this->app->get_table_data('web_to_lead');
        }

        $data['title'] = _l('web_to_lead');
        $this->load->view('admin/leads/forms', $data);
    }

    public function delete_form($id)
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }

        $success = $this->leads_model->delete_form($id);
        if ($success) {
            set_alert('success', _l('deleted', _l('web_to_lead_form')));
        }

        redirect(admin_url('leads/forms'));
    }

    // Sources
    /* Manage leads sources */
    public function sources()
    {
        if (!is_admin()) {
            access_denied('Leads Sources');
        }
        $data['sources'] = $this->leads_model->get_source();
        $data['title']   = 'Leads sources';
        $this->load->view('admin/leads/manage_sources', $data);
    }

    /* Add or update leads sources */
    public function source()
    {
        if (!is_admin() && get_option('staff_members_create_inline_lead_source') == '0') {
            access_denied('Leads Sources');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }

                $id = $this->leads_model->add_source($data);

                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('lead_source')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->leads_model->update_source($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('lead_source')));
                }
            }
        }
    }

    /* Delete leads source */
    public function delete_source($id)
    {
        if (!is_admin()) {
            access_denied('Delete Lead Source');
        }
        if (!$id) {
            redirect(admin_url('leads/sources'));
        }
        $response = $this->leads_model->delete_source($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_source_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('lead_source')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_source_lowercase')));
        }
        redirect(admin_url('leads/sources'));
    }

    // Statuses
    /* View leads statuses */
    public function statuses()
    {
        if (!is_admin()) {
            access_denied('Leads Statuses');
        }
        $data['statuses'] = $this->leads_model->get_status();
        $data['title']    = 'Leads statuses';
        $this->load->view('admin/leads/manage_statuses', $data);
    }

    /* Add or update leads status */
    public function status()
    {
        if (!is_admin() && get_option('staff_members_create_inline_lead_status') == '0') {
            access_denied('Leads Statuses');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }
                $id = $this->leads_model->add_status($data);
                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('lead_status')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->leads_model->update_status($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('lead_status')));
                }
            }
        }
    }

    /* Delete leads status from databae */
    public function delete_status($id)
    {
        if (!is_admin()) {
            access_denied('Leads Statuses');
        }
        if (!$id) {
            redirect(admin_url('leads/statuses'));
        }
        $response = $this->leads_model->delete_status($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_status_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('lead_status')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_status_lowercase')));
        }
        redirect(admin_url('leads/statuses'));
    }

    /* Add new lead note */
    public function add_note($rel_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($rel_id)) {
            ajax_access_denied();
        }

        if ($this->input->post()) {
            $data = $this->input->post();

            if ($data['contacted_indicator'] == 'yes') {
                $contacted_date         = to_sql_date($data['custom_contact_date'], true);
                $data['date_contacted'] = $contacted_date;
            }

            unset($data['contacted_indicator']);
            unset($data['custom_contact_date']);

            // Causing issues with duplicate ID or if my prefixed file for lead.php is used
            $data['description'] = isset($data['lead_note_description']) ? $data['lead_note_description'] : $data['description'];

            if (isset($data['lead_note_description'])) {
                unset($data['lead_note_description']);
            }

            $note_id = $this->misc_model->add_note($data, 'lead', $rel_id);

            if ($note_id) {
                if (isset($contacted_date)) {
                    $this->db->where('id', $rel_id);
                    $this->db->update(db_prefix() . 'leads', [
                        'lastcontact' => $contacted_date,
                    ]);
                    if ($this->db->affected_rows() > 0) {
                        $this->leads_model->log_lead_activity($rel_id, 'not_lead_activity_contacted', false, serialize([
                            get_staff_full_name(get_staff_user_id()),
                            _dt($contacted_date),
                        ]));
                    }
                }
            }
        }
        echo json_encode(['leadView' => $this->_get_lead_data($rel_id), 'id' => $rel_id]);
    }

    public function email_integration_folders()
    {
        if (!is_admin()) {
            ajax_access_denied('Leads Test Email Integration');
        }

        app_check_imap_open_function();

        $imap = new Imap(
            $this->input->post('email'),
            $this->input->post('password', false),
            $this->input->post('imap_server'),
            $this->input->post('encryption')
        );

        try {
            echo json_encode($imap->getSelectableFolders());
        } catch (ConnectionErrorException $e) {
            echo json_encode([
                'alert_type' => 'warning',
                'message'    => $e->getMessage(),
            ]);
        }
    }

    public function test_email_integration()
    {
        if (!is_admin()) {
            access_denied('Leads Test Email Integration');
        }

        app_check_imap_open_function(admin_url('leads/email_integration'));

        $mail     = $this->leads_model->get_email_integration();
        $password = $mail->password;

        if (false == $this->encryption->decrypt($password)) {
            set_alert('danger', _l('failed_to_decrypt_password'));
            redirect(admin_url('leads/email_integration'));
        }

        $imap = new Imap(
            $mail->email,
            $this->encryption->decrypt($password),
            $mail->imap_server,
            $mail->encryption
        );

        try {
            $connection = $imap->testConnection();

            try {
                $connection->getMailbox($mail->folder);
                set_alert('success', _l('lead_email_connection_ok'));
            } catch (MailboxDoesNotExistException $e) {
                set_alert('danger', str_replace(["\n", 'Mailbox'], ['<br />', 'Folder'], addslashes($e->getMessage())));
            }
        } catch (ConnectionErrorException $e) {
            $error = str_replace("\n", '<br />', addslashes($e->getMessage()));
            set_alert('danger', _l('lead_email_connection_not_ok') . '<br /><br /><b>' . $error . '</b>');
        }

        redirect(admin_url('leads/email_integration'));
    }

    public function email_integration()
    {
        if (!is_admin()) {
            access_denied('Leads Email Intregration');
        }
        if ($this->input->post()) {
            $data             = $this->input->post();
            $data['password'] = $this->input->post('password', false);

            if (isset($data['fakeusernameremembered'])) {
                unset($data['fakeusernameremembered']);
            }
            if (isset($data['fakepasswordremembered'])) {
                unset($data['fakepasswordremembered']);
            }

            $success = $this->leads_model->update_email_integration($data);
            if ($success) {
                set_alert('success', _l('leads_email_integration_updated'));
            }
            redirect(admin_url('leads/email_integration'));
        }
        $data['roles']    = $this->roles_model->get();
        $data['sources']  = $this->leads_model->get_source();
        $data['statuses'] = $this->leads_model->get_status();

        $data['members'] = $this->staff_model->get('', [
            'active'       => 1,
            'is_not_staff' => 0,
        ]);

        $data['title'] = _l('leads_email_integration');
        $data['mail']  = $this->leads_model->get_email_integration();

        $data['bodyclass'] = 'leads-email-integration';
        $this->load->view('admin/leads/email_integration', $data);
    }

    public function change_status_color()
    {
        if ($this->input->post() && is_admin()) {
            $this->leads_model->change_status_color($this->input->post());
        }
    }

    public function import()
    {
        if (!is_admin() && get_option('allow_non_admin_members_to_import_leads') != '1') {
            access_denied('Leads Import');
        }

        $dbFields = $this->db->list_fields(db_prefix() . 'leads');
        array_push($dbFields, 'tags');

        $this->load->library('import/import_leads', [], 'import');
        $this->import->setDatabaseFields($dbFields)
        ->setCustomFields(get_custom_fields('leads'));

        if ($this->input->post('download_sample') === 'true') {
            $this->import->downloadSample();
        }

        if ($this->input->post()
            && isset($_FILES['file_csv']['name']) && $_FILES['file_csv']['name'] != '') {
            $this->import->setSimulation($this->input->post('simulate'))
                          ->setTemporaryFileLocation($_FILES['file_csv']['tmp_name'])
                          ->setFilename($_FILES['file_csv']['name'])
                          ->perform();

            $data['total_rows_post'] = $this->import->totalRows();

            if (!$this->import->isSimulation()) {
                set_alert('success', _l('import_total_imported', $this->import->totalImported()));
            }
        }

        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();
        $data['members']  = $this->staff_model->get('', ['is_not_staff' => 0, 'active' => 1]);

        $data['title'] = _l('import');
        $this->load->view('admin/leads/import', $data);
    }

    public function validate_unique_field()
    {
        if ($this->input->post()) {

            // First we need to check if the field is the same
            $lead_id = $this->input->post('lead_id');
            $field   = $this->input->post('field');
            $value   = $this->input->post($field);

            if ($lead_id != '') {
                $this->db->select($field);
                $this->db->where('id', $lead_id);
                $row = $this->db->get(db_prefix() . 'leads')->row();
                if ($row->{$field} == $value) {
                    echo json_encode(true);
                    die();
                }
            }

            echo total_rows(db_prefix() . 'leads', [ $field => $value ]) > 0 ? 'false' : 'true';
        }
    }

    public function bulk_action()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        hooks()->do_action('before_do_bulk_action_for_leads');
        $total_deleted = 0;
        if ($this->input->post()) {
            $ids                   = $this->input->post('ids');
            $status                = $this->input->post('status');
            $source                = $this->input->post('source');
            $assigned              = $this->input->post('assigned');
            $visibility            = $this->input->post('visibility');
            $tags                  = $this->input->post('tags');
            $last_contact          = $this->input->post('last_contact');
            $lost                  = $this->input->post('lost');
            $has_permission_delete = staff_can('delete',  'leads');
            if (is_array($ids)) {
                foreach ($ids as $id) {
                    if ($this->input->post('mass_delete')) {
                        if ($has_permission_delete) {
                            if ($this->leads_model->delete($id)) {
                                $total_deleted++;
                            }
                        }
                    } else {
                        if ($status || $source || $assigned || $last_contact || $visibility) {
                            $update = [];
                            if ($status) {
                                // We will use the same function to update the status
                                $this->leads_model->update_lead_status([
                                    'status' => $status,
                                    'leadid' => $id,
                                ]);
                            }
                            if ($source) {
                                $update['source'] = $source;
                            }
                            if ($assigned) {
                                $update['assigned'] = $assigned;
                            }
                            if ($last_contact) {
                                $last_contact          = to_sql_date($last_contact, true);
                                $update['lastcontact'] = $last_contact;
                            }

                            if ($visibility) {
                                if ($visibility == 'public') {
                                    $update['is_public'] = 1;
                                } else {
                                    $update['is_public'] = 0;
                                }
                            }

                            if (count($update) > 0) {
                                $this->db->where('id', $id);
                                $this->db->update(db_prefix() . 'leads', $update);
                            }
                        }
                        if ($tags) {
                            handle_tags_save($tags, $id, 'lead');
                        }
                        if ($lost == 'true') {
                            $this->leads_model->mark_as_lost($id);
                        }
                    }
                }
            }
        }

        if ($this->input->post('mass_delete')) {
            set_alert('success', _l('total_leads_deleted', $total_deleted));
        }
    }

    public function download_files($lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }

        $files = $this->leads_model->get_lead_attachments($lead_id);

        if (count($files) == 0) {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        }

        $path = get_upload_path_by_type('lead') . $lead_id;

        $this->load->library('zip');

        foreach ($files as $file) {
            $this->zip->read_file($path . '/' . $file['file_name']);
        }

        $this->zip->download('files.zip');
        $this->zip->clear_data();
    }
	
	public function get_column_order()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $staff_id = get_staff_user_id();
        $columns = $this->get_table_columns();
        $saved_order = $this->db->get_where(db_prefix() . 'leads_table_order', ['staff_id' => $staff_id])->row();

        $column_order = $saved_order ? json_decode($saved_order->column_order, true) : array_column($columns, 'id');

        // Validate saved order
        $valid_order = array_intersect($column_order, array_column($columns, 'id'));
        if (count($valid_order) !== count($column_order)) {
            $column_order = array_column($columns, 'id');
        }

        echo json_encode([
            'success' => true,
            'columns' => $columns,
            'column_order' => $column_order,
        ]);
    }

    public function save_column_order()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $staff_id = get_staff_user_id();
        $column_order = $this->input->post('column_order', false);

        if (!$column_order || !is_array($column_order)) {
            echo json_encode(['success' => false, 'message' => _l('leads_table_order_invalid_data')]);
            return;
        }

        // Validate column IDs
        $valid_columns = array_column($this->get_table_columns(), 'id');
        $invalid_columns = array_diff($column_order, $valid_columns);
        if (!empty($invalid_columns)) {
            echo json_encode(['success' => false, 'message' => _l('leads_table_order_invalid_columns')]);
            return;
        }

        $data = [
            'staff_id' => $staff_id,
            'column_order' => json_encode($column_order),
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        $this->db->where('staff_id', $staff_id);
        if ($this->db->get(db_prefix() . 'leads_table_order')->num_rows() > 0) {
            $this->db->update(db_prefix() . 'leads_table_order', $data);
        } else {
            $this->db->insert(db_prefix() . 'leads_table_order', $data);
        }

        echo json_encode(['success' => true, 'message' => _l('leads_table_order_updated')]);
    }

    private function get_table_columns()
    {
        $columns = [
            ['id' => 'name', 'name' => _l('leads_dt_name')],
            ['id' => 'date-created', 'name' => _l('leads_dt_datecreated')],
            ['id' => 'phone', 'name' => _l('leads_dt_phonenumber')],
            ['id' => 'number', 'name' => _l('the_number_sign')],
            ['id' => 'email', 'name' => _l('leads_dt_email')],
            ['id' => 'last-contact', 'name' => _l('leads_dt_last_contact')],
            ['id' => 'tags', 'name' => _l('tags')],
            ['id' => 'company', 'name' => _l('lead_company')],
            ['id' => 'checkbox', 'name' => _l('checkbox')],
            ['id' => 'lead-value', 'name' => _l('leads_dt_lead_value')],
            ['id' => 'assigned', 'name' => _l('leads_dt_assigned')],
            ['id' => 'status', 'name' => _l('leads_dt_status')],
            ['id' => 'source', 'name' => _l('leads_source')],
        ];

        if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
            $columns[] = ['id' => 'consent', 'name' => _l('gdpr_consent')];
        }

        $custom_fields = get_custom_fields('leads', ['show_on_table' => 1]);
        foreach ($custom_fields as $field) {
            $columns[] = ['id' => 'custom_field_' . $field['id'], 'name' => $field['name']];
        }

        return hooks()->apply_filters('leads_table_order_columns', $columns);
    }
}








// leads Model 

<?php

use app\services\AbstractKanban;

defined('BASEPATH') or exit('No direct script access allowed');

class Leads_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get lead
     * @param  string $id Optional - leadid
     * @return mixed
     */
    public function get($id = '', $where = [])
    {
        $this->db->select('*,' . db_prefix() . 'leads.name, ' . db_prefix() . 'leads.id,' . db_prefix() . 'leads_status.name as status_name,' . db_prefix() . 'leads_sources.name as source_name');
        $this->db->join(db_prefix() . 'leads_status', db_prefix() . 'leads_status.id=' . db_prefix() . 'leads.status', 'left');
        $this->db->join(db_prefix() . 'leads_sources', db_prefix() . 'leads_sources.id=' . db_prefix() . 'leads.source', 'left');

        $this->db->where($where);
        if (is_numeric($id)) {
            $this->db->where(db_prefix() . 'leads.id', $id);
            $lead = $this->db->get(db_prefix() . 'leads')->row();
            if ($lead) {
                if ($lead->from_form_id != 0) {
                    $lead->form_data = $this->get_form([
                        'id' => $lead->from_form_id,
                    ]);
                }
                $lead->attachments = $this->get_lead_attachments($id);
                $lead->public_url  = leads_public_url($id);
            }

            return $lead;
        }

        return $this->db->get(db_prefix() . 'leads')->result_array();
    }

    /**
     * Get lead by given email
     *
     * @since 2.8.0
     *
     * @param  string $email
     *
     * @return \strClass|null
     */
    public function get_lead_by_email($email)
    {
        $this->db->where('email', $email);
        $this->db->limit(1);

        return $this->db->get('leads')->row();
    }

    /**
     * Add new lead to database
     * @param mixed $data lead data
     * @return mixed false || leadid
     */
    public function add($data)
    {
        if (isset($data['custom_contact_date']) || isset($data['custom_contact_date'])) {
            if (isset($data['contacted_today'])) {
                $data['lastcontact'] = date('Y-m-d H:i:s');
                unset($data['contacted_today']);
            } else {
                $data['lastcontact'] = to_sql_date($data['custom_contact_date'], true);
            }
        }

        if (isset($data['is_public']) && ($data['is_public'] == 1 || $data['is_public'] === 'on')) {
            $data['is_public'] = 1;
        } else {
            $data['is_public'] = 0;
        }

        if (!isset($data['country']) || isset($data['country']) && $data['country'] == '') {
            $data['country'] = 0;
        }

        if (isset($data['custom_contact_date'])) {
            unset($data['custom_contact_date']);
        }

        $data['description'] = nl2br($data['description']);
        $data['dateadded']   = date('Y-m-d H:i:s');
        $data['addedfrom']   = get_staff_user_id();

        $data = hooks()->apply_filters('before_lead_added', $data);

        $tags = '';
        if (isset($data['tags'])) {
            $tags = $data['tags'];
            unset($data['tags']);
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            unset($data['custom_fields']);
        }

        $data['address'] = trim($data['address']);
        $data['address'] = nl2br($data['address']);

        $data['email'] = trim($data['email']);
        $this->db->insert(db_prefix() . 'leads', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Lead Added [ID: ' . $insert_id . ']');
            $this->log_lead_activity($insert_id, 'not_lead_activity_created');

            handle_tags_save($tags, $insert_id, 'lead');

            if (isset($custom_fields)) {
                handle_custom_fields_post($insert_id, $custom_fields);
            }

            $this->lead_assigned_member_notification($insert_id, $data['assigned']);
            hooks()->do_action('lead_created', $insert_id);

            return $insert_id;
        }

        return false;
    }

    public function lead_assigned_member_notification($lead_id, $assigned, $integration = false)
    {
        if (empty($assigned) || $assigned == 0) {
            return;
        }

        if ($integration == false) {
            if ($assigned == get_staff_user_id()) {
                return false;
            }
        }

        $name = $this->db->select('name')->from(db_prefix() . 'leads')->where('id', $lead_id)->get()->row()->name;

        $notification_data = [
            'description'     => ($integration == false) ? 'not_assigned_lead_to_you' : 'not_lead_assigned_from_form',
            'touserid'        => $assigned,
            'link'            => '#leadid=' . $lead_id,
            'additional_data' => ($integration == false ? serialize([
                $name,
            ]) : serialize([])),
        ];

        if ($integration != false) {
            $notification_data['fromcompany'] = 1;
        }

        if (add_notification($notification_data)) {
            pusher_trigger_notification([$assigned]);
        }

        $this->db->select('email');
        $this->db->where('staffid', $assigned);
        $email = $this->db->get(db_prefix() . 'staff')->row()->email;

        send_mail_template('lead_assigned', $lead_id, $email);

        $this->db->where('id', $lead_id);
        $this->db->update(db_prefix() . 'leads', [
            'dateassigned' => date('Y-m-d'),
        ]);

        $not_additional_data = [
            e(get_staff_full_name()),
            '<a href="' . admin_url('profile/' . $assigned) . '" target="_blank">' . e(get_staff_full_name($assigned)) . '</a>',
        ];

        if ($integration == true) {
            unset($not_additional_data[0]);
            array_values(($not_additional_data));
        }

        $not_additional_data = serialize($not_additional_data);

        $not_desc = ($integration == false ? 'not_lead_activity_assigned_to' : 'not_lead_activity_assigned_from_form');
        $this->log_lead_activity($lead_id, $not_desc, $integration, $not_additional_data);

        hooks()->do_action('after_lead_assigned_member_notification_sent', $lead_id);
    }

    /**
     * Update lead
     * @param  array $data lead data
     * @param  mixed $id   leadid
     * @return boolean
     */
    public function update($data, $id)
    {
        $current_lead_data = $this->get($id);
        $current_status    = $this->get_status($current_lead_data->status);
        if ($current_status) {
            $current_status_id = $current_status->id;
            $current_status    = $current_status->name;
        } else {
            if ($current_lead_data->junk == 1) {
                $current_status = _l('lead_junk');
            } elseif ($current_lead_data->lost == 1) {
                $current_status = _l('lead_lost');
            } else {
                $current_status = '';
            }
            $current_status_id = 0;
        }

        $affectedRows = 0;
        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            if (handle_custom_fields_post($id, $custom_fields)) {
                $affectedRows++;
            }
            unset($data['custom_fields']);
        }
        if (!defined('API')) {
            if (isset($data['is_public'])) {
                $data['is_public'] = 1;
            } else {
                $data['is_public'] = 0;
            }

            if (!isset($data['country']) || isset($data['country']) && $data['country'] == '') {
                $data['country'] = 0;
            }

            if (isset($data['description'])) {
                $data['description'] = nl2br($data['description']);
            }
        }

        if (isset($data['lastcontact']) && $data['lastcontact'] == '' || isset($data['lastcontact']) && $data['lastcontact'] == null) {
            $data['lastcontact'] = null;
        } elseif (isset($data['lastcontact'])) {
            $data['lastcontact'] = to_sql_date($data['lastcontact'], true);
        }

        if (isset($data['tags'])) {
            if (handle_tags_save($data['tags'], $id, 'lead')) {
                $affectedRows++;
            }
            unset($data['tags']);
        }

        if (isset($data['remove_attachments'])) {
            foreach ($data['remove_attachments'] as $key => $val) {
                $attachment = $this->get_lead_attachments($id, $key);
                if ($attachment) {
                    $this->delete_lead_attachment($attachment->id);
                }
            }
            unset($data['remove_attachments']);
        }

        $data['address'] = trim($data['address']);
        $data['address'] = nl2br($data['address']);

        $data['email'] = trim($data['email']);

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', $data);
        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
            if (isset($data['status']) && $current_status_id != $data['status']) {
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'leads', [
                    'last_status_change' => date('Y-m-d H:i:s'),
                ]);
                $new_status_name = $this->get_status($data['status'])->name;
                $this->log_lead_activity($id, 'not_lead_activity_status_updated', false, serialize([
                    get_staff_full_name(),
                    $current_status,
                    $new_status_name,
                ]));

                hooks()->do_action('lead_status_changed', [
                    'lead_id'    => $id,
                    'old_status' => $current_status_id,
                    'new_status' => $data['status'],
                ]);
            }

            if (($current_lead_data->junk == 1 || $current_lead_data->lost == 1) && $data['status'] != 0) {
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'leads', [
                    'junk' => 0,
                    'lost' => 0,
                ]);
            }

            if (isset($data['assigned'])) {
                if ($current_lead_data->assigned != $data['assigned'] && (!empty($data['assigned']) && $data['assigned'] != 0)) {
                    $this->lead_assigned_member_notification($id, $data['assigned']);
                }
            }
            log_activity('Lead Updated [ID: ' . $id . ']');

            hooks()->do_action('after_lead_updated', $id);

            return true;
        }
        if ($affectedRows > 0) {
            hooks()->do_action('after_lead_updated', $id);
            return true;
        }

        return false;
    }

    /**
     * Delete lead from database and all connections
     * @param  mixed $id leadid
     * @return boolean
     */
    public function delete($id)
    {
        $affectedRows = 0;

        hooks()->do_action('before_lead_deleted', $id);

        $lead = $this->get($id);

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads');
        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Deleted [Deleted by: ' . get_staff_full_name() . ', ID: ' . $id . ']');

            $attachments = $this->get_lead_attachments($id);
            foreach ($attachments as $attachment) {
                $this->delete_lead_attachment($attachment['id']);
            }

            // Delete the custom field values
            $this->db->where('relid', $id);
            $this->db->where('fieldto', 'leads');
            $this->db->delete(db_prefix() . 'customfieldsvalues');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_activity_log');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_integration_emails');

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $this->db->delete(db_prefix() . 'notes');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'reminders');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'taggables');

            $this->load->model('proposals_model');
            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $proposals = $this->db->get(db_prefix() . 'proposals')->result_array();

            foreach ($proposals as $proposal) {
                $this->proposals_model->delete($proposal['id']);
            }

            // Get related tasks
            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $tasks = $this->db->get(db_prefix() . 'tasks')->result_array();
            foreach ($tasks as $task) {
                $this->tasks_model->delete_task($task['id']);
            }

            if (is_gdpr()) {
                $this->db->where('(description LIKE "%' . $lead->email . '%" OR description LIKE "%' . $lead->name . '%" OR description LIKE "%' . $lead->phonenumber . '%")');
                $this->db->delete(db_prefix() . 'activity_log');
            }

            $affectedRows++;
        }
        if ($affectedRows > 0) {
            hooks()->do_action('after_lead_deleted', $id);
            return true;
        }

        return false;
    }

    /**
     * Mark lead as lost
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_lost($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_lost');

            log_activity('Lead Marked as Lost [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_lost', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as lost
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_lost($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_lost');

            log_activity('Lead Unmarked as Lost [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Mark lead as junk
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_junk($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_junk');

            log_activity('Lead Marked as Junk [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_junk', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as junk
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_junk($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_junk');
            log_activity('Lead Unmarked as Junk [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Get lead attachments
     * @since Version 1.0.4
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_attachments($id = '', $attachment_id = '', $where = [])
    {
        $this->db->where($where);
        $idIsHash = !is_numeric($attachment_id) && strlen($attachment_id) == 32;
        if (is_numeric($attachment_id) || $idIsHash) {
            $this->db->where($idIsHash ? 'attachment_key' : 'id', $attachment_id);

            return $this->db->get(db_prefix() . 'files')->row();
        }
        $this->db->where('rel_id', $id);
        $this->db->where('rel_type', 'lead');
        $this->db->order_by('dateadded', 'DESC');

        return $this->db->get(db_prefix() . 'files')->result_array();
    }

    public function add_attachment_to_database($lead_id, $attachment, $external = false, $form_activity = false)
    {
        $this->misc_model->add_attachment_to_database($lead_id, 'lead', $attachment, $external);

        if ($form_activity == false) {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_added_attachment');
        } else {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_log_attachment', true, serialize([
                $form_activity,
            ]));
        }

        // No notification when attachment is imported from web to lead form
        if ($form_activity == false) {
            $lead         = $this->get($lead_id);
            $not_user_ids = [];
            if ($lead->addedfrom != get_staff_user_id()) {
                array_push($not_user_ids, $lead->addedfrom);
            }
            if ($lead->assigned != get_staff_user_id() && $lead->assigned != 0) {
                array_push($not_user_ids, $lead->assigned);
            }
            $notifiedUsers = [];
            foreach ($not_user_ids as $uid) {
                $notified = add_notification([
                    'description'     => 'not_lead_added_attachment',
                    'touserid'        => $uid,
                    'link'            => '#leadid=' . $lead_id,
                    'additional_data' => serialize([
                        $lead->name,
                    ]),
                ]);
                if ($notified) {
                    array_push($notifiedUsers, $uid);
                }
            }
            pusher_trigger_notification($notifiedUsers);
        }
    }

    /**
     * Delete lead attachment
     * @param  mixed $id attachment id
     * @return boolean
     */
    public function delete_lead_attachment($id)
    {
        $attachment = $this->get_lead_attachments('', $id);
        $deleted    = false;

        if ($attachment) {
            if (empty($attachment->external)) {
                unlink(get_upload_path_by_type('lead') . $attachment->rel_id . '/' . $attachment->file_name);
            }
            $this->db->where('id', $attachment->id);
            $this->db->delete(db_prefix() . 'files');
            if ($this->db->affected_rows() > 0) {
                $deleted = true;
                log_activity('Lead Attachment Deleted [ID: ' . $attachment->rel_id . ']');
            }

            if (is_dir(get_upload_path_by_type('lead') . $attachment->rel_id)) {
                // Check if no attachments left, so we can delete the folder also
                $other_attachments = list_files(get_upload_path_by_type('lead') . $attachment->rel_id);
                if (count($other_attachments) == 0) {
                    // okey only index.html so we can delete the folder also
                    delete_dir(get_upload_path_by_type('lead') . $attachment->rel_id);
                }
            }
        }

        return $deleted;
    }

    // Sources

    /**
     * Get leads sources
     * @param  mixed $id Optional - Source ID
     * @return mixed object if id passed else array
     */
    public function get_source($id = false)
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);

            return $this->db->get(db_prefix() . 'leads_sources')->row();
        }

        $this->db->order_by('name', 'asc');

        return $this->db->get(db_prefix() . 'leads_sources')->result_array();
    }

    /**
     * Add new lead source
     * @param mixed $data source data
     */
    public function add_source($data)
    {
        $this->db->insert(db_prefix() . 'leads_sources', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Source Added [SourceID: ' . $insert_id . ', Name: ' . $data['name'] . ']');
        }

        return $insert_id;
    }

    /**
     * Update lead source
     * @param  mixed $data source data
     * @param  mixed $id   source id
     * @return boolean
     */
    public function update_source($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_sources', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Source Updated [SourceID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead source from database
     * @param  mixed $id source id
     * @return mixed
     */
    public function delete_source($id)
    {
        $current = $this->get_source($id);
        // Check if is already using in table
        if (is_reference_in_table('source', db_prefix() . 'leads', $id) || is_reference_in_table('lead_source', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_sources');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_source') == $id) {
                update_option('leads_default_source', '');
            }
            log_activity('Leads Source Deleted [SourceID: ' . $id . ']');

            return true;
        }

        return false;
    }

    // Statuses

    /**
     * Get lead statuses
     * @param  mixed $id status id
     * @return mixed      object if id passed else array
     */
    public function get_status($id = '', $where = [])
    {
        if (is_numeric($id)) {
            $this->db->where($where);
            $this->db->where('id', $id);

            return $this->db->get(db_prefix() . 'leads_status')->row();
        }

        $whereKey = md5(serialize($where));
      
        $statuses = $this->app_object_cache->get('leads-all-statuses-'.$whereKey);

        if (!$statuses) {
            $this->db->where($where);
            $this->db->order_by('statusorder', 'asc');

            $statuses = $this->db->get(db_prefix() . 'leads_status')->result_array();
            $this->app_object_cache->add('leads-all-statuses-'.$whereKey, $statuses);
        }

        return $statuses;
    }

    /**
     * Add new lead status
     * @param array $data lead status data
     */
    public function add_status($data)
    {
        if (isset($data['color']) && $data['color'] == '') {
            $data['color'] = hooks()->apply_filters('default_lead_status_color', '#757575');
        }

        if (!isset($data['statusorder'])) {
            $data['statusorder'] = total_rows(db_prefix() . 'leads_status') + 1;
        }

        $this->db->insert(db_prefix() . 'leads_status', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Status Added [StatusID: ' . $insert_id . ', Name: ' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_status($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_status', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Status Updated [StatusID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead status from database
     * @param  mixed $id status id
     * @return boolean
     */
    public function delete_status($id)
    {
        $current = $this->get_status($id);
        // Check if is already using in table
        if (is_reference_in_table('status', db_prefix() . 'leads', $id) || is_reference_in_table('lead_status', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_status');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_status') == $id) {
                update_option('leads_default_status', '');
            }
            log_activity('Leads Status Deleted [StatusID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Update canban lead status when drag and drop
     * @param  array $data lead data
     * @return boolean
     */
    public function update_lead_status($data)
    {
        $this->db->select('status');
        $this->db->where('id', $data['leadid']);
        $_old = $this->db->get(db_prefix() . 'leads')->row();

        $old_status = '';

        if ($_old) {
            $old_status = $this->get_status($_old->status);
            if ($old_status) {
                $old_status = $old_status->name;
            }
        }

        $affectedRows   = 0;
        $current_status = $this->get_status($data['status'])->name;

        $this->db->where('id', $data['leadid']);
        $this->db->update(db_prefix() . 'leads', [
            'status' => $data['status'],
        ]);

        $_log_message = '';

        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
            if ($current_status != $old_status && $old_status != '') {
                $_log_message    = 'not_lead_activity_status_updated';
                $additional_data = serialize([
                    get_staff_full_name(),
                    $old_status,
                    $current_status,
                ]);

                hooks()->do_action('lead_status_changed', [
                    'lead_id'    => $data['leadid'],
                    'old_status' => $old_status,
                    'new_status' => $current_status,
                ]);
            }
            $this->db->where('id', $data['leadid']);
            $this->db->update(db_prefix() . 'leads', [
                'last_status_change' => date('Y-m-d H:i:s'),
            ]);
        }

        if (isset($data['order'])) {
            AbstractKanban::updateOrder($data['order'], 'leadorder', 'leads', $data['status']);
        }

        if ($affectedRows > 0) {
            if ($_log_message == '') {
                return true;
            }

            $this->log_lead_activity($data['leadid'], $_log_message, false, $additional_data);

            return true;
        }

        return false;
    }

    /* Ajax */

    /**
     * All lead activity by staff
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_activity_log($id)
    {
        $sorting = hooks()->apply_filters('lead_activity_log_default_sort', 'ASC');

        $this->db->where('leadid', $id);
        $this->db->order_by('date', $sorting);

        return $this->db->get(db_prefix() . 'lead_activity_log')->result_array();
    }

    public function staff_can_access_lead($id, $staff_id = '')
    {
        $staff_id = $staff_id == '' ? get_staff_user_id() : $staff_id;

        if (has_permission('leads', $staff_id, 'view')) {
            return true;
        }

        $CI = &get_instance();

        if (total_rows(db_prefix() . 'leads', 'id="' . $CI->db->escape_str($id) . '" AND (assigned=' . $CI->db->escape_str($staff_id) . ' OR is_public=1 OR addedfrom=' . $CI->db->escape_str($staff_id) . ')') > 0) {
            return true;
        }

        return false;
    }

    /**
     * Add lead activity from staff
     * @param  mixed  $id          lead id
     * @param  string  $description activity description
     */
    public function log_lead_activity($id, $description, $integration = false, $additional_data = '')
    {
        $log = [
            'date'            => date('Y-m-d H:i:s'),
            'description'     => $description,
            'leadid'          => $id,
            'staffid'         => get_staff_user_id(),
            'additional_data' => $additional_data,
            'full_name'       => get_staff_full_name(get_staff_user_id()),
        ];
        if ($integration == true) {
            $log['staffid']   = 0;
            $log['full_name'] = '[CRON]';
        }

        $this->db->insert(db_prefix() . 'lead_activity_log', $log);

        return $this->db->insert_id();
    }

    /**
     * Get email integration config
     * @return object
     */
    public function get_email_integration()
    {
        $this->db->where('id', 1);

        return $this->db->get(db_prefix() . 'leads_email_integration')->row();
    }

    /**
     * Get lead imported email activity
     * @param  mixed $id leadid
     * @return array
     */
    public function get_mail_activity($id)
    {
        $this->db->where('leadid', $id);
        $this->db->order_by('dateadded', 'asc');

        return $this->db->get(db_prefix() . 'lead_integration_emails')->result_array();
    }

    /**
     * Update email integration config
     * @param  mixed $data All $_POST data
     * @return boolean
     */
    public function update_email_integration($data)
    {
        $this->db->where('id', 1);
        $original_settings = $this->db->get(db_prefix() . 'leads_email_integration')->row();

        $data['create_task_if_customer']        = isset($data['create_task_if_customer']) ? 1 : 0;
        $data['active']                         = isset($data['active']) ? 1 : 0;
        $data['delete_after_import']            = isset($data['delete_after_import']) ? 1 : 0;
        $data['notify_lead_imported']           = isset($data['notify_lead_imported']) ? 1 : 0;
        $data['only_loop_on_unseen_emails']     = isset($data['only_loop_on_unseen_emails']) ? 1 : 0;
        $data['notify_lead_contact_more_times'] = isset($data['notify_lead_contact_more_times']) ? 1 : 0;
        $data['mark_public']                    = isset($data['mark_public']) ? 1 : 0;
        $data['responsible']                    = !isset($data['responsible']) ? 0 : $data['responsible'];

        if ($data['notify_lead_contact_more_times'] != 0 || $data['notify_lead_imported'] != 0) {
            if (isset($data['notify_type']) && $data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        // Check if not empty $data['password']
        // Get original
        // Decrypt original
        // Compare with $data['password']
        // If equal unset
        // If not encrypt and save
        if (!empty($data['password'])) {
            $or_decrypted = $this->encryption->decrypt($original_settings->password);
            if ($or_decrypted == $data['password']) {
                unset($data['password']);
            } else {
                $data['password'] = $this->encryption->encrypt($data['password']);
            }
        }

        $this->db->where('id', 1);
        $this->db->update(db_prefix() . 'leads_email_integration', $data);
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    public function change_status_color($data)
    {
        $this->db->where('id', $data['status_id']);
        $this->db->update(db_prefix() . 'leads_status', [
            'color' => $data['color'],
        ]);
    }

    public function update_status_order($data)
    {
        foreach ($data['order'] as $status) {
            $this->db->where('id', $status[0]);
            $this->db->update(db_prefix() . 'leads_status', [
                'statusorder' => $status[1],
            ]);
        }
    }

    public function get_form($where)
    {
        $this->db->where($where);

        return $this->db->get(db_prefix() . 'web_to_lead')->row();
    }

    public function add_form($data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);
        $data['form_key']           = app_generate_hash();

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $data['dateadded'] = date('Y-m-d H:i:s');

        $this->db->insert(db_prefix() . 'web_to_lead', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Web to Lead Form Added [' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_form($id, $data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'web_to_lead', $data);

        return ($this->db->affected_rows() > 0 ? true : false);
    }

    public function delete_form($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'web_to_lead');

        $this->db->where('from_form_id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'from_form_id' => 0,
        ]);

        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Form Deleted [' . $id . ']');

            return true;
        }

        return false;
    }

    private function _do_lead_web_to_form_responsibles($data)
    {
        if (isset($data['notify_lead_imported'])) {
            $data['notify_lead_imported'] = 1;
        } else {
            $data['notify_lead_imported'] = 0;
        }

        if ($data['responsible'] == '') {
            $data['responsible'] = 0;
        }
        if ($data['notify_lead_imported'] != 0) {
            if ($data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        return $data;
    }

    public function do_kanban_query($status, $search = '', $page = 1, $sort = [], $count = false)
    {
        _deprecated_function('Leads_model::do_kanban_query', '2.9.2', 'LeadsKanban class');

        $kanBan = (new LeadsKanban($status))
            ->search($search)
            ->page($page)
            ->sortBy($sort['sort'] ?? null, $sort['sort_by'] ?? null);

        if ($count) {
            return $kanBan->countAll();
        }

        return $kanBan->get();
    }
}




└── leads
    ├── _kan_ban_card.php
    ├── convert_to_customer.php
    ├── email_integration.php
    ├── formbuilder.php
    ├── forms.php
    ├── import.php
    ├── index.html
    ├── kan-ban.php
    ├── lead.php
    ├── leads_attachments_template.php
    ├── manage_leads.php
    ├── manage_sources.php
    ├── manage_statuses.php
    ├── profile.php
    └── status.php

    this are view files of leads 

<?php defined('BASEPATH') or exit('No direct script access allowed');
$lead_already_client_tooltip = '';
$lead_is_client              = $lead['is_lead_client'] !== '0';
if ($lead_is_client) {
    $lead_already_client_tooltip = ' data-toggle="tooltip" title="' . _l('lead_have_client_profile') . '"';
}
if ($lead['status'] == $status['id']) { ?>
<li data-lead-id="<?= e($lead['id']); ?>" <?= e($lead_already_client_tooltip); ?>
    class="lead-kan-ban<?= $lead['assigned'] == get_staff_user_id() ? ' current-user-lead' : ''; ?><?= $lead_is_client && get_option('lead_lock_after_convert_to_customer') == 1 && ! is_admin() ? ' not-sortable' : ''; ?>">
    <div class="panel-body lead-body">
        <div class="tw-flex lead-name">
            <?php if ($lead['assigned'] != 0) { ?>
            <a href="<?= admin_url('profile/' . $lead['assigned']); ?>"
                data-placement="right" data-toggle="tooltip"
                title="<?= e(get_staff_full_name($lead['assigned'])); ?>"
                class="mtop8 tw-mr-1.5">
                <?= staff_profile_image($lead['assigned'], [
                    'staff-profile-image-xs',
                ]); ?></a>
            <?php } ?>
            <a href="<?= admin_url('leads/index/' . e($lead['id'])); ?>"
                title="#<?= e($lead['id']) . ' - ' . e($lead['lead_name']); ?>"
                onclick="init_lead(<?= e($lead['id']); ?>);return false;"
                class="tw-block tw-min-w-0 tw-font-medium">
                <span class="mtop10 mbot10 tw-truncate tw-block">
                    #<?= e($lead['id']) . ' - ' . e($lead['lead_name']); ?>
                </span>
            </a>
        </div>
        <div class="row">
            <div class="col-md-12">
                <div class="tw-flex">
                    <div class="tw-grow tw-mr-2">
                        <p class="tw-text-sm tw-mb-0">
                            <?= _l('leads_canban_source', $lead['source_name']); ?>
                        </p>
                        <?php $lead_value = $lead['lead_value'] != 0 ? app_format_money($lead['lead_value'], $base_currency->symbol) : '--'; ?>
                        <p class="tw-text-sm tw-mb-0">
                            <?= e(_l('leads_canban_lead_value', $lead_value)); ?>
                        </p>
                    </div>
                    <div class="text-right">
                        <?php if (is_date($lead['lastcontact']) && $lead['lastcontact'] != '0000-00-00 00:00:00') { ?>
                        <small
                            class="text-dark tw-text-sm"><?= _l('leads_dt_last_contact'); ?>
                            <span class="bold">
                                <span class="text-has-action" data-toggle="tooltip"
                                    data-title="<?= e(_dt($lead['lastcontact'])); ?>">
                                    <?= e(time_ago($lead['lastcontact'])); ?>
                                </span>
                            </span>
                        </small><br />
                        <?php } ?>
                        <small
                            class="text-dark"><?= _l('lead_created'); ?>:
                            <span class="bold">
                                <span class="text-has-action" data-toggle="tooltip"
                                    data-title="<?= e(_dt($lead['dateadded'])); ?>">
                                    <?= e(time_ago($lead['dateadded'])); ?>
                                </span>
                            </span>
                        </small><br />
                        <?php hooks()->do_action('before_leads_kanban_card_icons', $lead); ?>
                        <span class="mright5 mtop5 inline-block text-muted" data-toggle="tooltip" data-placement="left"
                            data-title="<?= _l('leads_canban_notes', $lead['total_notes']); ?>">
                            <i class="fa-regular fa-note-sticky"></i>
                            <?= e($lead['total_notes']); ?>
                        </span>
                        <span class="mtop5 inline-block text-muted" data-placement="left" data-toggle="tooltip"
                            data-title="<?= _l('lead_kan_ban_attachments', $lead['total_files']); ?>">
                            <i class="fa fa-paperclip"></i>
                            <?= e($lead['total_files']); ?>
                        </span>
                        <?php hooks()->do_action('after_leads_kanban_card_icons', $lead); ?>
                    </div>
                </div>
            </div>

            <?php if ($lead['tags']) { ?>
            <div class="col-md-12">
                <div class="kanban-tags tw-text-sm tw-inline-flex">
                    <?= render_tags($lead['tags']); ?>
                </div>
            </div>
            <?php } ?>
            <a href="#" class="pull-right text-muted kan-ban-expand-top"
                onclick="slideToggle('#kan-ban-expand-<?= e($lead['id']); ?>'); return false;">
                <i class="fa fa-expand" aria-hidden="true"></i>
            </a>
            <div class="clearfix no-margin"></div>
            <div id="kan-ban-expand-<?= e($lead['id']); ?>"
                class="padding-10" style="display:none;">
                <div class="clearfix"></div>
                <hr class="hr-10" />
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_title'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['title'] != '' ? $lead['title'] : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_add_edit_email'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= $lead['email'] != '' ? '<a href="mailto:' . e($lead['email']) . '">' . e($lead['email']) . '</a>' : '-' ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_website'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= $lead['website'] != '' ? '<a href="' . e(maybe_add_http($lead['website'])) . '" target="_blank">' . e($lead['website']) . '</a>' : '-' ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_add_edit_phonenumber'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= $lead['phonenumber'] != '' ? '<a href="tel:' . e($lead['phonenumber']) . '">' . e($lead['phonenumber']) . '</a>' : '-' ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_company'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['company'] != '' ? $lead['company'] : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_address'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['address'] != '' ? $lead['address'] : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_city'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['city'] != '' ? $lead['city'] : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_state'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['state'] != '' ? $lead['state'] : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_country'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['country'] != 0 ? get_country($lead['country'])->short_name : '-') ?>
                </p>
                <p class="text-muted lead-field-heading">
                    <?= _l('lead_zip'); ?>
                </p>
                <p class="bold tw-text-sm">
                    <?= e($lead['zip'] != '' ? $lead['zip'] : '-') ?>
                </p>
            </div>
        </div>
    </div>
</li>
<?php }
?>






<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<div class="modal fade" id="convert_lead_to_client_modal" tabindex="-1" role="dialog" aria-labelledby="myModalLabel">
    <div class="modal-dialog modal-lg" role="document">
        <?php echo form_open('admin/leads/convert_to_customer', ['id' => 'lead_to_client_form']); ?>
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal"><span
                        aria-hidden="true">&times;</span></button>
                <h4 class="modal-title" id="myModalLabel">
                    <?php echo _l('lead_convert_to_client'); ?>
                </h4>
            </div>
            <div class="modal-body">
                <?php echo form_hidden('leadid', $lead->id); ?>
                <?php if (mb_strpos($lead->name, ' ') !== false) {
    $_temp     = explode(' ', $lead->name);
    $firstname = $_temp[0];
    if (isset($_temp[2])) {
        $lastname = $_temp[1] . ' ' . $_temp[2];
    } else {
        $lastname = $_temp[1];
    }
} else {
    $lastname  = '';
    $firstname = $lead->name;
}
           ?>
                <?php echo form_hidden('default_language', $lead->default_language); ?>
                <?php echo render_input('firstname', 'lead_convert_to_client_firstname', $firstname); ?>
                <?php echo render_input('lastname', 'lead_convert_to_client_lastname', $lastname); ?>
                <?php echo render_input('title', 'contact_position', $lead->title); ?>
                <?php echo render_input('email', 'lead_convert_to_email', $lead->email); ?>
                <?php echo render_input('company', 'lead_company', $lead->company); ?>
                <?php echo render_input('phonenumber', 'lead_convert_to_client_phone', $lead->phonenumber); ?>
                <?php echo render_input('website', 'client_website', $lead->website); ?>
                <?php echo render_textarea('address', 'client_address', $lead->address); ?>
                <?php echo render_input('city', 'client_city', $lead->city); ?>
                <?php echo render_input('state', 'client_state', $lead->state); ?>
                <?php
           $countries                = get_all_countries();
           $customer_default_country = get_option('customer_default_country');
           $selected                 = ($lead->country != 0 ? $lead->country : $customer_default_country);
           echo render_select('country', $countries, [ 'country_id', [ 'short_name']], 'clients_country', $selected, ['data-none-selected-text' => _l('dropdown_non_selected_tex')]);
           ?>
                <?php echo render_input('zip', 'clients_zip', $lead->zip); ?>
                <?php
           $not_mergable_customer_fields = ['userid', 'datecreated', 'leadid', 'default_language', 'default_currency', 'active'];
           $not_mergable_contact_fields  = ['id', 'userid', 'datecreated', 'is_primary', 'password', 'new_pass_key', 'new_pass_key_requested', 'last_ip', 'last_login', 'last_password_change', 'active', 'profile_image', 'direction'];
           $customer_fields              = $this->db->list_fields(db_prefix() . 'clients');
           $contact_fields               = $this->db->list_fields(db_prefix() . 'contacts');
           $custom_fields                = get_custom_fields('leads');
           $found_custom_fields          = false;
           foreach ($custom_fields as $field) {
               $value = get_custom_field_value($lead->id, $field['id'], 'leads');
               if ($value == '') {
                   continue;
               }
               $found_custom_fields = true;
           }
        if ($found_custom_fields == true) {
            echo '<h4 class="bold text-center mtop30">' . _l('copy_custom_fields_convert_to_customer') . '</h4><hr />';
        }
      foreach ($custom_fields as $field) {
          $value = get_custom_field_value($lead->id, $field['id'], 'leads');
          if ($value == '') {
              continue;
          } ?>
                <p class="bold text-info"><?php echo e($field['name']); ?> (<?php echo $value; ?>)</p>
                <hr />
                <p class="bold no-margin"><?php echo _l('leads_merge_customer'); ?></p>
                <div class="radio radio-primary">
                    <input type="radio" data-field-id="<?php echo e($field['id']); ?>" id="m_1_<?php echo e($field['id']); ?>"
                        class="include_leads_custom_fields" checked
                        name="include_leads_custom_fields[<?php echo e($field['id']); ?>]" value="1">
                    <label for="m_1_<?php echo e($field['id']); ?>" class="bold">
                        <span data-toggle="tooltip"
                            data-title="<?php echo _l('copy_custom_fields_convert_to_customer_help'); ?>"><i
                                class="fa fa-info-circle"></i></span> <?php echo _l('lead_merge_custom_field'); ?>
                    </label>
                </div>
                <div class="radio radio-primary">
                    <input type="radio" data-field-id="<?php echo e($field['id']); ?>" id="m_2_<?php echo e($field['id']); ?>"
                        class="include_leads_custom_fields"
                        name="include_leads_custom_fields[<?php echo e($field['id']); ?>]" value="2">
                    <label for="m_2_<?php echo e($field['id']); ?>" class="bold">
                        <?php echo _l('lead_merge_custom_field_existing'); ?>
                    </label>
                </div>
                <div class="hide" id="merge_db_field_<?php echo e($field['id']); ?>">
                    <hr />
                    <select name="merge_db_fields[<?php echo e($field['id']); ?>]" class="selectpicker" data-width="100%"
                        data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                        <option value=""></option>
                        <?php foreach ($customer_fields as $c_field) {
              if (!in_array($c_field, $not_mergable_customer_fields)) {
                  echo '<option value="' . $c_field . '">' . str_replace('_', ' ', ucfirst($c_field)) . '</option>';
              }
          } ?>
                    </select>
                    <hr />
                </div>
                <p class="bold"><?php echo _l('leads_merge_contact'); ?></p>
                <div class="radio radio-primary">
                    <input type="radio" data-field-id="<?php echo e($field['id']); ?>" id="m_3_<?php echo e($field['id']); ?>"
                        class="include_leads_custom_fields"
                        name="include_leads_custom_fields[<?php echo e($field['id']); ?>]" value="3">
                    <label for="m_3_<?php echo e($field['id']); ?>" class="bold">
                        <?php echo _l('leads_merge_as_contact_field'); ?>
                    </label>
                </div>
                <div class="radio radio-primary">
                    <input type="radio" data-field-id="<?php echo e($field['id']); ?>" id="m_4_<?php echo e($field['id']); ?>"
                        class="include_leads_custom_fields"
                        name="include_leads_custom_fields[<?php echo e($field['id']); ?>]" value="4">
                    <label for="m_4_<?php echo e($field['id']); ?>" class="bold">
                        <span data-toggle="tooltip"
                            data-title="<?php echo _l('copy_custom_fields_convert_to_customer_help'); ?>"><i
                                class="fa fa-info-circle"></i></span>
                        <?php echo _l('lead_merge_custom_field'); ?>
                    </label>
                </div>
                <div class="hide" id="merge_db_contact_field_<?php echo e($field['id']); ?>">
                    <hr />
                    <select name="merge_db_contact_fields[<?php echo e($field['id']); ?>]" class="selectpicker"
                        data-width="100%" data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                        <option value=""></option>
                        <?php foreach ($contact_fields as $c_field) {
              if (!in_array($c_field, $not_mergable_contact_fields)) {
                  echo '<option value="' . $c_field . '">' . str_replace('_', ' ', ucfirst($c_field)) . '</option>';
              }
          } ?>
                    </select>
                </div>
                <hr />
                <div class="radio radio-primary">
                    <input type="radio" data-field-id="<?php echo e($field['id']); ?>" id="m_5_<?php echo e($field['id']); ?>"
                        class="include_leads_custom_fields"
                        name="include_leads_custom_fields[<?php echo e($field['id']); ?>]" value="5">
                    <label for="m_5_<?php echo e($field['id']); ?>" class="bold">
                        <?php echo _l('lead_dont_merge_custom_field'); ?>
                    </label>
                </div>
                <hr />
                <?php
      } ?>
                <?php echo form_hidden('original_lead_email', $lead->email); ?>

                <!-- fake fields are a workaround for chrome autofill getting the wrong fields -->
                <input type="text" class="fake-autofill-field" name="fakeusernameremembered" value='' tabindex="-1" />
                <input type="password" class="fake-autofill-field" name="fakepasswordremembered" value=''
                    tabindex="-1" />

                <div class="client_password_set_wrapper">
                    <label for="password" class="control-label"><?php echo _l('client_password'); ?></label>
                    <div class="input-group">
                        <input type="password" class="form-control password" name="password" autocomplete="off">
                        <span class="input-group-addon tw-border-l-0">
                            <a href="#password" class="show_password"
                                onclick="showPassword('password');return false;"><i class="fa fa-eye"></i></a>
                        </span>
                        <span class="input-group-addon">
                            <a href="#" class="generate_password" onclick="generatePassword(this);return false;"><i
                                    class="fa fa-refresh"></i></a>
                        </span>
                    </div>
                </div>
                <?php if (is_email_template_active('contact-set-password')) { ?>
                <div class="checkbox checkbox-primary">
                    <input type="checkbox" name="send_set_password_email" id="send_set_password_email">
                    <label for="send_set_password_email">
                        <?php echo _l('client_send_set_password_email'); ?>
                    </label>
                </div>
                <?php } ?>
                <?php if (is_email_template_active('new-client-created')) { ?>
                <div class="checkbox checkbox-primary">
                    <input type="checkbox" name="donotsendwelcomeemail" id="donotsendwelcomeemail">
                    <label for="donotsendwelcomeemail"><?php echo _l('client_do_not_send_welcome_email'); ?></label>
                </div>
                <?php } ?>
                <?php if (total_rows(db_prefix() . 'notes', ['rel_type' => 'lead', 'rel_id' => $lead->id]) > 0) { ?>
                <div class="checkbox checkbox-primary">
                    <input type="checkbox" name="transfer_notes" id="transfer_notes">
                    <label for="transfer_notes"><?php echo _l('transfer_lead_notes_to_customer'); ?></label>
                </div>
                <?php } ?>
                <?php if (is_gdpr() && get_option('gdpr_enable_consent_for_contacts') == '1' && count($purposes) > 0) { ?>
                <div class="checkbox checkbox-primary">
                    <input type="checkbox" name="transfer_consent" id="transfer_consent">
                    <label for="transfer_consent"><?php echo _l('transfer_consent'); ?></label>
                </div>
                <?php } ?>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default"
                    onclick="init_lead(<?php echo e($lead->id); ?>); return false;"
                    data-dismiss="modal"><?php echo _l('back_to_lead'); ?></button>
                <button type="submit" data-form="#lead_to_client_form" autocomplete="off"
                    data-loading-text="<?php echo _l('wait_text'); ?>"
                    class="btn btn-primary"><?php echo _l('submit'); ?></button>
            </div>
        </div>
        <?php echo form_close(); ?>
    </div>
</div>
<script>
validate_lead_convert_to_client_form();
init_selectpicker();
</script>










<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <h4
                    class="tw-font-semibold tw-text-lg tw-text-neutral-700 tw-flex tw-justify-between tw-items-center tw-mt-0">
                    <?= e($title); ?>
                    <a href="<?= admin_url('spam_filters/view/leads'); ?>"
                        class="btn btn-primary">
                        <?= _l('spam_filters'); ?>
                    </a>
                </h4>
                <div class="panel_s">
                    <?= form_open($this->uri->uri_string(), ['id' => 'leads-email-integration']); ?>
                    <div class="panel-body">
                        <?php hooks()->do_action('before_leads_email_integration_form'); ?>

                        <?php if (! function_exists('iconv')) { ?>
                        <div class="alert alert-danger no-mbot">
                            You need to enable <b>iconv</b> php extension in order to use this feature. You can enable
                            it via php.ini or contact your hosting provider to enable this extension.
                        </div>
                        <?php } ?>
                        <?php if ($mail->email != '' && total_rows(db_prefix() . 'departments', ['email' => $mail->email]) > 0) { ?>
                        <div class="alert alert-danger no-mbot">
                            The email address
                            <b><?= e($mail->email); ?></b> is
                            used <b>also</b> as a <b>support
                                department</b> email. For leads email integration you must configure unique email if you
                            want everything to work properly.
                        </div>
                        <?php } ?>
                        <!-- fake fields are a workaround for chrome autofill getting the wrong fields -->
                        <input type="text" class="fake-autofill-field" name="fakeusernameremembered" value=''
                            tabindex="-1" />
                        <input type="password" class="fake-autofill-field" name="fakepasswordremembered" value=''
                            tabindex="-1" />
                        <div class="row -tw-mt-8">
                            <div class="col-md-12">
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="active" id="active" class="ays-ignore" <?php if ($mail->active == 1) {
                                        echo 'checked';
                                    } ?>>
                                    <label
                                        for="active"><?= _l('leads_email_active'); ?></label>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <?= render_input('imap_server', 'leads_email_integration_imap', $mail->imap_server); ?>
                                <?= render_input('email', 'leads_email_integration_email', $mail->email); ?>
                                <?php $ps = $mail->password;
if (! empty($ps)) {
    if ($this->encryption->decrypt($ps) == false) {
        $ps = $ps;
    } else {
        $ps = $this->encryption->decrypt($ps);
    }
}
echo render_input('password', 'leads_email_integration_password', $ps, 'password', ['autocomplete' => 'off']); ?>
                                <div class="form-group">
                                    <label
                                        for="encryption"><?= _l('leads_email_encryption'); ?></label><br />
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" name="encryption" value="tls" id="tls" <?php if ($mail->encryption == 'tls') {
                                            echo 'checked';
                                        } ?>>
                                        <label for="tls">TLS</label>
                                    </div>
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" name="encryption" value="ssl" id="ssl" <?php if ($mail->encryption == 'ssl') {
                                            echo 'checked';
                                        } ?>>
                                        <label for="ssl">SSL</label>
                                    </div>
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" name="encryption" value="" id="no_enc" <?php if ($mail->encryption == '') {
                                            echo 'checked';
                                        } ?>>
                                        <label
                                            for="no_enc"><?= _l('leads_email_integration_folder_no_encryption'); ?></label>
                                    </div>
                                </div>
                                <div class="form-group">
                                    <label for="folder" class="control-label">
                                        <?= _l('leads_email_integration_folder'); ?>
                                        <a href="#" onclick="retrieve_leads_email_integration_folders(); return false;">
                                            <i class="fa fa-refresh hidden" id="folders-loader"></i>
                                            <?= _l('retrieve_folders'); ?>
                                        </a>
                                    </label>
                                    <select name="folder" class="form-control selectpicker" id="folder">
                                        <option
                                            value="<?= e($mail->folder); ?>">
                                            <?= e($mail->folder); ?>
                                        </option>
                                    </select>
                                </div>
                                <?= render_input('check_every', 'leads_email_integration_check_every', $mail->check_every, 'number', ['min' => hooks()->apply_filters('leads_email_integration_check_every', 10), 'data-ays-ignore' => true]); ?>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="only_loop_on_unseen_emails" class="ays-ignore"
                                        id="only_loop_on_unseen_emails" <?php if ($mail->only_loop_on_unseen_emails == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label for="only_loop_on_unseen_emails"><i class="fa-regular fa-circle-question"
                                            data-toggle="tooltip"
                                            data-title="<?= _l('leads_email_integration_only_check_unseen_emails_help'); ?>"></i>
                                        <?= _l('leads_email_integration_only_check_unseen_emails'); ?></label>
                                </div>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" class="ays-ignore" name="create_task_if_customer"
                                        id="create_task_if_customer" <?php if ($mail->create_task_if_customer == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label for="create_task_if_customer"><i class="fa-regular fa-circle-question"
                                            data-toggle="tooltip"
                                            data-title="<?= _l('create_the_duplicate_form_data_as_task_help'); ?>"></i>
                                        <?= _l('lead_is_contact_create_task'); ?></label>
                                </div>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="delete_after_import" class="ays-ignore"
                                        id="delete_after_import" <?php if ($mail->delete_after_import == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label for="delete_after_import">
                                        <?= _l('delete_mail_after_import'); ?></label>
                                </div>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="mark_public" class="ays-ignore" id="mark_public"
                                        <?php if ($mail->mark_public == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label for="mark_public">
                                        <?= _l('auto_mark_as_public'); ?></label>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <?php
                                        $selected = $mail->lead_status;
if ($selected == 0) {
    $selected = get_option('leads_default_status');
}

echo render_leads_status_select($statuses, $selected, 'leads_email_integration_default_status', 'lead_status', ['data-ays-ignore' => true], true);

$selected = $mail->lead_source;
if ($selected == 0) {
    $selected = get_option('leads_default_source');
}
echo render_leads_source_select($sources, $selected, 'leads_email_integration_default_source', 'lead_source', ['data-ays-ignore' => true]);
$selected = '';

foreach ($members as $staff) {
    if ($mail->responsible == $staff['staffid']) {
        $selected = $staff['staffid'];
    }
}
?>
                                <?= render_select('responsible', $members, ['staffid', ['firstname', 'lastname']], 'leads_email_integration_default_assigned', $selected, ['data-ays-ignore' => true]); ?>
                                <hr />
                                <label for=""
                                    class="control-label"><?= _l('notification_settings'); ?></label>
                                <div class="clearfix"></div>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="notify_lead_imported" id="notify_lead_imported"
                                        class="ays-ignore" <?php if ($mail->notify_lead_imported == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label
                                        for="notify_lead_imported"><?= _l('leads_email_integration_notify_when_lead_imported'); ?></label>
                                </div>
                                <div class="checkbox checkbox-primary">
                                    <input type="checkbox" name="notify_lead_contact_more_times" class="ays-ignore"
                                        id="notify_lead_contact_more_times" <?php if ($mail->notify_lead_contact_more_times == 1) {
                                            echo 'checked';
                                        } ?>>
                                    <label
                                        for="notify_lead_contact_more_times"><?= _l('leads_email_integration_notify_when_lead_contact_more_times'); ?></label>
                                </div>
                                <div class="select-notification-settings<?php if ($mail->notify_lead_imported == '0' && $mail->notify_lead_contact_more_times == '0') {
                                    echo ' hide';
                                } ?>">
                                    <hr />
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" name="notify_type" class="ays-ignore" id="roles"
                                            value="roles" <?php if ($mail->notify_type == 'roles') {
                                                echo 'checked';
                                            } ?>>
                                        <label
                                            for="roles"><?= _l('staff_with_roles'); ?></label>
                                    </div>
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" name="notify_type" class="ays-ignore" value="specific_staff"
                                            id="specific_staff" <?php if ($mail->notify_type == 'specific_staff') {
                                                echo 'checked';
                                            } ?>>
                                        <label
                                            for="specific_staff"><?= _l('specific_staff_members'); ?></label>
                                    </div>
                                    <div class="radio radio-primary radio-inline">
                                        <input type="radio" class="ays-ignore" name="notify_type" id="assigned"
                                            value="assigned" <?php if ($mail->notify_type == 'assigned') {
                                                echo 'checked';
                                            } ?>>
                                        <label
                                            for="assigned"><?= _l('notify_assigned_user'); ?></label>
                                    </div>
                                    <div class="clearfix mtop15"></div>
                                    <div id="role_notify" class="<?php if ($mail->notify_type != 'roles') {
                                        echo 'hide';
                                    } ?>">
                                        <?php
                                          $selected = [];
if ($mail->notify_type == 'roles') {
    $selected = unserialize($mail->notify_ids);
}
?>
                                        <?= render_select('notify_ids_roles[]', $roles, ['roleid', ['name']], 'leads_email_integration_notify_roles', $selected, ['multiple' => true, 'data-ays-ignore' => true]); ?>
                                    </div>
                                    <div id="specific_staff_notify" class="<?php if ($mail->notify_type != 'specific_staff') {
                                        echo 'hide';
                                    } ?>">
                                        <?php
                                    $selected = [];
if ($mail->notify_type == 'specific_staff') {
    $selected = unserialize($mail->notify_ids);
}
?>
                                        <?= render_select('notify_ids_staff[]', $members, ['staffid', ['firstname', 'lastname']], 'leads_email_integration_notify_staff', $selected, ['multiple' => true, 'data-ays-ignore' => true]); ?>


                                    </div>

                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="panel-footer text-right">
                        <a href="<?= admin_url('leads/test_email_integration'); ?>"
                            class="btn btn-default test-leads-email-integration"><?= _l('leads_email_integration_test_connection'); ?></a>
                        <button type="submit"
                            class="btn btn-primary"><?= _l('submit'); ?></button>
                    </div>
                    <?= form_close(); ?>
                </div>
            </div>
        </div>
    </div>
</div>
<?php init_tail(); ?>
<script>
    $(function() {

        var $create_task_if_customer = $('#create_task_if_customer');

        $('#leads-email-integration').on('dirty.areYouSure', function() {
            // Enable save button only as the form is dirty.
            $('.test-leads-email-integration').addClass('disabled');
        });

        $('#leads-email-integration').on('clean.areYouSure', function() {
            // Form is clean so nothing to save - disable the save button.
            $('.test-leads-email-integration').removeClass('disabled');
        });

        $('#notify_lead_imported,#notify_lead_contact_more_times').on('change', function() {
            if ($('#notify_lead_imported').prop('checked') == false && $(
                    '#notify_lead_contact_more_times')
                .prop('checked') == false) {
                $('.select-notification-settings').addClass('hide');
            } else {
                $('.select-notification-settings').removeClass('hide');
            }
        });

        var validationObject = {
            lead_source: 'required',
            lead_status: 'required',
            imap_server: 'required',
            password: 'required',
            port: 'required',
            email: {
                required: true
            },
            check_every: {
                required: true,
                number: true
            },
            folder: {
                required: true,
            },
            responsible: {
                required: {
                    depends: function(element) {
                        var isRequiredByNotifyType = ($('input[name="notify_type"]:checked').val() ==
                            'assigned') ? true : false;
                        var isRequiredByAssignTask = ($create_task_if_customer.is(':checked')) ? true :
                            false;
                        var isRequired = isRequiredByNotifyType || isRequiredByAssignTask;
                        if (isRequired) {
                            $('[for="responsible"]').find('.req').removeClass('hide');
                        } else {
                            $(element).next('p.text-danger').remove();
                            $('[for="responsible"]').find('.req').addClass('hide');
                        }
                        return isRequired;
                    }
                }
            }
        };

        appValidateForm($('#leads-email-integration'), validationObject);

        var $notifyTypeInput = $('input[name="notify_type"]');

        $notifyTypeInput.on('change', function() {
            $('#leads-email-integration').validate().checkForm()
        });

        $create_task_if_customer.on('change', function() {
            $('#leads-email-integration').validate().checkForm()
        });

        $create_task_if_customer.trigger('change');

    });

    function retrieve_leads_email_integration_folders() {
        retrieve_imap_folders(admin_url + 'departments/folders', {
            email: $('input[name="email"]').val(),
            password: $('input[name="password"]').val(),
            host: $('input[name="imap_server"]').val(),
            encryption: $('input[name="encryption"]:checked').val()
        })
    }
</script>
</body>

</html>








<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<?php if (isset($form)) {
    echo form_hidden('form_id', $form->id);
} ?>
<div id="wrapper">
    <div class="content">
        <div
            class="<?= ! isset($form) ? 'tw-max-w-4xl tw-mx-auto' : 'tw-w-full'; ?>">
            <?php if (! isset($form)) { ?>
            <div class="alert alert-info">
                <?= _l('form_builder_create_form_first'); ?>
            </div>
            <?php } ?>
            <?php if (isset($form)) { ?>
            <div class="horizontal-scrollable-tabs">
                <div class="scroller arrow-left"><i class="fa fa-angle-left"></i></div>
                <div class="scroller arrow-right"><i class="fa fa-angle-right"></i></div>
                <div class="horizontal-tabs">
                    <ul class="nav nav-tabs nav-tabs-segmented nav-tabs-horizontal tw-mb-3" role="tablist">
                        <li role="presentation" class="active">
                            <a href="#tab_form_build" aria-controls="tab_form_build" role="tab" data-toggle="tab">
                                <i class="fa-solid fa-wrench menu-icon"></i>
                                <?= _l('form_builder'); ?>
                            </a>
                        </li>
                        <li role="presentation">
                            <a href="#tab_form_information" aria-controls="tab_form_information" role="tab"
                                data-toggle="tab">
                                <i class="fa-regular fa-file-lines menu-icon"></i>
                                <?= _l('form_information'); ?>
                            </a>
                        </li>
                        <li role="presentation">
                            <a href="#tab_form_integration" aria-controls="tab_form_integration" role="tab"
                                data-toggle="tab">
                                <i class="fa-solid fa-code menu-icon"></i>
                                <?= _l('form_integration_code'); ?>
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
            <?php } ?>
            <?php if (! isset($form)) { ?>
            <h4 class="tw-mt-0 tw-font-bold tw-text-lg tw-text-neutral-700">
                <?= _l('new_form'); ?>
            </h4>
            <?php } ?>
            <div class="panel_s">
                <div class="panel-body">
                    <div class="tab-content">
                        <?php if (isset($form)) { ?>
                        <div role="tabpanel" class="tab-pane active" id="tab_form_build">
                            <div id="build-wrap"></div>
                        </div>
                        <div role="tabpanel" class="tab-pane" id="tab_form_integration">
                            <p><?= _l('form_integration_code_help'); ?>
                            </p>
                            <textarea class="form-control"
                                rows="2"><iframe width="600" height="850" src="<?= site_url('forms/wtl/' . $form->form_key); ?>" frameborder="0" sandbox="allow-top-navigation allow-forms allow-scripts allow-same-origin allow-popups" allowfullscreen></iframe></textarea>
                            <h4 class="tw-my-5 bold">Share direct link</h4>
                            <p>
                                <span class="label label-default">
                                    <a href="<?= site_url('forms/wtl/' . $form->form_key) . '?styled=1'; ?>"
                                        target="_blank">
                                        <?= site_url('forms/wtl/' . $form->form_key) . '?styled=1'; ?>
                                    </a>
                                </span>
                                <br />
                                <br />
                                <span class="label label-default">
                                    <a href="<?= site_url('forms/wtl/' . $form->form_key) . '?styled=1&with_logo=1'; ?>"
                                        target="_blank">
                                        <?= site_url('forms/wtl/' . $form->form_key) . '?styled=1&with_logo=1'; ?>
                                    </a>
                                </span>
                            </p>
                            <hr />
                            <p class="bold mtop15">When placing the iframe snippet code consider the following:</p>
                            <p
                                class="<?= strpos(site_url(), 'http://') !== false ? 'bold text-success' : ''; ?>">
                                1. If the protocol of your installation is http use a http page inside the iframe.
                            </p>
                            <p
                                class="<?= strpos(site_url(), 'https://') !== false ? 'bold text-success' : ''; ?>">
                                2. If the protocol of your installation is https use a https page inside the
                                iframe.
                            </p>
                            <p>
                                None SSL installation will need to place the link in non ssl eq. landing page and
                                backwards.
                            </p>
                        </div>
                        <?php } ?>
                        <div role="tabpanel"
                            class="tab-pane<?= ! isset($form) ? ' active' : ''; ?>"
                            id="tab_form_information">
                            <div
                                class="<?= isset($form) ? 'tw-mx-auto tw-max-w-3xl' : ''; ?>">

                                <div class="horizontal-scrollable-tabs">
                                    <div class="scroller arrow-left"><i class="fa fa-angle-left"></i></div>
                                    <div class="scroller arrow-right"><i class="fa fa-angle-right"></i></div>
                                    <div class="horizontal-tabs">
                                        <ul class="nav nav-tabs nav-tabs-segmented nav-tabs-horizontal" role="tablist">
                                            <li role="presentation" class="active">
                                                <a href="#tab_config_general" aria-controls="tab_config_general"
                                                    role="tab" data-toggle="tab">
                                                    General
                                                </a>
                                            </li>
                                            <li role="presentation">
                                                <a href="#tab_config_branding" aria-controls="tab_config_branding"
                                                    role="tab" data-toggle="tab">
                                                    Branding
                                                </a>
                                            </li>
                                            <li role="presentation">
                                                <a href="#tab_config_submission" aria-controls="tab_config_submission"
                                                    role="tab" data-toggle="tab">
                                                    Submission
                                                </a>
                                            </li>
                                            <li role="presentation">
                                                <a href="#tab_config_notifications"
                                                    aria-controls="tab_config_notifications" role="tab"
                                                    data-toggle="tab">
                                                    Notifications
                                                </a>
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                                <?= form_open($this->uri->uri_string(), ['id' => 'form_info']); ?>
                                <div class="tab-content">
                                    <div role="tabpanel" class="tab-pane active" id="tab_config_general">
                                        <?php $value = (isset($form) ? $form->name : ''); ?>
                                        <?= render_input('name', 'form_name', $value); ?>
                                        <?php if (get_option('recaptcha_secret_key') != '' && get_option('recaptcha_site_key') != '') { ?>
                                        <div class="form-group">
                                            <label
                                                for=""><?= _l('form_recaptcha'); ?></label><br />
                                            <div class="radio radio-inline radio-danger">
                                                <input type="radio" name="recaptcha" id="racaptcha_0" value="0"
                                                    <?= isset($form) && $form->recaptcha == 0 || ! isset($form) ? 'checked' : ''; ?>>
                                                <label
                                                    for="recaptcha_0"><?= _l('settings_no'); ?></label>
                                            </div>
                                            <div class="radio radio-inline radio-success">
                                                <input type="radio" name="recaptcha" id="recaptcha_1" value="1"
                                                    <?= isset($form) && $form->recaptcha == 1 ? 'checked' : ''; ?>>
                                                <label
                                                    for="recaptcha_1"><?= _l('settings_yes'); ?></label>
                                            </div>
                                        </div>
                                        <?php } ?>
                                        <div class="form-group select-placeholder">
                                            <div class="tw-mb-1.5">
                                                <label for="language" class="control-label">
                                                    <?= _l('form_lang_validation'); ?>
                                                </label>
                                                <select name="language" id="language" class="form-control selectpicker"
                                                    data-none-selected-text="<?= _l('dropdown_non_selected_tex'); ?>">
                                                    <option value=""></option>
                                                    <?php foreach ($languages as $availableLanguage) { ?>
                                                    <option
                                                        value="<?= e($availableLanguage); ?>"
                                                        <?= (isset($form) && $form->language == $availableLanguage) || (! isset($form) && get_option('active_language') == $availableLanguage) ? 'selected' : ''; ?>>
                                                        <?= e(ucfirst($availableLanguage)); ?>
                                                    </option>
                                                    <?php } ?>
                                                </select>
                                                <span>
                                                    <?= _l('form_lang_validation_help'); ?>
                                                </span>
                                            </div>
                                        </div>

                                        <div class="form-group">
                                            <?php $value = (isset($form) ? $form->lead_name_prefix : ''); ?>
                                            <?= render_input('lead_name_prefix', 'lead_name_prefix', $value, 'text', [], [], 'mbot5'); ?>
                                            <span><?= _l('lead_name_prefix_help') ?></span>
                                        </div>
                                        <?= render_leads_source_select($sources, (isset($form) ? $form->lead_source : get_option('leads_default_source')), 'lead_import_source', 'lead_source');

echo render_leads_status_select($statuses, (isset($form) ? $form->lead_status : get_option('leads_default_status')), 'lead_import_status', 'lead_status', [], true);

$selected = '';

foreach ($members as $staff) {
    if (isset($form) && $form->responsible == $staff['staffid']) {
        $selected = $staff['staffid'];
    }
}
?>
                                        <?= render_select('responsible', $members, ['staffid', ['firstname', 'lastname']], 'leads_import_assignee', $selected); ?>
										<?php
// Customer selection
$selected_client = '';
foreach ($clients as $client) {
    if (isset($form) && $form->responsible_client == $client['userid']) {
        $selected_client = $client['userid'];
    }
}
?>
<?= render_select('responsible_client', $clients, ['userid', ['company']], 'leads_import_customer', $selected_client); ?>

                                    </div>
                                    <div role="tabpanel" class="tab-pane" id="tab_config_branding">
                                        <?php $value = (isset($form) ? $form->submit_btn_name : 'Submit'); ?>
                                        <?= render_input('submit_btn_name', 'form_btn_submit_text', $value); ?>
                                        <div class="md:tw-grid md:tw-grid-cols-2 md:tw-gap-4">
                                            <div class="[&_.input-group]:!tw-mb-0">
                                                <?php $value = (isset($form) ? $form->submit_btn_bg_color : '#84c529'); ?>
                                                <?= render_color_picker('submit_btn_bg_color', _l('submit_button_bg_color'), $value); ?>
                                            </div>
                                            <div class="[&_.input-group]:!tw-mb-0">
                                                <?php $value = (isset($form) ? $form->submit_btn_text_color : '#ffffff'); ?>
                                                <?= render_color_picker('submit_btn_text_color', _l('submit_button_text_color'), $value); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div role="tabpanel" class="tab-pane" id="tab_config_submission">
                                        <label for="" class="control-label bold">
                                            <?= _l('form_submit_success_action'); ?>?
                                        </label>
                                        <div class="radio radio-primary">
                                            <input type="radio" name="submit_action" value="0" id="success_message"
                                                <?= isset($form) && $form->submit_action == '0' || ! isset($form) ? 'checked' : ''; ?>>
                                            <label for="success_message">
                                                <?= _l('form_submit_success_display_thank_you'); ?>
                                            </label>
                                        </div>
                                        <div class="radio radio-primary">
                                            <input type="radio" name="submit_action" value="1" id="website_redirect"
                                                <?= isset($form) && $form->submit_action == '1' ? 'checked' : ''; ?>>
                                            <label for="website_redirect">
                                                <?= _l('form_submit_success_redirect_to_website'); ?>
                                            </label>
                                        </div>
                                        <?php $value = (isset($form) ? $form->success_submit_msg : ''); ?>
                                        <?= render_textarea('success_submit_msg', 'form_success_submit_msg', $value); ?>
                                        <?php $value = (isset($form) ? $form->submit_redirect_url : ''); ?>
                                        <?= render_input('submit_redirect_url', 'form_submit_website_url', $value, 'url'); ?>
                                        <div class="checkbox checkbox-primary">
                                            <input type="checkbox" name="mark_public" id="mark_public"
                                                <?= isset($form) && $form->mark_public == 1 ? 'checked' : ''; ?>>
                                            <label for="mark_public">
                                                <?= _l('auto_mark_as_public'); ?>
                                            </label>
                                        </div>
                                        <div class="checkbox checkbox-primary">
                                            <input type="checkbox" name="allow_duplicate" id="allow_duplicate"
                                                <?= isset($form) && $form->allow_duplicate == 1 || ! isset($form) ? 'checked' : ''; ?>>
                                            <label for="allow_duplicate">
                                                <?= e(_l('form_allow_duplicate', _l('lead_lowercase'))); ?>
                                            </label>
                                        </div>
                                        <div
                                            class="duplicate-settings-wrapper row<?= isset($form) && $form->allow_duplicate == 1 || ! isset($form) ? ' hide' : ''; ?>">

                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label for="track_duplicate_field">
                                                        <?= _l('track_duplicate_by_field'); ?>
                                                    </label><br />
                                                    <select class="selectpicker track_duplicate_field" data-width="100%"
                                                        name="track_duplicate_field" id="track_duplicate_field"
                                                        data-none-selected-text="<?= _l('dropdown_non_selected_tex'); ?>">
                                                        <option value=""></option>
                                                        <?php foreach ($db_fields as $field) { ?>
                                                        <option
                                                            value="<?= e($field->name); ?>"
                                                            <?= isset($form) && $form->track_duplicate_field == $field->name ? 'selected' : ''; ?>
                                                            <?= isset($form) && $form->track_duplicate_field_and == $field->name ? 'disabled' : ''; ?>>
                                                            <?= e($field->label); ?>
                                                        </option>
                                                        <?php } ?>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label for="track_duplicate_field_and">
                                                        <?= _l('and_track_duplicate_by_field'); ?>
                                                    </label><br />
                                                    <select class="selectpicker track_duplicate_field_and"
                                                        data-width="100%" name="track_duplicate_field_and"
                                                        id="track_duplicate_field_and"
                                                        data-none-selected-text="<?= _l('dropdown_non_selected_tex'); ?>">
                                                        <option value=""></option>
                                                        <?php foreach ($db_fields as $field) { ?>
                                                        <option
                                                            value="<?= e($field->name); ?>"
                                                            <?= isset($form) && $form->track_duplicate_field_and == $field->name ? ' selected' : ''; ?>
                                                            <?= isset($form) && $form->track_duplicate_field == $field->name ? 'disabled' : ''; ?>>
                                                            <?= e($field->label); ?>
                                                        </option>
                                                        <?php } ?>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-12">
                                                <div class="checkbox checkbox-primary">
                                                    <input type="checkbox" name="create_task_on_duplicate"
                                                        id="create_task_on_duplicate"
                                                        <?= (isset($form) && $form->create_task_on_duplicate == 1 || ! isset($form)) ? 'checked' : ''; ?>>
                                                    <label for="create_task_on_duplicate">
                                                        <i class="fa-regular fa-circle-question" data-toggle="tooltip"
                                                            data-title="<?= _l('create_the_duplicate_form_data_as_task_help'); ?>">
                                                        </i>
                                                        <?= e(_l('create_the_duplicate_form_data_as_task', _l('lead_lowercase'))); ?>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div role="tabpanel" class="tab-pane" id="tab_config_notifications">
                                        <p class="tw-font-semibold tw-mb-3">
                                            <?= _l('notification_settings'); ?>
                                        </p>
                                        <div class="checkbox">
                                            <input type="checkbox" name="notify_lead_imported" id="notify_lead_imported"
                                                <?= isset($form) && $form->notify_lead_imported == 1 || ! isset($form) ? 'checked' : ''; ?>>
                                            <label for="notify_lead_imported">
                                                <?= _l('leads_email_integration_notify_when_lead_imported'); ?>
                                            </label>
                                        </div>
                                        <div
                                            class="select-notification-settings tw-mt-6<?= isset($form) && $form->notify_lead_imported == '0' ? ' hide' : ''; ?>">
                                            <div class="radio radio-inline">
                                                <input type="radio" name="notify_type" value="specific_staff"
                                                    id="specific_staff"
                                                    <?= isset($form) && $form->notify_type == 'specific_staff' || ! isset($form) ? 'checked' : ''; ?>>
                                                <label
                                                    for="specific_staff"><?= _l('specific_staff_members'); ?></label>
                                            </div>
                                            <div class="radio radio-inline">
                                                <input type="radio" name="notify_type" id="roles" value="roles"
                                                    <?= isset($form) && $form->notify_type == 'roles' ? 'checked' : ''; ?>>
                                                <label
                                                    for="roles"><?= _l('staff_with_roles'); ?></label>
                                            </div>
                                            <div class="radio radio-inline">
                                                <input type="radio" name="notify_type" id="assigned" value="assigned"
                                                    <?= isset($form) && $form->notify_type == 'assigned' ? 'checked' : ''; ?>>
                                                <label
                                                    for="assigned"><?= _l('notify_assigned_user'); ?></label>
                                            </div>
                                            <div class="clearfix mtop15"></div>
                                            <div id="specific_staff_notify"
                                                class="<?= isset($form) && $form->notify_type != 'specific_staff' ? 'hide' : ''; ?>">
                                                <?php
                       $selected = [];
if (isset($form) && $form->notify_type == 'specific_staff') {
    $selected = unserialize($form->notify_ids);
}
?>
                                                <?= render_select('notify_ids_staff[]', $members, ['staffid', ['firstname', 'lastname']], 'leads_email_integration_notify_staff', $selected, ['multiple' => true]); ?>
                                            </div>
                                            <div id="role_notify"
                                                class="<?= isset($form) && $form->notify_type != 'roles' || ! isset($form) ? 'hide' : ''; ?>">
                                                <?php
$selected = [];
if (isset($form) && $form->notify_type == 'roles') {
    $selected = unserialize($form->notify_ids);
}
?>
                                                <?= render_select('notify_ids_roles[]', $roles, ['roleid', ['name']], 'leads_email_integration_notify_roles', $selected, ['multiple' => true]); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="text-right">
                                        <button type="submit" class="btn btn-primary">
                                            <?= _l('submit'); ?>
                                        </button>
                                    </div>
                                </div>
                                <?= form_close(); ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php init_tail(); ?>
<script
    src="<?= base_url('assets/plugins/form-builder/form-builder.min.js'); ?>">
</script>
<script>
    var buildWrap = document.getElementById('build-wrap');
    var formData = <?= json_encode($formData); ?> ;

    if (formData.length) {
        // If user paste with styling eq from some editor word and the Codeigniter XSS feature remove and apply xss=remove, may break the json.
        formData = formData.replace(/=\\/gm, "=''");
    }
</script>
<?php $this->load->view('admin/includes/_form_js_formatter'); ?>
<script>
    $(function() {

        $('body').on('blur', '.form-field.editing', function() {
            $.Shortcuts.start();
        });

        $('body').on('focus', '.form-field.editing', function() {
            $.Shortcuts.stop();
        });

        var formBuilder = $(buildWrap).formBuilder(fbOptions);
        var $create_task_on_duplicate = $('#create_task_on_duplicate');

        $('#allow_duplicate').on('change', function() {
            $('.duplicate-settings-wrapper').toggleClass('hide');
        });

        $('#notify_lead_imported').on('change', function() {
            $('.select-notification-settings').toggleClass('hide');
        });

        $('#track_duplicate_field,#track_duplicate_field_and').on('change', function() {
            var selector = ($(this).hasClass('track_duplicate_field') ? 'track_duplicate_field_and' :
                'track_duplicate_field')
            $('#' + selector + ' option').removeAttr('disabled', true);
            var val = $(this).val();
            if (val !== '') {
                $('#' + selector + ' option[value="' + val + '"]').attr('disabled', true);
            }
            $('#' + selector + '').selectpicker('refresh');
        });

        setTimeout(function() {
            $(".form-builder-save").wrap("<div class='btn-bottom-toolbar text-right'></div>");
            $btnToolbar = $('body').find('#tab_form_build .btn-bottom-toolbar');
            $btnToolbar = $('#tab_form_build').append($btnToolbar);
            $btnToolbar.find('.btn').addClass('btn-primary');
        }, 100);

        $('body').on('click', '.save-template', function() {
            $.post(admin_url + 'leads/save_form_data', {
                formData: formBuilder.formData,
                id: $('input[name="form_id"]').val()
            }).done(function(response) {
                response = JSON.parse(response);
                if (response.success == true) {
                    alert_float('success', response.message);
                }
            });
        });

        appValidateForm('#form_info', {
            name: 'required',
            lead_source: 'required',
            lead_status: 'required',
            language: 'required',
            submit_btn_name: 'required',
            submit_btn_bg_color: 'required',
            submit_btn_text_color: 'required',
            responsible: {
                required: {
                    depends: function(element) {
                        var isRequiredByNotifyType = ($('input[name="notify_type"]:checked')
                            .val() ==
                            'assigned') ? true : false;
                        var isRequiredByAssignTask = ($create_task_on_duplicate.is(':checked')) ?
                            true :
                            false;
                        var isRequired = isRequiredByNotifyType || isRequiredByAssignTask;
                        if (isRequired) {
                            $('[for="responsible"]').find('.req').removeClass('hide');
                        } else {
                            $(element).next('p.text-danger').remove();
                            $('[for="responsible"]').find('.req').addClass('hide');
                        }
                        return isRequired;
                    }
                }
            },
			responsible_client: 'required',
            success_submit_msg: {
                required: {
                    depends: function(element) {
                        var isRequired = ($('input[name="submit_action"]:checked').val() === '0') ?
                            true : false;
                        if (isRequired) {
                            $('[for="success_submit_msg"]').find('.req').removeClass('hide');
                        } else {
                            $(element).next('p.text-danger').remove();
                            $('[for="success_submit_msg"]').find('.req').addClass('hide');
                        }
                        return isRequired;
                    }
                }
            },
            submit_redirect_url: {
                required: {
                    depends: function(element) {
                        var isRequired = ($('input[name="submit_action"]:checked').val() === '1') ?
                            true : false;
                        if (isRequired) {
                            $('[for="submit_redirect_url"]').find('.req').removeClass('hide');
                        } else {
                            $(element).next('p.text-danger').remove();
                            $('[for="submit_redirect_url"]').find('.req').addClass('hide');
                        }
                        return isRequired;
                    }
                }
            }
        });

        var $notifyTypeInput = $('input[name="notify_type"]');
        $notifyTypeInput.on('change', function() {
            $('#form_info').validate().checkForm()
        });
        $notifyTypeInput.trigger('change');

        $create_task_on_duplicate.on('change', function() {
            $('#form_info').validate().checkForm()
        });

        $create_task_on_duplicate.trigger('change');

        var $submitActionInput = $('input[name="submit_action"]');
        $submitActionInput.on('change', function() {
            $('#form_info').validate().checkForm();

            if ($('input[name="submit_action"]:checked').val() === '1') {
                $('[app-field-wrapper="submit_redirect_url"]').removeClass('hide');
                $('[app-field-wrapper="success_submit_msg"]').addClass('hide');
            } else {
                $('[app-field-wrapper="success_submit_msg"]').removeClass('hide');
                $('[app-field-wrapper="submit_redirect_url"]').addClass('hide');
            }
        });
        $submitActionInput.trigger('change');
    });
</script>
</body>

</html>









<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="tw-mb-2">
                    <a href="<?php echo admin_url('leads/form'); ?>" class="btn btn-primary">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?php echo _l('new_form'); ?>
                    </a>
                </div>
                <div class="panel_s">
                    <div class="panel-body panel-table-full">
                        <?php hooks()->do_action('forms_table_start'); ?>
                        <?php render_datatable([
                              _l('id'),
                              _l('form_name'),
							  _l('leads_import_assignee'),
                              _l('leads_import_customer'),
                              _l('total_submissions'),
                              _l('leads_dt_datecreated'),
                              ], 'web-to-lead'); ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php init_tail(); ?>
<script>
$(function() {
    initDataTable('.table-web-to-lead', window.location.href);
});
</script>
</body>

</html>














<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="collapse" id="importHints">
                    <div class="panel_s">
                        <div class="panel-body tw-bg-gradient-to-l tw-from-transparent tw-to-neutral-50">
                            <?= $this->import->importGuidelinesInfoHtml(); ?>
                        </div>
                    </div>
                </div>
                <div class="tw-flex tw-justify-between tw-items-center tw-mb-3">
                    <h4 class="tw-my-0 tw-font-bold tw-text-lg tw-text-neutral-700 tw-flex tw-items-center tw-gap-x-2">
                        <i class="fa fa-question-circle tw-cursor-pointer" data-toggle="collapse" href="#importHints"
                            aria-expanded="false" aria-controls="importHints"></i>
                        <?= _l('import_leads'); ?>
                    </h4>
                    <?= $this->import->downloadSampleFormHtml(); ?>
                </div>
                <div class="panel_s">
                    <div class="panel-body">
                        <?= $this->import->maxInputVarsWarningHtml(); ?>

                        <?php if (! $this->import->isSimulation()) { ?>
                        <?= $this->import->createSampleTableHtml(); ?>
                        <?php } else { ?>
                        <div class="tw-mb-6">
                            <?= $this->import->simulationDataInfo(); ?>
                        </div>
                        <?= $this->import->createSampleTableHtml(true); ?>
                        <?php } ?>
                        <div class="tw-mt-4">
                            <div class="row">
                                <div class="col-md-4">
                                    <?= form_open_multipart($this->uri->uri_string(), ['id' => 'import_form']); ?>
                                    <?= form_hidden('leads_import', 'true'); ?>
                                    <?= render_input('file_csv', 'choose_csv_file', '', 'file'); ?>
                                    <?= render_leads_status_select($statuses, ($this->input->post('status') ? $this->input->post('status') : get_option('leads_default_status')), _l('lead_import_status') . ' (fallback)', 'status', [], true);
echo render_leads_source_select($sources, ($this->input->post('source') ? $this->input->post('source') : get_option('leads_default_source')), _l('lead_import_source') . ' (fallback)');
?>
                                    <?= render_select('responsible', $members, ['staffid', ['firstname', 'lastname']], 'leads_import_assignee', $this->input->post('responsible')); ?>
                                    <div class="form-group">
                                        <button type="button"
                                            class="btn btn-primary import btn-import-submit"><?= _l('import'); ?></button>
                                        <button type="button"
                                            class="btn btn-default simulate btn-import-submit"><?= _l('simulate_import'); ?></button>
                                    </div>
                                    <?= form_close(); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php init_tail(); ?>
<script
    src="<?= base_url('assets/plugins/jquery-validation/additional-methods.min.js'); ?>">
</script>
<script>
    $(function() {
        appValidateForm($('#import_form'), {
            file_csv: {
                required: true,
                extension: "csv"
            },
            source: 'required',
            status: 'required'
        });
    });
</script>
</body>

</html>








<!DOCTYPE html>
<html>
<head>
	<title>403 Forbidden</title>
</head>
<body>

<p>Directory access is forbidden.</p>

</body>
</html>












<?php defined('BASEPATH') or exit('No direct script access allowed');
$is_admin = is_admin();
$i        = 0;
foreach ($statuses as $status) {
    $kanBan = new \app\services\leads\LeadsKanban($status['id']);
    $kanBan->search($this->input->get('search'))
    ->sortBy($this->input->get('sort_by'), $this->input->get('sort'));
    if ($this->input->get('refresh')) {
        $kanBan->refresh($this->input->get('refresh')[$status['id']] ?? null);
    }
    $leads       = $kanBan->get();
    $total_leads = count($leads);
    $total_pages = $kanBan->totalPages();

    $settings = '';
    foreach (get_system_favourite_colors() as $color) {
        $color_selected_class = 'cpicker-small';
        if ($color == $status['color']) {
            $color_selected_class = 'cpicker-big';
        }
        $settings .= "<div class='kanban-cpicker cpicker " . $color_selected_class . "' data-color='" . $color . "' style='background:" . $color . ';border:1px solid ' . $color . "'></div>";
    } ?>
<ul class="kan-ban-col" data-col-status-id="<?php echo e($status['id']); ?>" data-total-pages="<?php echo e($total_pages); ?>"
    data-total="<?php echo e($total_leads); ?>">
    <li class="kan-ban-col-wrapper">
        <div class="border-right panel_s">
            <?php
        $status_color = '';
    if (!empty($status['color'])) {
        $status_color = 'style="background:' . $status['color'] . ';border:1px solid ' . $status['color'] . '"';
    } ?>
            <div class="panel-heading tw-bg-neutral-700 tw-text-white"
                <?php if ($status['isdefault'] == 1) { ?>data-toggle="tooltip"
                data-title="<?php echo _l('leads_converted_to_client') . ' - ' . _l('client'); ?>" <?php } ?>
                <?php echo $status_color; ?> data-status-id="<?php echo e($status['id']); ?>">
                <i class="fa fa-reorder pointer"></i>
                <span class="heading pointer tw-ml-1" <?php if ($is_admin) { ?>
                    data-order="<?php echo e($status['statusorder']); ?>" data-color="<?php echo e($status['color']); ?>"
                    data-name="<?php echo e($status['name']); ?>"
                    onclick="edit_status(this,<?php echo e($status['id']); ?>); return false;"
                    <?php } ?>><?php echo e($status['name']); ?>
                </span> -
                <?php echo app_format_money(
        $summary[$statusSummaryIndex = array_search($status['id'], array_column($summary, 'id'))]['value'],
        $base_currency
    ); ?> - <small><?php echo $summary[$statusSummaryIndex]['total'] . ' ' . _l('leads') ?></small>
                <a href="#" onclick="return false;" class="pull-right color-white kanban-color-picker kanban-stage-color-picker<?php if ($status['isdefault'] == 1) {
        echo ' kanban-stage-color-picker-last';
    } ?>" data-placement="bottom" data-toggle="popover" data-content="
            <div class='text-center'>
              <button type='button' return false;' class='btn btn-primary btn-block mtop10 new-lead-from-status'>
                <?php echo _l('new_lead'); ?>
              </button>
            </div>
            <?php if (is_admin()) {?>
            <hr />
            <div class='kan-ban-settings cpicker-wrapper'>
              <?php echo $settings; ?>
            </div><?php } ?>" data-html="true" data-trigger="focus">
                    <i class="fa fa-angle-down"></i>
                </a>
            </div>
            <div class="kan-ban-content-wrapper">
                <div class="kan-ban-content">
                    <ul class="status leads-status sortable" data-lead-status-id="<?php echo e($status['id']); ?>">
                        <?php
                foreach ($leads as $lead) {
                    $this->load->view('admin/leads/_kan_ban_card', ['lead' => $lead, 'status' => $status, 'base_currency' => $base_currency]);
                } ?>
                        <?php if ($total_leads > 0) { ?>
                        <li class="text-center not-sortable kanban-load-more"
                            data-load-status="<?php echo e($status['id']); ?>">
                            <a href="#" class="btn btn-default btn-block<?php if ($total_pages <= 1 || $kanBan->getPage() === $total_pages) {
                    echo ' disabled';
                } ?>" data-page="<?php echo $kanBan->getPage(); ?>"
                                onclick="kanban_load_more(<?php echo e($status['id']); ?>, this, 'leads/leads_kanban_load_more', 315, 360); return false;"
                                ;>
                                <?php echo _l('load_more'); ?>
                            </a>
                        </li>
                        <?php } ?>
                        <li class="text-center not-sortable mtop30 kanban-empty<?php if ($total_leads > 0) {
                    echo ' hide';
                } ?>">
                            <h4>
                                <i class="fa-solid fa-circle-notch" aria-hidden="true"></i><br /><br />
                                <?php echo _l('no_leads_found'); ?>
                            </h4>
                        </li>
                    </ul>
                </div>
            </div>
    </li>
</ul>
<?php $i++;
} ?>













<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<div class="modal-header">
    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
        <span aria-hidden="true">&times;</span>
    </button>
    <div class="tw-flex tw-justify-between tw-items-center">
        <h4 class="modal-title tw-flex tw-items-center">
            <?php if (isset($lead)) { ?>
            <?php $name = ! empty($lead->name) ? $lead->name : (! empty($lead->company) ? $lead->company : _l('lead')); ?>
            #<?= $lead->id; ?> -
            <?= $name; ?>
            <div class="tw-ml-3 -tw-mt-px">
                <?php if ($lead->lost == 1) { ?>
                <span
                    class="label label-danger"><?= _l('lead_lost'); ?></span>
                <?php } elseif ($lead->junk == 1) { ?>
                <span
                    class="label label-warning"><?= _l('lead_junk'); ?></span>
                <?php } elseif (total_rows(db_prefix() . 'clients', ['leadid' => $lead->id])) { ?>
                <span
                    class="label label-success"><?= _l('lead_is_client'); ?></span>
                <?php } ?>
            </div>
            <?php } else { ?>
            <?= _l('add_new', _l('lead_lowercase')); ?>
            <?php } ?>
        </h4>

        <?php if (isset($lead)) { ?>
        <a href="#"
            class="lead-print-btn tw-text-neutral-500 hover:tw-text-neutral-800 focus:tw-text-neutral-800 tw-mt-1 tw-space-x-1.5 tw-mx-4"
            onclick="print_lead_information(); return false;">
            <i class="fa-solid fa-print"></i>
            <span><?= _l('print'); ?></span>
        </a>
        <?php } ?>
    </div>

</div>
<div class="modal-body">
    <div class="row">
        <div class="col-md-12">
            <?php if (isset($lead)) {
                echo form_hidden('leadid', $lead->id);
            } ?>
            <div class="top-lead-menu">
                <?php if (isset($lead)) { ?>
                <div class="horizontal-scrollable-tabs tw-mb-10">
                    <div class="scroller arrow-left"><i class="fa fa-angle-left"></i></div>
                    <div class="scroller arrow-right"><i class="fa fa-angle-right"></i></div>
                    <div class="horizontal-tabs">
                        <ul class="nav nav-tabs nav-tabs-horizontal nav-tabs-segmented<?= ! isset($lead) ? ' lead-new' : ''?>"
                            role="tablist">
                            <li role="presentation" class="active">
                                <a href="#tab_lead_profile" aria-controls="tab_lead_profile" role="tab"
                                    data-toggle="tab">
                                    <i class="fa-regular fa-user menu-icon"></i>
                                    <?= _l('lead_profile'); ?>
                                </a>
                            </li>
                            <?php if (isset($lead)) { ?>
                            <?php if (count($mail_activity) > 0 || isset($show_email_activity) && $show_email_activity) { ?>
                            <li role="presentation">
                                <a href="#tab_email_activity" aria-controls="tab_email_activity" role="tab"
                                    data-toggle="tab">
                                    <i class="fa-regular fa-envelope menu-icon"></i>
                                    <?= hooks()->apply_filters('lead_email_activity_subject', _l('lead_email_activity')); ?>
                                </a>
                            </li>
                            <?php } ?>
                            <li role="presentation">
                                <a href="#tab_proposals_leads"
                                    onclick="initDataTable('.table-proposals-lead', admin_url + 'proposals/proposal_relations/' + <?= e($lead->id); ?> + '/lead','undefined', 'undefined','undefined',[6,'desc']);"
                                    aria-controls="tab_proposals_leads" role="tab" data-toggle="tab">
                                    <i class="fa-regular fa-file-lines menu-icon"></i>
                                    <?= _l('proposals');
                                if ($total_proposals > 0) {
                                    echo ' <span class="badge">' . $total_proposals . '</span>';
                                }
                                ?>
                                </a>
                            </li>
                            <li role="presentation">
                                <a href="#tab_tasks_leads"
                                    onclick="init_rel_tasks_table(<?= e($lead->id); ?>,'lead','.table-rel-tasks-leads');"
                                    aria-controls="tab_tasks_leads" role="tab" data-toggle="tab">
                                    <i class="fa-regular fa-circle-check menu-icon"></i>
                                    <?= _l('tasks');
                                if ($total_tasks > 0) {
                                    echo ' <span class="badge">' . $total_tasks . '</span>';
                                }
                                ?>
                                </a>
                            </li>
                            <li role="presentation">
                                <a href="#attachments" aria-controls="attachments" role="tab" data-toggle="tab">
                                    <i class="fa-solid fa-paperclip menu-icon"></i>
                                    <?= _l('lead_attachments');
                                if ($total_attachments > 0) {
                                    echo ' <span class="badge">' . $total_attachments . '</span>';
                                }
                                ?>
                                </a>
                            </li>
                            <li role="presentation">
                                <a href="#lead_reminders"
                                    onclick="initDataTable('.table-reminders-leads', admin_url + 'misc/get_reminders/' + <?= e($lead->id); ?> + '/' + 'lead', undefined, undefined,undefined,[1, 'asc']);"
                                    aria-controls="lead_reminders" role="tab" data-toggle="tab">
                                    <i class="fa-regular fa-bell menu-icon"></i>
                                    <?= _l('leads_reminders_tab');
                                if ($total_reminders > 0) {
                                    echo ' <span class="badge">' . $total_reminders . '</span>';
                                }
                                ?>
                                </a>
                            </li>
                            <li role="presentation">
                                <a href="#lead_notes" aria-controls="lead_notes" role="tab" data-toggle="tab">
                                    <i class="fa-regular fa-note-sticky menu-icon"></i>
                                    <?= _l('lead_add_edit_notes');
                                if ($total_notes > 0) {
                                    echo ' <span class="badge">' . $total_notes . '</span>';
                                }
                                ?>
                                </a>
                            </li>
                            <li role="presentation">
                                <a href="#lead_activity" aria-controls="lead_activity" role="tab" data-toggle="tab">
                                    <i class="fa-solid fa-grip-lines-vertical menu-icon"></i>
                                    <?= _l('lead_add_edit_activity'); ?>
                                </a>
                            </li>
                            <?php if (is_gdpr() && (get_option('gdpr_enable_lead_public_form') == '1' || get_option('gdpr_enable_consent_for_leads') == '1')) { ?>
                            <li role="presentation">
                                <a href="#gdpr" aria-controls="gdpr" role="tab" data-toggle="tab">
                                    <i class="fa-solid fa-gavel menu-icon"></i>
                                    <?= _l('gdpr_short'); ?>
                                </a>
                            </li>
                            <?php } ?>
                            <?php } ?>
                            <?php hooks()->do_action('after_lead_lead_tabs', $lead ?? null); ?>
                        </ul>
                    </div>
                </div>
                <?php } ?>
            </div>
            <!-- Tab panes -->
            <div class="tab-content">
                <!-- from leads modal -->
                <div role="tabpanel" class="tab-pane active" id="tab_lead_profile">
                    <?php $this->load->view('admin/leads/profile'); ?>
                </div>
                <?php if (isset($lead)) { ?>
                <?php if (count($mail_activity) > 0 || isset($show_email_activity) && $show_email_activity) { ?>
                <div role="tabpanel" class="tab-pane" id="tab_email_activity">
                    <?php hooks()->do_action('before_lead_email_activity', ['lead' => $lead, 'email_activity' => $mail_activity]); ?>
                    <?php foreach ($mail_activity as $_mail_activity) { ?>
                    <div class="lead-email-activity">
                        <div class="media-left">
                            <i class="fa-regular fa-envelope"></i>
                        </div>
                        <div class="media-body">
                            <h4 class="bold no-margin lead-mail-activity-subject">
                                <?= e($_mail_activity['subject']); ?>
                                <br />
                                <small
                                    class="text-muted display-block mtop5 font-medium-xs"><?= e(_dt($_mail_activity['dateadded'])); ?></small>
                            </h4>
                            <div class="lead-mail-activity-body">
                                <hr />
                                <?= process_text_content_for_display($_mail_activity['body']); ?>
                            </div>
                            <hr />
                        </div>
                    </div>
                    <div class="clearfix"></div>
                    <?php } ?>
                    <?php hooks()->do_action('after_lead_email_activity', ['lead_id' => $lead->id, 'emails' => $mail_activity]); ?>
                </div>
                <?php } ?>
                <?php if (is_gdpr() && (get_option('gdpr_enable_lead_public_form') == '1' || get_option('gdpr_enable_consent_for_leads') == '1' || (get_option('gdpr_data_portability_leads') == '1') && is_admin())) { ?>
                <div role="tabpanel" class="tab-pane" id="gdpr">
                    <?php if (get_option('gdpr_enable_lead_public_form') == '1') { ?>
                    <a href="<?= e($lead->public_url); ?>"
                        target="_blank" class="mtop5">
                        <?= _l('view_public_form'); ?>
                    </a>
                    <?php } ?>
                    <?php if (get_option('gdpr_data_portability_leads') == '1' && is_admin()) { ?>
                    <?php
                                if (get_option('gdpr_enable_lead_public_form') == '1') {
                                    echo ' | ';
                                }
                        ?>
                    <a
                        href="<?= admin_url('leads/export/' . $lead->id); ?>">
                        <?= _l('dt_button_export'); ?>
                    </a>
                    <?php } ?>
                    <?php if (get_option('gdpr_enable_lead_public_form') == '1' || (get_option('gdpr_data_portability_leads') == '1' && is_admin())) { ?>
                    <hr class="-tw-mx-3.5" />
                    <?php } ?>
                    <?php if (get_option('gdpr_enable_consent_for_leads') == '1') { ?>
                    <h4 class="no-mbot">
                        <?= _l('gdpr_consent'); ?>
                    </h4>
                    <?php $this->load->view('admin/gdpr/lead_consent'); ?>
                    <hr />
                    <?php } ?>
                </div>
                <?php } ?>
                <div role="tabpanel" class="tab-pane" id="lead_activity">
                    <div>
                        <div class="activity-feed">
                            <?php foreach ($activity_log as $log) { ?>
                            <div class="feed-item">
                                <div class="date">
                                    <span class="text-has-action" data-toggle="tooltip"
                                        data-title="<?= e(_dt($log['date'])); ?>">
                                        <?= e(time_ago($log['date'])); ?>
                                    </span>
                                </div>
                                <div class="text">
                                    <?php if ($log['staffid'] != 0) { ?>
                                    <a
                                        href="<?= admin_url('profile/' . $log['staffid']); ?>">
                                        <?= staff_profile_image($log['staffid'], ['staff-profile-xs-image pull-left mright5']);
                                        ?>
                                    </a>
                                    <?php
                                    }
                                $additional_data = '';
                                if (! empty($log['additional_data'])) {
                                    $additional_data = unserialize($log['additional_data']);
                                    echo ($log['staffid'] == 0) ? _l($log['description'], $additional_data) : e($log['full_name']) . ' - ' . _l($log['description'], $additional_data);
                                } else {
                                    echo e($log['full_name']) . ' - ';

                                    if ($log['custom_activity'] == 0) {
                                        echo e(_l($log['description']));
                                    } else {
                                        echo process_text_content_for_display(_l($log['description'], '', false));
                                    }
                                }
                                ?>
                                </div>
                            </div>
                            <?php } ?>
                        </div>
                        <div class="col-md-12">
                            <?= render_textarea('lead_activity_textarea', '', '', ['placeholder' => _l('enter_activity')], [], 'mtop15'); ?>
                            <div class="text-right">
                                <button id="lead_enter_activity"
                                    class="btn btn-primary"><?= _l('submit'); ?></button>
                            </div>
                        </div>
                        <div class="clearfix"></div>
                    </div>
                </div>
                <div role="tabpanel" class="tab-pane" id="tab_proposals_leads">
                    <?php if (staff_can('create', 'proposals')) { ?>
                    <a href="<?= admin_url('proposals/proposal?rel_type=lead&rel_id=' . $lead->id); ?>"
                        class="btn btn-primary mbot25"><?= _l('new_proposal'); ?></a>
                    <?php } ?>
                    <?php if (total_rows(db_prefix() . 'proposals', ['rel_type' => 'lead', 'rel_id' => $lead->id]) > 0 && (staff_can('create', 'proposals') || staff_can('edit', 'proposals'))) { ?>
                    <a href="#" class="btn btn-primary mbot25" data-toggle="modal"
                        data-target="#sync_data_proposal_data"><?= _l('sync_data'); ?></a>
                    <?php $this->load->view('admin/proposals/sync_data', ['related' => $lead, 'rel_id' => $lead->id, 'rel_type' => 'lead']); ?>
                    <?php } ?>
                    <?php
                        $table_data = [
                            _l('proposal') . ' #',
                            _l('proposal_subject'),
                            _l('proposal_total'),
                            _l('proposal_date'),
                            _l('proposal_open_till'),
                            _l('tags'),
                            _l('proposal_date_created'),
                            _l('proposal_status'),
                        ];
                    $custom_fields = get_custom_fields('proposal', ['show_on_table' => 1]);

                    foreach ($custom_fields as $field) {
                        array_push($table_data, [
                            'name'     => $field['name'],
                            'th_attrs' => ['data-type' => $field['type'], 'data-custom-field' => 1],
                        ]);
                    }
                    $table_data = hooks()->apply_filters('proposals_relation_table_columns', $table_data);
                    render_datatable($table_data, 'proposals-lead', [], [
                        'data-last-order-identifier' => 'proposals-relation',
                        'data-default-order'         => get_table_last_order('proposals-relation'),
                    ]);
                    ?>
                </div>
                <div role="tabpanel" class="tab-pane" id="tab_tasks_leads">
                    <?php init_relation_tasks_table(['data-new-rel-id' => $lead->id, 'data-new-rel-type' => 'lead'], 'tasksFilters'); ?>
                </div>
                <div role="tabpanel" class="tab-pane" id="lead_reminders">
                    <a href="#" data-toggle="modal" class="btn btn-primary"
                        data-target=".reminder-modal-lead-<?= e($lead->id); ?>"><i
                            class="fa-regular fa-bell"></i>
                        <?= _l('lead_set_reminder_title'); ?></a>
                    <hr />
                    <?php render_datatable([_l('reminder_description'), _l('reminder_date'), _l('reminder_staff'), _l('reminder_is_notified')], 'reminders-leads'); ?>
                </div>
                <div role="tabpanel" class="tab-pane" id="attachments">
                    <?= form_open('admin/leads/add_lead_attachment', ['class' => 'dropzone mtop15 mbot15', 'id' => 'lead-attachment-upload']); ?>
                    <?= form_close(); ?>
                    <?php if (get_option('dropbox_app_key') != '') { ?>
                    <hr />
                    <div class=" pull-left">
                        <?php if (count($lead->attachments) > 0) { ?>
                        <a href="<?= admin_url('leads/download_files/' . $lead->id); ?>"
                            class="bold">
                            <?= _l('download_all'); ?>
                            (.zip)
                        </a>
                        <?php } ?>
                    </div>
                    <div class="tw-flex tw-justify-end tw-items-center tw-space-x-2">
                        <button class="gpicker">
                            <i class="fa-brands fa-google" aria-hidden="true"></i>
                            <?= _l('choose_from_google_drive'); ?>
                        </button>
                        <div id="dropbox-chooser-lead"></div>
                    </div>
                    <div class=" clearfix"></div>
                    <?php } ?>
                    <?php if (count($lead->attachments) > 0) { ?>
                    <div class="mtop20" id="lead_attachments">
                        <?php $this->load->view('admin/leads/leads_attachments_template', ['attachments' => $lead->attachments]); ?>
                    </div>
                    <?php } ?>
                </div>
                <div role="tabpanel" class="tab-pane" id="lead_notes">
                    <?= form_open(admin_url('leads/add_note/' . $lead->id), ['id' => 'lead-notes']); ?>
                    <div class="form-group">
                        <textarea id="lead_note_description" name="lead_note_description" class="form-control"
                            rows="4"></textarea>
                    </div>
                    <div class="lead-select-date-contacted hide">
                        <?= render_datetime_input('custom_contact_date', 'lead_add_edit_datecontacted', '', ['data-date-end-date' => date('Y-m-d')]); ?>
                    </div>
                    <div class="radio radio-primary">
                        <input type="radio" name="contacted_indicator" id="contacted_indicator_yes" value="yes">
                        <label
                            for="contacted_indicator_yes"><?= _l('lead_add_edit_contacted_this_lead'); ?></label>
                    </div>
                    <div class="radio radio-primary">
                        <input type="radio" name="contacted_indicator" id="contacted_indicator_no" value="no" checked>
                        <label
                            for="contacted_indicator_no"><?= _l('lead_not_contacted'); ?></label>
                    </div>
                    <button type="submit"
                        class="btn btn-primary pull-right"><?= _l('lead_add_edit_add_note'); ?></button>
                    <?= form_close(); ?>
                    <div class="clearfix"></div>
                    <hr />
                    <?php
                    $len = count($notes);
                    $i   = 0;

                    foreach ($notes as $note) { ?>
                    <div class="media lead-note">
                        <a href="<?= admin_url('profile/' . $note['addedfrom']); ?>"
                            target="_blank">
                            <?= staff_profile_image($note['addedfrom'], ['staff-profile-image-small', 'pull-left mright10']); ?>
                        </a>
                        <div class="media-body">
                            <?php if ($note['addedfrom'] == get_staff_user_id() || is_admin()) { ?>
                            <a href="#" class="pull-right text-muted"
                                onclick="delete_lead_note(this,<?= e($note['id']); ?>, <?= e($lead->id); ?>);return false;">
                                <i class="fa-regular fa-trash-can"></i>
                            </a>

                            <a href="#" class="pull-right text-muted tw-mr-3"
                                onclick="toggle_edit_note(<?= e($note['id']); ?>);return false;">
                                <i class="fa-regular fa-pen-to-square"></i>
                            </a>
                            <?php } ?>

                            <a href="<?= admin_url('profile/' . $note['addedfrom']); ?>"
                                target="_blank">
                                <h5 class="media-heading tw-font-semibold tw-mb-0">
                                    <?php if (! empty($note['date_contacted'])) { ?>
                                    <span data-toggle="tooltip"
                                        data-title="<?= e(_dt($note['date_contacted'])); ?>">
                                        <i class="fa fa-phone-square text-success" aria-hidden="true"></i>
                                    </span>
                                    <?php } ?>
                                    <?= e(get_staff_full_name($note['addedfrom'])); ?>
                                </h5>
                                <span class="tw-text-sm tw-text-neutral-500">
                                    <?= e(_l('lead_note_date_added', _dt($note['dateadded']))); ?>
                                </span>
                            </a>

                            <div data-note-description="<?= e($note['id']); ?>"
                                class="text-muted tw-leading-relaxed mtop10">
                                <?= process_text_content_for_display($note['description']); ?>
                            </div>
                            <div data-note-edit-textarea="<?= e($note['id']); ?>"
                                class="hide mtop15">
                                <?= render_textarea('note', '', $note['description']); ?>
                                <div class="text-right">
                                    <button type="button" class="btn btn-default"
                                        onclick="toggle_edit_note(<?= e($note['id']); ?>);return false;">
                                        <?= _l('cancel'); ?>
                                    </button>
                                    <button type="button" class="btn btn-primary"
                                        onclick="edit_note(<?= e($note['id']); ?>);">
                                        <?= _l('update_note'); ?>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php if ($i >= 0 && $i != $len - 1) {
                            echo '<hr />';
                        }
                        ?>
                    </div>
                    <?php $i++;
                    } ?>
                </div>
                <?php } ?>
                <?php hooks()->do_action('after_lead_tabs_content', $lead ?? null); ?>
            </div>
        </div>
    </div>
</div>
<?php hooks()->do_action('lead_modal_profile_bottom', (isset($lead) ? $lead->id : '')); ?>
















<?php defined('BASEPATH') or exit('No direct script access allowed');
$data = '<div class="row">';
foreach($attachments as $attachment) {
    $attachment_url = site_url('download/file/lead_attachment/'.$attachment['id']);
    if(!empty($attachment['external'])){
        $attachment_url = $attachment['external_link'];
    }
    $data .= '<div class="display-block lead-attachment-wrapper">';
    $data .= '<div class="col-md-10">';
    $data .= '<div class="pull-left"><i class="'.e(get_mime_class($attachment['filetype'])).'"></i></div>';
    $data .= '<a href="'.e($attachment_url).'" target="_blank">'.e($attachment['file_name']).'</a>';
    $data .= '<p class="text-muted">'.e($attachment["filetype"]).'</p>';
    $data .= '</div>';
    $data .= '<div class="col-md-2 text-right">';
    if($attachment['staffid'] == get_staff_user_id() || is_admin()){
    $data .= '<a href="#" class="text-danger" onclick="delete_lead_attachment(this,'.$attachment['id'].', '.$attachment['rel_id'].'); return false;"><i class="fa fa fa-times"></i></a>';
    }
    $data .= '</div>';
    $data .= '<div class="clearfix"></div><hr/>';
    $data .= '</div>';
}
$data .= '</div>';
echo $data;


















<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="_buttons tw-mb-2 sm:tw-mb-4">
                    <a href="#" onclick="init_lead(); return false;"
                        class="btn btn-primary mright5 pull-left display-block">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?php echo _l('new_lead'); ?>
                    </a>
                    <?php if (is_admin() || get_option('allow_non_admin_members_to_import_leads') == '1') { ?>
                    <a href="<?php echo admin_url('leads/import'); ?>"
                        class="btn btn-primary pull-left display-block hidden-xs">
                        <i class="fa-solid fa-upload tw-mr-1"></i>
                        <?php echo _l('import_leads'); ?>
                    </a>
                    <?php } ?>
                    <div class="row">
                        <div class="col-sm-5 ">
                            <a href="#" class="btn btn-default btn-with-tooltip" data-toggle="tooltip"
                                data-title="<?php echo _l('leads_summary'); ?>" data-placement="top"
                                onclick="slideToggle('.leads-overview'); return false;"><i
                                    class="fa fa-bar-chart"></i></a>
                            <a href="<?php echo admin_url('leads/switch_kanban/' . $switch_kanban); ?>"
                                class="btn btn-default mleft5 hidden-xs" data-toggle="tooltip" data-placement="top"
                                data-title="<?php echo $switch_kanban == 1 ? _l('leads_switch_to_kanban') : _l('switch_to_list_view'); ?>">
                                <?php if ($switch_kanban == 1) { ?>
                                <i class="fa-solid fa-grip-vertical"></i>
                                <?php } else { ?>
                                <i class="fa-solid fa-table-list"></i>
                                <?php }; ?>
                            </a>
                        </div>
                        <div class="col-sm-4 col-xs-12 pull-right leads-search">
                            <?php if ($this->session->userdata('leads_kanban_view') == 'true') { ?>
                            <div data-toggle="tooltip" data-placement="top"
                                data-title="<?php echo _l('search_by_tags'); ?>">
                                <?php echo render_input('search', '', '', 'search', ['data-name' => 'search', 'onkeyup' => 'leads_kanban();', 'placeholder' => _l('leads_search')], [], 'no-margin') ?>
                            </div>
                            <?php } else { ?>
                            <div id="vueApp" class="tw-inline pull-right">
								<div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
									<a href="#" class="btn btn-default leads-table-order-btn" data-toggle="modal" data-target="#leads-table-order-modal" title="<?php echo _l('leads_table_order_configure'); ?>">
										<i class="fa fa-columns tw-mr-1"></i> <?php echo _l('leads_table_order_configure'); ?>
									</a>
								</div>
                                <app-filters
                                    id="<?php echo $table->id(); ?>"
                                    view="<?php echo $table->viewName(); ?>"
                                    :rules="<?php echo app\services\utilities\Js::from($this->input->get('status') ? $table->findRule('status')->setValue([$this->input->get('status')]) : []); ?>"
                                    :saved-filters="<?php echo $table->filtersJs(); ?>"
                                    :available-rules="<?php echo $table->rulesJs(); ?>">
                                </app-filters>
                            </div>
                            <?php } ?>
                            <?php echo form_hidden('sort_type'); ?>
                            <?php echo form_hidden('sort', (get_option('default_leads_kanban_sort') != '' ? get_option('default_leads_kanban_sort_type') : '')); ?>
                        </div>
                    </div>
                    <div class="clearfix"></div>
                    <div class="hide leads-overview tw-mt-2 sm:tw-mt-4 tw-mb-4 sm:tw-mb-0">
                        <h4 class="tw-mt-0 tw-font-semibold tw-text-lg">
                            <?php echo _l('leads_summary'); ?>
                        </h4>
                        <div class="tw-flex tw-flex-wrap tw-flex-col lg:tw-flex-row tw-w-full tw-gap-3 lg:tw-gap-6">
                            <?php
                           foreach ($summary as $status) { ?>
                            <div
                                class="lg:tw-border-r lg:tw-border-solid lg:tw-border-neutral-300 tw-flex-1 tw-flex tw-items-center last:tw-border-r-0">
                                <span class="tw-font-semibold tw-mr-3 rtl:tw-ml-3 tw-text-lg">
                                    <?php
                                          if (isset($status['percent'])) {
                                              echo '<span data-toggle="tooltip" data-title="' . $status['total'] . '">' . $status['percent'] . '%</span>';
                                          } else {
                                              // Is regular status
                                              echo $status['total'];
                                          }
                                       ?>
                                </span>
                                <span style="color:<?php echo e($status['color']); ?>"
                                    class="<?php echo isset($status['junk']) || isset($status['lost']) ? 'text-danger' : ''; ?>">
                                    <?php echo e($status['name']); ?>
                                </span>
                            </div>
                            <?php } ?>
                        </div>

                    </div>
                </div>
                <div class="<?php echo $isKanBan ? '' : 'panel_s' ; ?>">
                    <div class="<?php echo $isKanBan ? '' : 'panel-body' ; ?>">
                        <div class="tab-content">
                            <?php
                        if ($isKanBan) { ?>
                            <div class="active kan-ban-tab" id="kan-ban-tab" style="overflow:auto;">
                                <div class="kanban-leads-sort">
                                    <span class="bold"><?php echo _l('leads_sort_by'); ?>: </span>
                                    <a href="#" onclick="leads_kanban_sort('dateadded'); return false"
                                        class="dateadded">
                                        <?php if (get_option('default_leads_kanban_sort') == 'dateadded') {
                            echo '<i class="kanban-sort-icon fa fa-sort-amount-' . strtolower(get_option('default_leads_kanban_sort_type')) . '"></i> ';
                        } ?><?php echo _l('leads_sort_by_datecreated'); ?>
                                    </a>
                                    |
                                    <a href="#" onclick="leads_kanban_sort('leadorder');return false;"
                                        class="leadorder">
                                        <?php if (get_option('default_leads_kanban_sort') == 'leadorder') {
                            echo '<i class="kanban-sort-icon fa fa-sort-amount-' . strtolower(get_option('default_leads_kanban_sort_type')) . '"></i> ';
                        } ?><?php echo _l('leads_sort_by_kanban_order'); ?>
                                    </a>
                                    |
                                    <a href="#" onclick="leads_kanban_sort('lastcontact');return false;"
                                        class="lastcontact">
                                        <?php if (get_option('default_leads_kanban_sort') == 'lastcontact') {
                            echo '<i class="kanban-sort-icon fa fa-sort-amount-' . strtolower(get_option('default_leads_kanban_sort_type')) . '"></i> ';
                        } ?><?php echo _l('leads_sort_by_lastcontact'); ?>
                                    </a>
                                </div>
                                <div class="row">
                                    <div class="container-fluid leads-kan-ban">
                                        <div id="kan-ban"></div>
                                    </div>
                                </div>
                            </div>
                            <?php } else { ?>
                            <div class="row" id="leads-table">
                                <div class="col-md-12">
                                    <a href="#" data-toggle="modal" data-table=".table-leads"
                                        data-target="#leads_bulk_actions"
                                        class="hide bulk-actions-btn table-btn"><?php echo _l('bulk_actions'); ?></a>
                                    <div class="modal fade bulk_actions" id="leads_bulk_actions" tabindex="-1"
                                        role="dialog">
                                        <div class="modal-dialog" role="document">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <button type="button" class="close" data-dismiss="modal"
                                                        aria-label="Close"><span
                                                            aria-hidden="true">&times;</span></button>
                                                    <h4 class="modal-title"><?php echo _l('bulk_actions'); ?></h4>
                                                </div>
                                                <div class="modal-body">
                                                    <?php if (staff_can('delete',  'leads')) { ?>
                                                    <div class="checkbox checkbox-danger">
                                                        <input type="checkbox" name="mass_delete" id="mass_delete">
                                                        <label
                                                            for="mass_delete"><?php echo _l('mass_delete'); ?></label>
                                                    </div>
                                                    <hr class="mass_delete_separator" />
                                                    <?php } ?>
                                                    <div id="bulk_change">
                                                        <div class="form-group">
                                                            <div class="checkbox checkbox-primary checkbox-inline">
                                                                <input type="checkbox" name="leads_bulk_mark_lost"
                                                                    id="leads_bulk_mark_lost" value="1">
                                                                <label for="leads_bulk_mark_lost">
                                                                    <?php echo _l('lead_mark_as_lost'); ?>
                                                                </label>
                                                            </div>
                                                        </div>
                                                        <?php echo render_select('move_to_status_leads_bulk', $statuses, ['id', 'name'], 'ticket_single_change_status'); ?>
                                                        <?php
                                             echo render_select('move_to_source_leads_bulk', $sources, ['id', 'name'], 'lead_source');
                                             echo render_datetime_input('leads_bulk_last_contact', 'leads_dt_last_contact');
                                             echo render_select('assign_to_leads_bulk', $staff, ['staffid', ['firstname', 'lastname']], 'leads_dt_assigned');
                                             ?>
                                                        <div class="form-group">
                                                            <?php echo '<p><b><i class="fa fa-tag" aria-hidden="true"></i> ' . _l('tags') . ':</b></p>'; ?>
                                                            <input type="text" class="tagsinput" id="tags_bulk"
                                                                name="tags_bulk" value="" data-role="tagsinput">
                                                        </div>
                                                        <hr />
                                                        <div class="form-group no-mbot">
                                                            <div class="radio radio-primary radio-inline">
                                                                <input type="radio" name="leads_bulk_visibility"
                                                                    id="leads_bulk_public" value="public">
                                                                <label for="leads_bulk_public">
                                                                    <?php echo _l('lead_public'); ?>
                                                                </label>
                                                            </div>
                                                            <div class="radio radio-primary radio-inline">
                                                                <input type="radio" name="leads_bulk_visibility"
                                                                    id="leads_bulk_private" value="private">
                                                                <label for="leads_bulk_private">
                                                                    <?php echo _l('private'); ?>
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-default"
                                                        data-dismiss="modal"><?php echo _l('close'); ?></button>
                                                    <a href="#" class="btn btn-primary"
                                                        onclick="leads_bulk_action(this); return false;"><?php echo _l('confirm'); ?></a>
                                                </div>
                                            </div>
                                            <!-- /.modal-content -->
                                        </div>
                                        <!-- /.modal-dialog -->
                                    </div>
                                    <!-- /.modal -->
                                    <?php

                              $table_data  = [];
                              $_table_data = [
                                '<span class="hide"> - </span><div class="checkbox mass_select_all_wrap"><input type="checkbox" id="mass_select_all" data-to-table="leads"><label></label></div>',
                                [
                                 'name'     => _l('the_number_sign'),
                                 'th_attrs' => ['class' => 'toggleable', 'id' => 'th-number'],
                               ],
                               [
                                 'name'     => _l('leads_dt_name'),
                                 'th_attrs' => ['class' => 'toggleable', 'id' => 'th-name'],
                               ],
                              ];
                              if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
                                  $_table_data[] = [
                                    'name'     => _l('gdpr_consent') . ' (' . _l('gdpr_short') . ')',
                                    'th_attrs' => ['id' => 'th-consent', 'class' => 'not-export'],
                                 ];
                              }
                              $_table_data[] = [
                               'name'     => _l('lead_company'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-company'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_dt_email'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-email'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_dt_phonenumber'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-phone'],
                              ];
                              $_table_data[] = [
                                 'name'     => _l('leads_dt_lead_value'),
                                 'th_attrs' => ['class' => 'toggleable', 'id' => 'th-lead-value'],
                                ];
                              $_table_data[] = [
                               'name'     => _l('tags'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-tags'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_dt_assigned'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-assigned'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_dt_status'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-status'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_source'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-source'],
                              ];
                              $_table_data[] = [
                               'name'     => _l('leads_dt_last_contact'),
                               'th_attrs' => ['class' => 'toggleable', 'id' => 'th-last-contact'],
                              ];
                              $_table_data[] = [
                                'name'     => _l('leads_dt_datecreated'),
                                'th_attrs' => ['class' => 'date-created toggleable', 'id' => 'th-date-created'],
                              ];
                              foreach ($_table_data as $_t) {
                                  array_push($table_data, $_t);
                              }
                              $custom_fields = get_custom_fields('leads', ['show_on_table' => 1]);
                              foreach ($custom_fields as $field) {
                                  array_push($table_data, [
                                   'name'     => $field['name'],
                                   'th_attrs' => ['data-type' => $field['type'], 'data-custom-field' => 1],
                                ]);
                              }
                              $table_data = hooks()->apply_filters('leads_table_columns', $table_data);
                               ?>
                                    <div class="panel-table-full">
                                        <?php
                                 render_datatable(
                                   $table_data,
                                   'leads',
                                   ['customizable-table number-index-2'],
                                   [
                                    'id'                         => 'leads',
                                    'data-last-order-identifier' => 'leads',
                                    'data-default-order'         => get_table_last_order('leads'),
                                 ]
                               );
                                ?>
                                    </div>
                                </div>
                            </div>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<script id="hidden-columns-table-leads" type="text/json">
<?php echo get_staff_meta(get_staff_user_id(), 'hidden-columns-table-leads'); ?>
</script>
<?php include_once(APPPATH . 'views/admin/leads/status.php'); ?>
<?php init_tail(); ?>
<script>
var openLeadID = '<?php echo e($leadid); ?>';
$(function() {
    leads_kanban();
    $('#leads_bulk_mark_lost').on('change', function() {
        $('#move_to_status_leads_bulk').prop('disabled', $(this).prop('checked') == true);
        $('#move_to_status_leads_bulk').selectpicker('refresh')
    });
    $('#move_to_status_leads_bulk').on('change', function() {
        if ($(this).selectpicker('val') != '') {
            $('#leads_bulk_mark_lost').prop('disabled', true);
            $('#leads_bulk_mark_lost').prop('checked', false);
        } else {
            $('#leads_bulk_mark_lost').prop('disabled', false);
        }
    });
});
</script>
<script src="<?= base_url('assets/js/leads_table_order.js'); ?>"></script>
</body>

</html>

















<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="tw-mb-2">
                    <a href="#" onclick="new_source(); return false;" class="btn btn-primary">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?= _l('lead_new_source'); ?>
                    </a>
                </div>
                <div class="panel_s">
                    <div class="panel-body panel-table-full">
                        <?php if (count($sources) > 0) { ?>
                        <table class="table dt-table" data-order-col="1" data-order-type="asc">
                            <thead>
                                <th><?= _l('id'); ?>
                                </th>
                                <th><?= _l('leads_sources_table_name'); ?>
                                </th>
                                <th><?= _l('options'); ?>
                                </th>
                            </thead>
                            <tbody>
                                <?php foreach ($sources as $source) { ?>
                                <tr>
                                    <td><?= e($source['id']); ?>
                                    </td>
                                    <td><a href="#" class="tw-font-medium"
                                            onclick="edit_source(this,<?= e($source['id']); ?>); return false"
                                            data-name="<?= e($source['name']); ?>"><?= e($source['name']); ?></a><br />
                                        <span class="text-muted">
                                            <?= _l('leads_table_total', total_rows(db_prefix() . 'leads', ['source' => $source['id']])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="tw-flex tw-items-center tw-space-x-2">
                                            <a href="#"
                                                onclick="edit_source(this,<?= e($source['id']); ?>); return false"
                                                data-name="<?= e($source['name']); ?>"
                                                class="tw-text-neutral-500 hover:tw-text-neutral-700 focus:tw-text-neutral-700">
                                                <i class="fa-regular fa-pen-to-square fa-lg"></i>
                                            </a>
                                            <a href="<?= admin_url('leads/delete_source/' . $source['id']); ?>"
                                                class="tw-text-neutral-500 hover:tw-text-neutral-700 focus:tw-text-neutral-700 _delete">
                                                <i class="fa-regular fa-trash-can fa-lg"></i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                        <?php } else { ?>
                        <p class="no-margin">
                            <?= _l('leads_sources_not_found'); ?>
                        </p>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<div class="modal fade" id="source" tabindex="-1" role="dialog">
    <div class="modal-dialog">
        <?= form_open(admin_url('leads/source')); ?>
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span
                        aria-hidden="true">&times;</span></button>
                <h4 class="modal-title">
                    <span
                        class="edit-title"><?= _l('edit_source'); ?></span>
                    <span
                        class="add-title"><?= _l('lead_new_source'); ?></span>
                </h4>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-12">
                        <div id="additional"></div>
                        <?= render_input('name', 'leads_source_add_edit_name'); ?>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default"
                    data-dismiss="modal"><?= _l('close'); ?></button>
                <button type="submit"
                    class="btn btn-primary"><?= _l('submit'); ?></button>
            </div>
        </div>
        <!-- /.modal-content -->
        <?= form_close(); ?>
    </div>
    <!-- /.modal-dialog -->
</div>
<!-- /.modal -->
<?php init_tail(); ?>
<script>
    $(function() {
        appValidateForm($('form'), {
            name: 'required'
        }, manage_leads_sources);
        $('#source').on('hidden.bs.modal', function(event) {
            $('#additional').html('');
            $('#source input[name="name"]').val('');
            $('.add-title').removeClass('hide');
            $('.edit-title').removeClass('hide');
        });
    });

    function manage_leads_sources(form) {
        var data = $(form).serialize();
        var url = form.action;
        $.post(url, data).done(function(response) {
            window.location.reload();
        });
        return false;
    }

    function new_source() {
        $('#source').modal('show');
        $('.edit-title').addClass('hide');
    }

    function edit_source(invoker, id) {
        var name = $(invoker).data('name');
        $('#additional').append(hidden_input('id', id));
        $('#source input[name="name"]').val(name);
        $('#source').modal('show');
        $('.add-title').addClass('hide');
    }
</script>
</body>

</html>


















<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="tw-mb-2">
                    <a href="#" onclick="new_status(); return false;" class="btn btn-primary">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?= _l('lead_new_status'); ?>
                    </a>
                </div>
                <div class="panel_s">
                    <div class="panel-body panel-table-full">
                        <?php if (count($statuses) > 0) { ?>
                        <table class="table dt-table" data-order-col="1" data-order-type="asc">
                            <thead>
                                <th><?= _l('id'); ?>
                                </th>
                                <th><?= _l('leads_status_table_name'); ?>
                                </th>
                                <th><?= _l('options'); ?>
                                </th>
                            </thead>
                            <tbody>
                                <?php foreach ($statuses as $status) { ?>
                                <tr>
                                    <td>
                                        <?= e($status['id']); ?>
                                    </td>
                                    <td>
                                        <a href="#" class="tw-font-medium"
                                            onclick="edit_status(this,<?= e($status['id']); ?>);return false;"
                                            data-color="<?= e($status['color']); ?>"
                                            data-name="<?= e($status['name']); ?>"
                                            data-order="<?= e($status['statusorder']); ?>"><?= e($status['name']); ?></a><br />
                                        <span class="text-muted">
                                            <?= _l('leads_table_total', total_rows(db_prefix() . 'leads', ['status' => $status['id']])); ?></span>
                                    </td>
                                    <td>
                                        <div class="tw-flex tw-items-center tw-space-x-2">
                                            <a href="#"
                                                onclick="edit_status(this,<?= e($status['id']); ?>);return false;"
                                                data-color="<?= e($status['color']); ?>"
                                                data-name="<?= e($status['name']); ?>"
                                                data-order="<?= e($status['statusorder']); ?>"
                                                class="tw-text-neutral-500 hover:tw-text-neutral-700 focus:tw-text-neutral-700">
                                                <i class="fa-regular fa-pen-to-square fa-lg"></i>
                                            </a>
                                            <?php if ($status['isdefault'] == 0) { ?>
                                            <a href="<?= admin_url('leads/delete_status/' . $status['id']); ?>"
                                                class="tw-text-neutral-500 hover:tw-text-neutral-700 focus:tw-text-neutral-700 _delete">
                                                <i class="fa-regular fa-trash-can fa-lg"></i>
                                            </a>
                                            <?php } ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                        <?php } else { ?>
                        <p class="no-margin">
                            <?= _l('lead_statuses_not_found'); ?>
                        </p>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php include_once APPPATH . 'views/admin/leads/status.php'; ?>
<?php init_tail(); ?>
</body>

</html>














<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<div class="lead-wrapper<?= $openEdit == true ? ' open-edit' : ''; ?>"
    <?= isset($lead) && ($lead->junk == 1 || $lead->lost == 1) ? 'lead-is-just-or-lost' : ''; ?>>

    <?php if (isset($lead)) { ?>
    <div class="tw-flex tw-items-center tw-justify-end tw-space-x-1.5">

        <?php
                       $client                  = false;
        $convert_to_client_tooltip_email_exists = '';
        if (total_rows(db_prefix() . 'contacts', ['email' => $lead->email]) > 0 && total_rows(db_prefix() . 'clients', ['leadid' => $lead->id]) == 0) {
            $convert_to_client_tooltip_email_exists = _l('lead_email_already_exists');
            $text                                   = _l('lead_convert_to_client');
        } elseif (total_rows(db_prefix() . 'clients', ['leadid' => $lead->id])) {
            $client = true;
        } else {
            $text = _l('lead_convert_to_client');
        }
        ?>

        <?php if ($lead_locked == false) { ?>
        <div
            class="lead-edit<?= isset($lead) ? ' hide' : ''; ?>">
            <button type="button" class="btn btn-primary lead-top-btn lead-save-btn"
                onclick="document.getElementById('lead-form-submit').click();">
                <?= _l('submit'); ?>
            </button>
        </div>
        <?php } ?>
        <?php if ($client && (staff_can('view', 'customers') || is_customer_admin(get_client_id_by_lead_id($lead->id)))) { ?>
        <a data-toggle="tooltip" class="btn btn-primary lead-top-btn lead-view" data-placement="top"
            title="<?= _l('lead_converted_edit_client_profile'); ?>"
            href="<?= admin_url('clients/client/' . get_client_id_by_lead_id($lead->id)); ?>">
            <i class="fa-regular fa-user"></i>
        </a>
        <?php } ?>
        <?php if (total_rows(db_prefix() . 'clients', ['leadid' => $lead->id]) == 0) { ?>
        <a href="#" data-toggle="tooltip"
            data-title="<?= e($convert_to_client_tooltip_email_exists); ?>"
            class="btn btn-primary lead-convert-to-customer lead-top-btn lead-view"
            onclick="convert_lead_to_customer(<?= e($lead->id); ?>); return false;">
            <i class="fa-regular fa-user"></i>
            <?= e($text); ?>
        </a>
        <?php } ?>

        <div
            class="<?= $lead_locked == true ? ' hide' : ''; ?>">
            <a href="#" lead-edit data-toggle="tooltip"
                data-title="<?= _l('edit'); ?>"
                class="btn btn-default lead-top-btn !tw-px-3">

                <i class="fa-regular fa-pen-to-square"></i>
            </a>
        </div>

        <div class="btn-group" id="lead-more-btn">
            <a href="#" class="btn btn-default dropdown-toggle lead-top-btn" data-toggle="dropdown" aria-haspopup="true"
                aria-expanded="false">
                <?= _l('more'); ?>
                <span class="caret"></span>
            </a>
            <ul class="dropdown-menu dropdown-menu-right" id="lead-more-dropdown">
                <?php if ($lead->junk == 0) {
                    if ($lead->lost == 0 && (total_rows(db_prefix() . 'clients', ['leadid' => $lead->id]) == 0)) { ?>
                <li>
                    <a href="#"
                        onclick="lead_mark_as_lost(<?= e($lead->id); ?>); return false;">
                        <i class="fa fa-mars"></i>
                        <?= _l('lead_mark_as_lost'); ?>
                    </a>
                </li>
                <?php } elseif ($lead->lost == 1) { ?>
                <li>
                    <a href="#"
                        onclick="lead_unmark_as_lost(<?= e($lead->id); ?>); return false;">
                        <i class="fa fa-smile-o"></i>
                        <?= _l('lead_unmark_as_lost'); ?>
                    </a>
                </li>
                <?php } ?>
                <?php
                } ?>
                <!-- mark as junk -->
                <?php if ($lead->lost == 0) {
                    if ($lead->junk == 0 && (total_rows(db_prefix() . 'clients', ['leadid' => $lead->id]) == 0)) { ?>
                <li>
                    <a href="#"
                        onclick="lead_mark_as_junk(<?= e($lead->id); ?>); return false;">
                        <i class="fa fa fa-times"></i>
                        <?= _l('lead_mark_as_junk'); ?>
                    </a>
                </li>
                <?php } elseif ($lead->junk == 1) { ?>
                <li>
                    <a href="#"
                        onclick="lead_unmark_as_junk(<?= e($lead->id); ?>); return false;">
                        <i class="fa fa-smile-o"></i>
                        <?= _l('lead_unmark_as_junk'); ?>
                    </a>
                </li>
                <?php } ?>
                <?php } ?>
                <?php if ((staff_can('delete', 'leads') && $lead_locked == false) || is_admin()) { ?>
                <li>
                    <a href="<?= admin_url('leads/delete/' . $lead->id); ?>"
                        class="text-danger delete-text _delete" data-toggle="tooltip" title="">
                        <i class="fa-regular fa-trash-can"></i>
                        <?= _l('lead_edit_delete_tooltip'); ?>
                    </a>
                </li>
                <?php } ?>
            </ul>
        </div>
    </div>
    <?php } ?>

    <div class="clearfix no-margin"></div>

    <?php if (isset($lead)) { ?>

    <div class="row mbot15" style="margin-top:12px;">
        <hr class="no-margin" />
    </div>

    <div class="alert alert-warning hide mtop20" role="alert" id="lead_proposal_warning">
        <?= _l('proposal_warning_email_change', [_l('lead_lowercase'), _l('lead_lowercase'), _l('lead_lowercase')]); ?>
        <hr />
        <a href="#"
            onclick="update_all_proposal_emails_linked_to_lead(<?= e($lead->id); ?>); return false;"
            class="alert-link">
            <?= _l('update_proposal_email_yes'); ?>
        </a>
        <br />
        <a href="#"
            onclick="init_lead_modal_data(<?= e($lead->id); ?>); return false;"
            class="alert-link">
            <?= _l('update_proposal_email_no'); ?>
        </a>
    </div>
    <?php } ?>
    <?= form_open((isset($lead) ? admin_url('leads/lead/' . $lead->id) : admin_url('leads/lead')), ['id' => 'lead_form']); ?>
    <div class="row">
        <div class="lead-view<?= ! isset($lead) ? ' hide' : ''; ?>"
            id="leadViewWrapper">
            <div class="col-md-4 col-xs-12 lead-information-col">
                <div class="lead-info-heading">
                    <h4>
                        <?= _l('lead_info'); ?>
                    </h4>
                </div>
                <dl>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_add_edit_name'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 lead-name">
                        <?= isset($lead) && $lead->name != '' ? e($lead->name) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_title'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->title != '' ? e($lead->title) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_add_edit_email'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->email != '' ? '<a href="mailto:' . e($lead->email) . '">' . e($lead->email) . '</a>' : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_website'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->website != '' ? '<a href="' . e(maybe_add_http($lead->website)) . '" target="_blank">' . e($lead->website) . '</a>' : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_add_edit_phonenumber'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->phonenumber != '' ? '<a href="tel:' . e($lead->phonenumber) . '">' . e($lead->phonenumber) . '</a>' : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_value'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->lead_value != 0 ? e(app_format_money($lead->lead_value, $base_currency->id)) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_company'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->company != '' ? e($lead->company) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_address'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->address != '' ? process_text_content_for_display($lead->address) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_city'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->city != '' ? e($lead->city) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_state'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->state != '' ? e($lead->state) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_country'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->country != 0 ? e(get_country($lead->country)->short_name) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_zip'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->zip != '' ? e($lead->zip) : '-' ?>
                    </dd>
                </dl>
            </div>
            <div class="col-md-4 col-xs-12 lead-information-col">
                <div class="lead-info-heading">
                    <h4>
                        <?= _l('lead_general_info'); ?>
                    </h4>
                </div>
                <dl>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500 no-mtop">
                        <?= _l('lead_add_edit_status'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-2 mbot15">
                        <?php if (isset($lead)) {
                            echo $lead->status_name != '' ? ('<span class="lead-status-' . e($lead->status) . ' label' . (empty($lead->color) ? ' label-default' : '') . '" style="color:' . e($lead->color) . ';border:1px solid ' . adjust_hex_brightness($lead->color, 0.4) . ';background: ' . adjust_hex_brightness($lead->color, 0.04) . ';">' . e($lead->status_name) . '</span>') : '-';
                        } else {
                            echo '-';
                        } ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_add_edit_source'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot15">
                        <?= isset($lead) && $lead->source_name != '' ? e($lead->source_name) : '-' ?>
                    </dd>
                    <?php if (! is_language_disabled()) { ?>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('localization_default_language'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot15">
                        <?= isset($lead) && $lead->default_language != '' ? e(ucfirst($lead->default_language)) : _l('system_default_string') ?>
                    </dd>
                    <?php } ?>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_add_edit_assigned'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot15">
                        <?= isset($lead) && $lead->assigned != 0 ? e(get_staff_full_name($lead->assigned)) : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('tags'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot10">
                        <?php if (isset($lead)) {
                            $tags = get_tags_in($lead->id, 'lead');
                            if (count($tags) > 0) {
                                echo render_tags($tags);
                                echo '<div class="clearfix"></div>';
                            } else {
                                echo '-';
                            }
                        } ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('leads_dt_datecreated'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->dateadded != '' ? '<span class="text-has-action" data-toggle="tooltip" data-title="' . e(_dt($lead->dateadded)) . '">' . e(time_ago($lead->dateadded)) . '</span>' : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('leads_dt_last_contact'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= isset($lead) && $lead->lastcontact != '' ? '<span class="text-has-action" data-toggle="tooltip" data-title="' . e(_dt($lead->lastcontact)) . '">' . e(time_ago($lead->lastcontact)) . '</span>' : '-' ?>
                    </dd>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_public'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot15">
                        <?php if (isset($lead)) {
                            if ($lead->is_public == 1) {
                                echo _l('lead_is_public_yes');
                            } else {
                                echo _l('lead_is_public_no');
                            }
                        } else {
                            echo '-';
                        } ?>
                    </dd>
                    <?php if (isset($lead) && $lead->from_form_id != 0) { ?>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('web_to_lead_form'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 mbot15">
                        <?= e($lead->form_data->name); ?>
                    </dd>
                    <?php } ?>
                </dl>
            </div>
            <div class="col-md-4 col-xs-12 lead-information-col">
                <?php if (total_rows(db_prefix() . 'customfields', ['fieldto' => 'leads', 'active' => 1]) > 0 && isset($lead)) { ?>
                <div class="lead-info-heading">
                    <h4>
                        <?= _l('custom_fields'); ?>
                    </h4>
                </div>
                <dl>
                    <?php foreach (get_custom_fields('leads') as $field) { ?>
                    <?php $value = get_custom_field_value($lead->id, $field['id'], 'leads'); ?>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500 no-mtop">
                        <?= e($field['name']); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1 tw-break-words">
                        <?= $value != '' ? $value : '-' ?>
                    </dd>
                    <?php } ?>
                    <?php } ?>
                </dl>
            </div>
            <div class="clearfix"></div>
            <div class="col-md-12">
                <dl>
                    <dt class="lead-field-heading tw-font-normal tw-text-neutral-500">
                        <?= _l('lead_description'); ?>
                    </dt>
                    <dd class="tw-text-neutral-900 tw-mt-1">
                        <?= process_text_content_for_display((isset($lead) && $lead->description != '' ? $lead->description : '-')); ?>
                    </dd>
                </dl>
            </div>
        </div>
        <div class="clearfix"></div>
        <div
            class="lead-edit<?= isset($lead) ? ' hide' : ''; ?>">
            <div class="col-md-4">
                <?php
        $selected = '';
if (isset($lead)) {
    $selected = $lead->status;
} elseif (isset($status_id)) {
    $selected = $status_id;
}
echo render_leads_status_select($statuses, $selected, 'lead_add_edit_status');
?>
            </div>
            <div class="col-md-4">
                <?= render_leads_source_select($sources, (isset($lead) ? $lead->source : get_option('leads_default_source')), 'lead_add_edit_source'); ?>
            </div>
            <div class="col-md-4">
                <?php
               $assigned_attrs = [];
$selected                      = (isset($lead) ? $lead->assigned : get_staff_user_id());
if (isset($lead)
   && $lead->assigned == get_staff_user_id()
   && $lead->addedfrom != get_staff_user_id()
   && ! is_admin($lead->assigned)
   && staff_cant('view', 'leads')
) {
    $assigned_attrs['disabled'] = true;
}
echo render_select('assigned', $members, ['staffid', ['firstname', 'lastname']], 'lead_add_edit_assigned', $selected, $assigned_attrs); ?>
            </div>
            <div class="clearfix"></div>
            <hr class="mtop5 mbot10" />
            <div class="col-md-12">
                <div class="form-group no-mbot" id="inputTagsWrapper">
                    <label for="tags" class="control-label"><i class="fa fa-tag" aria-hidden="true"></i>
                        <?= _l('tags'); ?></label>
                    <input type="text" class="tagsinput" id="tags" name="tags"
                        value="<?= isset($lead) ? prep_tags_input(get_tags_in($lead->id, 'lead')) : ''; ?>"
                        data-role="tagsinput">
                </div>
            </div>
            <div class="clearfix"></div>
            <hr class="no-mtop mbot15" />
            <div class="col-md-6">
                <?php $value = (isset($lead) ? $lead->name : ''); ?>
                <?= render_input('name', 'lead_add_edit_name', $value); ?>
                <?php $value = (isset($lead) ? $lead->title : ''); ?>
                <?= render_input('title', 'lead_title', $value); ?>
                <?php $value = (isset($lead) ? $lead->email : ''); ?>
                <?= render_input('email', 'lead_add_edit_email', $value); ?>
                <?php if ((isset($lead) && empty($lead->website)) || ! isset($lead)) {
                    $value = (isset($lead) ? $lead->website : '');
                    echo render_input('website', 'lead_website', $value);
                } else { ?>
                <div class="form-group">
                    <label
                        for="website"><?= _l('lead_website'); ?></label>
                    <div class="input-group">
                        <input type="text" name="website" id="website"
                            value="<?= e($lead->website); ?>"
                            class="form-control">
                        <div class="input-group-addon">
                            <span>
                                <a href="<?= e(maybe_add_http($lead->website)); ?>"
                                    target="_blank" tabindex="-1">
                                    <i class="fa fa-globe"></i>
                                </a>
                            </span>
                        </div>
                    </div>
                </div>
                <?php }
                $value = (isset($lead) ? $lead->phonenumber : ''); ?>
                <?= render_input('phonenumber', 'lead_add_edit_phonenumber', $value); ?>
                <div class="form-group">
                    <label
                        for="lead_value"><?= _l('lead_value'); ?></label>
                    <div class="input-group" data-toggle="tooltip"
                        title="<?= _l('lead_value_tooltip'); ?>">
                        <input type="number" class="form-control" name="lead_value" value="<?php if (isset($lead)) {
                            echo $lead->lead_value;
                        }?>">
                        <div class="input-group-addon">
                            <?= e($base_currency->symbol); ?>
                        </div>
                    </div>
                    </label>
                </div>
                <?php $value = (isset($lead) ? $lead->company : ''); ?>
                <?= render_input('company', 'lead_company', $value); ?>
            </div>
            <div class="col-md-6">
                <?php $value = (isset($lead) ? $lead->address : ''); ?>
                <?= render_textarea('address', 'lead_address', $value, ['rows' => 1, 'style' => 'height:36px;font-size:100%;']); ?>
                <?php $value = (isset($lead) ? $lead->city : ''); ?>
                <?= render_input('city', 'lead_city', $value); ?>
                <?php $value = (isset($lead) ? $lead->state : ''); ?>
                <?= render_input('state', 'lead_state', $value); ?>
                <?php
                           $countries = get_all_countries();
$customer_default_country             = get_option('customer_default_country');
$selected                             = (isset($lead) ? $lead->country : $customer_default_country);
echo render_select('country', $countries, ['country_id', ['short_name']], 'lead_country', $selected, ['data-none-selected-text' => _l('dropdown_non_selected_tex')]);
?>
                <?php $value = (isset($lead) ? $lead->zip : ''); ?>
                <?= render_input('zip', 'lead_zip', $value); ?>
                <?php if (! is_language_disabled()) { ?>
                <div class="form-group">
                    <label for="default_language"
                        class="control-label"><?= _l('localization_default_language'); ?></label>
                    <select name="default_language" data-live-search="true" id="default_language"
                        class="form-control selectpicker"
                        data-none-selected-text="<?= _l('dropdown_non_selected_tex'); ?>">
                        <option value="">
                            <?= _l('system_default_string'); ?>
                        </option>
                        <?php foreach ($this->app->get_available_languages() as $availableLanguage) {
                            $selected = '';
                            if (isset($lead)) {
                                if ($lead->default_language == $availableLanguage) {
                                    $selected = 'selected';
                                }
                            } ?>
                        <option value="<?= e($availableLanguage); ?>"
                            <?= e($selected); ?>>
                            <?= e(ucfirst($availableLanguage)); ?>
                        </option>
                        <?php
                        } ?>
                    </select>
                </div>
                <?php } ?>
            </div>
            <div class="col-md-12">
                <?php $value = (isset($lead) ? $lead->description : ''); ?>
                <?= render_textarea('description', 'lead_description', $value); ?>
                <div class="row">
                    <div class="col-md-12">
                        <?php if (! isset($lead)) { ?>
                        <div class="lead-select-date-contacted hide">
                            <?= render_datetime_input('custom_contact_date', 'lead_add_edit_datecontacted', '', ['data-date-end-date' => date('Y-m-d')]); ?>
                        </div>
                        <?php } else { ?>
                        <?= render_datetime_input('lastcontact', 'leads_dt_last_contact', _dt($lead->lastcontact), ['data-date-end-date' => date('Y-m-d')]); ?>
                        <?php } ?>
                        <div
                            class="checkbox-inline checkbox<?= isset($lead) ? ' hide' : ''; ?><?= isset($lead) && (is_lead_creator($lead->id) || staff_can('edit', 'leads')) ? ' lead-edit' : ''; ?>">
                            <input type="checkbox" name="is_public"
                                <?= isset($lead) ? 'checked' : ''; ?>
                            id="lead_public">
                            <label for="lead_public">
                                <?= _l('lead_public'); ?>
                            </label>
                        </div>
                        <?php if (! isset($lead)) { ?>
                        <div class="checkbox-inline checkbox checkbox-primary">
                            <input type="checkbox" name="contacted_today" id="contacted_today" checked>
                            <label for="contacted_today">
                                <?= _l('lead_add_edit_contacted_today'); ?>
                            </label>
                        </div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="col-md-12 mtop15">
                <?php $rel_id = (isset($lead) ? $lead->id : false); ?>
                <?= render_custom_fields('leads', $rel_id); ?>
            </div>
            <div class="clearfix"></div>
        </div>
    </div>
    <?php if (isset($lead)) { ?>
    <div class="lead-latest-activity tw-mb-3 lead-view">
        <div class="lead-info-heading">
            <h4><?= _l('lead_latest_activity'); ?>
            </h4>
        </div>
        <div id="lead-latest-activity" class="pleft5"></div>
    </div>
    <?php } ?>
    <?php if ($lead_locked == false) { ?>
    <div
        class="lead-edit<?= isset($lead) ? ' hide' : ''; ?>">
        <hr class="-tw-mx-5 tw-border-neutral-200" />
        <button type="submit" class="btn btn-primary pull-right lead-save-btn" id="lead-form-submit">
            <?= _l('submit'); ?>
        </button>
        <button type=" button" class="btn btn-default pull-right mright5" data-dismiss="modal">
            <?= _l('close'); ?>
        </button>
    </div>
    <?php } ?>
    <div class="clearfix"></div>
    <?= form_close(); ?>
</div>
<?php if (isset($lead) && $lead_locked == true) { ?>
<script>
    $(function() {
        // Set all fields to disabled if lead is locked
        $.each($('.lead-wrapper').find('input, select, textarea'), function() {
            $(this).attr('disabled', true);
            if ($(this).is('select')) {
                $(this).selectpicker('refresh');
            }
        });
    });
</script>
<?php } ?>


















<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<div class="modal fade" id="status" tabindex="-1" role="dialog">
   <div class="modal-dialog">
      <?php echo form_open(admin_url('leads/status'), ['id' => 'leads-status-form']); ?>
      <div class="modal-content">
         <div class="modal-header">
            <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
            <h4 class="modal-title">
               <span class="edit-title"><?php echo _l('edit_status'); ?></span>
               <span class="add-title"><?php echo _l('lead_new_status'); ?></span>
            </h4>
         </div>
         <div class="modal-body">
            <div class="row">
               <div class="col-md-12">
                  <div id="additional"></div>
                  <?php echo render_input('name', 'leads_status_add_edit_name'); ?>
                  <?php echo render_color_picker('color', _l('leads_status_color')); ?>
                  <?php echo render_input('statusorder', 'leads_status_add_edit_order', total_rows(db_prefix() . 'leads_status') + 1, 'number'); ?>
               </div>
            </div>
         </div>
         <div class="modal-footer">
            <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
            <button type="submit" class="btn btn-primary"><?php echo _l('submit'); ?></button>
         </div>
      </div>
      <!-- /.modal-content -->
      <?php echo form_close(); ?>
   </div>
   <!-- /.modal-dialog -->
</div>
<!-- /.modal -->
<script>
  window.addEventListener('load', function () {
    appValidateForm($("body").find('#leads-status-form'), {
        name: 'required'
    }, manage_leads_statuses);
    $('#status').on("hidden.bs.modal", function (event) {
        $('#additional').html('');
        $('#status input[name="name"]').val('');
        $('#status input[name="color"]').val('');
        $('#status input[name="statusorder"]').val('');
        $('.add-title').removeClass('hide');
        $('.edit-title').removeClass('hide');
        $('#status input[name="statusorder"]').val($('table tbody tr').length + 1);
    });
});

// Create lead new status
function new_status() {
    $('#status').modal('show');
    $('.edit-title').addClass('hide');
}

// Edit status function which init the data to the modal
function edit_status(invoker, id) {
    $('#additional').append(hidden_input('id', id));
    $('#status input[name="name"]').val($(invoker).data('name'));
    $('#status .colorpicker-input').colorpicker('setValue', $(invoker).data('color'));
    $('#status input[name="statusorder"]').val($(invoker).data('order'));
    $('#status').modal('show');
    $('.add-title').addClass('hide');
}

// Form handler function for leads status
function manage_leads_statuses(form) {
    var data = $(form).serialize();
    var url = form.action;
    $.post(url, data).done(function (response) {
        window.location.reload();
    });
    return false;
}
</script>







i  have this leads module already working properly 

now i am making a new module called deals in that module i have created below module





<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('deals/deals_model');
    }

    public function index()
    {
        if ($this->input->get('view') === 'kanban') {
            return $this->kanban();
        }

        $data['title'] = _l('deals');
        $data['deals'] = $this->deals_model->get_all();
        // dd($data);
        $this->load->view('list', $data);
    }

public function kanban()
{
        $data['title'] = _l('deals_kanban');
        $data['stages'] = $this->deals_model->get_stages_with_deals();
        $this->load->view('kanban', $data);
    }

    public function create()
    {
        //dd(admin_url('deals'));
        if ($this->input->post()) {
            $post = $this->input->post();
            if (empty($post['pipeline_id'])) {
                $post['pipeline_id'] = $this->deals_model->get_default_pipeline_id();
            }
            if (empty($post['stage_id'])) {
                $post['stage_id'] = $this->deals_model->get_default_open_stage_id();
            }
            $this->deals_model->create($post);
            set_alert('success', _l('added_successfully', _l('deal')));
            
            return redirect(admin_url('deals'));
        }
        $data['title'] = _l('new_deal');
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $this->load->view('create', $data);
    }

    public function edit($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($this->input->post()) {
            $post = $this->input->post();
            $this->deals_model->update_deal($id, $post);
            set_alert('success', _l('updated_successfully', _l('deal')));
            return redirect(admin_url('deals/view/'.$id));
        }
        $data['title'] = _l('edit');
        $data['deal'] = $deal;
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $data['stages'] = $this->deals_model->get_stages($deal['pipeline_id']);
        $this->load->view('create', $data);
    }


    /**
     * Update the deal value (amount) via AJAX.
     * Expects POST 'deal_value'.
     * Returns JSON: ['success' => bool]
     */
    public function update_amount($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $amount = $this->input->post('deal_value');
        if (!is_numeric($amount) || $amount < 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid amount']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['deal_value' => $amount]);
        echo json_encode(['success' => (bool)$updated]);
    }

    /**
     * Update the expected close date via AJAX.
     * Expects POST 'expected_close_date'.
     * Returns JSON: ['success' => bool]
     */
    public function update_close_date($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $date = $this->input->post('expected_close_date');
        if (!$date || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            echo json_encode(['success' => false, 'message' => 'Invalid date']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['expected_close_date' => $date]);
        echo json_encode(['success' => (bool)$updated]);
    }

    public function update_details($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($this->input->post()) {
            $post = $this->input->post();
            $this->deals_model->update_deal($id, $post);
            set_alert('success', _l('updated_successfully', _l('deal')));
        }
        redirect(admin_url('deals/view/'.$id));
    }

    public function upload_attachment($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        if ($this->input->post()) {
            $config['upload_path'] = './uploads/deals/';
            $config['allowed_types'] = 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx|txt';
            $config['max_size'] = 2048;
            
            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0755, true);
            }
            
            $this->load->library('upload', $config);
            
            if ($this->upload->do_upload('attachment')) {
                $upload_data = $this->upload->data();
                $attachment_data = [
                    'deal_id' => $id,
                    'file_name' => $upload_data['file_name'],
                    'description' => $this->input->post('description'),
                    'dateadded' => date('Y-m-d H:i:s')
                ];
                $this->deals_model->add_attachment($attachment_data);
                set_alert('success', _l('file_uploaded_successfully'));
            } else {
                set_alert('danger', $this->upload->display_errors());
            }
        }
        redirect(admin_url('deals/view/'.$id));
    }

    public function delete($id)
    {
        if ($this->deals_model->delete($id)) {
            set_alert('success', _l('deleted', _l('deal')));
        }
        redirect(admin_url('deals'));
    }

    public function table()
    {
        $this->app->get_table_data('deals', [
            'select' => [
                'd.id',
                'd.name',
                'd.company',
                'd.deal_value',
                'd.dateadded',
                'd.status_final',
                'dp.name as pipeline_name',
                'ds.name as stage_name',
                'CONCAT(s.firstname, " ", s.lastname) as assigned_staff'
            ],
            'from' => db_prefix() . 'deals d',
            'join' => [
                'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = d.pipeline_id',
                'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = d.stage_id',
                'LEFT JOIN ' . db_prefix() . 'staff s ON s.staffid = d.addedfrom'
            ]
        ]);
    }

    public function get_stages($pipeline_id)
    {
        $stages = $this->deals_model->get_stages($pipeline_id);
        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => true,
            'stages' => $stages
        ]));
    }

    public function bulk_action()
    {
        $action = $this->input->post('bulk_action');
        $ids = $this->input->post('ids');
        
        if (!$ids || !$action) {
            $this->output->set_content_type('application/json')->set_output(json_encode([
                'success' => false,
                'message' => _l('no_action_selected')
            ]));
            return;
        }

        $success = false;
        $message = '';

        switch ($action) {
            case 'delete':
                foreach ($ids as $id) {
                    $this->deals_model->delete($id);
                }
                $success = true;
                $message = _l('deals_deleted_successfully');
                break;
                
            case 'move_pipeline':
                $pipeline_id = $this->input->post('pipeline_id');
                $stage_id = $this->input->post('stage_id');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, [
                        'pipeline_id' => $pipeline_id,
                        'stage_id' => $stage_id
                    ]);
                }
                $success = true;
                $message = _l('deals_moved_successfully');
                break;
                
            case 'assign':
                $assigned_to = $this->input->post('assigned_to');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, ['addedfrom' => $assigned_to]);
                }
                $success = true;
                $message = _l('deals_assigned_successfully');
                break;
        }

        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => $success,
            'message' => $message
        ]));
    }

    public function view($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        $data['deal'] = $deal;
        $pipelineId = $deal['pipeline_id'] ?: $this->deals_model->get_default_pipeline_id();
        $data['stages'] = $this->deals_model->get_stages($pipelineId);
        $data['lost_reasons'] = $this->deals_model->get_lost_reasons();
        $data['title'] = $deal['name'];
        $this->load->view('view', $data);
    }

    public function set_stage($id)
    {
        $stageId = (int)$this->input->post('stage_id');
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($deal['status_final'] === 'won' || $deal['status_final'] === 'lost') {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Cannot change stage for won/lost deals']);
                return;
            }
            return redirect(admin_url('deals/view/'.$id));
        }
        
        $old_stage = $this->deals_model->get_stage_name($deal['stage_id']);
        $new_stage = $this->deals_model->get_stage_name($stageId);
        
        $this->deals_model->update_deal($id, ['stage_id'=>$stageId]);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Stage changed from ' . $old_stage . ' to ' . $new_stage, null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            redirect(admin_url('deals/view/'.$id));
        }
    }

    public function update_status($id)
    {
        $status = $this->input->post('status'); // open|won|lost
        $old_status = $this->deals_model->get_one($id)['status_final'];
        $this->deals_model->update_status($id, $status);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Status changed from ' . $old_status . ' to ' . $status, null, 'activities');
        
        set_alert('success', _l('updated_successfully', _l('deal')));
        redirect($_SERVER['HTTP_REFERER']);
    }

    public function mark_won($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        $this->deals_model->mark_won($id);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Deal marked as won', null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    public function mark_lost($id)
    {
        $reasonId = $this->input->post('lost_reason_id') ? (int)$this->input->post('lost_reason_id') : null;
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        $reason_name = '';
        if ($reasonId) {
            $reason = $this->deals_model->get_lost_reason($reasonId);
            $reason_name = $reason ? $reason['name'] : '';
        }
        
        $this->deals_model->mark_lost($id, $reasonId);
        
        // Log activity
        $activity_message = 'Deal marked as lost';
        if ($reason_name) {
            $activity_message .= ' - Reason: ' . $reason_name;
        }
        $this->deals_model->add_deal_activity($id, $activity_message, null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    // Deal Contacts Methods
    public function add_contact_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        

        $contact_id = (int)$this->input->post('contact_id');
        if ($contact_id) {
            // Add existing contact to deal
            $this->deals_model->add_contact_to_deal($deal_id, $contact_id);
            $this->deals_model->add_deal_activity($deal_id, 'Contact added to deal', null, 'activities');
            echo json_encode(['success' => true, 'message' => 'Contact added']);
        } else {
            // Create new contact and add to deal
            $contact_data = [
                'firstname' => $this->input->post('firstname'),
                'lastname' => $this->input->post('lastname'),
                'email' => $this->input->post('email'),
                'phonenumber' => $this->input->post('phonenumber'),
            ];

            $new_contact_id = $this->deals_model->create_module_contact($contact_data);
            
            if ($new_contact_id) {
                $this->deals_model->add_contact_to_deal($deal_id, $new_contact_id);
                $this->deals_model->add_deal_activity($deal_id, 'New contact created and added to deal', null, 'activities');
                echo json_encode(['success' => true, 'message' => 'Contact created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
            }
        }
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_contact_from_deal($deal_id, $contact_id);
        $this->deals_model->add_deal_activity($deal_id, 'Contact removed from deal', null, 'activities');
        echo json_encode(['success' => true, 'message' => 'Contact removed']);
    }

    public function get_deal_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_deal_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    public function get_available_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_available_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    // Deal Companies Methods
    public function add_company_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $company_id = (int)$this->input->post('company_id');
        if ($company_id) {
            // Add existing company to deal
            $this->deals_model->add_company_to_deal($deal_id, $company_id);
            $this->deals_model->add_deal_activity($deal_id, 'Company added to deal', null, 'activities');
            echo json_encode(['success' => true, 'message' => 'Company added']);
        } else {
            // Create new company and add to deal
            $company_data = [
                'name' => $this->input->post('company_name'),
                'email' => $this->input->post('company_email'),
                'phone' => $this->input->post('company_phone'),
                'website' => $this->input->post('company_website'),
                'address' => $this->input->post('company_address')
            ];

            $new_company_id = $this->deals_model->create_company($company_data);
            
            if ($new_company_id) {
                $this->deals_model->add_company_to_deal($deal_id, $new_company_id);
                $this->deals_model->add_deal_activity($deal_id, 'New company created and added to deal', null, 'activities');
                echo json_encode(['success' => true, 'message' => 'Company created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create company']);
            }
        }
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_company_from_deal($deal_id, $company_id);
        $this->deals_model->add_deal_activity($deal_id, 'Company removed from deal', null, 'activities');
        echo json_encode(['success' => true, 'message' => 'Company removed']);
    }

    public function get_deal_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_deal_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function get_available_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_available_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function add_activity($deal_id)
{
    if (!$this->input->is_ajax_request()) {
        show_404();
    }
    $deal = $this->deals_model->get_one($deal_id);
    if (!$deal) {
        echo json_encode(['success' => false, 'message' => 'Deal not found']);
        return;
    }
    $payload = $this->input->post();
    $id = $this->deals_model->add_deal_activity($deal_id, $payload);
    echo json_encode(['success' => (bool)$id, 'id' => $id]);
}

    // Deal Attachments Methods
    public function add_attachment_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Upload new attachment and add to deal
        $config['upload_path'] = './uploads/deals/';
        $config['allowed_types'] = 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx|txt';
        $config['max_size'] = 2048;
        
        if (!is_dir($config['upload_path'])) {
            mkdir($config['upload_path'], 0755, true);
        }
        
        $this->load->library('upload', $config);
        
        if ($this->upload->do_upload('attachment')) {
            $upload_data = $this->upload->data();
            $attachment_data = [
                'file_name' => $upload_data['file_name'],
                'filetype' => $upload_data['file_type'],
                'description' => $this->input->post('description'),
                'staffid' => get_staff_user_id(),
                'dateadded' => date('Y-m-d H:i:s')
            ];
            
            $new_attachment_id = $this->deals_model->add_attachment($attachment_data);
            
            if ($new_attachment_id) {
                $this->deals_model->add_attachment_to_deal($deal_id, $new_attachment_id);
                $this->deals_model->add_deal_activity($deal_id, 'New attachment uploaded and added to deal', null, 'documents');
                echo json_encode(['success' => true, 'message' => 'Attachment added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to save attachment']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => $this->upload->display_errors()]);
        }
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_attachment_from_deal($deal_id, $attachment_id);
        $this->deals_model->add_deal_activity($deal_id, 'Attachment removed from deal', null, 'documents');
        echo json_encode(['success' => true, 'message' => 'Attachment removed']);
    }

    public function get_deal_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $attachments = $this->deals_model->get_deal_attachments($deal_id);
        echo json_encode(['success' => true, 'attachments' => $attachments]);
    }

    public function get_available_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $attachments = $this->deals_model->get_available_attachments($deal_id);
        echo json_encode(['success' => true, 'attachments' => $attachments]);
    }

    public function download_attachment($attachment_id)
    {
        $attachment = $this->deals_model->get_attachment($attachment_id);
        if (!$attachment) {
            show_404();
        }
        $file_path = FCPATH . 'uploads/deals/' . $attachment['file_name'];
        if (!file_exists($file_path)) {
            show_404();
        }
        $this->load->helper('download');
        force_download($file_path, null);
    }

    // Deal Activities Methods
    public function get_deal_activities($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $activities = $this->deals_model->get_deal_activities($deal_id);
        echo json_encode(['success' => true, 'activities' => $activities]);
    }
    public function update_activity_status($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $status = $this->input->post('status');
        $success = $this->deals_model->update_activity_status($id, $status);
        echo json_encode(['success' => $success]);
    }

    public function get_staff_ajax()
{
    if (!$this->input->is_ajax_request()) {
        show_404();
    }
    echo json_encode(['success' => true, 'staff' => $this->deals_model->get_staff()]);
}
    public function add_note($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $description = trim((string)$this->input->post('description'));
        if ($description === '') {
            echo json_encode(['success' => false, 'message' => 'Description required']);
            return;
        }
        $id = $this->deals_model->add_note($deal_id, $description);
        echo json_encode(['success' => (bool)$id, 'id' => $id]);
    }

    public function get_notes($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $notes = $this->deals_model->get_notes($deal_id);
        echo json_encode(['success' => true, 'notes' => $notes]);
    }

    public function add_call($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $payload = $this->input->post();
        $id = $this->deals_model->add_call($deal_id, $payload);
        echo json_encode(['success' => (bool)$id, 'id' => $id]);
    }
    public function get_calls($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $calls = $this->deals_model->get_calls($deal_id);
        echo json_encode(['success' => true, 'calls' => $calls]);
    }

    public function test_db($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Test creating a contact
        $contact_data = [
            'firstname' => 'Test',
            'lastname' => 'Contact',
            'email' => 'test' . time() . '@example.com',
            'phonenumber' => '+1234567890'
        ];

        $contact_id = $this->deals_model->create_module_contact($contact_data);
        
        if ($contact_id) {
            // Test linking contact to deal
            $link_id = $this->deals_model->add_contact_to_deal($deal_id, $contact_id);
            
            if ($link_id) {
                // Test activity logging
                $activity_id = $this->deals_model->add_deal_activity($deal_id, 'Test contact added', null, 'activities');
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Database test successful',
                    'contact_id' => $contact_id,
                    'link_id' => $link_id,
                    'activity_id' => $activity_id
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to link contact to deal']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
        }
    }
}



<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Settings extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('deals/deals_model');
    }

    public function index()
    {
        // This controller is used only for AJAX actions from the settings view
        show_404();
    }

    public function pipelines()
    {
        // Re-order via drag & drop
        if ($this->input->post('action') === 'reorder_stages') {
            $ok = $this->deals_model->reorder_stages((int)$this->input->post('pipeline_id'), $this->input->post('order') ?? []);
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'create') {
            $id = $this->deals_model->create_pipeline($this->input->post('name'));
            return $this->json(['success' => (bool)$id, 'id' => $id]);
        }
        if ($this->input->post('action') === 'update') {
            $ok = $this->deals_model->update_pipeline((int)$this->input->post('id'), ['name'=>$this->input->post('name')]);
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'delete') {
            $ok = $this->deals_model->delete_pipeline((int)$this->input->post('id'));
            return $this->json(['success' => $ok]);
        }

        // Manage stages order and CRUD
        if ($this->input->post('action') === 'create_stage') {
            $id = $this->deals_model->create_stage((int)$this->input->post('pipeline_id'), [
                'name' => $this->input->post('name'),
                'position' => (int)$this->input->post('position') ?? 0,
                'is_won' => (int)$this->input->post('is_won') ?? 0,
                'is_lost' => (int)$this->input->post('is_lost') ?? 0,
            ]);
            return $this->json(['success' => (bool)$id, 'id' => $id]);
        }
        if ($this->input->post('action') === 'update_stage') {
            $id = (int)$this->input->post('id');
            $data = [];
            if ($this->input->post('name') !== null) { $data['name'] = $this->input->post('name'); }
            if ($this->input->post('position') !== null) { $data['position'] = (int)$this->input->post('position'); }
            if ($this->input->post('win_probability') !== null) { $data['win_probability'] = (int)$this->input->post('win_probability'); }
            if ($this->input->post('is_won') !== null) { $data['is_won'] = (int)$this->input->post('is_won'); }
            if ($this->input->post('is_lost') !== null) { $data['is_lost'] = (int)$this->input->post('is_lost'); }
            $ok = !empty($data) ? $this->deals_model->update_stage($id, $data) : true;
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'delete_stage') {
            $ok = $this->deals_model->delete_stage((int)$this->input->post('id'));
            return $this->json(['success' => $ok]);
        }

        // Fetch pipelines + stages
        $pipelines = $this->deals_model->get_pipelines();
        foreach ($pipelines as &$p) {
            $p['stages'] = $this->deals_model->get_stages($p['id']);
        }

        $data['pipelines'] = $pipelines;
        $this->load->view('settings', $data);
    }

    public function lost_reasons()
    {
        if ($this->input->post('action') === 'create') {
            $id = $this->deals_model->create_lost_reason($this->input->post('name'));
            return $this->json(['success'=> (bool)$id, 'id'=>$id]);
        }
        if ($this->input->post('action') === 'update') {
            $ok = $this->deals_model->update_lost_reason((int)$this->input->post('id'), $this->input->post('name'));
            return $this->json(['success'=>$ok]);
        }
        if ($this->input->post('action') === 'delete') {
            $ok = $this->deals_model->delete_lost_reason((int)$this->input->post('id'));
            return $this->json(['success'=>$ok]);
        }

        $data['reasons'] = $this->deals_model->get_lost_reasons();
        $this->load->view('lost_reasons', $data);
    }

    private function json($payload)
    {
        $this->output->set_content_type('application/json')->set_output(json_encode($payload));
    }
}






<?php

/*
 * Inject sidebar menu and links for deals module
 */

hooks()->add_action('admin_init', function () {
    $CI = &get_instance();

    $CI->app_menu->add_sidebar_menu_item('deals', [
        'slug'     => 'deals',
        'name'     => _l('deals'),
        'icon'     => 'fa-solid fa-handshake',
        'href'     => admin_url('deals'),
        'position' => 16,
    ]);
});


<?php

hooks()->add_action('app_admin_head', function () {
    echo '<style>.deal-status-badge{padding:3px 6px;border-radius:4px;font-size:11px}</style>';
});

hooks()->add_action('app_admin_footer', function () {
    echo '<script>window.dealsModuleLoaded=true;</script>';
});




<?php
$lang['deals'] = 'Deals';
$lang['deals_kanban'] = 'Deals Kanban';
$lang['new_deal'] = 'New Deal';
$lang['edit_deal'] = 'Edit Deal';
$lang['deal'] = 'Deal';
$lang['mark_as_open'] = 'Open';
$lang['mark_as_won'] = 'Won';
$lang['mark_as_lost'] = 'Lost';
$lang['switch_to_list_view'] = 'List View';
$lang['deal_value'] = 'Amount';
$lang['expected_close_date'] = 'Expected Close Date';
$lang['pipeline'] = 'Pipeline';
$lang['stage'] = 'Stage';
$lang['pipelines'] = 'Pipelines';
$lang['stages'] = 'Stages';
$lang['lost_reasons'] = 'Lost Reasons';
$lang['add_contact'] = 'Add Contact';
$lang['add_company'] = 'Add Company';
$lang['add_attachment'] = 'Add Attachment';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['no_activities_found'] = 'No activities found';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['view'] = 'View';
$lang['add'] = 'Add';
$lang['edit'] = 'Edit';
$lang['update'] = 'Update';
$lang['save'] = 'Save';
$lang['close'] = 'Close';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['file_uploaded_successfully'] = 'File uploaded successfully';
$lang['select_pipeline'] = 'Select Pipeline';
$lang['select_stage'] = 'Select Stage';
$lang['pipeline_name'] = 'Pipeline Name';
$lang['enter_pipeline_name'] = 'Enter pipeline name';
$lang['enter_stage_name'] = 'Enter stage name';
$lang['delete_pipeline_confirm'] = 'Are you sure you want to delete this pipeline?';
$lang['delete_stage_confirm'] = 'Are you sure you want to delete this stage?';
$lang['pipeline_name_required'] = 'Pipeline name is required';
$lang['deal_name'] = 'Deal Name';
$lang['assigned_to'] = 'Assigned To';
$lang['total_deals'] = 'Total Deals';
$lang['total_value'] = 'Total Value';
$lang['deals_summary'] = 'Deals Summary';
$lang['deals_table_order_configure'] = 'Configure Table Columns';
$lang['move_to_pipeline'] = 'Move to Pipeline';
$lang['move_to_stage'] = 'Move to Stage';
$lang['assign_to'] = 'Assign To';
$lang['select_staff'] = 'Select Staff';
$lang['no_action_selected'] = 'No action selected';
$lang['deals_deleted_successfully'] = 'Deals deleted successfully';
$lang['deals_moved_successfully'] = 'Deals moved successfully';
$lang['deals_assigned_successfully'] = 'Deals assigned successfully';
$lang['company_name'] = 'Company Name';
$lang['company_email'] = 'Company Email';
$lang['company_phone'] = 'Company Phone';
$lang['firstname'] = 'First Name';
$lang['lastname'] = 'Last Name';
$lang['email'] = 'Email';
$lang['phonenumber'] = 'Phone Number';
$lang['please_select_lost_reason'] = 'Please select a lost reason';
$lang['won'] = 'Won';
$lang['lost'] = 'Lost';
$lang['lost_reason'] = 'Lost Reason';
$lang['select_lost_reason'] = 'Select Lost Reason';
$lang['cancel'] = 'Cancel';
$lang['submit'] = 'Submit';
$lang['dropdown_non_selected_tex'] = 'Nothing selected';
$lang['updated_successfully'] = 'Updated successfully';
$lang['deleted'] = 'Deleted successfully';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['no_activities_found'] = 'No activities found';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['view'] = 'View';
$lang['add'] = 'Add';
$lang['edit'] = 'Edit';
$lang['update'] = 'Update';
$lang['save'] = 'Save';
$lang['close'] = 'Close';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['file_uploaded_successfully'] = 'File uploaded successfully';
$lang['select_pipeline'] = 'Select Pipeline';
$lang['select_stage'] = 'Select Stage';
$lang['products'] = 'Products';
$lang['tasks'] = 'Tasks';
$lang['reminders'] = 'Reminders';
$lang['emails'] = 'Emails';
$lang['calls'] = 'Calls';
$lang['meetings'] = 'Meetings';
$lang['documents'] = 'Documents';
$lang['expenses'] = 'Expenses';
$lang['invoices'] = 'Invoices';
$lang['estimates'] = 'Estimates';
$lang['proposals'] = 'Proposals';
$lang['contracts'] = 'Contracts';
$lang['credit_notes'] = 'Credit Notes';
$lang['templates'] = 'Templates';
$lang['automation'] = 'Automation';
$lang['webhooks'] = 'Webhooks';
$lang['custom_fields'] = 'Custom Fields';
$lang['probability'] = 'Probability';
$lang['currency'] = 'Currency';
$lang['tags'] = 'Tags';
$lang['archived'] = 'Archived';
$lang['archived_date'] = 'Archived Date';
$lang['archived_by'] = 'Archived By';
$lang['is_primary'] = 'Is Primary';
$lang['dateadded'] = 'Date Added';
$lang['staffid'] = 'Staff ID';
$lang['external'] = 'External';
$lang['external_link'] = 'External Link';
$lang['filetype'] = 'File Type';
$lang['subject'] = 'Subject';
$lang['message'] = 'Message';
$lang['from_email'] = 'From Email';
$lang['from_name'] = 'From Name';
$lang['to_email'] = 'To Email';
$lang['to_name'] = 'To Name';
$lang['cc'] = 'CC';
$lang['bcc'] = 'BCC';
$lang['sent'] = 'Sent';
$lang['sent_date'] = 'Sent Date';
$lang['call_type'] = 'Call Type';
$lang['call_status'] = 'Call Status';
$lang['call_duration'] = 'Call Duration';
$lang['call_recording'] = 'Call Recording';
$lang['call_notes'] = 'Call Notes';
$lang['call_date'] = 'Call Date';
$lang['meeting_subject'] = 'Meeting Subject';
$lang['meeting_description'] = 'Meeting Description';
$lang['meeting_date'] = 'Meeting Date';
$lang['meeting_duration'] = 'Meeting Duration';
$lang['meeting_location'] = 'Meeting Location';
$lang['meeting_type'] = 'Meeting Type';
$lang['meeting_status'] = 'Meeting Status';
$lang['meeting_notes'] = 'Meeting Notes';
$lang['document_name'] = 'Document Name';
$lang['document_type'] = 'Document Type';
$lang['document_file'] = 'Document File';
$lang['document_description'] = 'Document Description';
$lang['document_version'] = 'Document Version';
$lang['document_status'] = 'Document Status';
$lang['template_name'] = 'Template Name';
$lang['template_content'] = 'Template Content';
$lang['template_type'] = 'Template Type';
$lang['template_category'] = 'Template Category';
$lang['is_active'] = 'Is Active';
$lang['created_by'] = 'Created By';
$lang['automation_name'] = 'Automation Name';
$lang['automation_type'] = 'Automation Type';
$lang['trigger_condition'] = 'Trigger Condition';
$lang['action_type'] = 'Action Type';
$lang['action_data'] = 'Action Data';
$lang['webhook_name'] = 'Webhook Name';
$lang['webhook_url'] = 'Webhook URL';
$lang['webhook_events'] = 'Webhook Events';
$lang['webhook_secret'] = 'Webhook Secret';
$lang['event_type'] = 'Event Type';
$lang['payload'] = 'Payload';
$lang['response_code'] = 'Response Code';
$lang['response_body'] = 'Response Body';
$lang['status'] = 'Status';
$lang['field_name'] = 'Field Name';
$lang['field_type'] = 'Field Type';
$lang['field_label'] = 'Field Label';
$lang['field_placeholder'] = 'Field Placeholder';
$lang['field_options'] = 'Field Options';
$lang['field_required'] = 'Field Required';
$lang['field_order'] = 'Field Order';
$lang['field_visible'] = 'Field Visible';
$lang['field_value'] = 'Field Value';
$lang['qty'] = 'Quantity';
$lang['rate'] = 'Rate';
$lang['discount'] = 'Discount';
$lang['tax'] = 'Tax';
$lang['total'] = 'Total';
$lang['product_id'] = 'Product ID';
$lang['task_id'] = 'Task ID';
$lang['expense_id'] = 'Expense ID';
$lang['invoice_id'] = 'Invoice ID';
$lang['estimate_id'] = 'Estimate ID';
$lang['proposal_id'] = 'Proposal ID';
$lang['contract_id'] = 'Contract ID';
$lang['credit_note_id'] = 'Credit Note ID';
$lang['contact_id'] = 'Contact ID';
$lang['company_id'] = 'Company ID';
$lang['field_id'] = 'Field ID';
$lang['webhook_id'] = 'Webhook ID';
$lang['filter_name'] = 'Filter Name';
$lang['field'] = 'Field';
$lang['operator'] = 'Operator';
$lang['value'] = 'Value';
$lang['share_with_team'] = 'Share with Team';
$lang['is_default'] = 'Is Default';
$lang['created_at'] = 'Created At';
$lang['filter_name'] = 'Filter Name';
$lang['field'] = 'Field';
$lang['operator'] = 'Operator';
$lang['value'] = 'Value';
$lang['share_with_team'] = 'Share with Team';
$lang['is_default'] = 'Is Default';
$lang['created_at'] = 'Created At';

// Additional language strings for the updated deals module
$lang['all'] = 'All';
$lang['filter_by_all'] = 'Filter by All';
$lang['select_existing'] = 'Select Existing';
$lang['create_new'] = 'Create New';
$lang['select_contact'] = 'Select Contact';
$lang['select_company'] = 'Select Company';
$lang['select_attachment'] = 'Select Attachment';
$lang['upload_new'] = 'Upload New';
$lang['confirm_action_prompt'] = 'Are you sure you want to perform this action?';
$lang['pin_on_top'] = 'Pin on top';
$lang['notes'] = 'Notes';
$lang['calls'] = 'Calls';
$lang['documents'] = 'Documents';
$lang['emails'] = 'Emails';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['add_contact'] = 'Add Contact';
$lang['add_company'] = 'Add Company';
$lang['add_attachment'] = 'Add Attachment';
$lang['company_name'] = 'Company Name';
$lang['company_email'] = 'Company Email';
$lang['company_phone'] = 'Company Phone';
$lang['company_website'] = 'Company Website';
$lang['company_address'] = 'Company Address';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['close'] = 'Close';
$lang['save'] = 'Save';
$lang['firstname'] = 'First Name';
$lang['lastname'] = 'Last Name';
$lang['email'] = 'Email';
$lang['phonenumber'] = 'Phone Number';




model



<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals_model extends App_Model
{
    public function get_all()
    {
        return $this->db->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
    }

    public function get_stages_with_deals($pipelineId = null)
    {
        if (!$pipelineId) {
            $pipelineId = $this->get_default_pipeline_id();
        }
        $stages = $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
        foreach ($stages as &$stage) {
            $stage['deals'] = $this->db->where('stage_id', (int) $stage['id'])->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
        }
        return $stages;
    }

    public function get_one($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals')->row_array();
    }

    public function update_deal($id, array $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals');
        return $this->db->affected_rows() > 0;
    }

    public function add_attachment($data)
    {
        $this->db->insert(db_prefix() . 'deals_attachments', $data);
        return $this->db->insert_id();
    }

    public function get_attachments($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)->get(db_prefix() . 'deals_attachments')->result_array();
    }



    public function create($data)
    {
        $insert = [
            'name' => $data['name'] ?? null,
            'company' => $data['company'] ?? null,
            'description' => $data['description'] ?? null,
            'status' => $data['status'] ?? null,
            'pipeline_id' => $data['pipeline_id'] ?? $this->get_default_pipeline_id(),
            'stage_id' => $data['stage_id'] ?? $this->get_default_open_stage_id(),
            'deal_value' => $data['deal_value'] ?? 0,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'status_final' => 'open',
        ];
        $this->db->insert(db_prefix() . 'deals', $insert);
        return $this->db->insert_id();
    }

    public function update_status($id, $status)
    {
        $statusId = $this->map_status_to_stage($status);
        $payload = [
            'last_status_change' => date('Y-m-d H:i:s'),
            'status' => $statusId,
        ];
        if ($status === 'won' || $status === 'lost') {
            $payload['status_final'] = $status;
        }
        $this->db->where('id', $id)->update(db_prefix() . 'deals', $payload);
        return $this->db->affected_rows() > 0;
    }

    public function mark_won($id)
    {
        return $this->update_status($id, 'won');
    }

    public function mark_lost($id, $reasonId = null)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'lost',
            'lost_reason_id' => $reasonId,
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }

    private function map_status_to_stage($status)
    {
        $statusName = null;
        if ($status === 'won') {
            $statusName = 'Won';
        } elseif ($status === 'lost') {
            $statusName = 'Lost';
        } else {
            $statusName = 'Open';
        }

        $row = $this->db->where('name', $statusName)->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : $this->get_default_open_stage_id();
    }

    // Pipelines helpers
    public function get_default_pipeline_id()
    {
        $row = $this->db->order_by('`order`', 'ASC')->limit(1)->get(db_prefix() . 'deals_pipelines')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_default_open_stage_id()
    {
        $pipelineId = $this->get_default_pipeline_id();
        if (!$pipelineId) {
            return null;
        }
        $row = $this->db->where(['pipeline_id' => $pipelineId, 'is_won' => 0, 'is_lost' => 0])->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_pipelines()
    {
        return $this->db->order_by('`order`', 'ASC')->get(db_prefix() . 'deals_pipelines')->result_array();
    }

    public function create_pipeline($name)
    {
        $this->db->insert(db_prefix() . 'deals_pipelines', ['name' => $name, 'order' => (int) ($this->db->count_all(db_prefix() . 'deals_pipelines') + 1)]);
        return $this->db->insert_id();
    }

    public function update_pipeline($id, $data)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals_pipelines', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_pipeline($id)
    {
        $this->db->where('id', $id)->delete(db_prefix() . 'deals_pipelines');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages($pipelineId)
    {
        return $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
    }

    public function create_stage($pipelineId, $data)
    {
        $data['pipeline_id'] = (int) $pipelineId;
        if (!isset($data['position'])) {
            $data['position'] = (int) $this->db->where('pipeline_id', (int) $pipelineId)->count_all_results(db_prefix() . 'deals_stages') + 1;
        }
        $this->db->insert(db_prefix() . 'deals_stages', $data);
        return $this->db->insert_id();
    }

    public function update_stage($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_stages', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_stage($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_stages');
        return $this->db->affected_rows() > 0;
    }

    public function reorder_stages($pipelineId, $orderedIds)
    {
        $position = 1;
        foreach ((array) $orderedIds as $stageId) {
            $this->db->where('id', (int) $stageId)->where('pipeline_id', (int) $pipelineId)->update(db_prefix() . 'deals_stages', ['position' => $position]);
            $position++;
        }
        return true;
    }

    // Lost reasons CRUD
    public function get_lost_reasons()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_lost_reasons')->result_array();
    }

    public function get_lost_reason($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_lost_reasons')->row_array();
    }

    public function get_staff()
    {
        return $this->db->select('staffid, firstname, lastname')
            ->where('active', 1)
            ->order_by('firstname', 'ASC')
            ->get(db_prefix() . 'staff')
            ->result_array();
    }

    public function get_stage_name($stage_id)
    {
        $stage = $this->db->where('id', (int) $stage_id)->get(db_prefix() . 'deals_stages')->row_array();
        return $stage ? $stage['name'] : 'Unknown Stage';
    }

    public function create_lost_reason($name)
    {
        $this->db->insert(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->insert_id();
    }
    public function update_lost_reason($id, $name)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->affected_rows() > 0;
    }
    public function delete_lost_reason($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_lost_reasons');
        return $this->db->affected_rows() > 0;
    }

    // Deal Contacts Methods
    public function get_deal_contacts($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_contact_links l')
            ->join(db_prefix() . 'deals_contacts_book cb', 'cb.id = l.contact_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_contact_to_deal($deal_id, $contact_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'contact_id' => (int) $contact_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_contact_links', $data);
        return $this->db->insert_id();
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('contact_id', (int) $contact_id)
            ->delete(db_prefix() . 'deal_contact_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_contacts($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_contacts_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT contact_id FROM ' . db_prefix() . 'deal_contact_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.firstname', 'ASC')->get()->result_array();
    }

    public function create_module_contact($data)
    {
        $insert = [
            'firstname' => $data['firstname'] ?? '',
            'lastname' => $data['lastname'] ?? '',
            'email' => $data['email'] ?? '',
            'phonenumber' => $data['phonenumber'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1,
        ];
        $this->db->insert(db_prefix() . 'deals_contacts_book', $insert);
        return $this->db->insert_id();
    }

    // Deal Companies Methods
    public function get_deal_companies($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_company_links l')
            ->join(db_prefix() . 'deals_companies_book cb', 'cb.id = l.company_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_company_to_deal($deal_id, $company_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'company_id' => (int) $company_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_company_links', $data);
        return $this->db->insert_id();
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('company_id', (int) $company_id)
            ->delete(db_prefix() . 'deal_company_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_companies($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_companies_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT company_id FROM ' . db_prefix() . 'deal_company_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.name', 'ASC')->get()->result_array();
    }

    // Company CRUD Methods (module-specific)
    public function create_company($data)
    {
        // Check if email or phone already exists
        if (!empty($data['email'])) {
            $existing = $this->db->where('email', $data['email'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Email already exists', 'existing_id' => $existing['id']];
            }
        }

        if (!empty($data['phone'])) {
            $existing = $this->db->where('phone', $data['phone'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Phone number already exists', 'existing_id' => $existing['id']];
            }
        }

        $insert = [
            'name' => $data['name'] ?? '',
            'email' => $data['email'] ?? '',
            'phone' => $data['phone'] ?? '',
            'website' => $data['website'] ?? '',
            'address' => $data['address'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1
        ];

        $this->db->insert(db_prefix() . 'deals_companies_book', $insert);
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function get_company($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_companies_book')->row_array();
    }

    public function update_company($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_companies_book', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_company($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_companies_book');
        return $this->db->affected_rows() > 0;
    }

    // Deal Attachments Relations Methods
    public function get_deal_attachments($deal_id)
    {
        $this->db->select('da.*, dar.dateadded as linked_date')
            ->from(db_prefix() . 'deal_attachments_relations dar')
            ->join(db_prefix() . 'deals_attachments da', 'da.id = dar.attachment_id', 'left')
            ->where('dar.deal_id', (int) $deal_id)
            ->order_by('dar.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function get_attachment($attachment_id)
    {
        return $this->db->where('id', (int) $attachment_id)->get(db_prefix() . 'deals_attachments')->row_array();
    }

    public function add_attachment_to_deal($deal_id, $attachment_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'attachment_id' => (int) $attachment_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];

        $this->db->insert(db_prefix() . 'deal_attachments_relations', $data);
        return $this->db->insert_id();
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('attachment_id', (int) $attachment_id)
            ->delete(db_prefix() . 'deal_attachments_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_attachments($deal_id = null)
    {
        $this->db->select('da.*')
            ->from(db_prefix() . 'deals_attachments da');

        if ($deal_id) {
            $this->db->where('da.id NOT IN (SELECT attachment_id FROM ' . db_prefix() . 'deal_attachments_relations WHERE deal_id = ' . (int) $deal_id . ')');
        }

        return $this->db->order_by('da.dateadded', 'DESC')->get()->result_array();
    }

    // Deal Activities Methods with type
    public function get_deal_activities($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('date', 'DESC')
            ->get(db_prefix() . 'deals_activity')
            ->result_array();
    }

    public function add_deal_activity($deal_id, $payload)
    {
        $data = [
            'title' => $payload['title'] ?? '',
            'rel_id' => (int) $deal_id,
            'description' => $payload['description'] ?? '',
            'additional_data' => $payload['additional_data'] ?? null,
            'staffid' => get_staff_user_id(),
            'full_name' => get_staff_full_name(get_staff_user_id()),
            'date' => date('Y-m-d H:i:s'),
            'type' => $payload['type'] ?? 'activity',
            'activity_date' => $payload['activity_date'] ?? null,
            'reminder' => $payload['reminder'] ?? 0,
            'owner_id' => $payload['owner_id'] ?? get_staff_user_id(),
            'guests' => $payload['guests'] ?? null,
            'status' => $payload['status'] ?? 'pending',
        ];
        $this->db->insert(db_prefix() . 'deals_activity', $data);
        return $this->db->insert_id();
    }

    // Notes
    public function add_note($deal_id, $description)
    {
        $noteData = [
            'rel_id' => (int) $deal_id,
            'description' => (string) $description,
            'addedfrom' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        $this->db->insert(db_prefix() . 'deals_notes', $noteData);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Note added',
                'additional_data' => $description,
                'type' => 'notes'
            ]);
        }
        return $insertId;
    }
    public function update_activity_status($id, $status)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_activity', ['status' => $status]);
        return $this->db->affected_rows() > 0;
    }

    public function get_notes($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('dateadded', 'DESC')
            ->get(db_prefix() . 'deals_notes')
            ->result_array();
    }

    // Calls
    public function add_call($deal_id, $payload)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'outcome' => $payload['outcome'] ?? '',
            'call_notes' => $payload['call_notes'] ?? null,
            'call_date' => !empty($payload['call_date']) ? $payload['call_date'] : date('Y-m-d H:i:s'),
            'staffid' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        $this->db->insert(db_prefix() . 'deals_calls', $data);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Call logged',
                'additional_data' => json_encode([
                    'outcome' => $data['outcome'],
                    'date' => $data['call_date'],
                    'notes' => $data['call_notes']
                ]),
                'type' => 'calls'
            ]);
        }
        return $insertId;
    }

    public function get_calls($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)
            ->order_by('call_date', 'DESC')
            ->get(db_prefix() . 'deals_calls')
            ->result_array();
    }
}



└──views
    ├── _kan_ban_card.php
    ├── create.php
    ├── deals.css
    ├── kanban.php
    ├── list.php
    ├── lost_reasons.php
    ├── settings.php
    └── view.phpthis are view file and in last install and deals file

<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
 
<li data-deal-id="<?= e($deal['id']); ?>"
    <?= isset($deal['is_deal_client']) && $deal['is_deal_client'] !== '0' ? 'data-toggle="tooltip" title="' . _l('deal_have_client_profile') . '"' : ''; ?>
    class="deal-kan-ban<?= $deal['assigned'] == get_staff_user_id() ? ' current-user-deal' : ''; ?><?= isset($deal['is_deal_client']) && $deal['is_deal_client'] !== '0' && get_option('deal_lock_after_convert_to_customer') == 1 && !is_admin() ? ' not-sortable' : ''; ?>">
    <div class="panel-body deal-body">
        <div class="tw-flex deal-name">
            <?php if ($deal['assigned'] != 0) { ?>
                <a href="<?= admin_url('profile/' . $deal['assigned']); ?>" data-placement="right" data-toggle="tooltip"
                    title="<?= e(get_staff_full_name($deal['assigned'])); ?>" class="mtop8 tw-mr-1.5">
                    <?= staff_profile_image($deal['assigned'], ['staff-profile-image-xs']); ?>
                </a>
            <?php } ?>
            <a href="javascript:void(0)" title="#<?= e($deal['id']) . ' - ' . e($deal['name'] ?? 'Unnamed Deal'); ?>"
                onclick="openViewDealModal(<?= e($deal['id']); ?>); return false;" class="tw-block tw-min-w-0 tw-font-medium">
                <span class="mtop10 mbot10 tw-truncate tw-block">
                    #<?= e($deal['id']) . ' - ' . e($deal['name'] ?? 'Unnamed Deal'); ?>
                </span>
            </a>
        </div>
        <div class="row">
            <div class="col-md-12">
                <div class="tw-flex">
                    <div class="tw-grow tw-mr-2">
                        <p class="tw-text-sm tw-mb-0">
                            <?= 'Source: ' . ($deal['source_name'] ?? 'Unknown'); ?>
                        </p>
                        <?php
                        $deal_value = isset($deal['deal_value']) && $deal['deal_value'] != 0
                            ? app_format_money($deal['deal_value'], $base_currency->symbol)
                            : '--';
                        ?>
                        <p class="tw-text-sm tw-mb-0">
                            <?= 'Deal Value: ' . $deal_value; ?>
                        </p>
 
                    </div>
                    <div class="text-right">
                        <?php if (isset($deal['lastcontact']) && is_date($deal['lastcontact']) && $deal['lastcontact'] != '0000-00-00 00:00:00') { ?>
                            <small class="text-dark tw-text-sm">
                                <?= _l('deals_dt_last_contact'); ?>
                                <span class="bold">
                                    <span class="text-has-action" data-toggle="tooltip" data-title="<?= e(_dt($deal['lastcontact'])); ?>">
                                        <?= e(time_ago($deal['lastcontact'])); ?>
                                    </span>
                                </span>
                            </small><br />
                        <?php } ?>
                        <small class="text-dark">
                            <?= _l('deal_created'); ?>:
                            <span class="bold">
                                <span class="text-has-action" data-toggle="tooltip" data-title="<?= e(_dt($deal['dateadded'])); ?>">
 
                                    <?= e(time_ago($deal['dateadded'])); ?>
                                </span>
                            </span>
                        </small><br />
                        <?php hooks()->do_action('before_deals_kanban_card_icons', $deal); ?>
                        <span class="mright5 mtop5 inline-block text-muted" data-toggle="tooltip" data-placement="left"
                            data-title="<?= _l('deals_canban_notes', $deal['total_notes'] ?? 0); ?>">
                            <i class="fa-regular fa-note-sticky"></i> <?= e($deal['total_notes'] ?? 0); ?>
                        </span>
                        <span class="mtop5 inline-block text-muted" data-placement="left" data-toggle="tooltip"
                            data-title="<?= _l('deal_kan_ban_attachments', $deal['total_files'] ?? 0); ?>">
                            <i class="fa fa-paperclip"></i> <?= e($deal['total_files'] ?? 0); ?>
                        </span>
                        <?php hooks()->do_action('after_deals_kanban_card_icons', $deal); ?>
                    </div>
                </div>
            </div>
            <?php if (!empty($deal['tags'] ?? [])) { ?>
                <div class="col-md-12">
                    <div class="kanban-tags tw-text-sm tw-inline-flex">
                        <?= render_tags($deal['tags'] ?? []); ?>
                    </div>
                </div>
            <?php } ?>
            <a href="javascript:void(0)" class="pull-right text-muted kan-ban-expand-top"
                onclick="$('#kan-ban-expand-<?= e($deal['id']); ?>').slideToggle(); return false;">
                <i class="fa fa-expand" aria-hidden="true"></i>
            </a>
 
            <div class="clearfix no-margin"></div>
            <div id="kan-ban-expand-<?= e($deal['id']); ?>" class="padding-10" style="display:none;">
                <div class="clearfix"></div>
                <hr class="hr-10" />
                <p class="text-muted deal-field-heading"><?= _l('deal_title'); ?></p>
                <p class="bold tw-text-sm"><?= e($deal['name'] ?? '-'); ?></p>
                <p class="text-muted deal-field-heading"><?= _l('deal_add_edit_email'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['email']) ? '<a href="mailto:' . e($deal['email']) . '">' . e($deal['email']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_website'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['website']) ? '<a href="' . e(maybe_add_http($deal['website'])) . '" target="_blank">' . e($deal['website']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_add_edit_phonenumber'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['phonenumber']) ? '<a href="tel:' . e($deal['phonenumber']) . '">' . e($deal['phonenumber']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_company'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['company']) ? $deal['company'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_address'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['address']) ? $deal['address'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_city'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['city']) ? $deal['city'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_state'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['state']) ? $deal['state'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_country'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['country']) && $deal['country'] != 0 ? get_country($deal['country'])->short_name : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_zip'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['zip']) ? $deal['zip'] : '-'); ?></p>
 
            </div>
        </div>
</li>






<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-8 col-md-offset-2">
        <div class="panel_s">
          <div class="panel-body">
            <h4 class="tw-mt-0"><?php echo isset($deal) ? _l('edit_deal') : _l('new_deal'); ?></h4>
            <hr class="hr-panel-heading" />
            <?php echo form_open(admin_url(isset($deal) ? 'deals/edit/'.$deal['id'] : 'deals/create')); ?>
              <div class="row">
                <div class="col-md-6">
                  <?php echo render_input('name', 'name', isset($deal) ? $deal['name'] : ''); ?>
                </div>
                <div class="col-md-6">
                  <?php echo render_input('company', 'company', isset($deal) ? $deal['company'] : ''); ?>
                </div>
                <div class="col-md-6">
                  <div class="form-group">
                    <label for="pipeline_id" class="control-label"><?php echo _l('pipeline'); ?></label>
                    <select name="pipeline_id" id="pipeline_id" class="selectpicker" data-width="100%" data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                      <?php foreach(($pipelines ?? []) as $pipeline){ ?>
                        <option value="<?php echo (int)$pipeline['id']; ?>" <?php echo (isset($deal) && $deal['pipeline_id'] == $pipeline['id']) ? 'selected' : ''; ?>>
                          <?php echo html_entity_decode($pipeline['name']); ?>
                        </option>
                      <?php } ?>
                    </select>
                  </div>
                </div>
                <?php if (isset($deal) && isset($stages)): ?>
                <div class="col-md-6">
                  <div class="form-group">
                    <label for="stage_id" class="control-label"><?php echo _l('stage'); ?></label>
                    <select name="stage_id" id="stage_id" class="selectpicker" data-width="100%" data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                      <?php foreach($stages as $stage){ ?>
                        <option value="<?php echo (int)$stage['id']; ?>" <?php echo ($deal['stage_id'] == $stage['id']) ? 'selected' : ''; ?>>
                          <?php echo html_entity_decode($stage['name']); ?>
                        </option>
                      <?php } ?>
                    </select>
                  </div>
                </div>
                <?php endif; ?>
                <div class="col-md-12">
                  <?php echo render_textarea('description', 'description', isset($deal) ? $deal['description'] : ''); ?>
                </div>
                <div class="col-md-4">
                  <?php echo render_input('deal_value', 'deal_value', isset($deal) ? $deal['deal_value'] : '', 'number'); ?>
                </div>
                <div class="col-md-4">
                  <?php echo render_date_input('expected_close_date', 'expected_close_date', isset($deal) ? $deal['expected_close_date'] : ''); ?>
                </div>
              </div>
              <div class="text-right">
                <button type="submit" class="btn btn-primary"><?php echo _l('submit'); ?></button>
              </div>
            <?php echo form_close(); ?>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>
<?php init_tail(); ?>






/* Deals Module Custom Styles */

/* Pipeline Stages */
.pipeline-stage {
    display: inline-flex;
    align-items: center;
    margin: 0 10px;
}

.pipeline-stage .stage-button {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    border: none;
    color: white;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
}

.pipeline-stage .stage-button:hover {
    transform: scale(1.1);
}

.pipeline-stage .stage-button:disabled {
    cursor: not-allowed;
    opacity: 0.7;
}

.pipeline-stage .stage-name {
    margin: 0 10px;
    font-weight: 500;
    color: #333;
}

.pipeline-stage .stage-arrow {
    margin: 0 10px;
    color: #ccc;
}

/* Activity Cards */
.activity-card {
    border-left: 4px solid #007cba;
    padding-left: 15px;
    margin-bottom: 20px;
    background: #f9f9f9;
    padding: 15px;
    border-radius: 5px;
}

.activity-card .activity-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 10px;
}

.activity-card .activity-description {
    font-weight: 500;
    color: #333;
    margin-bottom: 5px;
}

.activity-card .activity-user {
    font-size: 12px;
    color: #666;
}

.activity-card .activity-time {
    font-size: 12px;
    color: #999;
    text-align: right;
}

.activity-card .activity-actions {
    text-align: right;
}

/* Contact, Company, Attachment Items */
.item-list {
    margin-bottom: 10px;
}

.item-list .item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid #eee;
}

.item-list .item:last-child {
    border-bottom: none;
}

.item-list .item-info {
    display: flex;
    align-items: center;
}

.item-list .item-icon {
    margin-right: 8px;
    color: #007cba;
}

.item-list .item-name {
    font-weight: 500;
    color: #333;
}

.item-list .item-remove {
    background: #dc3545;
    color: white;
    border: none;
    border-radius: 3px;
    padding: 2px 6px;
    font-size: 10px;
    cursor: pointer;
}

.item-list .item-remove:hover {
    background: #c82333;
}

/* Modal Tabs */
.modal-tabs {
    margin-bottom: 20px;
}

.modal-tabs .nav-tabs {
    border-bottom: 2px solid #dee2e6;
}

.modal-tabs .nav-tabs > li > a {
    border: none;
    color: #666;
    padding: 10px 15px;
}

.modal-tabs .nav-tabs > li.active > a {
    border: none;
    border-bottom: 2px solid #007cba;
    color: #007cba;
    background: transparent;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 40px 20px;
    color: #999;
}

.empty-state .empty-icon {
    font-size: 48px;
    margin-bottom: 15px;
    color: #ccc;
}

.empty-state .empty-text {
    font-size: 14px;
    color: #999;
}

/* Responsive Design */
@media (max-width: 768px) {
    .pipeline-stage {
        margin: 5px;
    }
    
    .pipeline-stage .stage-name {
        display: none;
    }
    
    .activity-card .activity-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .activity-card .activity-time {
        text-align: left;
        margin-top: 5px;
    }
}

/* Custom Colors for Stages */
.stage-open { background-color: #28B8DA !important; }
.stage-won { background-color: #84c529 !important; }
.stage-lost { background-color: #fc2d42 !important; }
.stage-completed { background-color: #84c529 !important; }
.stage-pending { background-color: #e3e3e3 !important; }

/* Badge Styles */
.badge {
    background-color: #007cba;
    color: white;
    padding: 2px 6px;
    border-radius: 10px;
    font-size: 10px;
    margin-left: 5px;
}

/* Form Enhancements */
.form-group label {
    font-weight: 500;
    color: #333;
    margin-bottom: 5px;
}

.form-control {
    border-radius: 4px;
    border: 1px solid #ddd;
    padding: 8px 12px;
}

.form-control:focus {
    border-color: #007cba;
    box-shadow: 0 0 0 0.2rem rgba(0, 124, 186, 0.25);
}

/* Button Enhancements */
.btn-primary {
    background-color: #007cba;
    border-color: #007cba;
}

.btn-primary:hover {
    background-color: #005a8b;
    border-color: #005a8b;
}

.btn-success {
    background-color: #84c529;
    border-color: #84c529;
}

.btn-danger {
    background-color: #fc2d42;
    border-color: #fc2d42;
}






<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-12">
        <div class="panel_s">
          <div class="panel-body">
            <div class="tw-flex tw-justify-between tw-items-center">
              <h4 class="tw-m-0"><?php echo _l('deals_kanban'); ?></h4>
              <a href="<?php echo admin_url('deals'); ?>" class="btn btn-default"><i class="fa fa-list"></i> <?php echo _l('switch_to_list_view'); ?></a>
            </div>
            <hr class="hr-panel-heading" />
            <div class="kanban-wrapper">
              <div class="row">
                <?php foreach ($stages as $stage) { ?>
                  <div class="col-md-3">
                    <div class="kanban-col">
                      <div class="kanban-col-header">
                        <strong><?php echo html_entity_decode($stage['name']); ?></strong>
                      </div>
                      <div class="kanban-items">
                        <?php foreach ($stage['deals'] as $deal) { ?>
                          <div class="kanban-card panel_s">
                            <div class="panel-body">
                              <div class="tw-flex tw-justify-between tw-items-center">
                                <div>
                                  <strong><?php echo html_entity_decode($deal['name']); ?></strong>
                                  <div class="text-muted">#<?php echo (int)$deal['id']; ?> · <?php echo html_entity_decode($deal['company']); ?></div>
                                </div>
                                <div class="text-right">
                                  <div><?php echo app_format_money((float)$deal['deal_value'], get_base_currency()); ?></div>
                                </div>
                              </div>
                              <div class="tw-mt-2">
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-flex tw-gap-2">
                                  <input type="hidden" name="status" value="open">
                                  <button class="btn btn-xs btn-default" type="submit"><?php echo _l('mark_as_open'); ?></button>
                                </form>
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-inline-block tw-ml-2">
                                  <input type="hidden" name="status" value="won">
                                  <button class="btn btn-xs btn-success" type="submit"><?php echo _l('mark_as_won'); ?></button>
                                </form>
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-inline-block tw-ml-2">
                                  <input type="hidden" name="status" value="lost">
                                  <button class="btn btn-xs btn-danger" type="submit"><?php echo _l('mark_as_lost'); ?></button>
                                </form>
                              </div>
                            </div>
                          </div>
                        <?php } ?>
                      </div>
                    </div>
                  </div>
                <?php } ?>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>
<?php init_tail(); ?>








<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="_buttons tw-mb-2 sm:tw-mb-4">
                    <a href="<?php echo admin_url('deals/create'); ?>" class="btn btn-primary mright5 pull-left display-block">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?php echo _l('new_deal'); ?>
                    </a>
                    <a href="<?php echo admin_url('deals?view=kanban'); ?>" class="btn btn-default pull-left display-block mleft5">
                        <i class="fa-solid fa-grip-vertical tw-mr-1"></i>
                        <?php echo _l('deals_kanban'); ?>
                    </a>
                    <div class="row">
                        <div class="col-sm-5">
                            <a href="#" class="btn btn-default btn-with-tooltip" data-toggle="tooltip"
                                data-title="<?php echo _l('deals_summary'); ?>" data-placement="top"
                                onclick="slideToggle('.deals-overview'); return false;">
                                <i class="fa fa-bar-chart"></i>
                            </a>
                        </div>
                        <div class="col-sm-4 col-xs-12 pull-right deals-search">
                            <div class="tw-inline pull-right">
                                <div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
                                    <a href="#" class="btn btn-default deals-table-order-btn" data-toggle="modal" data-target="#deals-table-order-modal" title="<?php echo _l('deals_table_order_configure'); ?>">
                                        <i class="fa fa-columns tw-mr-1"></i> <?php echo _l('deals_table_order_configure'); ?>
                                    </a>
                                </div>
                            </div>
                            <?php echo form_hidden('sort_type'); ?>
                            <?php echo form_hidden('sort', ''); ?>
                        </div>
                    </div>
                    <div class="clearfix"></div>
                    
                    <!-- Deals Summary Overview -->
                    <div class="hide deals-overview tw-mt-2 sm:tw-mt-4 tw-mb-4 sm:tw-mb-0">
                        <h4 class="tw-mt-0 tw-font-semibold tw-text-lg">
                            <?php echo _l('deals_summary'); ?>
                        </h4>
                        <div class="tw-flex tw-flex-wrap tw-flex-col lg:tw-flex-row tw-w-full tw-gap-3 lg:tw-gap-6">
                            <div class="lg:tw-border-r lg:tw-border-solid lg:tw-border-neutral-300 tw-flex-1 tw-flex tw-items-center last:tw-border-r-0">
                                <span class="tw-font-semibold tw-mr-3 rtl:tw-ml-3 tw-text-lg">
                                    <?php echo count($deals); ?>
                                </span>
                                <span class="tw-text-neutral-600"><?php echo _l('total_deals'); ?></span>
                            </div>
                            <div class="lg:tw-border-r lg:tw-border-solid lg:tw-border-neutral-300 tw-flex-1 tw-flex tw-items-center last:tw-border-r-0">
                                <span class="tw-font-semibold tw-mr-3 rtl:tw-ml-3 tw-text-lg">
                                    <?php 
                                    $total_value = 0;
                                    foreach($deals as $deal) {
                                        $total_value += (float)$deal['deal_value'];
                                    }
                                    echo app_format_money($total_value, get_base_currency());
                                    ?>
                                </span>
                                <span class="tw-text-neutral-600"><?php echo _l('total_value'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="panel_s">
                    <div class="panel-body">
                        <div class="row" id="deals-table">
                            <div class="col-md-12">
                                <!-- Bulk Actions Modal -->
                                <div class="modal fade bulk_actions" id="deals_bulk_actions" tabindex="-1" role="dialog">
                                    <div class="modal-dialog" role="document">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                                    <span aria-hidden="true">&times;</span>
                                                </button>
                                                <h4 class="modal-title"><?php echo _l('bulk_actions'); ?></h4>
                                            </div>
                                            <div class="modal-body">
                                                <div class="checkbox checkbox-danger">
                                                    <input type="checkbox" name="mass_delete" id="mass_delete">
                                                    <label for="mass_delete"><?php echo _l('mass_delete'); ?></label>
                                                </div>
                                                <hr class="mass_delete_separator" />
                                                <div id="bulk_change">
                                                    <div class="form-group">
                                                        <label for="move_to_pipeline_bulk"><?php echo _l('move_to_pipeline'); ?></label>
                                                        <select name="move_to_pipeline_bulk" id="move_to_pipeline_bulk" class="selectpicker" data-width="100%">
                                                            <option value=""><?php echo _l('select_pipeline'); ?></option>
                                                            <?php foreach($pipelines ?? [] as $pipeline) { ?>
                                                                <option value="<?php echo $pipeline['id']; ?>"><?php echo $pipeline['name']; ?></option>
                                                            <?php } ?>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
                                                <a href="#" class="btn btn-primary" onclick="deals_bulk_action(this); return false;"><?php echo _l('confirm'); ?></a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Deals Table -->
                                <div class="panel-table-full">
                                    <div class="table-responsive">
                                        <table class="table dt-table table-deals" data-order-col="0" data-order-type="desc">
                                            <thead>
                                                <tr>
                                                    <th><input type="checkbox" id="mass_select_all" data-to-table="deals"></th>
                                                    <th class="th-deal-number"><?php echo _l('the_number_sign'); ?></th>
                                                    <th class="th-deal-name"><?php echo _l('deal_name'); ?></th>
                                                    <th class="th-deal-company"><?php echo _l('company'); ?></th>
                                                    <th class="th-deal-pipeline"><?php echo _l('pipeline'); ?></th>
                                                    <th class="th-deal-stage"><?php echo _l('stage'); ?></th>
                                                    <th class="th-deal-value"><?php echo _l('deal_value'); ?></th>
                                                    <th class="th-deal-assigned"><?php echo _l('assigned_to'); ?></th>
                                                    <th class="th-deal-status"><?php echo _l('status'); ?></th>
                                                    <th class="th-deal-created"><?php echo _l('dateadded'); ?></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($deals as $deal) { ?>
                                                    <tr>
                                                        <td><input type="checkbox" class="individual" data-to-table="deals" value="<?php echo $deal['id']; ?>"></td>
                                                        <td><?php echo '#' . $deal['id']; ?></td>
                                                        <td>
                                                            <a href="<?php echo admin_url('deals/view/' . $deal['id']); ?>" class="tw-font-medium">
                                                                <?php echo html_entity_decode($deal['name']); ?>
                                                            </a>
                                                            <div class="row-options">
                                                                <a href="<?php echo admin_url('deals/view/' . $deal['id']); ?>"><?php echo _l('view'); ?></a>
                                                                <span> | </span>
                                                                <a href="<?php echo admin_url('deals/edit/' . $deal['id']); ?>"><?php echo _l('edit'); ?></a>
                                                                <span> | </span>
                                                                <a href="<?php echo admin_url('deals/delete/' . $deal['id']); ?>" class="text-danger _delete"><?php echo _l('delete'); ?></a>
                                                            </div>
                                                        </td>
                                                        <td><?php echo html_entity_decode($deal['company']); ?></td>
                                                        <td><?php echo isset($deal['pipeline_name']) ? $deal['pipeline_name'] : '-'; ?></td>
                                                        <td><?php echo isset($deal['stage_name']) ? $deal['stage_name'] : '-'; ?></td>
                                                        <td><?php echo app_format_money((float)$deal['deal_value'], get_base_currency()); ?></td>
                                                        <td><?php echo isset($deal['assigned_staff']) ? $deal['assigned_staff'] : '-'; ?></td>
                                                        <td>
                                                            <span class="label label-<?php echo ($deal['status_final'] == 'won' ? 'success' : ($deal['status_final'] == 'lost' ? 'danger' : 'default')); ?>">
                                                                <?php echo ucfirst($deal['status_final']); ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo _dt($deal['dateadded']); ?></td>
                                                    </tr>
                                                <?php } ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php init_tail(); ?>

<script>
$(function() {
    // Initialize DataTable
    $('.table-deals').DataTable({
        "order": [[ 1, "desc" ]],
        "pageLength": 25,
        "responsive": true,
        "columnDefs": [
            { "orderable": false, "targets": 0 }
        ]
    });
    
    // Handle mass select all
    $('#mass_select_all').on('change', function() {
        var checked = $(this).prop('checked');
        $('.individual').prop('checked', checked);
        toggleBulkActions();
    });
    
    // Handle individual checkboxes
    $('.individual').on('change', function() {
        toggleBulkActions();
        var allChecked = $('.individual:checked').length === $('.individual').length;
        $('#mass_select_all').prop('checked', allChecked);
    });
    
    function toggleBulkActions() {
        var checkedBoxes = $('.individual:checked').length;
        if (checkedBoxes > 0) {
            $('.bulk-actions-btn').removeClass('hide');
        } else {
            $('.bulk-actions-btn').addClass('hide');
        }
    }
});

function deals_bulk_action(button) {
    var checkedIds = [];
    $('.individual:checked').each(function() {
        checkedIds.push($(this).val());
    });
    
    if (checkedIds.length === 0) {
        alert('<?php echo _l('no_action_selected'); ?>');
        return;
    }
    
    var massDelete = $('#mass_delete').prop('checked');
    var pipelineId = $('#move_to_pipeline_bulk').val();
    
    if (massDelete) {
        if (confirm('<?php echo _l('confirm_bulk_delete'); ?>')) {
            // Perform delete action
            $.post(admin_url + 'deals/bulk_delete', {ids: checkedIds}, function(response) {
                if (response.success) {
                    location.reload();
                }
            });
        }
    } else if (pipelineId) {
        // Move to pipeline
        $.post(admin_url + 'deals/bulk_move', {ids: checkedIds, pipeline_id: pipelineId}, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    }
}
</script>







<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php
$CI = &get_instance();
if (!isset($reasons)) {
  $CI->load->model('deals/deals_model');
  $reasons = $CI->deals_model->get_lost_reasons();
}
?>

<div class="panel_s">
  <div class="panel-body">
    <div class="row">
      <div class="col-md-6">
        <div class="input-group">
          <input type="text" id="lost_reason_name" class="form-control" placeholder="<?php echo _l('lost_reason'); ?>">
          <span class="input-group-btn">
            <button class="btn btn-primary" onclick="lrCreate()"><?php echo _l('add'); ?></button>
          </span>
        </div>
        <ul class="list-group tw-mt-3">
          <?php foreach (($reasons ?? []) as $r) { ?>
            <li class="list-group-item tw-flex tw-justify-between tw-items-center">
              <span><?php echo html_entity_decode($r['name']); ?></span>
              <span>
                <button class="btn btn-xs btn-default" onclick="lrEdit(<?php echo (int)$r['id']; ?>,'<?php echo html_escape($r['name']); ?>')"><i class="fa fa-pencil"></i></button>
                <button class="btn btn-xs btn-danger" onclick="lrDelete(<?php echo (int)$r['id']; ?>)"><i class="fa fa-trash"></i></button>
              </span>
            </li>
          <?php } ?>
        </ul>
      </div>
    </div>
  </div>
</div>

<script>
var CSRF = { name: '<?php echo $this->security->get_csrf_token_name(); ?>', value: '<?php echo $this->security->get_csrf_hash(); ?>' };
function lrCreate(){
  var name = document.getElementById('lost_reason_name').value.trim();
  if(!name) return;
  var f = new FormData(); f.append('action','create'); f.append('name', name); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
function lrEdit(id, prev){
  var name = prompt('Lost reason', prev || '');
  if(name===null) return;
  var f = new FormData(); f.append('action','update'); f.append('id', id); f.append('name', name); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
function lrDelete(id){
  if(!confirm('Delete?')) return;
  var f = new FormData(); f.append('action','delete'); f.append('id', id); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
</script>








<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php
$CI = &get_instance();
if (!isset($pipelines)) {
    $CI->load->model('deals/deals_model');
    $pipelines = $CI->deals_model->get_pipelines();
}
// Ensure stages are present for each pipeline
if (is_array($pipelines)) {
    foreach ($pipelines as $idx => $p) {
        if (!isset($p['stages'])) {
            $pipelines[$idx]['stages'] = $CI->deals_model->get_stages($p['id']);
        }
    }
}
$selectedPipelineId = isset($pipelines[0]['id']) ? (int)$pipelines[0]['id'] : 0;
?>

<h4 class="tw-mb-4"><?php echo _l('pipelines'); ?></h4>

<div class="panel_s">
  <div class="panel-body">
    <div class="row">
      <div class="col-md-12">
        <h5><?php echo _l('deals'); ?> <?php echo _l('pipelines'); ?></h5>
        
        <!-- Pipeline Management Section -->
        <div class="form-group">
          <label class="control-label"><?php echo _l('pipelines'); ?></label>
          <select id="pipeline_select" class="selectpicker" data-width="100%" onchange="onPipelineChange()">
            <option value=""><?php echo _l('select_pipeline'); ?></option>
            <?php foreach (($pipelines ?? []) as $p) { ?>
              <option value="<?php echo (int)$p['id']; ?>" <?php echo ((int)$p['id']===$selectedPipelineId?'selected':''); ?> data-name="<?php echo html_escape($p['name']); ?>"><?php echo html_entity_decode($p['name']); ?></option>
            <?php } ?>
          </select>
        </div>
        
        <!-- Add/Edit Pipeline Form -->
        <div class="form-group">
          <div class="input-group">
            <input type="text" id="pipeline_name" class="form-control" placeholder="<?php echo _l('pipeline_name'); ?>">
            <span class="input-group-btn">
              <button id="pipeline_action_btn" class="btn btn-primary" type="button" onclick="createOrUpdatePipeline()"><?php echo _l('add'); ?></button>
            </span>
          </div>
        </div>
        
        <!-- Pipeline List -->
        <ul class="list-group" id="pipelines-list" style="max-height:60vh; overflow:auto;">
          <?php foreach (($pipelines ?? []) as $p) { ?>
            <li class="list-group-item pipeline-item" data-id="<?php echo (int)$p['id']; ?>" style="<?php echo ((int)$p['id']!==$selectedPipelineId?'display:none;':''); ?>">
              <div class="tw-flex tw-justify-between tw-items-center">
                <strong><?php echo html_entity_decode($p['name']); ?></strong>
                <div>
                  <button class="btn btn-xs btn-default" onclick="editPipeline(<?php echo (int)$p['id']; ?>)"><i class="fa fa-pencil"></i></button>
                  <button class="btn btn-xs btn-danger" onclick="deletePipeline(<?php echo (int)$p['id']; ?>)"><i class="fa fa-trash"></i></button>
                </div>
              </div>
              
              <!-- Stages Section -->
              <div class="tw-mt-2">
                <small><?php echo _l('stages'); ?></small>
                <ul class="list-unstyled sortable-stages" data-pipeline="<?php echo (int)$p['id']; ?>">
                  <?php foreach (($p['stages'] ?? []) as $s) { ?>
                    <li class="stage-item tw-py-2 tw-flex tw-items-center tw-justify-between" data-id="<?php echo (int)$s['id']; ?>">
                      <div class="tw-flex tw-items-center tw-gap-2">
                        <i class="fa fa-grip-vertical text-muted handle" style="cursor: move;"></i>
                        <input class="form-control input-sm" style="width:260px" value="<?php echo html_entity_decode($s['name']); ?>" onchange="updateStage(<?php echo (int)$s['id']; ?>,{name:this.value})">
                      </div>
                      <div class="tw-flex tw-items-center tw-gap-2">
                        <div class="tw-flex tw-items-center" style="width:260px">
                          <input type="range" min="0" max="100" value="<?php echo (int)($s['win_probability'] ?? 100); ?>" class="form-control" oninput="document.getElementById('prob_<?php echo (int)$s['id']; ?>').innerText=this.value;" onchange="updateStage(<?php echo (int)$s['id']; ?>,{win_probability:this.value})">
                          <span id="prob_<?php echo (int)$s['id']; ?>" class="tw-ml-2"><?php echo (int)($s['win_probability'] ?? 100); ?></span>
                        </div>
                        <button class="btn btn-xs btn-danger" onclick="deleteStage(<?php echo (int)$s['id']; ?>)"><i class="fa fa-trash"></i></button>
                      </div>
                    </li>
                  <?php } ?>
                </ul>
                <button class="btn btn-xs btn-success" onclick="addStage(<?php echo (int)$p['id']; ?>)"><i class="fa fa-plus"></i> <?php echo _l('add'); ?> <?php echo _l('stage'); ?></button>
              </div>
            </li>
          <?php } ?>
        </ul>
      </div>
    </div>
  </div>
</div>

<!-- Include jQuery UI for drag and drop -->
<link rel="stylesheet" href="//code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
<script src="https://code.jquery.com/ui/1.13.2/jquery-ui.js"></script>

<style>
.sortable-stages {
    min-height: 20px;
}

.stage-item {
    background: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    margin-bottom: 5px;
    padding: 8px 12px;
    transition: all 0.2s ease;
}

.stage-item:hover {
    background: #e9ecef;
    border-color: #adb5bd;
}

.stage-item.ui-sortable-helper {
    background: #fff;
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    transform: rotate(2deg);
}

.stage-item.ui-sortable-placeholder {
    background: #f8f9fa;
    border: 2px dashed #dee2e6;
    visibility: visible !important;
    height: 40px;
}

.handle {
    cursor: move;
    color: #6c757d;
}

.handle:hover {
    color: #495057;
}

.pipeline-item {
    margin-bottom: 20px;
    border: 1px solid #dee2e6;
    border-radius: 6px;
}

.pipeline-item .list-group-item {
    border: none;
    border-radius: 6px;
}

.form-group {
    margin-bottom: 20px;
}

.btn-xs {
    padding: 2px 8px;
    font-size: 12px;
}
</style>

<script>
var CSRF = { name: '<?php echo $this->security->get_csrf_token_name(); ?>', value: '<?php echo $this->security->get_csrf_hash(); ?>' };

function createOrUpdatePipeline(){
  var select = document.getElementById('pipeline_select');
  var id = select.value;
  var name = document.getElementById('pipeline_name').value.trim();
  if (!name) { 
    alert('<?php echo _l('pipeline_name_required'); ?>');
    return; 
  }
  
  if (id) { 
    request('update', {id:id, name:name}); 
  } else { 
    request('create', {name:name}); 
  }
}

function onPipelineChange(){
  var select = document.getElementById('pipeline_select');
  var id = select.value;
  var name = select.options[select.selectedIndex] ? select.options[select.selectedIndex].getAttribute('data-name') : '';
  
  // populate input and button state
  document.getElementById('pipeline_name').value = name || '';
  document.getElementById('pipeline_action_btn').innerText = name ? '<?php echo _l('edit'); ?>' : '<?php echo _l('add'); ?>';
  
  // toggle pipeline blocks
  Array.prototype.forEach.call(document.querySelectorAll('.pipeline-item'), function(el){
    el.style.display = (String(el.getAttribute('data-id')) === String(id)) ? '' : 'none';
  });
}

function editPipeline(id){
  var name = prompt('<?php echo _l('enter_pipeline_name'); ?>');
  if(name && name.trim()){ 
    request('update', {id:id, name:name.trim()}); 
  }
}

function deletePipeline(id){
  if(confirm('<?php echo _l('delete_pipeline_confirm'); ?>')){ 
    request('delete', {id:id}); 
  }
}

function addStage(pipeline_id){
  var name = prompt('<?php echo _l('enter_stage_name'); ?>');
  if(name && name.trim()){ 
    request('create_stage', {pipeline_id:pipeline_id, name:name.trim()}); 
  }
}

function updateStage(id, payload){
  payload = payload || {};
  payload.id = id;
  payload.action = 'update_stage';
  var form = new FormData();
  Object.keys(payload).forEach(function(k){ form.append(k, payload[k]); });
  form.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { method:'POST', body: form, credentials: 'same-origin' })
    .then(r=>r.json()).then(()=>location.reload());
}

function deleteStage(id){
  if(confirm('<?php echo _l('delete_stage_confirm'); ?>')){ 
    request('delete_stage', {id:id}); 
  }
}

function request(action, payload){
  var form = new FormData();
  form.append('action', action);
  Object.keys(payload||{}).forEach(function(k){ form.append(k, payload[k]); });
  form.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { method:'POST', body: form, credentials: 'same-origin' })
    .then(r=>r.json())
    .then(()=>location.reload());
}

// Initialize drag and drop functionality
$(document).ready(function() {
  // Make stages sortable within each pipeline
  $('.sortable-stages').sortable({
    handle: '.handle',
    axis: 'y',
    cursor: 'move',
    opacity: 0.8,
    update: function(event, ui) {
      var pipeline = $(this).data('pipeline');
      var order = $(this).children('li').map(function(){ 
        return $(this).data('id'); 
      }).get();
      
      // Send reorder request to server
      var form = new FormData();
      form.append('action', 'reorder_stages');
      form.append('pipeline_id', pipeline);
      order.forEach(function(id){ 
        form.append('order[]', id); 
      });
      form.append(CSRF.name, CSRF.value);
      
      fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { 
        method: 'POST', 
        body: form, 
        credentials: 'same-origin' 
      }).then(r => r.json()).then(data => {
        if (data.success) {
          // Show success message
          console.log('Stages reordered successfully');
        }
      });
    }
  });
  
  // Initialize default view
  onPipelineChange();
});
</script>









<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<link rel="stylesheet" href="<?php echo base_url('modules/deals/views/deals.css'); ?>">
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-12">
        <div class="panel_s">
          <div class="panel-body">
            <!-- Header Section -->
            <div class="row">
              <div class="col-md-8">
                <h4 class="tw-m-0"><?php echo html_entity_decode($deal['name']); ?></h4>
                <p class="text-muted tw-mt-1">
                  <?php echo _l('deal_value'); ?>:
                  <?php echo app_format_money((float) $deal['deal_value'], get_base_currency()); ?> |
                  <?php echo _l('expected_close_date'); ?>:
                  <?php echo isset($deal['expected_close_date']) ? _d($deal['expected_close_date']) : '-'; ?>
                </p>
                <p class="text-muted tw-mt-1">
                  <?php echo _l('created_at'); ?>: <?php echo _dt($deal['dateadded']); ?>
                </p>
              </div>
              <div class="col-md-4 text-right">
                <div class="tw-flex tw-gap-2 tw-justify-end">
                  <button type="button" class="btn btn-info btn-xs" onclick="testDatabase()">
                    <i class="fa fa-database"></i> Test DB
                  </button>
                  <?php if ($deal['status_final'] === 'won') { ?>
                    <span class="btn btn-success" style="cursor: default;"><?php echo _l('won'); ?></span>
                  <?php } elseif ($deal['status_final'] === 'lost') { ?>
                    <span class="btn btn-danger" style="cursor: default;"><?php echo _l('lost'); ?></span>
                    <?php if (isset($deal['lost_reason_name']) && $deal['lost_reason_name']) { ?>
                      <span class="btn btn-default"
                        style="cursor: default;"><?php echo html_entity_decode($deal['lost_reason_name']); ?></span>
                    <?php } ?>
                  <?php } else { ?>
                    <button type="button" class="btn btn-success"
                      onclick="markDealWon(<?php echo (int) $deal['id']; ?>)"><?php echo _l('mark_as_won'); ?></button>
                    <button type="button" class="btn btn-danger"
                      onclick="showLostReasonModal()"><?php echo _l('mark_as_lost'); ?></button>
                  <?php } ?>
                </div>
              </div>
            </div>

            <hr class="hr-panel-heading" />

            <!-- Pipeline Stages -->
            <div class="tw-flex tw-gap-2 tw-flex-wrap tw-mb-4 tw-justify-center">
              <?php
              $final = $deal['status_final'];
              $currentStage = (int) $deal['stage_id'];
              $stageCount = count($stages);

              foreach ($stages as $index => $stage) {
                $stageId = (int) $stage['id'];
                if ($final === 'won') {
                  $bg = '#84c529';
                } elseif ($final === 'lost') {
                  $bg = '#fc2d42';
                } else {
                  $currentIndex = -1;
                  foreach ($stages as $idx => $s) {
                    if ((int) $s['id'] === $currentStage) {
                      $currentIndex = $idx;
                      break;
                    }
                  }
                  if ($index < $currentIndex) {
                    $bg = '#84c529';
                  } elseif ($index === $currentIndex) {
                    $bg = '#FFD600';
                  } else {
                    $bg = '#e3e3e3';
                  }
                }
                ?>
                <div class="tw-flex tw-items-center">
                  <button type="button" class="btn"
                    style="background: <?php echo $bg; ?>; color:#fff; border-radius: 50%; width: 40px; height: 40px; padding: 0; font-size: 16px;"
                    onclick="setDealStage(<?php echo (int) $deal['id']; ?>, <?php echo (int) $stage['id']; ?>)" <?php echo ($final === 'won' || $final === 'lost') ? 'disabled' : ''; ?>>
                    <?php
                    if ($final === 'won' || $final === 'lost') {
                      // Always show check icon for all stages if deal is won/lost
                      echo '<i class="fa fa-check"></i>';
                    } else {
                      $currentIndex = -1;
                      foreach ($stages as $idx => $s) {
                        if ((int) $s['id'] === $currentStage) {
                          $currentIndex = $idx;
                          break;
                        }
                      }
                      if ($index < $currentIndex) {
                        // Completed stage: check icon
                        echo '<i class="fa fa-check"></i>';
                      } elseif ($index === $currentIndex) {
                        // Current stage: dot icon
                        echo '<i class="fa fa-exclamation"></i>';
                      } else {
                        // Future stage: show number
                        echo ($index + 1);
                      }
                    }
                    ?>
                  </button>
                  <span class="tw-mx-2 tw-text-sm tw-font-medium"><?php echo html_entity_decode($stage['name']); ?></span>
                  <?php if ($index < $stageCount - 1) { ?>
                    <i class="fa fa-arrow-right tw-mx-2 tw-text-gray-400"></i>
                  <?php } ?>
                </div>
              <?php } ?>
            </div>

            <!-- Main Content: Left Sidebar (33%) and Right Content (66%) -->
            <div class="row">
              <!-- Left Sidebar - Details, Contacts, Companies, Attachments -->
              <div class="col-md-4">
                <!-- Details Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title"><?php echo _l('details'); ?></h5>
                  </div>
                  <div class="modern-card-body">
                    <div class="form-group deal-value-group" style="position:relative;">
                      <label class="tw-font-medium"><?php echo _l('deal_value'); ?></label>
                      <p class="tw-text-lg tw-font-bold tw-text-green-600 deal-value-display"
                        style="margin-bottom:0; display:inline-block;">
                        <?php echo app_format_money((float) $deal['deal_value'], get_base_currency()); ?>
                        <button class="btn btn-xs btn-default tw-ml-2 edit-amount-btn"
                          style="display:none; position:relative; top:-2px;" onclick="editDealAmount()">
                          <i class="fa fa-edit"></i>
                        </button>
                      </p>
                      <form id="edit-amount-form" style="display:none; margin-top:8px;">
                        <div class="input-group input-group-sm" style="max-width:180px;">
                          <input type="number" step="0.01" min="0" class="form-control" id="edit-amount-input"
                            value="<?php echo (float) $deal['deal_value']; ?>">
                          <span class="input-group-btn">
                            <button class="btn btn-success" type="submit"><i class="fa fa-check"></i></button>
                            <button class="btn btn-default" type="button" onclick="cancelEditAmount()"><i
                                class="fa fa-times"></i></button>
                          </span>
                        </div>
                      </form>
                    </div>
                    <div class="form-group close-date-group" style="position:relative;">
                      <label class="tw-font-medium"><?php echo _l('expected_close_date'); ?></label>
                      <p class="tw-text-gray-700 close-date-display" style="margin-bottom:0; display:inline-block;">
                        <?php echo isset($deal['expected_close_date']) ? _d($deal['expected_close_date']) : '-'; ?>
                        <button class="btn btn-xs btn-default tw-ml-2 edit-close-date-btn"
                          style="display:none; position:relative; top:-2px;" onclick="editCloseDate()">
                          <i class="fa fa-edit"></i>
                        </button>
                      </p>
                      <form id="edit-close-date-form" style="display:none; margin-top:8px;">
                        <div class="input-group input-group-sm" style="max-width:180px;">
                          <input type="date" class="form-control" id="edit-close-date-input"
                            value="<?php echo isset($deal['expected_close_date']) ? date('Y-m-d', strtotime($deal['expected_close_date'])) : ''; ?>">
                          <span class="input-group-btn">
                            <button class="btn btn-success" type="submit"><i class="fa fa-check"></i></button>
                            <button class="btn btn-default" type="button" onclick="cancelEditCloseDate()"><i
                                class="fa fa-times"></i></button>
                          </span>
                        </div>
                      </form>
                    </div>
                  </div>
                </div>

                <!-- Contacts Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('contacts'); ?>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addContact()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="contacts-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-users fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_contacts_found'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>

                <!-- Companies Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('companies'); ?> <span id="companies-count">(0)</span>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addCompany()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="companies-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-building fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_company_associated'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>

                <!-- Attachments Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('attachments'); ?>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addAttachment()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="attachments-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-paperclip fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_attachments_uploaded'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              <!-- Right Side - Activities -->
              <div class="col-md-8">
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title"><?php echo _l('activities'); ?></h5>
                  </div>
                  <div class="modern-card-body">
                    <!-- Activity Tabs -->
                    <ul class="nav nav-tabs" role="tablist">
                      <li role="presentation" class="active">
                        <a href="#all-activities" aria-controls="all-activities" role="tab"
                          data-toggle="tab"><?php echo _l('all'); ?></a>
                      </li>
                      <li role="presentation">
                        <a href="#activities-tab" aria-controls="activities-tab" role="tab"
                          data-toggle="tab"><?php echo _l('activities'); ?> <span class="badge"
                            id="badge-activities">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#emails-tab" aria-controls="emails-tab" role="tab"
                          data-toggle="tab"><?php echo _l('emails'); ?> <span class="badge"
                            id="badge-emails">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#documents-tab" aria-controls="documents-tab" role="tab"
                          data-toggle="tab"><?php echo _l('documents'); ?> <span class="badge"
                            id="badge-documents">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#calls-tab" aria-controls="calls-tab" role="tab"
                          data-toggle="tab"><?php echo _l('calls'); ?> <span class="badge" id="badge-calls">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#notes-tab" aria-controls="notes-tab" role="tab"
                          data-toggle="tab"><?php echo _l('notes'); ?> <span class="badge" id="badge-notes">0</span></a>
                      </li>
                    </ul>

                    <!-- Tab Content -->
                    <div class="tab-content tw-mt-3">
                      <div role="tabpanel" class="tab-pane active" id="all-activities">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <select class="form-control tw-w-auto" id="activity-filter">
                            <option value="all"><?php echo _l('filter_by_all'); ?></option>
                            <option value="activities"><?php echo _l('activities'); ?></option>
                            <option value="emails"><?php echo _l('emails'); ?></option>
                            <option value="documents"><?php echo _l('documents'); ?></option>
                            <option value="calls"><?php echo _l('calls'); ?></option>
                            <option value="notes"><?php echo _l('notes'); ?></option>
                          </select>
                        </div>
                        <div class="timeline-container" id="activities-list">
                          <!-- Activities will be loaded here -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="activities-tab">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="activities-search"
                            placeholder="Search...">
                          <button class="btn btn-primary btn-sm"
                            onclick="toggleAddActivityForm()"><?php echo _l('add_activity'); ?></button>
                        </div>
                        <div id="add-activity-form" style="display: none;" class="tw-mb-3">
                          <div class="form-group">
                            <label>* Title</label>
                            <input type="text" id="activity-title" class="form-control" required>
                          </div>
                          <div class="tw-flex tw-gap-2">
                            <div class="form-group tw-flex-1">
                              <label>Date</label>
                              <input type="date" id="activity-date" class="form-control"
                                value="<?php echo date('Y-m-d'); ?>">
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>Time</label>
                              <input type="time" id="activity-time" class="form-control"
                                value="<?php echo date('H:i'); ?>">
                            </div>
                          </div>
                          <div class="tw-flex tw-gap-2">
                            <div class="form-group tw-flex-1">
                              <label>Reminder</label>
                              <input type="number" id="reminder-num" class="form-control" min="0" value="30">
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>&nbsp;</label>
                              <select id="reminder-unit" class="form-control">
                                <option value="minutes">minutes</option>
                                <option value="hours">hours</option>
                                <option value="days">days</option>
                              </select>
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>&nbsp;</label>
                              <select id="reminder-when" class="form-control">
                                <option value="before due">before due</option>
                                <option value="after due">after due</option>
                              </select>
                            </div>
                          </div>
                          <div class="form-group">
                            <label>* Owner</label>
                            <select id="activity-owner" class="form-control">
                              <option value="">Admin</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label>+ Guests</label>
                            <select id="activity-guests" class="form-control" multiple>
                            </select>
                          </div>
                          <div class="form-group">
                            <label>+ Description</label>
                            <textarea id="activity-description" class="form-control" rows="4"></textarea>
                          </div>
                          <div class="form-group">
                            <label>Note</label>
                            <textarea id="activity-note" class="form-control"
                              placeholder="Notes are private and visible only for the sales reps." rows="4"></textarea>
                          </div>
                          <p>Associated with 1 record</p>
                          <label><input type="checkbox" id="activity-completed"> Mark as completed</label>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" onclick="toggleAddActivityForm()">Cancel</button>
                            <button class="btn btn-primary" onclick="saveActivity()">Add Activity</button>
                          </div>
                        </div>
                        <div class="tw-flex tw-overflow-x-auto tw-gap-2 tw-mb-3" id="activity-periods">
                          <button class="active" data-period="all">All (1)</button>
                          <button data-period="today">Today (0)</button>
                          <button data-period="tomorrow">Tomorrow (0)</button>
                          <button data-period="this_week">This Week (0)</button>
                          <button data-period="next_week">Next Week (0)</button>
                          <button data-period="done">Done (0)</button>
                        </div>
                        <div class="list-container" id="activities-only-list">
                          <!-- Activities only -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="emails-tab">
                        <div class="timeline-container" id="emails-list">
                          <!-- Emails -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="documents-tab">
                        <div class="timeline-container" id="documents-list">
                          <!-- Documents -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="calls-tab">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="calls-search"
                            placeholder="Search calls...">
                          <button class="btn btn-primary btn-sm" onclick="toggleLogCallForm()">+ Log Call</button>
                        </div>
                        <div id="log-call-form" style="display: none;" class="tw-mb-3 tw-bg-gray-50 tw-p-4 tw-rounded">
                          <div class="form-group">
                            <label for="call-type"><span class="text-danger">*</span> Call Type</label>
                            <select id="call-type" class="form-control" required>
                              <option value="">Select Type</option>
                              <option value="inbound">Inbound</option>
                              <option value="outbound">Outbound</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label for="call-status"><span class="text-danger">*</span> Call Status</label>
                            <select id="call-status" class="form-control" required>
                              <option value="">Select Status</option>
                              <option value="completed">Completed</option>
                              <option value="missed">Missed</option>
                              <option value="scheduled">Scheduled</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label for="call-duration">Call Duration (minutes)</label>
                            <input type="number" min="0" id="call-duration" class="form-control"
                              placeholder="Duration in minutes">
                          </div>
                          <div class="form-group">
                            <label for="call-notes">Call Notes</label>
                            <textarea id="call-notes" class="form-control" rows="3"
                              placeholder="E.g. Spoke with client, discussed next steps..."></textarea>
                          </div>
                          <div class="form-group">
                            <label for="call-date"><span class="text-danger">*</span> Call Date &amp; Time</label>
                            <input type="datetime-local" id="call-date" class="form-control"
                              value="<?php echo date('Y-m-d\TH:i'); ?>">
                          </div>
                          <p class="tw-text-xs tw-text-gray-500">Associated with 1 record</p>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" type="button" onclick="toggleLogCallForm()">Cancel</button>
                            <button class="btn btn-primary" type="button" onclick="saveCall()">Log Call</button>
                          </div>
                        </div>
                        <div class="list-container" id="calls-list">
                          <!-- Calls -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="notes-tab">
                        <div class="tw-mb-3">
                          <h5>Manage Notes</h5>
                          <p>You can create notes for you and your team and keep track of important info here.</p>
                        </div>
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="notes-search" placeholder="Search...">
                          <button class="btn btn-primary btn-sm" onclick="toggleAddNoteForm()">+ Add Note</button>
                        </div>
                        <div id="add-note-form" style="display: none;" class="tw-mb-3">
                          <div class="form-group">
                            <textarea id="note-description" class="form-control" rows="4"
                              placeholder="Paragraph"></textarea>
                          </div>
                          <p>Notes are private and visible only for the sales reps.</p>
                          <p>Associated with 1 record</p>
                          <label><input type="checkbox" id="create-followup-note"> Create follow up task</label>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" onclick="toggleAddNoteForm()">Cancel</button>
                            <button class="btn btn-primary" onclick="saveNote()">Add Note</button>
                          </div>
                        </div>
                        <div class="list-container" id="notes-list">
                          <!-- Notes -->
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Lost Reason Modal -->
<div class="modal fade" id="lostReasonModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title"><?php echo _l('mark_as_lost'); ?></h4>
      </div>
      <div class="modal-body">
        <div class="form-group">
          <label><?php echo _l('lost_reason'); ?></label>
          <select id="lost-reason-select" class="form-control" required>
            <option value="">-- <?php echo _l('select_lost_reason'); ?> --</option>
            <?php foreach (($lost_reasons ?? []) as $r) { ?>
              <option value="<?php echo (int) $r['id']; ?>"><?php echo html_entity_decode($r['name']); ?></option>
            <?php } ?>
          </select>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('cancel'); ?></button>
        <button type="button" class="btn btn-danger"
          onclick="markDealLost(<?php echo (int) $deal['id']; ?>)"><?php echo _l('mark_as_lost'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Contact Modal -->
<div class="modal fade" id="contactModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title" id="contactModalTitle"><?php echo _l('add_contact'); ?></h4>
      </div>
      <div class="modal-body">
        <ul class="nav nav-tabs" role="tablist">
          <li role="presentation" class="active">
            <a href="#select-contact" aria-controls="select-contact" role="tab"
              data-toggle="tab"><?php echo _l('select_existing'); ?></a>
          </li>
          <li role="presentation">
            <a href="#create-contact" aria-controls="create-contact" role="tab"
              data-toggle="tab"><?php echo _l('create_new'); ?></a>
          </li>
        </ul>

        <div class="tab-content tw-mt-3">
          <div role="tabpanel" class="tab-pane active" id="select-contact">
            <div class="form-group">
              <label><?php echo _l('select_contact'); ?></label>
              <select class="form-control" id="existing-contact-select">
                <option value=""><?php echo _l('select_contact'); ?></option>
              </select>
            </div>
          </div>

          <div role="tabpanel" class="tab-pane" id="create-contact">
            <div class="form-group">
              <label><?php echo _l('firstname'); ?></label>
              <input type="text" id="contact-firstname" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('lastname'); ?></label>
              <input type="text" id="contact-lastname" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('email'); ?></label>
              <input type="email" id="contact-email" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('phonenumber'); ?></label>
              <input type="text" id="contact-phonenumber" class="form-control">
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveContact()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Company Modal -->
<div class="modal fade" id="companyModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title" id="companyModalTitle"><?php echo _l('add_company'); ?></h4>
      </div>
      <div class="modal-body">
        <ul class="nav nav-tabs" role="tablist">
          <li role="presentation" class="active">
            <a href="#select-company" aria-controls="select-company" role="tab"
              data-toggle="tab"><?php echo _l('select_existing'); ?></a>
          </li>
          <li role="presentation">
            <a href="#create-company" aria-controls="create-company" role="tab"
              data-toggle="tab"><?php echo _l('create_new'); ?></a>
          </li>
        </ul>

        <div class="tab-content tw-mt-3">
          <div role="tabpanel" class="tab-pane active" id="select-company">
            <div class="form-group">
              <label><?php echo _l('select_company'); ?></label>
              <select class="form-control" id="existing-company-select">
                <option value=""><?php echo _l('select_company'); ?></option>
              </select>
            </div>
          </div>

          <div role="tabpanel" class="tab-pane" id="create-company">
            <div class="form-group">
              <label><?php echo _l('company_name'); ?></label>
              <input type="text" id="company-name" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('company_email'); ?></label>
              <input type="email" id="company-email" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_phone'); ?></label>
              <input type="text" id="company-phone" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_website'); ?></label>
              <input type="url" id="company-website" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_address'); ?></label>
              <textarea id="company-address" class="form-control" rows="3"></textarea>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveCompany()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Attachment Modal -->
<div class="modal fade" id="attachmentModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title"><?php echo _l('add_attachment'); ?></h4>
      </div>
      <div class="modal-body">
        <div class="form-group">
          <label><?php echo _l('file'); ?></label>
          <input type="file" id="attachment-file" class="form-control" required>
        </div>
        <div class="form-group">
          <label><?php echo _l('description'); ?></label>
          <textarea id="attachment-description" class="form-control" rows="3"></textarea>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveAttachment()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1" role="dialog">
  <div class="modal-dialog modal-lg" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title">Preview</h4>
      </div>
      <div class="modal-body">
        <div id="preview-content"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<?php init_tail(); ?>



<script>
  var dealId = <?php echo (int) $deal['id']; ?>;
  var CSRF = {
    name: '<?php echo $this->security->get_csrf_token_name(); ?>',
    value: '<?php echo $this->security->get_csrf_hash(); ?>'
  };

  var staffMap = {};

  $(function () {
    loadContacts(); loadCompanies(); loadAttachments(); loadActivities();
    loadAvailableContacts(); loadAvailableCompanies(); loadStaff();

    $('#activities-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#activities-only-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });
    $('#calls-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#calls-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });
    $('#notes-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#notes-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });

    $('#activity-filter').on('change', filterActivities);
    $('a[data-toggle="tab"]').on('shown.bs.tab', placeTimelineDots);
  });

  function parseIfString(response) {
    if (typeof response === 'string') {
      try {
        return JSON.parse(response);
      } catch (e) {
        return { __parseError: true, error: e, raw: response };
      }
    }
    return response;
  }

  function getActivityIconData(type) {
    // Use only one definition, prefer the one with more types and consistent with your payload
    switch (type) {
      case 'calls':
      case 'call':
        return { icon: 'fa-phone', bg: '#10b981', color: '#fff' };
      case 'emails':
        return { icon: 'fa-envelope', bg: '#6366f1', color: '#fff' };
      case 'notes':
      case 'note':
        return { icon: 'fa-sticky-note-o', bg: '#f59e42', color: '#fff' };
      case 'documents':
        return { icon: 'fa-file-text-o', bg: '#fbbf24', color: '#fff' };
      case 'stage':
        return { icon: 'fa-arrow-right', bg: '#3b82f6', color: '#fff' };
      case 'status':
        return { icon: 'fa-check-circle', bg: '#84c529', color: '#fff' };
      case 'contact':
        return { icon: 'fa-user-plus', bg: '#10b981', color: '#fff' };
      case 'company':
        return { icon: 'fa-building', bg: '#6366f1', color: '#fff' };
      case 'attachment':
        return { icon: 'fa-paperclip', bg: '#f59e42', color: '#fff' };
      default:
        return { icon: 'fa-clock-o', bg: '#6b7280', color: '#fff' };
    }
  }

  function getActivityBadge(type) {
    switch (type) {
      case 'calls': return 'call';
      case 'emails': return 'email';
      case 'notes': return 'note';
      case 'documents': return 'document';
      case 'stage': return 'stage';
      case 'status': return 'status';
      default: return 'activity';
    }
  }
  function isRowType(type) { return ['stage', 'status', 'contact', 'company', 'attachment', 'emails', 'documents'].indexOf(type || '') !== -1; }
  function isCardType(type) { return ['activities', 'activity', 'notes', 'note', 'calls', 'call'].indexOf(type || '') !== -1; }

  function loadStaff() {
    $.get(admin_url + 'deals/get_staff_ajax', function (res) {
      res = parseIfString(res);
      if (res.success && res.staff) {
        res.staff.forEach(function (s) {
          $('#activity-owner').append('<option value="' + s.staffid + '">' + s.firstname + ' ' + s.lastname + '</option>');
          $('#activity-guests').append('<option value="' + s.staffid + '">' + s.firstname + ' ' + s.lastname + '</option>');
        });
      }
    });
  }

  function cancelActivity() {
    $('#activities-add-form').hide();
    $('#activities-controls').show();
  }

  function cancelCall() {
    $('#calls-add-form').hide();
    $('#calls-controls').show();
  }

  function cancelNote() {
    $('#notes-add-form').hide();
    $('#notes-controls').show();
    $('#add-note-btn').show();
  }

  // Only one saveActivity, saveCall, saveNote function, matching your backend payload
  function saveActivity() {
    var title = $('#activity-title').val() ? $('#activity-title').val().trim() : '';
    if (!title) {
      alert('Title is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('title', title);
    formData.append('activity_date', $('#activity-date').val());
    formData.append('description', $('#activity-description').val());
    formData.append('status', $('#activity-completed').prop('checked') ? 'completed' : 'pending');
    formData.append('owner_id', $('#activity-owner').val());
    formData.append('guests', JSON.stringify($('#activity-guests').val() || []));
    // Reminder calculation
    var num = parseInt($('#reminder-num').val() || 0);
    var unit = $('#reminder-unit').val();
    var when = $('#reminder-when').val();
    var factor = unit === 'minutes' ? 60 : unit === 'hours' ? 3600 : 86400;
    var seconds = num * factor;
    if (when === 'after due') seconds = -seconds;
    formData.append('reminder', seconds);

    $.ajax({
      url: admin_url + 'deals/add_activity/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelActivity();
          loadActivities();
        } else {
          alert('Error adding activity');
        }
      }
    });
  }

  function saveCall() {
    var outcome = $('#call-outcome').val();
    if (!outcome) {
      alert('Outcome is required');
      return;
    }
    var callNotes = $('#call-description').val();
    var callDate = $('#call-date').val();

    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('outcome', outcome);
    // If callDate is empty, let backend default to now
    if (callDate) {
      formData.append('call_date', callDate);
    }
    if (callNotes) {
      formData.append('call_notes', callNotes);
    }

    $.ajax({
      url: admin_url + 'deals/add_call/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelCall();
          loadActivities();
        } else {
          alert('Error adding call: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding call');
      }
    });
  }

  function saveNote() {
    var description = $('#note-description').val() ? $('#note-description').val().trim() : '';
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);

    $.ajax({
      url: admin_url + 'deals/add_note/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelNote();
          loadActivities();
        } else {
          alert('Error adding note');
        }
      }
    });
  }

  function toggleActivityStatus(id, checked) {
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('status', checked ? 'completed' : 'pending');

    $.ajax({
      url: admin_url + 'deals/update_activity_status/' + id,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          loadActivities();
        }
      }
    });
  }

  function filterActivitiesByPeriod(period) {
    var now = moment();
    $('#activities-only-list .activity-card').show();
    if (period !== 'all') {
      $('#activities-only-list .activity-card').hide();
      $('#activities-only-list .activity-card').filter(function () {
        var date = $(this).data('date');
        var status = $(this).data('status');
        if (period === 'done') return status === 'completed';
        if (!date) return false;
        var mDate = moment(date);
        if (period === 'today') return mDate.isSame(now, 'day');
        if (period === 'tomorrow') return mDate.isSame(now.clone().add(1, 'day'), 'day');
        if (period === 'this_week') return mDate.isSame(now, 'week');
        if (period === 'next_week') return mDate.isSame(now.clone().add(1, 'week'), 'week');
      }).show();
    }
  }

  function loadActivities() {
    $.get(admin_url + 'deals/get_deal_activities/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        var errorHtml = '<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing activities response</p></div>';
        $('#activities-list,#activities-only-list,#emails-list,#documents-list,#calls-list,#notes-list').html(errorHtml);
        return;
      }
      if (!(response.success && response.activities && response.activities.length)) {
        var noHtml = '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_activity_found"); ?></p></div>';
        $('#activities-list,#activities-only-list').html(noHtml);
        $('#emails-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_emails_found"); ?></p></div>');
        $('#documents-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_documents_found"); ?></p></div>');
        $('#calls-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_calls_found"); ?></p></div>');
        $('#notes-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_notes_found"); ?></p></div>');
        $('#badge-activities,#badge-emails,#badge-documents,#badge-calls,#badge-notes').text('0');
        return;
      }

      var htmlAll = '', htmlActivitiesOnly = '', htmlEmails = '', htmlDocuments = '', htmlCalls = '', htmlNotes = '';
      var countActivities = 0, countEmails = 0, countDocuments = 0, countCalls = 0, countNotes = 0;

      response.activities.forEach(function (a) {
        var t = a.type || 'activity';
        var icon = getActivityIconData(t);
        var badge = getActivityBadge(t);

        function renderRow() {
          var row = '';
          row += '<div class="timeline-item timeline-row" data-type="' + t + '" data-icon="' + icon.icon + '" data-icon-bg="' + icon.bg + '" data-icon-color="' + icon.color + '">';
          row += '<span class="dot"></span>';
          row += '<p class="timeline-row-text">' + (a.description || '—') + ' <span class="timeline-row-meta">-  ' + (a.full_name || '—') + ' -  ' + (a.date ? moment(a.date).format('MMM DD, YYYY HH:mm') : '—') + '</span></p>';
          row += '</div>';
          return row;
        }
        function renderCard() {
          var card = '';
          card += '<div class="timeline-item timeline-card" data-type="' + t + '" data-icon="' + icon.icon + '" data-icon-bg="' + icon.bg + '" data-icon-color="' + icon.color + '">';
          card += '<span class="dot"></span>';
          card += '<div class="timeline-content">';
          card += '<div class="timeline-header">';
          card += '<div class="timeline-info">';
          card += '<h6 class="timeline-title">' + (a.title || a.description || 'No description') + '</h6>';
          card += '<p class="timeline-meta"><i class="fa fa-user"></i> ' + (a.full_name || 'Unknown user') + ' <span class="meta-sep">&bull;</span> <i class="fa fa-calendar"></i> ' + (a.date ? moment(a.date).format('MMM DD, YYYY HH:mm') : 'Unknown date') + '</p>';
          card += '</div>';
          card += '<span class="activity-badge ' + badge + '">' + (t || 'activity') + '</span>';
          card += '</div>';
          if (a.additional_data) {
            try {
              var addData = typeof a.additional_data === 'string' ? JSON.parse(a.additional_data) : a.additional_data;
              var desc = '';
              if (addData.outcome !== undefined) {
                desc += '<div><strong>Outcome:</strong> ' + (addData.outcome ? addData.outcome : '-') + '</div>';
              }
              if (addData.date !== undefined) {
                // Format date to DD-MM-YYYY (Indian format)
                var dateVal = addData.date;
                var formattedDate = '-';
                if (dateVal) {
                  var d = new Date(dateVal);
                  if (!isNaN(d.getTime())) {
                    var day = ('0' + d.getDate()).slice(-2);
                    var month = ('0' + (d.getMonth() + 1)).slice(-2);
                    var year = d.getFullYear();
                    formattedDate = day + '-' + month + '-' + year;
                  } else if (typeof dateVal === 'string' && dateVal.length >= 10) {
                    // fallback: try to parse as YYYY-MM-DD or YYYYMMDD
                    var m = dateVal.match(/^(\d{4})-?(\d{2})-?(\d{2})/);
                    if (m) {
                      formattedDate = m[3] + '-' + m[2] + '-' + m[1];
                    } else {
                      formattedDate = dateVal;
                    }
                  }
                }
                desc += '<div><strong>Date:</strong> ' + formattedDate + '</div>';
              }
              if (addData.notes !== undefined) {
                desc += '<div><strong>Notes:</strong> ' + (addData.notes ? addData.notes : '-') + '</div>';
              }
              card += '<div class="timeline-description">' + desc + '</div>';
            } catch (e) {
              card += '<div class="timeline-description">' + a.additional_data + '</div>';
            }
          }
          if (t === 'activity' || t === 'activities') {
            // card += '<div class="tw-mt-2"><label class="tw-text-xs"><input type="checkbox" ' + ((a.status === 'completed') ? 'checked' : '') + ' onchange="toggleActivityStatus(' + a.id + ', this.checked)"> Completed</label></div>';
          }
          card += '<div class="timeline-actions">';
          card += '<button class="btn btn-pin" onclick="pinActivity(' + a.id + ')" title="Pin to Top"><i class="fa fa-thumb-tack"></i></button>';
          card += '<button class="btn btn-edit" onclick="editActivity(' + a.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          card += '<button class="btn btn-delete" onclick="deleteActivity(' + a.id + ')" title="Delete"><i class="fa fa-trash"></i></button>';
          card += '</div>';
          card += '</div>';
          card += '</div>';
          return card;
        }

        var block = isRowType(t) ? renderRow() : renderCard();
        htmlAll += block;

        switch (t) {
          case 'emails': countEmails++; htmlEmails += block; break;
          case 'documents': countDocuments++; htmlDocuments += block; break;
          case 'calls': case 'call': countCalls++; htmlCalls += block; break;
          case 'notes': case 'note': countNotes++; htmlNotes += block; break;
          default: countActivities++; htmlActivitiesOnly += block; break;
        }
      });

      $('#activities-list').html(htmlAll);
      $('#activities-only-list').html(htmlActivitiesOnly || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_activity_found"); ?></p></div>');
      $('#emails-list').html(htmlEmails || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_emails_found"); ?></p></div>');
      $('#documents-list').html(htmlDocuments || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_documents_found"); ?></p></div>');
      $('#calls-list').html(htmlCalls || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_calls_found"); ?></p></div>');
      $('#notes-list').html(htmlNotes || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_notes_found"); ?></p></div>');

      $('#badge-activities').text(countActivities);
      $('#badge-emails').text(countEmails);
      $('#badge-documents').text(countDocuments);
      $('#badge-calls').text(countCalls);
      $('#badge-notes').text(countNotes);

      placeTimelineDots();
    }).fail(function () {
      var errorHtml = '<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading activities</p></div>';
      $('#activities-list,#activities-only-list,#emails-list,#documents-list,#calls-list,#notes-list').html(errorHtml);
    });
  }


  function getActivityIconData(type) {
    switch (type) {
      case 'calls':
      case 'call':
        return { icon: 'fa-phone', bg: '#ff69b4', color: '#fff' }; // Pink as in screenshot for calls
      case 'emails':
        return { icon: 'fa-envelope', bg: '#6366f1', color: '#fff' };
      case 'notes':
      case 'note':
        return { icon: 'fa-sticky-note', bg: '#ff69b4', color: '#fff' }; // Pink for notes
      case 'documents':
        return { icon: 'fa-file-text', bg: '#fbbf24', color: '#fff' };
      case 'activity':
        return { icon: 'fa-check', bg: '#84c529', color: '#fff' }; // Green check for activities
      case 'stage':
        return { icon: 'fa-arrow-right', bg: '#add8e6', color: '#000' }; // Light blue arrow for stage changes
      case 'status':
        return { icon: 'fa-trophy', bg: '#add8e6', color: '#000' }; // Light blue for won/lost
      case 'contact':
      case 'company':
      case 'attachment':
        return { icon: 'fa-link', bg: '#add8e6', color: '#000' }; // Light blue for associations
      default:
        return { icon: 'fa-info-circle', bg: '#add8e6', color: '#fff' }; // Default light blue
    }
  }

  function loadContacts() {
    console.log('Loading contacts for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_contacts/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse contacts response:', response.error, response.raw);
        $('#contacts-list').html('<div class="error-state"><i class="fa fa-exclamation-triangle"></i><h4>Error parsing contacts response</h4></div>');
        return;
      }
      console.log('Contacts response:', response.success, response.contacts, (response.success && response.contacts && response.contacts.length > 0));
      if (response.success && response.contacts && response.contacts.length > 0) {
        var html = '';
        response.contacts.forEach(function (contact) {
          html += '<div class="contact-item">';
          html += '<div class="contact-header">';
          html += '<div class="contact-avatar">';
          html += '<i class="fa fa-user"></i>';
          html += '</div>';
          html += '<div class="contact-info">';
          html += '<h6 class="contact-name">' + (contact.firstname || '') + ' ' + (contact.lastname || '') + '</h6>';
          if (contact.email) {
            html += '<div class="contact-detail"><i class="fa fa-envelope"></i> ' + contact.email + '</div>';
          }
          if (contact.phonenumber) {
            html += '<div class="contact-detail"><i class="fa fa-phone"></i> ' + contact.phonenumber + '</div>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="contact-actions">';
          html += '<button class="btn btn-sm btn-outline-primary" onclick="editContact(' + contact.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeContact(' + contact.id + ')" title="Remove"><i class="fa fa-times"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#contacts-list').html(html);
        console.log('Contacts HTML updated:', html);
      } else {
        console.log('No contacts found or error in response');
        $('#contacts-list').html('<div class="empty-state"><i class="fa fa-users"></i><h4><?php echo _l("no_contacts_found"); ?></h4><p><?php echo _l("add_contacts_to_get_started"); ?></p></div>');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading contacts:', error);
      console.error('Response:', xhr.responseText);
      $('#contacts-list').html('<div class="error-state"><i class="fa fa-exclamation-triangle"></i><h4>Error loading contacts</h4></div>');
    });
  }

  function loadCompanies() {
    console.log('Loading companies for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_companies/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse companies response:', response.error, response.raw);
        $('#companies-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing companies response</p></div>');
        $('#companies-count').text('(0)');
        return;
      }
      console.log('Companies response:', response);
      if (response.success && response.companies && response.companies.length > 0) {
        var html = '';
        response.companies.forEach(function (company) {
          html += '<div class="company-item">';
          html += '<div class="company-header">';
          html += '<div class="company-avatar"><i class="fa fa-building"></i></div>';
          html += '<div class="company-info">';
          html += '<h6 class="company-name">' + (company.name || '') + '</h6>';
          if (company.email) {
            html += '<div class="company-detail"><i class="fa fa-envelope"></i> ' + company.email + '</div>';
          }
          if (company.phone) {
            html += '<div class="company-detail"><i class="fa fa-phone"></i> ' + company.phone + '</div>';
          }
          if (company.website) {
            html += '<div class="company-detail"><i class="fa fa-globe"></i> <a href="' + company.website + '" target="_blank">' + company.website + '</a></div>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="company-actions">';
          html += '<button class="btn btn-sm btn-outline-primary" onclick="editCompany(' + company.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeCompany(' + company.id + ')" title="Remove"><i class="fa fa-times"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#companies-list').html(html);
        $('#companies-count').text('(' + response.companies.length + ')');
        console.log('Companies HTML updated:', html);
      } else {
        console.log('No companies found or error in response');
        $('#companies-list').html('<div class="text-muted text-center tw-py-4"><i class="fa fa-building fa-2x tw-mb-2"></i><p><?php echo _l("no_company_associated"); ?></p></div>');
        $('#companies-count').text('(0)');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading companies:', error);
      console.error('Response:', xhr.responseText);
      $('#companies-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading companies</p></div>');
      $('#companies-count').text('(0)');
    });
  }

  function isImage(filename) {
    return /\.(gif|jpe?g|tiff?|png|webp|bmp)$/i.test(filename);
  }

  function previewAttachment(id, filename) {
    var url = admin_url + 'deals/download_attachment/' + id;
    var content = '';
    if (isImage(filename)) {
      content = '<img src="' + url + '" alt="' + filename + '" style="max-width:100%; height:auto;">';
    } else if (filename.toLowerCase().endsWith('.pdf')) {
      content = '<iframe src="' + url + '" width="100%" height="600px" frameborder="0"></iframe>';
    } else {
      content = '<p>Preview not available for this file type. <a href="' + url + '">Download</a> to view.</p>';
    }
    $('#preview-content').html(content);
    $('#previewModal').modal('show');
  }

  function loadAttachments() {
    console.log('Loading attachments for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_attachments/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse attachments response:', response.error, response.raw);
        $('#attachments-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing attachments response</p></div>');
        return;
      }
      console.log('Attachments response:', response);
      if (response.success && response.attachments && response.attachments.length > 0) {
        var html = '';
        response.attachments.forEach(function (attachment) {
          html += '<div class="attachment-item tw-flex tw-items-center tw-justify-between tw-mb-3">';
          html += '<div class="tw-flex tw-items-center">';
          var url = admin_url + 'deals/download_attachment/' + attachment.id;
          if (isImage(attachment.file_name)) {
            html += '<img src="' + url + '" style="max-width:300px; max-height:300px; width:auto; height:auto; object-fit:contain;" class="tw-rounded tw-mr-3" alt="' + attachment.file_name + '">';
          } else {
            html += '<i class="fa fa-file tw-text-4xl tw-mr-3 tw-text-gray-400"></i>';
          }
          html += '<div>';
          html += '<a href="#" onclick="previewAttachment(' + attachment.id + ', \'' + attachment.file_name + '\')" class="tw-font-medium">' + (attachment.file_name || 'Unknown file') + '</a>';
          if (attachment.description) {
            html += '<p class="tw-text-sm tw-text-gray-600">' + attachment.description + '</p>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="tw-flex tw-gap-2">';
          html += '<a href="' + url + '" class="btn btn-sm btn-outline-primary"><i class="fa fa-download"></i></a>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeAttachment(' + attachment.id + ')"><i class="fa fa-trash"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#attachments-list').html(html);
        console.log('Attachments HTML updated:', html);
      } else {
        console.log('No attachments found or error in response');
        $('#attachments-list').html('<div class="text-muted text-center tw-py-4"><i class="fa fa-paperclip fa-2x tw-mb-2"></i><p><?php echo _l("no_attachments_uploaded"); ?></p></div>');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading attachments:', error);
      console.error('Response:', xhr.responseText);
      $('#attachments-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading attachments</p></div>');
    });
  }

  // --- TIMELINE ICONS ON LINE, NOT IN CARD ---


  function loadAvailableContacts() {
    $.get(admin_url + 'deals/get_available_contacts/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse available contacts response:', response.error, response.raw);
        $('#existing-contact-select').html('<option value=""><?php echo _l("select_contact"); ?></option>');
        return;
      }
      console.log('Contacts response:', response);
      if (response.success) {
        var html = '<option value=""><?php echo _l("select_contact"); ?></option>';
        if (response.contacts && response.contacts.length > 0) {
          response.contacts.forEach(function (contact) {
            html += '<option value="' + contact.id + '">' + contact.firstname + ' ' + contact.lastname + ' (' + contact.email + ')</option>';
          });
        }
        $('#existing-contact-select').html(html);
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading contacts:', error);
    });
  }

  function loadAvailableCompanies() {
    $.get(admin_url + 'deals/get_available_companies/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse available companies response:', response.error, response.raw);
        $('#existing-company-select').html('<option value=""><?php echo _l("select_company"); ?></option>');
        return;
      }
      console.log('Companies response:', response);
      if (response.success) {
        var html = '<option value=""><?php echo _l("select_company"); ?></option>';
        if (response.companies && response.companies.length > 0) {
          response.companies.forEach(function (company) {
            html += '<option value="' + company.id + '">' + company.name + '</option>';
          });
        }
        $('#existing-company-select').html(html);
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading companies:', error);
    });
  }
  function toggleAddActivityForm() {
    $('#add-activity-form').toggle();
  }

  function toggleLogCallForm() {
    $('#log-call-form').toggle();
  }

  function toggleAddNoteForm() {
    $('#add-note-form').toggle();
  }
  function addComment(activityId) {
    var desc = prompt('Add comment');
    if (desc) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('description', desc);
      $.ajax({
        url: admin_url + 'deals/add_comment/' + activityId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.success) {
            loadActivities();
          }
        }
      });
    }
  }

  function addContact() {
    $('#contactModalTitle').text('<?php echo _l("add_contact"); ?>');
    // Clear form fields
    $('#contact-firstname').val('');
    $('#contact-lastname').val('');
    $('#contact-email').val('');
    $('#contact-phonenumber').val('');
    $('#existing-contact-select').val('');
    $('#contactModal').modal('show');
  }

  function addCompany() {
    $('#companyModalTitle').text('<?php echo _l("add_company"); ?>');
    // Clear form fields
    $('#company-name').val('');
    $('#company-email').val('');
    $('#company-phone').val('');
    $('#company-website').val('');
    $('#company-address').val('');
    $('#existing-company-select').val('');
    $('#companyModal').modal('show');
  }

  function addAttachment() {
    // Clear form fields
    $('#attachment-file').val('');
    $('#attachment-description').val('');
    $('#attachmentModal').modal('show');
  }

  function showLostReasonModal() {
    $('#lost-reason-select').val('');
    $('#lostReasonModal').modal('show');
  }

  function editContact(id) {
    alert('Edit contact functionality will be implemented for ID: ' + id);
  }

  function editCompany(id) {
    alert('Edit company functionality will be implemented for ID: ' + id);
  }

  function saveContact() {
    var contactId = $('#existing-contact-select').val();
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);

    if (contactId) {
      formData.append('contact_id', contactId);
    } else {
      formData.append('firstname', $('#contact-firstname').val());
      formData.append('lastname', $('#contact-lastname').val());
      formData.append('email', $('#contact-email').val());
      formData.append('phonenumber', $('#contact-phonenumber').val());
    }

    $.ajax({
      url: admin_url + 'deals/add_contact_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save contact response: ' + response.error);
          return;
        }
        console.log('Save contact response:', response);
        if (response.success) {
          $('#contactModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving contact:', error);
        alert('Error saving contact: ' + error);
      }
    });
  }

  function saveCompany() {
    var companyId = $('#existing-company-select').val();
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);

    if (companyId) {
      formData.append('company_id', companyId);
    } else {
      formData.append('company_name', $('#company-name').val());
      formData.append('company_email', $('#company-email').val());
      formData.append('company_phone', $('#company-phone').val());
      formData.append('company_website', $('#company-website').val());
      formData.append('company_address', $('#company-address').val());
    }

    $.ajax({
      url: admin_url + 'deals/add_company_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save company response: ' + response.error);
          return;
        }
        console.log('Save company response:', response);
        if (response.success) {
          $('#companyModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving company:', error);
        alert('Error saving company: ' + error);
      }
    });
  }

  function saveAttachment() {
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);
    formData.append('attachment', $('#attachment-file')[0].files[0]);
    formData.append('description', $('#attachment-description').val());

    $.ajax({
      url: admin_url + 'deals/add_attachment_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save attachment response: ' + response.error);
          return;
        }
        console.log('Save attachment response:', response);
        if (response.success) {
          $('#attachmentModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving attachment:', error);
        alert('Error saving attachment: ' + error);
      }
    });
  }

  function removeContact(contactId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_contact_from_deal/' + dealId + '/' + contactId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove contact response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function removeCompany(companyId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_company_from_deal/' + dealId + '/' + companyId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove company response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function removeAttachment(attachmentId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_attachment_from_deal/' + dealId + '/' + attachmentId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove attachment response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function setDealStage(dealId, stageId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('stage_id', stageId);

      $.ajax({
        url: admin_url + 'deals/set_stage/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing set stage response: ' + response.error);
            return;
          }
          window.location.reload();
        }
      });
    }
  }

  function markDealWon(dealId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/mark_won/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing mark won response: ' + response.error);
            return;
          }
          window.location.reload();
        }
      });
    }
  }

  function markDealLost(dealId) {
    var lostReasonId = $('#lost-reason-select').val();
    if (lostReasonId === '') {
      alert('<?php echo _l("please_select_lost_reason"); ?>');
      return;
    }
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('lost_reason_id', lostReasonId);

      $.ajax({
        url: admin_url + 'deals/mark_lost/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing mark lost response: ' + response.error);
            return;
          }
          $('#lostReasonModal').modal('hide');
          window.location.reload();
        }
      });
    }
  }



  function pinActivity(activityId) {
    // Implementation for pinning activity
    alert('Pin activity functionality will be implemented');
  }

  // Activity Management Functions
  function addActivity() {
    $('#activity-description').val('');
    $('#addActivityModal').modal('show');
  }

  function logCall() {
    $('#call-type').val('outbound');
    $('#call-status').val('completed');
    $('#call-duration').val('');
    $('#call-notes').val('');
    $('#call-date').val(new Date().toISOString().slice(0, 16));
    $('#logCallModal').modal('show');
  }

  function addNote() {
    $('#note-description').val('');
    $('#addNoteModal').modal('show');
  }

  function editActivity(activityId) {
    // Edit activity functionality
    console.log('Editing activity:', activityId);
    // Add your edit logic here
  }

  function deleteActivity(activityId) {
    // Delete activity functionality
    if (confirm('Are you sure you want to delete this activity?')) {
      console.log('Deleting activity:', activityId);
      // Add your delete logic here
    }
  }

  function filterActivities() {
    var filterValue = $('#activity-filter').val();
    console.log('Filtering activities by:', filterValue);

    if (filterValue === 'all') {
      $('.timeline-item').show();
    } else {
      $('.timeline-item').hide();
      $('.timeline-item[data-type="' + filterValue + '"]').show();
    }
  }

  function refreshAllData() {
    loadContacts();
    loadCompanies();
    loadAttachments();
    loadActivities();
    loadAvailableContacts();
    loadAvailableCompanies();
  }

  function saveActivity() {
    var description = $('#activity-description').val().trim();
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);
    $.ajax({
      url: admin_url + 'deals/add_activity/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#addActivityModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding activity: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding activity');
      }
    });
  }

  function saveCall() {
    var payload = {
      call_type: $('#call-type').val(),
      call_status: $('#call-status').val(),
      call_duration: $('#call-duration').val(),
      call_notes: $('#call-notes').val(),
      call_date: $('#call-date').val()
    };
    if (!payload.call_type || !payload.call_status) {
      alert('Call type and status are required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    for (var key in payload) {
      formData.append(key, payload[key] || '');
    }
    $.ajax({
      url: admin_url + 'deals/add_call/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#logCallModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding call: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding call');
      }
    });
  }

  function saveNote() {
    var description = $('#note-description').val().trim();
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);
    $.ajax({
      url: admin_url + 'deals/add_note/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#addNoteModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding note: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding note');
      }
    });
  }

  function placeTimelineDots() {
    $('.timeline-item').each(function () {
      var $it = $(this);
      var icon = $it.data('icon') || 'fa-clock-o';
      var bg = $it.data('icon-bg') || '#6b7280';
      var color = $it.data('icon-color') || '#fff';
      var $dot = $it.find('.dot');
      $dot.html('<i class="fa ' + icon + '"></i>').css({ background: bg, color: color });
    });
  }

  function filterActivities() {
    var v = $('#activity-filter').val();
    if (v === 'all') $('.timeline-item').show();
    else { $('.timeline-item').hide(); $('.timeline-item[data-type="' + v + '"]').show(); }
    placeTimelineDots();
  }
  function toggleAddActivityForm() { $('#add-activity-form').toggle(); }
  function toggleLogCallForm() { $('#log-call-form').toggle(); }
  function toggleAddNoteForm() { $('#add-note-form').toggle(); }

  function editActivity(id) { /* open modal, load item, submit to edit endpoints */ }
  function deleteActivity(id) { if (confirm('Delete this item?')) { /* call delete endpoint then reload */ } }
  function pinActivity(id) { /* optional */ }
  // Initialize activity filter
  $(document).ready(function () {
    $('#activity-filter').on('change', filterActivities);
  });

  // Remove duplicate/alternate saveActivity, saveCall, saveNote functions
  // Only keep the above versions that match your backend payload

  function placeTimelineDots() {
    $('.timeline-item').each(function () {
      var $it = $(this);
      var icon = $it.data('icon') || 'fa-clock-o';
      var bg = $it.data('icon-bg') || '#6b7280';
      var color = $it.data('icon-color') || '#fff';
      var $dot = $it.find('.dot');
      $dot.html('<i class="fa ' + icon + '"></i>').css({ background: bg, color: color });
    });
  }

  function filterActivities() {
    var v = $('#activity-filter').val();
    if (v === 'all') $('.timeline-item').show();
    else { $('.timeline-item').hide(); $('.timeline-item[data-type="' + v + '"]').show(); }
    placeTimelineDots();
  }

  // Remove duplicate toggleAddActivityForm, toggleLogCallForm, toggleAddNoteForm, etc.
  // Only keep one version of each function

  // Initialize activity filter
  $(document).ready(function () {
    $('#activity-filter').on('change', filterActivities);
  });











  $(function () {
    // Show edit icon on hover for amount
    $('.deal-value-group').hover(
      function () { $(this).find('.edit-amount-btn').show(); },
      function () { if (!$('#edit-amount-form').is(':visible')) $(this).find('.edit-amount-btn').hide(); }
    );
    // Show edit icon on hover for close date
    $('.close-date-group').hover(
      function () { $(this).find('.edit-close-date-btn').show(); },
      function () { if (!$('#edit-close-date-form').is(':visible')) $(this).find('.edit-close-date-btn').hide(); }
    );

    // Amount form submit
    $('#edit-amount-form').on('submit', function (e) {
      e.preventDefault();
      saveDealAmount();
    });

    // Close date form submit
    $('#edit-close-date-form').on('submit', function (e) {
      e.preventDefault();
      saveCloseDate();
    });
  });

  function editDealAmount() {
    $('.deal-value-display').hide();
    $('#edit-amount-form').show();
    $('.edit-amount-btn').hide();
    $('#edit-amount-input').focus();
  }
  function cancelEditAmount() {
    $('#edit-amount-form').hide();
    $('.deal-value-display').show();
    $('.edit-amount-btn').hide();
  }
  function saveDealAmount() {
    var amount = parseFloat($('#edit-amount-input').val());
    if (isNaN(amount) || amount < 0) {
      alert('Please enter a valid amount');
      return;
    }
    var formData = new FormData();
    formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', '<?php echo $this->security->get_csrf_hash(); ?>');
    formData.append('deal_value', amount);
    $.ajax({
      url: admin_url + 'deals/update_amount/<?php echo (int) $deal['id']; ?>',
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        try { response = typeof response === 'string' ? JSON.parse(response) : response; } catch (e) { }
        if (response && response.success) {
          location.reload();
        } else {
          alert('Failed to update amount');
        }
      },
      error: function () { alert('Failed to update amount'); }
    });
  }

  function editCloseDate() {
    $('.close-date-display').hide();
    $('#edit-close-date-form').show();
    $('.edit-close-date-btn').hide();
    $('#edit-close-date-input').focus();
  }
  function cancelEditCloseDate() {
    $('#edit-close-date-form').hide();
    $('.close-date-display').show();
    $('.edit-close-date-btn').hide();
  }
  function saveCloseDate() {
    var date = $('#edit-close-date-input').val();
    if (!date) {
      alert('Please select a date');
      return;
    }
    var formData = new FormData();
    formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', '<?php echo $this->security->get_csrf_hash(); ?>');
    formData.append('expected_close_date', date);
    $.ajax({
      url: admin_url + 'deals/update_close_date/<?php echo (int) $deal['id']; ?>',
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        try { response = typeof response === 'string' ? JSON.parse(response) : response; } catch (e) { }
        if (response && response.success) {
          location.reload();
        } else {
          alert('Failed to update close date');
        }
      },
      error: function () { alert('Failed to update close date'); }
    });
  }
</script>


<style>
  /* Contact, Company, and Attachment Items - Simpler */
  .contact-item,
  .company-item,
  .attachment-item {
    background: #fff;
    /* White for clean look */
    border-radius: 4px;
    padding: 12px;
    margin-bottom: 8px;
    border: 1px solid #e5e7eb;
    transition: all 0.3s ease;
    position: relative;
  }

  .contact-item:hover,
  .company-item:hover,
  .attachment-item:hover {
    border-color: #d1d5db;
    /* Subtle hover */
  }

  .contact-actions,
  .company-actions {
    position: absolute;
    right: 12px;
    top: 50%;
    transform: translateY(-50%);
    opacity: 0;
    transition: all 0.3s ease;
    display: flex;
    gap: 6px;
  }

  .contact-item:hover .contact-actions,
  .company-item:hover .company-actions {
    opacity: 1;
  }

  .contact-header,
  .company-header {
    display: flex;
    align-items: center;
    margin-bottom: 6px;
  }

  .contact-avatar,
  .company-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: #3b82f6;
    color: #fff;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 12px;
  }

  .contact-name,
  .company-name {
    font-weight: 500;
    color: #1f2937;
  }

  .contact-detail,
  .company-detail {
    color: #6b7280;
    font-size: 13px;
    margin-bottom: 3px;
    padding-left: 44px;
  }



  /* Timeline spine */
  .timeline-container {
    position: relative;
    padding: 10px 0 10px 36px;
  }

  .timeline-container::before {
    content: '';
    position: absolute;
    left: 18px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: #e5e7eb;
    border-radius: 1px;
  }

  .timeline-item {
    position: relative;
    margin-bottom: 14px;
    min-height: 28px;
  }

  .timeline-item .dot {
    position: absolute;
    left: 0;
    top: 2px;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    box-shadow: 0 2px 8px rgba(0, 0, 0, .08);
    z-index: 2;
    font-size: 18px;
  }

  /* Single-line rows */
  .timeline-item.timeline-row .timeline-row-text {
    margin: 6px 0 0 50px;
    font-size: 13px;
    color: #334155;
  }

  .timeline-item.timeline-row .timeline-row-meta {
    color: #94a3b8;
    margin-left: 6px;
    font-size: 12px;
  }

  /* Cards */
  .timeline-item.timeline-card .timeline-content {
    margin-left: 50px;
    background: #fff;
    border: 1px solid #e5e7eb;
    border-radius: 6px;
    padding: 12px 14px;
  }

  .timeline-header {
    display: flex;
    align-items: center;
    margin-bottom: 6px;
  }

  .timeline-info {
    flex: 1;
  }

  .timeline-title {
    font-weight: 600;
    color: #1f2937;
    margin: 0;
    font-size: 14px;
  }

  .timeline-meta {
    color: #6b7280;
    font-size: 12px;
    margin: 0;
  }

  .meta-sep {
    margin: 0 6px;
    color: #cbd5e1;
  }

  .timeline-description {
    color: #475569;
    line-height: 1.5;
    margin: 8px 0 2px;
    font-size: 13px;
  }

  .timeline-actions {
    position: absolute;
    right: 8px;
    top: 6px;
    opacity: 0;
    transform: translateX(8px);
    transition: all .2s ease;
    display: flex;
    gap: 6px;
  }

  .timeline-item:hover .timeline-actions {
    opacity: 1;
    transform: translateX(0);
  }

  .timeline-actions .btn {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    font-size: 10px;
  }

  .timeline-actions .btn-edit {
    background: #3b82f6;
    color: #fff;
  }

  .timeline-actions .btn-delete {
    background: #ef4444;
    color: #fff;
  }

  .timeline-actions .btn-pin {
    background: #8b5cf6;
    color: #fff;
  }

  /* Badges */
  .activity-badge {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
    margin-left: 8px;
    background: #f3f4f6;
    color: #6b7280;
  }

  .activity-badge.call {
    background: #d1fae5;
    color: #10b981;
  }

  .activity-badge.email {
    background: #e0e7ff;
    color: #6366f1;
  }

  .activity-badge.note {
    background: #fef3c7;
    color: #f59e42;
  }

  .activity-badge.document {
    background: #fef9c3;
    color: #fbbf24;
  }

  .activity-badge.stage {
    background: #dbeafe;
    color: #3b82f6;
  }

  .activity-badge.status {
    background: #e9fbe5;
    color: #84c529;
  }

  .activity-badge.activity {
    background: #f3f4f6;
    color: #6b7280;
  }

  /* Cards in list-only tabs */
  #activities-only-list,
  #calls-list,
  #notes-list {
    padding: 0;
  }

  #activities-only-list .timeline-item.timeline-card .timeline-content,
  #calls-list .timeline-item.timeline-card .timeline-content,
  #notes-list .timeline-item.timeline-card .timeline-content {
    border-left: none;
    background: #fff;
  }

  /* General theme */
  .btn-primary {
    background: #6366f1;
    border-color: #6366f1;
  }

  .nav-tabs {
    background: #eff6ff;
    border: none;
    border-radius: 4px;
    padding: 6px;
  }

  .nav-tabs>li>a {
    border: none;
    border-radius: 4px;
    padding: 10px 16px;
    color: #333;
    font-weight: 500;
  }

  .nav-tabs>li.active>a {
    border-bottom: 2px solid #6366f1;
    background: transparent;
    color: #6366f1;
  }

  .badge {
    background: #ff0000;
    color: #fff;
    border-radius: 10px;
    padding: 3px 6px;
    font-size: 10px;
  }

  .form-control {
    border-radius: 8px;
    border: 1px solid #ddd;
  }

  .modern-card {
    background: #fff;
    border-radius: 4px;
    padding: 16px;
    border: 1px solid #e5e7eb;
    margin-bottom: 16px;
  }

  .modern-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 12px;
    padding-bottom: 12px;
    border-bottom: 1px solid #f3f4f6;
  }

  @media (max-width: 768px) {
    .timeline-container {
      padding-left: 34px;
    }

    .timeline-container::before {
      left: 16px;
    }

    .timeline-item .dot {
      width: 30px;
      height: 30px;
      font-size: 16px;
    }

    .timeline-item.timeline-card .timeline-content,
    .timeline-item.timeline-row .timeline-row-text {
      margin-left: 44px;
    }

    .timeline-actions {
      position: static;
      opacity: 1;
      transform: none;
      margin-top: 6px;
      justify-content: flex-end;
    }
  }
</style>















<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Install
{
    private $CI;

    public function __construct()
    {
        $this->CI = &get_instance();
    }

    public function run()
    {
        $this->CI->load->database();

        try {

            // Create deals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals` (
                 `id` INT NOT NULL AUTO_INCREMENT,
                    `hash` VARCHAR(65) NULL,
                    `name` VARCHAR(255) NULL,
                    `title` VARCHAR(100) NULL,
                    `company` VARCHAR(255) NULL,
                    `description` TEXT NULL,
                    `country` INT NULL,
                    `zip` VARCHAR(50) NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `address` TEXT NULL,
                    `assigned` INT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `from_form_id` INT NULL,
                    `status` INT NULL,
                    `source` INT NULL,
                    `lastcontact` DATETIME NULL,
                    `dateassigned` DATE NULL,
                    `last_status_change` DATETIME NULL,
                    `addedfrom` INT NULL,
                    `email` VARCHAR(100) NULL,
                    `website` VARCHAR(255) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `date_converted` DATETIME NULL,
                    `lost` TINYINT(1) DEFAULT 0,
                    `junk` TINYINT(1) DEFAULT 0,
                    `last_deal_status` INT NULL,
                    `is_imported_from_email_integration` TINYINT NULL,
                    `email_integration_uid` VARCHAR(30) NULL,
                    `is_public` VARCHAR(255) NULL,
                    `default_language` VARCHAR(50) NULL,
                    `client_id` INT NULL,
                    `deal_value` DECIMAL(15,2) DEFAULT 0.00,
                    `expected_close_date` DATE NULL,
                    `pipeline_id` INT NULL,
                    `stage_id` INT NULL,
                    `status_final` VARCHAR(10) NOT NULL DEFAULT "open",
                    `lost_reason_id` INT NULL,
                    `probability` TINYINT(3) DEFAULT 100,
                    `currency` INT NULL,
                    `tags` TEXT NULL,
                    `is_archived` TINYINT(1) DEFAULT 0,
                    `archived_date` DATETIME NULL,
                    `archived_by` INT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_pipeline_id` (`pipeline_id`),
                    KEY `idx_stage_id` (`stage_id`),
                    KEY `idx_status_final` (`status_final`),
                    KEY `idx_assigned` (`assigned`),
                    KEY `idx_dateadded` (`dateadded`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            if (!$this->CI->db->table_exists(db_prefix() . 'tags_deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'tags_deals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            //create web to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'web_to_deal')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'web_to_deal` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `name` VARCHAR(255) NOT NULL,
                `form_key` VARCHAR(32) NOT NULL UNIQUE,
                `form_data` TEXT,
                `recaptcha` TINYINT(1) DEFAULT 0,
                `language` VARCHAR(50),
                `lead_name_prefix` VARCHAR(50),
                `deal_source` INT,
                `deal_status` INT,
                `responsible` INT DEFAULT 0,
                `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                `submit_action` TINYINT(1) DEFAULT 0,
                `success_submit_msg` TEXT,
                `submit_redirect_url` VARCHAR(255),
                `mark_public` TINYINT(1) DEFAULT 0,
                `allow_duplicate` TINYINT(1) DEFAULT 1,
                `track_duplicate_field` VARCHAR(100),
                `track_duplicate_field_and` VARCHAR(100),
                `create_task_on_duplicate` TINYINT(1) DEFAULT 0,
                `notify_deal_imported` TINYINT(1) DEFAULT 1,
                `notify_type` ENUM("specific_staff", "roles", "assigned") DEFAULT NULL,
                `notify_ids` TEXT,
                `dateadded` DATETIME NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                $this->CI->db->query('CREATE INDEX `idx_form_key` ON `' . db_prefix() . 'web_to_deal` (`form_key`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_source` ON `' . db_prefix() . 'web_to_deal` (`deal_source`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_status` ON `' . db_prefix() . 'web_to_deal` (`deal_status`);');
                $this->CI->db->query('CREATE INDEX `idx_responsible` ON `' . db_prefix() . 'web_to_deal` (`responsible`);');
            }

            // Create deals_statuses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_statuses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_statuses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `statusorder` INT NOT NULL,
                    `isdefault` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages Open/Won/Lost
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'statusorder' => 1,
                    'isdefault' => 1,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Won',
                    'color' => '#84c529',
                    'statusorder' => 2,
                    'isdefault' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'statusorder' => 3,
                    'isdefault' => 0,
                ]);
            }

            // Pipelines table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_pipelines')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_pipelines` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(150) NOT NULL,
                    `order` INT DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
                // Seed default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_pipelines', ['name' => 'Sales Pipeline', 'order' => 1]);
                $defaultPipelineId = $this->CI->db->insert_id();

                // Stages table linked to pipeline
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_stages` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `pipeline_id` INT NOT NULL,
                    `name` VARCHAR(150) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `position` INT DEFAULT 0,
                    `win_probability` TINYINT(3) DEFAULT 100,
                    `is_won` TINYINT(1) DEFAULT 0,
                    `is_lost` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`),
                    KEY `pipeline_id` (`pipeline_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages into default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'position' => 1,
                    'win_probability' => 100,
                    'is_won' => 0,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Won',
                    'color' => '#84c529',
                    'position' => 2,
                    'win_probability' => 100,
                    'is_won' => 1,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'position' => 3,
                    'win_probability' => 0,
                    'is_won' => 0,
                    'is_lost' => 1,
                ]);
            }

            // Create deals attachments table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_attachments')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_attachments` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `file_name` VARCHAR(255) NOT NULL,
                    `filetype` VARCHAR(100) NULL,
                    `description` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `external` VARCHAR(100) NULL,
                    `external_link` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add win_probability to deals_stages if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals_stages')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_stages');
                if (!in_array('win_probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_stages` ADD `win_probability` TINYINT(3) DEFAULT 100 AFTER `position`');
                }
            }

            // Lost reasons table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_lost_reasons')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_lost_reasons` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(191) NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add pipeline_id and stage_id to deals if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `client_id`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
            }

            // Create deal_sources table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_sources')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_sources` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }



            // Create deal_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `date_contacted` DATETIME NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_activity table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_activity` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` VARCHAR(255) NOT NULL,
                    `additional_data` TEXT NULL,
                    `staffid` INT NULL,
                    `full_name` VARCHAR(100) NULL,
                    `date` DATETIME NOT NULL,
                    `type` VARCHAR(30) DEFAULT "activity",
                    `custom_activity` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Ensure "type" column exists
            if ($this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');
                if (!in_array('type', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `type` VARCHAR(30) DEFAULT "activity" AFTER `date`');
                }
            }

            // Create deal_email_integration table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_email_integration')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_email_integration` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `email` VARCHAR(100) NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    `created_at` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific contacts master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `firstname` VARCHAR(100) NOT NULL,
                    `lastname` VARCHAR(100) NULL,
                    `email` VARCHAR(191) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `email` (`email`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module contacts
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_contact_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_contact_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact_unique` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific companies master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(191) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unique_email` (`email`),
                    UNIQUE KEY `unique_phone` (`phone`),
                    KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module companies
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_company_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_company_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company_unique` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create companies table for storing company information
            if (!$this->CI->db->table_exists(db_prefix() . 'companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(100) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `zip` VARCHAR(50) NULL,
                    `country` INT NULL,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `name` (`name`),
                    KEY `active` (`active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_attachments_relations table for linking attachments to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_attachments_relations')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_attachments_relations` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `attachment_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_attachment_unique` (`deal_id`, `attachment_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `attachment_id` (`attachment_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_forms table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_forms')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_forms` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `form_key` VARCHAR(100) NOT NULL,
                    `lead_source` INT NULL,
                    `lead_status` INT NULL,
                    `responsible` INT NULL,
                    `language` VARCHAR(50) NULL,
                    `recaptcha` TINYINT(1) DEFAULT 0,
                    `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                    `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                    `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                    `success_submit_msg` TEXT NULL,
                    `submit_action` TINYINT(1) DEFAULT 0,
                    `submit_redirect_url` VARCHAR(255) NULL,
                    `mark_public` TINYINT(1) DEFAULT 0,
                    `allow_duplicate` TINYINT(1) DEFAULT 1,
                    `track_duplicate_field` VARCHAR(100) NULL,
                    `track_duplicate_field_and` VARCHAR(100) NULL,
                    `create_task_on_duplicate` TINYINT(1) DEFAULT 1,
                    `notify_lead_imported` TINYINT(1) DEFAULT 1,
                    `notify_type` VARCHAR(50) NULL,
                    `notify_ids` TEXT NULL,
                    `lead_name_prefix` VARCHAR(50) NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_saved_filters table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_saved_filters')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_saved_filters` (
                    `id` INT AUTO_INCREMENT PRIMARY KEY,
                    `filter_name` VARCHAR(255) NOT NULL,
                    `field` VARCHAR(100) NOT NULL,
                    `operator` VARCHAR(50) NOT NULL,
                    `value` VARCHAR(255) NOT NULL,
                    `share_with_team` TINYINT(1) DEFAULT 0,
                    `is_default` TINYINT(1) DEFAULT 0,
                    `created_by` INT NOT NULL,
                    `created_at` DATETIME NOT NULL,
                    UNIQUE KEY `filter_name` (`filter_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contacts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_companies table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_products table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_products')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_products` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `product_id` INT NOT NULL,
                    `qty` DECIMAL(15,2) NOT NULL DEFAULT 1.00,
                    `rate` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `discount` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `tax` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `total` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `product_id` (`product_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_tasks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_tasks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_tasks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `task_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_task` (`deal_id`, `task_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `task_id` (`task_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_fields table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_fields')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_fields` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `field_name` VARCHAR(100) NOT NULL,
                    `field_type` VARCHAR(50) NOT NULL,
                    `field_label` VARCHAR(255) NOT NULL,
                    `field_placeholder` VARCHAR(255) NULL,
                    `field_options` TEXT NULL,
                    `field_required` TINYINT(1) DEFAULT 0,
                    `field_order` INT DEFAULT 0,
                    `field_visible` TINYINT(1) DEFAULT 1,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_field_values table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_field_values')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_field_values` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `field_id` INT NOT NULL,
                    `field_value` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_field` (`deal_id`, `field_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `field_id` (`field_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_reminders table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_reminders')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_reminders` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `date` DATETIME NOT NULL,
                    `isnotified` TINYINT(1) DEFAULT 0,
                    `notify_by_email` TINYINT(1) DEFAULT 1,
                    `notify_by_sms` TINYINT(1) DEFAULT 0,
                    `creator` INT NOT NULL,
                    `datecreated` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `date` (`date`),
                    KEY `creator` (`creator`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_emails table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_emails')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_emails` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `subject` VARCHAR(255) NOT NULL,
                    `message` TEXT NOT NULL,
                    `from_email` VARCHAR(100) NOT NULL,
                    `from_name` VARCHAR(100) NOT NULL,
                    `to_email` VARCHAR(100) NOT NULL,
                    `to_name` VARCHAR(100) NOT NULL,
                    `cc` TEXT NULL,
                    `bcc` TEXT NULL,
                    `attachments` TEXT NULL,
                    `sent` TINYINT(1) DEFAULT 0,
                    `sent_date` DATETIME NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `sent` (`sent`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_calls table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_calls')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_calls` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `call_type` VARCHAR(50) NOT NULL,
                    `call_status` VARCHAR(50) NOT NULL,
                    `call_duration` INT NULL,
                    `call_recording` VARCHAR(255) NULL,
                    `call_notes` TEXT NULL,
                    `call_date` DATETIME NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `call_date` (`call_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_meetings table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_meetings')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_meetings` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `meeting_subject` VARCHAR(255) NOT NULL,
                    `meeting_description` TEXT NULL,
                    `meeting_date` DATETIME NOT NULL,
                    `meeting_duration` INT NULL,
                    `meeting_location` VARCHAR(255) NULL,
                    `meeting_type` VARCHAR(50) NULL,
                    `meeting_status` VARCHAR(50) NOT NULL,
                    `meeting_notes` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `meeting_date` (`meeting_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_documents table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_documents')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_documents` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `document_name` VARCHAR(255) NOT NULL,
                    `document_type` VARCHAR(100) NOT NULL,
                    `document_file` VARCHAR(255) NOT NULL,
                    `document_description` TEXT NULL,
                    `document_version` VARCHAR(20) NULL,
                    `document_status` VARCHAR(50) NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `document_status` (`document_status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_expenses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_expenses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_expenses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `expense_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_expense` (`deal_id`, `expense_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `expense_id` (`expense_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_invoices table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_invoices')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_invoices` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `invoice_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_invoice` (`deal_id`, `invoice_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `invoice_id` (`invoice_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_estimates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_estimates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_estimates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `estimate_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_estimate` (`deal_id`, `estimate_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `estimate_id` (`estimate_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_proposals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_proposals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_proposals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `proposal_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_proposal` (`deal_id`, `proposal_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `proposal_id` (`proposal_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contracts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contracts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contracts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contract_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contract` (`deal_id`, `contract_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contract_id` (`contract_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_credit_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_credit_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_credit_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `credit_note_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_credit_note` (`deal_id`, `credit_note_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `credit_note_id` (`credit_note_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_templates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_templates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_templates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `template_name` VARCHAR(255) NOT NULL,
                    `template_content` TEXT NOT NULL,
                    `template_type` VARCHAR(50) NOT NULL,
                    `template_category` VARCHAR(100) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `template_type` (`template_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_automation table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_automation')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_automation` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `automation_name` VARCHAR(255) NOT NULL,
                    `automation_type` VARCHAR(50) NOT NULL,
                    `trigger_condition` TEXT NOT NULL,
                    `action_type` VARCHAR(50) NOT NULL,
                    `action_data` TEXT NOT NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `automation_type` (`automation_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhooks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhooks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhooks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_name` VARCHAR(255) NOT NULL,
                    `webhook_url` VARCHAR(500) NOT NULL,
                    `webhook_events` TEXT NOT NULL,
                    `webhook_secret` VARCHAR(255) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhook_logs table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhook_logs')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhook_logs` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_id` INT NOT NULL,
                    `event_type` VARCHAR(100) NOT NULL,
                    `payload` TEXT NOT NULL,
                    `response_code` INT NULL,
                    `response_body` TEXT NULL,
                    `status` VARCHAR(50) NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `webhook_id` (`webhook_id`),
                    KEY `event_type` (`event_type`),
                    KEY `status` (`status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');

            if (!in_array('title', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `title` VARCHAR(255) NULL AFTER `id`');
            }
            if (!in_array('activity_date', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `activity_date` DATETIME NULL AFTER `title`');
            }
            if (!in_array('reminder', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `reminder` INT DEFAULT 0 AFTER `activity_date`');
            }
            if (!in_array('owner_id', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `owner_id` INT NULL AFTER `reminder`');
            }
            if (!in_array('guests', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `guests` TEXT NULL AFTER `owner_id`');
            }
            if (!in_array('status', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `status` VARCHAR(20) DEFAULT "pending" AFTER `guests`');
            }

            // For deals_calls, change call_status to outcome
            if (in_array('call_status', $fields = $this->CI->db->list_fields(db_prefix() . 'deals_calls'))) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` CHANGE `call_status` `outcome` VARCHAR(50) DEFAULT ""');
            }

            // Remove unnecessary columns if exist
            if (in_array('call_type', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_type`');
            }
            if (in_array('call_duration', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_duration`');
            }
            if (in_array('call_recording', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_recording`');
            }

            // Add missing fields to existing tables if they don't exist
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');

                if (!in_array('expected_close_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `expected_close_date` DATE NULL AFTER `deal_value`');
                }
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `expected_close_date`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
                if (!in_array('probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `probability` TINYINT(3) DEFAULT 100 AFTER `lost_reason_id`');
                }
                if (!in_array('currency', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `currency` INT NULL AFTER `probability`');
                }
                if (!in_array('tags', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `tags` TEXT NULL AFTER `currency`');
                }
                if (!in_array('is_archived', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `is_archived` TINYINT(1) DEFAULT 0 AFTER `tags`');
                }
                if (!in_array('archived_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_date` DATETIME NULL AFTER `is_archived`');
                }
                if (!in_array('archived_by', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_by` INT NULL AFTER `archived_date`');
                }
            }

            return true;
        } catch (Exception $e) {
            log_message('error', 'Deals module installation failed: ' . $e->getMessage());
            return false;
        }
    }

}

$installer = new Install();
$installer->run();












<?php

defined('BASEPATH') || exit('No direct script access allowed');

/*
    Module Name: Deals
    Description: Deals management with list, kanban, and status (open/won/lost)
    Version: 1.0.0
    Requires at least: 2.9.*
*/

define('DEALS_MODULE', 'deals');

// Load installer
require_once __DIR__ . '/install.php';

$CI = &get_instance();

// Activation hook
register_activation_hook(DEALS_MODULE, 'deals_module_activation_hook');
function deals_module_activation_hook()
{
    $CI = &get_instance();
    require_once __DIR__ . '/install.php';
}

// Register language files
register_language_files(DEALS_MODULE, [DEALS_MODULE]);

// Include assets and menu
require_once __DIR__ . '/includes/assets.php';
require_once __DIR__ . '/includes/sidebar_menu_links.php';

// Register settings section and pipelines manager tab
hooks()->add_action('admin_init', function () {
    $CI = &get_instance();
    $CI->app->add_settings_section('deals', [
        'title'    => _l('deals'),
        'position' => 18,
        'children' => [
            [
                'name' => _l('pipelines'),
                'view' => 'deals/settings',
                'position' => 10,
                'icon' => 'fa-solid fa-diagram-project',
            ],
            [
                'name' => _l('lost_reasons'),
                'view' => 'deals/lost_reasons',
                'position' => 15,
                'icon' => 'fa-regular fa-face-frown',
            ],
        ],
    ]);
});







<?php

use app\services\proposals\ProposalsPipeline;

defined('BASEPATH') or exit('No direct script access allowed');

class Proposals extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('proposals_model');
        $this->load->model('currencies_model');
    }

    public function index($proposal_id = '')
    {
        $this->list_proposals($proposal_id);
    }

    public function list_proposals($proposal_id = '')
    {
        close_setup_menu();

        if (staff_cant('view', 'proposals') && staff_cant('view_own', 'proposals') && get_option('allow_staff_view_estimates_assigned') == 0) {
            access_denied('proposals');
        }

        $isPipeline = $this->session->userdata('proposals_pipeline') == 'true';

        if ($isPipeline && !$this->input->get('status')) {
            $data['title']           = _l('proposals_pipeline');
            $data['bodyclass']       = 'proposals-pipeline';
            $data['switch_pipeline'] = false;
            // Direct access
            if (is_numeric($proposal_id)) {
                $data['proposalid'] = $proposal_id;
            } else {
                $data['proposalid'] = $this->session->flashdata('proposalid');
            }

            $this->load->view('admin/proposals/pipeline/manage', $data);
        } else {

            // Pipeline was initiated but user click from home page and need to show table only to filter
            if ($this->input->get('status') && $isPipeline) {
                $this->pipeline(0, true);
            }

            $data['proposal_id']           = $proposal_id;
            $data['switch_pipeline']       = true;
            $data['title']                 = _l('proposals');
            $data['proposal_statuses']     = $this->proposals_model->get_statuses();
            $data['proposals_sale_agents'] = $this->proposals_model->get_sale_agents();
            $data['years']                 = $this->proposals_model->get_proposals_years();
            $data['table'] = App_table::find('proposals');
            $this->load->view('admin/proposals/manage', $data);
        }
    }

    public function table()
    {
        if (
            staff_cant('view', 'proposals')
            && staff_cant('view_own', 'proposals')
            && get_option('allow_staff_view_proposals_assigned') == 0
        ) {
            ajax_access_denied();
        }

        App_table::find('proposals')->output();
    }

    public function proposal_relations($rel_id, $rel_type)
    {
        $this->app->get_table_data('proposals_relations', [
            'rel_id'   => $rel_id,
            'rel_type' => $rel_type,
        ]);
    }

    public function delete_attachment($id)
    {
        $file = $this->misc_model->get_file($id);
        if ($file->staffid == get_staff_user_id() || is_admin()) {
            echo $this->proposals_model->delete_attachment($id);
        } else {
            ajax_access_denied();
        }
    }

    public function clear_signature($id)
    {
        if (staff_can('delete',  'proposals')) {
            $this->proposals_model->clear_signature($id);
        }

        redirect(admin_url('proposals/list_proposals/' . $id));
    }

    public function sync_data()
    {
        if (staff_can('create',  'proposals') || staff_can('edit',  'proposals')) {
            $has_permission_view = staff_can('view',  'proposals');

            $this->db->where('rel_id', $this->input->post('rel_id'));
            $this->db->where('rel_type', $this->input->post('rel_type'));

            if (!$has_permission_view) {
                $this->db->where('addedfrom', get_staff_user_id());
            }

            $address = trim($this->input->post('address'));
            $address = nl2br($address);
            $this->db->update(db_prefix() . 'proposals', [
                'phone'   => $this->input->post('phone'),
                'zip'     => $this->input->post('zip'),
                'country' => $this->input->post('country'),
                'state'   => $this->input->post('state'),
                'address' => $address,
                'city'    => $this->input->post('city'),
            ]);

            if ($this->db->affected_rows() > 0) {
                echo json_encode([
                    'message' => _l('all_data_synced_successfully'),
                ]);
            } else {
                echo json_encode([
                    'message' => _l('sync_proposals_up_to_date'),
                ]);
            }
        }
    }

    public function proposal($id = '')
    {
        if ($this->input->post()) {
            $proposal_data = $this->input->post();
            if ($id == '') {
                if (staff_cant('create', 'proposals')) {
                    access_denied('proposals');
                }
				
                $id = $this->proposals_model->add($proposal_data);
                if ($id) {
                    set_alert('success', _l('added_successfully', _l('proposal')));
                    if ($this->set_proposal_pipeline_autoload($id)) {
                        redirect(admin_url('proposals'));
                    } else {
                        redirect(admin_url('proposals/list_proposals/' . $id));
                    }
                }
            } else {
                if (staff_cant('edit', 'proposals')) {
                    access_denied('proposals');
                }
                $success = $this->proposals_model->update($proposal_data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('proposal')));
                }
                if ($this->set_proposal_pipeline_autoload($id)) {
                    redirect(admin_url('proposals'));
                } else {
                    redirect(admin_url('proposals/list_proposals/' . $id));
                }
            }
        }
        if ($id == '') {
            $title = _l('add_new', _l('proposal'));
        } else {
            $data['proposal'] = $this->proposals_model->get($id);

            if (!$data['proposal'] || !user_can_view_proposal($id)) {
                blank_page(_l('proposal_not_found'));
            }

            $data['estimate']    = $data['proposal'];
            $data['is_proposal'] = true;
            $title               = _l('edit', _l('proposal'));
        }

        $this->load->model('taxes_model');
        $data['taxes'] = $this->taxes_model->get();
        $this->load->model('invoice_items_model');
        $data['ajaxItems'] = false;
        if (total_rows(db_prefix() . 'items') <= ajax_on_total_items()) {
            $data['items'] = $this->invoice_items_model->get_grouped();
        } else {
            $data['items']     = [];
            $data['ajaxItems'] = true;
        }
        $data['items_groups'] = $this->invoice_items_model->get_groups();

        $data['statuses']      = $this->proposals_model->get_statuses();
        $data['staff']         = $this->staff_model->get('', ['active' => 1]);
        $data['currencies']    = $this->currencies_model->get();
        $data['base_currency'] = $this->currencies_model->get_base_currency();

        $data['title'] = $title;
        $this->load->view('admin/proposals/proposal', $data);
    }

    public function get_template()
    {
        $name = $this->input->get('name');
        echo $this->load->view('admin/proposals/templates/' . $name, [], true);
    }

    public function send_expiry_reminder($id)
    {
        $canView = user_can_view_proposal($id);
        if (!$canView) {
            access_denied('proposals');
        } else {
            if (staff_cant('view', 'proposals') && staff_cant('view_own', 'proposals') && $canView == false) {
                access_denied('proposals');
            }
        }

        $success = $this->proposals_model->send_expiry_reminder($id);
        if ($success) {
            set_alert('success', _l('sent_expiry_reminder_success'));
        } else {
            set_alert('danger', _l('sent_expiry_reminder_fail'));
        }
        if ($this->set_proposal_pipeline_autoload($id)) {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        } else {
            redirect(admin_url('proposals/list_proposals/' . $id));
        }
    }

    public function clear_acceptance_info($id)
    {
        if (is_admin()) {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'proposals', get_acceptance_info_array(true));
        }

        redirect(admin_url('proposals/list_proposals/' . $id));
    }

    public function pdf($id)
    {
        if (!$id) {
            redirect(admin_url('proposals'));
        }

        $canView = user_can_view_proposal($id);
        if (!$canView) {
            access_denied('proposals');
        } else {
            if (staff_cant('view', 'proposals') && staff_cant('view_own', 'proposals') && $canView == false) {
                access_denied('proposals');
            }
        }

        $proposal = $this->proposals_model->get($id);

        try {
            $pdf = proposal_pdf($proposal);
        } catch (Exception $e) {
            $message = $e->getMessage();
            echo $message;
            if (strpos($message, 'Unable to get the size of the image') !== false) {
                show_pdf_unable_to_get_image_size_error();
            }
            die;
        }

        $type = 'D';

        if ($this->input->get('output_type')) {
            $type = $this->input->get('output_type');
        }

        if ($this->input->get('print')) {
            $type = 'I';
        }

        $proposal_number = format_proposal_number($id);
        $pdf->Output($proposal_number . '.pdf', $type);
    }

    public function get_proposal_data_ajax($id, $to_return = false)
    {
        if (staff_cant('view', 'proposals') && staff_cant('view_own', 'proposals') && get_option('allow_staff_view_proposals_assigned') == 0) {
            echo _l('access_denied');
            die;
        }

        $proposal = $this->proposals_model->get($id, [], true);

        if (!$proposal || !user_can_view_proposal($id)) {
            echo _l('proposal_not_found');
            die;
        }

        $this->app_mail_template->set_rel_id($proposal->id);
        $data = prepare_mail_preview_data('proposal_send_to_customer', $proposal->email);

        $merge_fields = [];

        $merge_fields[] = [
            [
                'name' => 'Items Table',
                'key'  => '{proposal_items}',
            ],
        ];

        $merge_fields = array_merge($merge_fields, $this->app_merge_fields->get_flat('proposals', 'other', '{email_signature}'));

        $data['proposal_statuses']     = $this->proposals_model->get_statuses();
        $data['members']               = $this->staff_model->get('', ['active' => 1]);
        $data['proposal_merge_fields'] = $merge_fields;
        $data['proposal']              = $proposal;
        $data['totalNotes']            = total_rows(db_prefix() . 'notes', ['rel_id' => $id, 'rel_type' => 'proposal']);
        if ($to_return == false) {
            $this->load->view('admin/proposals/proposals_preview_template', $data);
        } else {
            return $this->load->view('admin/proposals/proposals_preview_template', $data, true);
        }
    }

    public function add_note($rel_id)
    {
        if ($this->input->post() && user_can_view_proposal($rel_id)) {
            $this->misc_model->add_note($this->input->post(), 'proposal', $rel_id);
            echo $rel_id;
        }
    }

    public function get_notes($id)
    {
        if (user_can_view_proposal($id)) {
            $data['notes'] = $this->misc_model->get_notes($id, 'proposal');
            $this->load->view('admin/includes/sales_notes_template', $data);
        }
    }

    public function convert_to_estimate($id)
    {
        if (staff_cant('create', 'estimates')) {
            access_denied('estimates');
        }
        if ($this->input->post()) {
            $this->load->model('estimates_model');
            $estimate_id = $this->estimates_model->add($this->input->post());
            if ($estimate_id) {
                set_alert('success', _l('proposal_converted_to_estimate_success'));
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'proposals', [
                    'estimate_id' => $estimate_id,
                    'status'      => 3,
                ]);
                log_activity('Proposal Converted to Estimate [EstimateID: ' . $estimate_id . ', ProposalID: ' . $id . ']');

                hooks()->do_action('proposal_converted_to_estimate', ['proposal_id' => $id, 'estimate_id' => $estimate_id]);

                redirect(admin_url('estimates/estimate/' . $estimate_id));
            } else {
                set_alert('danger', _l('proposal_converted_to_estimate_fail'));
            }
            if ($this->set_proposal_pipeline_autoload($id)) {
                redirect(admin_url('proposals'));
            } else {
                redirect(admin_url('proposals/list_proposals/' . $id));
            }
        }
    }

    public function convert_to_invoice($id)
    {
        if (staff_cant('create', 'invoices')) {
            access_denied('invoices');
        }
        if ($this->input->post()) {
            $this->load->model('invoices_model');
            $invoice_id = $this->invoices_model->add($this->input->post());
            if ($invoice_id) {
                set_alert('success', _l('proposal_converted_to_invoice_success'));
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'proposals', [
                    'invoice_id' => $invoice_id,
                    'status'     => 3,
                ]);
                log_activity('Proposal Converted to Invoice [InvoiceID: ' . $invoice_id . ', ProposalID: ' . $id . ']');

                do_action_deprecated('proposal_converted_to_invoice', ['proposal_id' => $id, 'invoice_id' => $invoice_id], '3.1.6', 'after_proposal_converted_to_invoice');
                hooks()->do_action('after_proposal_converted_to_invoice', ['proposal_id' => $id, 'invoice_id' => $invoice_id]);

                redirect(admin_url('invoices/invoice/' . $invoice_id));
            } else {
                set_alert('danger', _l('proposal_converted_to_invoice_fail'));
            }
            if ($this->set_proposal_pipeline_autoload($id)) {
                redirect(admin_url('proposals'));
            } else {
                redirect(admin_url('proposals/list_proposals/' . $id));
            }
        }
    }

    public function get_invoice_convert_data($id)
    {
        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [
            'expenses_only !=' => 1,
        ]);
        $this->load->model('taxes_model');
        $data['taxes']         = $this->taxes_model->get();
        $data['currencies']    = $this->currencies_model->get();
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $this->load->model('invoice_items_model');
        $data['ajaxItems'] = false;
        if (total_rows(db_prefix() . 'items') <= ajax_on_total_items()) {
            $data['items'] = $this->invoice_items_model->get_grouped();
        } else {
            $data['items']     = [];
            $data['ajaxItems'] = true;
        }
        $data['items_groups'] = $this->invoice_items_model->get_groups();

        $data['staff']          = $this->staff_model->get('', ['active' => 1]);
        $data['proposal']       = $this->proposals_model->get($id);
        $data['billable_tasks'] = [];
        $data['add_items']      = $this->_parse_items($data['proposal']);

        if ($data['proposal']->rel_type == 'lead') {
            $this->db->where('leadid', $data['proposal']->rel_id);
            $data['customer_id'] = $this->db->get(db_prefix() . 'clients')->row()->userid;
        } else {
            $data['customer_id'] = $data['proposal']->rel_id;
            $data['project_id'] = $data['proposal']->project_id;
        }
        $data['custom_fields_rel_transfer'] = [
            'belongs_to' => 'proposal',
            'rel_id'     => $id,
        ];
        $this->load->view('admin/proposals/invoice_convert_template', $data);
    }

    public function get_estimate_convert_data($id)
    {
        $this->load->model('taxes_model');
        $data['taxes']         = $this->taxes_model->get();
        $data['currencies']    = $this->currencies_model->get();
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $this->load->model('invoice_items_model');
        $data['ajaxItems'] = false;
        if (total_rows(db_prefix() . 'items') <= ajax_on_total_items()) {
            $data['items'] = $this->invoice_items_model->get_grouped();
        } else {
            $data['items']     = [];
            $data['ajaxItems'] = true;
        }
        $data['items_groups'] = $this->invoice_items_model->get_groups();

        $data['staff']     = $this->staff_model->get('', ['active' => 1]);
        $data['proposal']  = $this->proposals_model->get($id);
        $data['add_items'] = $this->_parse_items($data['proposal']);

        $this->load->model('estimates_model');
        $data['estimate_statuses'] = $this->estimates_model->get_statuses();
        if ($data['proposal']->rel_type == 'lead') {
            $this->db->where('leadid', $data['proposal']->rel_id);
            $data['customer_id'] = $this->db->get(db_prefix() . 'clients')->row()->userid;
        } else {
            $data['customer_id'] = $data['proposal']->rel_id;
            $data['project_id'] = $data['proposal']->project_id;
        }

        $data['custom_fields_rel_transfer'] = [
            'belongs_to' => 'proposal',
            'rel_id'     => $id,
        ];

        $this->load->view('admin/proposals/estimate_convert_template', $data);
    }

    private function _parse_items($proposal)
    {
        $items = [];
        foreach ($proposal->items as $item) {
            $taxnames = [];
            $taxes    = get_proposal_item_taxes($item['id']);
            foreach ($taxes as $tax) {
                array_push($taxnames, $tax['taxname']);
            }
            $item['taxname']        = $taxnames;
            $item['parent_item_id'] = $item['id'];
            $item['id']             = 0;
            $items[]                = $item;
        }

        return $items;
    }

    /* Send proposal to email */
    public function send_to_email($id)
    {
        $canView = user_can_view_proposal($id);
        if (!$canView) {
            access_denied('proposals');
        } else {
            if (staff_cant('view', 'proposals') && staff_cant('view_own', 'proposals') && $canView == false) {
                access_denied('proposals');
            }
        }

        if ($this->input->post()) {
            try {
                $success = $this->proposals_model->send_proposal_to_email(
                    $id,
                    $this->input->post('attach_pdf'),
                    $this->input->post('cc')
                );
            } catch (Exception $e) {
                $message = $e->getMessage();
                echo $message;
                if (strpos($message, 'Unable to get the size of the image') !== false) {
                    show_pdf_unable_to_get_image_size_error();
                }
                die;
            }

            if ($success) {
                set_alert('success', _l('proposal_sent_to_email_success'));
            } else {
                set_alert('danger', _l('proposal_sent_to_email_fail'));
            }

            if ($this->set_proposal_pipeline_autoload($id)) {
                redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
            } else {
                redirect(admin_url('proposals/list_proposals/' . $id));
            }
        }
    }

    public function copy($id)
    {
        if (staff_cant('create', 'proposals')) {
            access_denied('proposals');
        }
        $new_id = $this->proposals_model->copy($id);
        if ($new_id) {
            set_alert('success', _l('proposal_copy_success'));
            $this->set_proposal_pipeline_autoload($new_id);
            redirect(admin_url('proposals/proposal/' . $new_id));
        } else {
            set_alert('success', _l('proposal_copy_fail'));
        }
        if ($this->set_proposal_pipeline_autoload($id)) {
            redirect(admin_url('proposals'));
        } else {
            redirect(admin_url('proposals/list_proposals/' . $id));
        }
    }

    public function mark_action_status($status, $id)
    {
        if (staff_cant('edit', 'proposals')) {
            access_denied('proposals');
        }
        $success = $this->proposals_model->mark_action_status($status, $id);
        if ($success) {
            set_alert('success', _l('proposal_status_changed_success'));
        } else {
            set_alert('danger', _l('proposal_status_changed_fail'));
        }
        if ($this->set_proposal_pipeline_autoload($id)) {
            redirect(admin_url('proposals'));
        } else {
            redirect(admin_url('proposals/list_proposals/' . $id));
        }
    }

    public function delete($id)
    {
        if (staff_cant('delete', 'proposals')) {
            access_denied('proposals');
        }
        $response = $this->proposals_model->delete($id);
        if ($response == true) {
            set_alert('success', _l('deleted', _l('proposal')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('proposal_lowercase')));
        }
        redirect(admin_url('proposals'));
    }

    public function get_relation_data_values($rel_id, $rel_type)
    {
		
        echo json_encode($this->proposals_model->get_relation_data_values($rel_id, $rel_type));
    }

    public function add_proposal_comment()
    {
        if ($this->input->post()) {
            echo json_encode([
                'success' => $this->proposals_model->add_comment($this->input->post()),
            ]);
        }
    }

    public function edit_comment($id)
    {
        if ($this->input->post()) {
            echo json_encode([
                'success' => $this->proposals_model->edit_comment($this->input->post(), $id),
                'message' => _l('comment_updated_successfully'),
            ]);
        }
    }

    public function get_proposal_comments($id)
    {
        $data['comments'] = $this->proposals_model->get_comments($id);
        $this->load->view('admin/proposals/comments_template', $data);
    }

    public function remove_comment($id)
    {
        $this->db->where('id', $id);
        $comment = $this->db->get(db_prefix() . 'proposal_comments')->row();
        if ($comment) {
            if ($comment->staffid != get_staff_user_id() && !is_admin()) {
                echo json_encode([
                    'success' => false,
                ]);
                die;
            }
            echo json_encode([
                'success' => $this->proposals_model->remove_comment($id),
            ]);
        } else {
            echo json_encode([
                'success' => false,
            ]);
        }
    }

    public function save_proposal_data()
    {
        if (staff_cant('edit', 'proposals') && staff_cant('create', 'proposals')) {
            header('HTTP/1.0 400 Bad error');
            echo json_encode([
                'success' => false,
                'message' => _l('access_denied'),
            ]);
            die;
        }
        $success = false;
        $message = '';

        $this->db->where('id', $this->input->post('proposal_id'));
        $this->db->update(db_prefix() . 'proposals', [
            'content' => html_purify($this->input->post('content', false)),
        ]);

        $success = $this->db->affected_rows() > 0;
        $message = _l('updated_successfully', _l('proposal'));

        echo json_encode([
            'success' => $success,
            'message' => $message,
        ]);
    }

    // Pipeline
    public function pipeline($set = 0, $manual = false)
    {
        if ($set == 1) {
            $set = 'true';
        } else {
            $set = 'false';
        }
        $this->session->set_userdata([
            'proposals_pipeline' => $set,
        ]);
        if ($manual == false) {
            redirect(admin_url('proposals'));
        }
    }

    public function pipeline_open($id)
    {
        if (staff_can('view',  'proposals') || staff_can('view_own',  'proposals') || get_option('allow_staff_view_proposals_assigned') == 1) {
            $data['proposal']      = $this->get_proposal_data_ajax($id, true);
            $data['proposal_data'] = $this->proposals_model->get($id);
            $this->load->view('admin/proposals/pipeline/proposal', $data);
        }
    }

    public function update_pipeline()
    {
        if (staff_can('edit',  'proposals')) {
            $this->proposals_model->update_pipeline($this->input->post());
        }
    }

    public function get_pipeline()
    {
        if (staff_can('view',  'proposals') || staff_can('view_own',  'proposals') || get_option('allow_staff_view_proposals_assigned') == 1) {
            $data['statuses'] = $this->proposals_model->get_statuses();
            $this->load->view('admin/proposals/pipeline/pipeline', $data);
        }
    }

    public function pipeline_load_more()
    {
        $status = $this->input->get('status');
        $page   = $this->input->get('page');

        $proposals = (new ProposalsPipeline($status))
        ->search($this->input->get('search'))
        ->sortBy(
            $this->input->get('sort_by'),
            $this->input->get('sort')
        )
        ->page($page)->get();

        foreach ($proposals as $proposal) {
            $this->load->view('admin/proposals/pipeline/_kanban_card', [
                'proposal' => $proposal,
                'status'   => $status,
            ]);
        }
    }

    public function set_proposal_pipeline_autoload($id)
    {
        if ($id == '') {
            return false;
        }

        if ($this->session->has_userdata('proposals_pipeline') && $this->session->userdata('proposals_pipeline') == 'true') {
            $this->session->set_flashdata('proposalid', $id);

            return true;
        }

        return false;
    }

    public function get_due_date()
    {
        if ($this->input->post()) {
            $date    = $this->input->post('date');
            $duedate = '';
            if (get_option('proposal_due_after') != 0) {
                $date    = to_sql_date($date);
                $d       = date('Y-m-d', strtotime('+' . get_option('proposal_due_after') . ' DAY', strtotime($date)));
                $duedate = _d($d);
                echo $duedate;
            }
        }
    }
}



<?php

use app\services\utilities\Date;
use app\services\tasks\TasksKanban;

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property-read Staff_model $staff_model
 * @property-read Tasks_model $tasks_model
 * @property-read Projects_model $projects_model
 */
class Tasks extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('projects_model');
    }

    /* Open also all taks if user access this /tasks url */
    public function index($id = '')
    {
        $this->list_tasks($id);
    }

    /* List all tasks */
    public function list_tasks($id = '')
    {
        close_setup_menu();
        // If passed from url
        $data['custom_view'] = $this->input->get('custom_view') ? $this->input->get('custom_view') : '';
        $data['taskid']      = $id;

        if ($this->input->get('kanban')) {
            $this->switch_kanban(0, true);
        }

        $data['switch_kanban'] = false;
        $data['bodyclass']     = 'tasks-page';

        if ($this->session->userdata('tasks_kanban_view') == 'true') {
            $data['switch_kanban'] = true;
            $data['bodyclass']     = 'tasks-page kan-ban-body';
        }

        $data['title'] = _l('tasks');
        $data['tasks_table'] = App_table::find('tasks');
        $this->load->view('admin/tasks/manage', $data);
    }

    public function table()
    {
        App_table::find('tasks')->output();
    }

    public function kanban()
    {
        echo $this->load->view('admin/tasks/kan_ban', [], true);
    }

    public function ajax_search_assign_task_to_timer()
    {
        if ($this->input->is_ajax_request()) {
            $q = $this->input->post('q');
            $q = trim($q);
            $this->db->select('name, id,' . tasks_rel_name_select_query() . ' as subtext');
            $this->db->from(db_prefix() . 'tasks');
            $this->db->where('' . db_prefix() . 'tasks.id IN (SELECT taskid FROM ' . db_prefix() . 'task_assigned WHERE staffid = ' . get_staff_user_id() . ')');
            //   $this->db->where('id NOT IN (SELECT task_id FROM '.db_prefix().'taskstimers WHERE staff_id = ' . get_staff_user_id() . ' AND end_time IS NULL)');
            $this->db->where('status != ', 5);
            $this->db->where('billed', 0);
            $this->db->group_start();
            $this->db->like('name', $q);
            $this->db->or_like(tasks_rel_name_select_query(), $q);
            $this->db->group_end();
            echo json_encode($this->db->get()->result_array());
        }
    }

    public function tasks_kanban_load_more()
    {
        $status = $this->input->get('status');
        $page   = $this->input->get('page');

        $tasks = (new TasksKanban($status))
            ->search($this->input->get('search'))
            ->sortBy(
                $this->input->get('sort_by'),
                $this->input->get('sort')
            )
            ->forProject($this->input->get('project_id') ?: null)
            ->page($page)->get();


        foreach ($tasks as $task) {
            $this->load->view('admin/tasks/_kan_ban_card', [
                'task'   => $task,
                'status' => $status,
            ]);
        }
    }

    public function update_order()
    {
        $this->tasks_model->update_order($this->input->post());
    }

    public function switch_kanban($set = 0, $manual = false)
    {
        if ($set == 1) {
            $set = 'false';
        } else {
            $set = 'true';
        }

        $this->session->set_userdata([
            'tasks_kanban_view' => $set,
        ]);

        if ($manual == false) {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        }
    }

    // Used in invoice add/edit
    public function get_billable_tasks_by_project($project_id)
    {
        if ($this->input->is_ajax_request() && (staff_can('edit',  'invoices') || staff_can('create',  'invoices'))) {
            $customer_id = get_client_id_by_project_id($project_id);
            echo json_encode($this->tasks_model->get_billable_tasks($customer_id, $project_id));
        }
    }

    // Used in invoice add/edit
    public function get_billable_tasks_by_customer_id($customer_id)
    {
        if ($this->input->is_ajax_request() && (staff_can('edit',  'invoices') || staff_can('create',  'invoices'))) {
            echo json_encode($this->tasks_model->get_billable_tasks($customer_id));
        }
    }

    public function update_task_description($id)
    {
        if (staff_can('edit',  'tasks')) {
            $data = hooks()->apply_filters('before_update_task', [
                'description' => html_purify($this->input->post('description', false)),
            ], $id);

            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'tasks', $data);

            hooks()->do_action('after_update_task', $id);
        }
    }

    public function detailed_overview()
    {
        $overview = [];

        $has_permission_create = staff_can('create',  'tasks');
        $has_permission_view   = staff_can('view',  'tasks');

        if (!$has_permission_view) {
            $staff_id = get_staff_user_id();
        } elseif ($this->input->post('member')) {
            $staff_id = $this->input->post('member');
        } else {
            $staff_id = '';
        }

        $month = ($this->input->post('month') ? $this->input->post('month') : date('m'));
        if ($this->input->post() && $this->input->post('month') == '') {
            $month = '';
        }

        $status = $this->input->post('status');

        $fetch_month_from = 'startdate';

        $year       = ($this->input->post('year') ? $this->input->post('year') : date('Y'));
        $project_id = $this->input->get('project_id');

        for ($m = 1; $m <= 12; $m++) {
            if ($month != '' && $month != $m) {
                continue;
            }

            // Task rel_name
            $sqlTasksSelect = '*,' . tasks_rel_name_select_query() . ' as rel_name';

            // Task logged time
            $selectLoggedTime = get_sql_calc_task_logged_time('tmp-task-id');
            // Replace tmp-task-id to be the same like tasks.id
            $selectLoggedTime = str_replace('tmp-task-id', db_prefix() . 'tasks.id', $selectLoggedTime);

            if (is_numeric($staff_id)) {
                $selectLoggedTime .= ' AND staff_id=' . $this->db->escape_str($staff_id);
                $sqlTasksSelect .= ',(' . $selectLoggedTime . ')';
            } else {
                $sqlTasksSelect .= ',(' . $selectLoggedTime . ')';
            }

            $sqlTasksSelect .= ' as total_logged_time';

            // Task checklist items
            $sqlTasksSelect .= ',' . get_sql_select_task_total_checklist_items();

            if (is_numeric($staff_id)) {
                $sqlTasksSelect .= ',(SELECT COUNT(id) FROM ' . db_prefix() . 'task_checklist_items WHERE taskid=' . db_prefix() . 'tasks.id AND finished=1 AND finished_from=' . $staff_id . ') as total_finished_checklist_items';
            } else {
                $sqlTasksSelect .= ',' . get_sql_select_task_total_finished_checklist_items();
            }

            // Task total comment and total files
            $selectTotalComments = ',(SELECT COUNT(id) FROM ' . db_prefix() . 'task_comments WHERE taskid=' . db_prefix() . 'tasks.id';
            $selectTotalFiles    = ',(SELECT COUNT(id) FROM ' . db_prefix() . 'files WHERE rel_id=' . db_prefix() . 'tasks.id AND rel_type="task"';

            if (is_numeric($staff_id)) {
                $sqlTasksSelect .= $selectTotalComments . ' AND staffid=' . $staff_id . ') as total_comments_staff';
                $sqlTasksSelect .= $selectTotalFiles . ' AND staffid=' . $staff_id . ') as total_files_staff';
            }

            $sqlTasksSelect .= $selectTotalComments . ') as total_comments';
            $sqlTasksSelect .= $selectTotalFiles . ') as total_files';

            // Task assignees
            $sqlTasksSelect .= ',' . get_sql_select_task_asignees_full_names() . ' as assignees' . ',' . get_sql_select_task_assignees_ids() . ' as assignees_ids';

            $this->db->select($sqlTasksSelect);

            $this->db->where('MONTH(' . $fetch_month_from . ')', $m);
            $this->db->where('YEAR(' . $fetch_month_from . ')', $year);

            if ($project_id && $project_id != '') {
                $this->db->where('rel_id', $project_id);
                $this->db->where('rel_type', 'project');
            }

            if (!$has_permission_view) {
                $sqlWhereStaff = '(id IN (SELECT taskid FROM ' . db_prefix() . 'task_assigned WHERE staffid=' . $staff_id . ')';

                // User dont have permission for view but have for create
                // Only show tasks createad by this user.
                if ($has_permission_create) {
                    $sqlWhereStaff .= ' OR addedfrom=' . get_staff_user_id();
                }

                $sqlWhereStaff .= ')';
                $this->db->where($sqlWhereStaff);
            } elseif ($has_permission_view) {
                if (is_numeric($staff_id)) {
                    $this->db->where('(id IN (SELECT taskid FROM ' . db_prefix() . 'task_assigned WHERE staffid=' . $staff_id . '))');
                }
            }

            if ($status) {
                $this->db->where('status', $status);
            }

            $this->db->order_by($fetch_month_from, 'ASC');
            array_push($overview, $m);
            $overview[$m] = $this->db->get(db_prefix() . 'tasks')->result_array();
        }

        unset($overview[0]);

        $overview = [
            'staff_id' => $staff_id,
            'detailed' => $overview,
        ];

        $data['members']  = $this->staff_model->get();
        $data['overview'] = $overview['detailed'];
        $data['years']    = $this->tasks_model->get_distinct_tasks_years(($this->input->post('month_from') ? $this->input->post('month_from') : 'startdate'));
        $data['staff_id'] = $overview['staff_id'];
        $data['title']    = _l('detailed_overview');
        $this->load->view('admin/tasks/detailed_overview', $data);
    }

    public function init_relation_tasks($rel_id, $rel_type)
    {
        if ($this->input->is_ajax_request()) {
           App_table::find('related_tasks')->output([
                'rel_id'   => $rel_id,
                'rel_type' => $rel_type,
           ]);
        }
    }

    /* Add new task or update existing */
    public function task($id = '')
    {
        if (staff_cant('edit', 'tasks') && staff_cant('create', 'tasks')) {
            ajax_access_denied();
        }

        $data = [];
        // FOr new task add directly from the projects milestones
        if ($this->input->get('milestone_id')) {
            $this->db->where('id', $this->input->get('milestone_id'));
            $milestone = $this->db->get(db_prefix() . 'milestones')->row();
            if ($milestone) {
                $data['_milestone_selected_data'] = [
                    'id'       => $milestone->id,
                    'due_date' => _d($milestone->due_date),
                ];
            }
        }
        if ($this->input->get('start_date')) {
            $data['start_date'] = $this->input->get('start_date');
        }
        if ($this->input->post()) {
            $data                = $this->input->post();
            $data['description'] = html_purify($this->input->post('description', false));
            if ($id == '') {
                if (staff_cant('create', 'tasks')) {
                    header('HTTP/1.0 400 Bad error');
                    echo json_encode([
                        'success' => false,
                        'message' => _l('access_denied'),
                    ]);
                    die;
                }
                $id      = $this->tasks_model->add($data);
                $_id     = false;
                $success = false;
                $message = '';
                if ($id) {
                    $success       = true;
                    $_id           = $id;
                    $message       = _l('added_successfully', _l('task'));
                    $uploadedFiles = handle_task_attachments_array($id);
                    if ($uploadedFiles && is_array($uploadedFiles)) {
                        foreach ($uploadedFiles as $file) {
                            $this->misc_model->add_attachment_to_database($id, 'task', [$file]);
                        }
                    }
                }
                echo json_encode([
                    'success' => $success,
                    'id'      => $_id,
                    'message' => $message,
                ]);
            } else {
                if (staff_cant('edit', 'tasks')) {
                    header('HTTP/1.0 400 Bad error');
                    echo json_encode([
                        'success' => false,
                        'message' => _l('access_denied'),
                    ]);
                    die;
                }
                $success = $this->tasks_model->update($data, $id);
                $message = '';
                if ($success) {
                    $message = _l('updated_successfully', _l('task'));
                }
                echo json_encode([
                    'success' => $success,
                    'message' => $message,
                    'id'      => $id,
                ]);
            }
            die;
        }

        $data['milestones']         = [];
        $data['checklistTemplates'] = $this->tasks_model->get_checklist_templates();
        if ($id == '') {
            $title = _l('add_new', _l('task'));
        } else {
            $data['task'] = $this->tasks_model->get($id);
            if ($data['task']->rel_type == 'project') {
                $data['milestones'] = $this->projects_model->get_milestones($data['task']->rel_id);
            }
            $title = _l('edit', _l('task')) . ' ' . $data['task']->name;
        }

        $data['project_end_date_attrs'] = [];
        if ($this->input->get('rel_type') == 'project' && $this->input->get('rel_id') || ($id !== '' && $data['task']->rel_type == 'project')) {
            $project = $this->projects_model->get($id === '' ? $this->input->get('rel_id') : $data['task']->rel_id);

            if ($project->deadline) {
                $data['project_end_date_attrs'] = [
                    'data-date-end-date' => $project->deadline,
                ];
            }
        }
        $data['members'] = $this->staff_model->get('', ['active' => 1]);
        $data['id']      = $id;
        $data['title']   = $title;
        $this->load->view('admin/tasks/task', $data);
    }

    public function copy()
    {
        if (staff_can('create',  'tasks')) {
            $new_task_id = $this->tasks_model->copy($this->input->post());
            $response    = [
                'new_task_id' => '',
                'alert_type'  => 'warning',
                'message'     => _l('failed_to_copy_task'),
                'success'     => false,
            ];
            if ($new_task_id) {
                $response['message']     = _l('task_copied_successfully');
                $response['new_task_id'] = $new_task_id;
                $response['success']     = true;
                $response['alert_type']  = 'success';
            }
            echo json_encode($response);
        }
    }

    public function get_billable_task_data($task_id)
    {
        $task              = $this->tasks_model->get_billable_task_data($task_id);
        $task->description = seconds_to_time_format($task->total_seconds) . ' ' . _l('hours');
        echo json_encode($task);
    }

    /**
     * Task ajax request modal
     * @param  mixed $taskid
     * @return mixed
     */
    public function get_task_data($taskid, $return = false)
    {
        $tasks_where = [];

        if (staff_cant('view', 'tasks')) {
            $tasks_where = get_tasks_where_string(false);
        }

        $task = $this->tasks_model->get($taskid, $tasks_where);

        if (!$task) {
            header('HTTP/1.0 404 Not Found');
            echo 'Task not found';
            die();
        }

        $data['checklistTemplates'] = $this->tasks_model->get_checklist_templates();
        $data['task']               = $task;
        $data['id']                 = $task->id;
        $data['staff']              = $this->staff_model->get('', ['active' => 1]);
        $data['reminders']          = $this->tasks_model->get_reminders($taskid);

        $data['task_staff_members'] = $this->tasks_model->get_staff_members_that_can_access_task($taskid);
        // For backward compatibilities
        $data['staff_reminders'] = $data['task_staff_members'];

        $data['hide_completed_items'] = get_staff_meta(get_staff_user_id(), 'task-hide-completed-items-' . $taskid);

        $data['project_deadline'] = null;
        if ($task->rel_type == 'project') {
            $data['project_deadline'] = get_project_deadline($task->rel_id);
        }

        if ($return == false) {
            $this->load->view('admin/tasks/view_task_template', $data);
        } else {
            return $this->load->view('admin/tasks/view_task_template', $data, true);
        }
    }

    public function add_reminder($task_id)
    {
        $message    = '';
        $alert_type = 'warning';
        if ($this->input->post()) {
            $success = $this->misc_model->add_reminder($this->input->post(), $task_id);
            if ($success) {
                $alert_type = 'success';
                $message    = _l('reminder_added_successfully');
            }
        }
        echo json_encode([
            'taskHtml'   => $this->get_task_data($task_id, true),
            'alert_type' => $alert_type,
            'message'    => $message,
        ]);
    }

    public function edit_reminder($id)
    {
        $reminder = $this->misc_model->get_reminders($id);
        if ($reminder && ($reminder->creator == get_staff_user_id() || is_admin()) && $reminder->isnotified == 0) {
            $success = $this->misc_model->edit_reminder($this->input->post(), $id);
            echo json_encode([
                'taskHtml'   => $this->get_task_data($reminder->rel_id, true),
                'alert_type' => 'success',
                'message'    => ($success ? _l('updated_successfully', _l('reminder')) : ''),
            ]);
        }
    }

    public function delete_reminder($rel_id, $id)
    {
        $success    = $this->misc_model->delete_reminder($id);
        $alert_type = 'warning';
        $message    = _l('reminder_failed_to_delete');
        if ($success) {
            $alert_type = 'success';
            $message    = _l('reminder_deleted');
        }
        echo json_encode([
            'taskHtml'   => $this->get_task_data($rel_id, true),
            'alert_type' => $alert_type,
            'message'    => $message,
        ]);
    }

    public function get_staff_started_timers($return = false)
    {
        $data['startedTimers'] = $this->misc_model->get_staff_started_timers();
        $_data['html']         = $this->load->view('admin/tasks/started_timers', $data, true);
        $_data['total_timers'] = count($data['startedTimers']);

        $timers = json_encode($_data);
        if ($return) {
            return $timers;
        }

        echo $timers;
    }

    public function save_checklist_item_template()
    {
        if (staff_can('create',  'checklist_templates')) {
            $id = $this->tasks_model->add_checklist_template($this->input->post('description'));
            echo json_encode(['id' => $id]);
        }
    }

    public function remove_checklist_item_template($id)
    {
        if (staff_can('delete',  'checklist_templates')) {
            $success = $this->tasks_model->remove_checklist_item_template($id);
            echo json_encode(['success' => $success]);
        }
    }

    public function init_checklist_items()
    {
        if ($this->input->is_ajax_request()) {
            if ($this->input->post()) {
                $post_data                       = $this->input->post();
                $data['task_id']                 = $post_data['taskid'];
                $data['checklists']              = $this->tasks_model->get_checklist_items($post_data['taskid']);
                $data['task_staff_members']      = $this->tasks_model->get_staff_members_that_can_access_task($data['task_id']);
                $data['current_user_is_creator'] = $this->tasks_model->is_task_creator(get_staff_user_id(), $data['task_id']);
                $data['hide_completed_items']    = get_staff_meta(get_staff_user_id(), 'task-hide-completed-items-' . $data['task_id']);

                $this->load->view('admin/tasks/checklist_items_template', $data);
            }
        }
    }

    public function task_tracking_stats($task_id)
    {
        $data['stats'] = json_encode($this->tasks_model->task_tracking_stats($task_id));
        $this->load->view('admin/tasks/tracking_stats', $data);
    }

    public function checkbox_action($listid, $value)
    {
        $this->db->where('id', $listid);
        $this->db->update(db_prefix() . 'task_checklist_items', [
            'finished' => $value,
        ]);

        if ($this->db->affected_rows() > 0) {
            if ($value == 1) {
                $this->db->where('id', $listid);
                $this->db->update(db_prefix() . 'task_checklist_items', [
                    'finished_from' => get_staff_user_id(),
                ]);
                hooks()->do_action('task_checklist_item_finished', $listid);
            }
        }
    }

    public function add_checklist_item()
    {
        if ($this->input->is_ajax_request()) {
            if ($this->input->post()) {
                echo json_encode([
                    'success' => $this->tasks_model->add_checklist_item($this->input->post()),
                ]);
            }
        }
    }

    public function update_checklist_order()
    {
        if ($this->input->is_ajax_request()) {
            if ($this->input->post()) {
                $this->tasks_model->update_checklist_order($this->input->post());
            }
        }
    }

    public function delete_checklist_item($id)
    {
        $list = $this->tasks_model->get_checklist_item($id);
        if (staff_can('delete',  'tasks') || $list->addedfrom == get_staff_user_id()) {
            if ($this->input->is_ajax_request()) {
                echo json_encode([
                    'success' => $this->tasks_model->delete_checklist_item($id),
                ]);
            }
        }
    }

    public function update_checklist_item()
    {
        if ($this->input->is_ajax_request()) {
            if ($this->input->post()) {
                $desc = $this->input->post('description');
                $desc = trim($desc);
                $this->tasks_model->update_checklist_item($this->input->post('listid'), $desc);
                echo json_encode(['can_be_template' => (total_rows(db_prefix() . 'tasks_checklist_templates', ['description' => $desc]) == 0)]);
            }
        }
    }

    public function make_public($task_id)
    {
        if (staff_cant('edit', 'tasks')) {
            json_encode([
                'success' => false,
            ]);
            die;
        }
        echo json_encode([
            'success'  => $this->tasks_model->make_public($task_id),
            'taskHtml' => $this->get_task_data($task_id, true),
        ]);
    }

    public function add_external_attachment()
    {
        if ($this->input->post()) {
            $this->tasks_model->add_attachment_to_database(
                $this->input->post('task_id'),
                $this->input->post('files'),
                $this->input->post('external')
            );
        }
    }

    /* Add new task comment / ajax */
    public function add_task_comment()
    {
        $data            = $this->input->post();
        $data['content'] = html_purify($this->input->post('content', false));
        if ($this->input->post('no_editor')) {
            $data['content'] = nl2br($this->input->post('content'));
        }
        $comment_id = false;
        if (
            $data['content'] != ''
            || (isset($_FILES['file']['name']) && is_array($_FILES['file']['name']) && count($_FILES['file']['name']) > 0)
        ) {
            $comment_id = $this->tasks_model->add_task_comment($data);
            if ($comment_id) {
                $commentAttachments = handle_task_attachments_array($data['taskid'], 'file');
                if ($commentAttachments && is_array($commentAttachments)) {
                    foreach ($commentAttachments as $file) {
                        $file['task_comment_id'] = $comment_id;
                        $this->misc_model->add_attachment_to_database($data['taskid'], 'task', [$file]);
                    }

                    if (count($commentAttachments) > 0) {
                        $this->db->query('UPDATE ' . db_prefix() . "task_comments SET content = CONCAT(content, '[task_attachment]')
                            WHERE id = " . $this->db->escape_str($comment_id));
                    }
                }
            }
        }
        echo json_encode([
            'success'  => $comment_id ? true : false,
            'taskHtml' => $this->get_task_data($data['taskid'], true),
        ]);
    }

    public function download_files($task_id, $comment_id = null)
    {
        $taskWhere = 'external IS NULL';

        if ($comment_id) {
            $taskWhere .= ' AND task_comment_id=' . $this->db->escape_str($comment_id);
        }

        if (staff_cant('view', 'tasks')) {
            $taskWhere .= ' AND ' . get_tasks_where_string(false);
        }

        $files = $this->tasks_model->get_task_attachments($task_id, $taskWhere);

        if (count($files) == 0) {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        }

        $path = get_upload_path_by_type('task') . $task_id;

        $this->load->library('zip');

        foreach ($files as $file) {
            $this->zip->read_file($path . '/' . $file['file_name']);
        }

        $this->zip->download('files.zip');
        $this->zip->clear_data();
    }

    /* Add new task follower / ajax */
    public function add_task_followers()
    {
        $task = $this->tasks_model->get($this->input->post('taskid'));

        if (staff_can('edit', 'tasks') ||
                ($task->current_user_is_creator && staff_can('create', 'tasks'))) {
            echo json_encode([
                'success'  => $this->tasks_model->add_task_followers($this->input->post()),
                'taskHtml' => $this->get_task_data($this->input->post('taskid'), true),
            ]);
        }
    }

    /* Add task assignees / ajax */
    public function add_task_assignees()
    {
        $task = $this->tasks_model->get($this->input->post('taskid'));

        if (staff_can('edit', 'tasks') ||
                ($task->current_user_is_creator && staff_can('create', 'tasks'))) {
            echo json_encode([
                'success'  => $this->tasks_model->add_task_assignees($this->input->post()),
                'taskHtml' => $this->get_task_data($this->input->post('taskid'), true),
            ]);
        }
    }

    public function edit_comment()
    {
        if ($this->input->post()) {
            $data            = $this->input->post();
            $data['content'] = html_purify($this->input->post('content', false));
            if ($this->input->post('no_editor')) {
                $data['content'] = nl2br(clear_textarea_breaks($this->input->post('content')));
            }
            $success = $this->tasks_model->edit_comment($data);
            $message = '';
            if ($success) {
                $message = _l('task_comment_updated');
            }
            echo json_encode([
                'success'  => $success,
                'message'  => $message,
                'taskHtml' => $this->get_task_data($data['task_id'], true),
            ]);
        }
    }

    /* Remove task comment / ajax */
    public function remove_comment($id)
    {
        echo json_encode([
            'success' => $this->tasks_model->remove_comment($id),
        ]);
    }

    /* Remove assignee / ajax */
    public function remove_assignee($id, $taskid)
    {
        $task = $this->tasks_model->get($taskid);

        if (staff_can('edit', 'tasks') ||
                ($task->current_user_is_creator && staff_can('create', 'tasks'))) {
            $success = $this->tasks_model->remove_assignee($id, $taskid);
            $message = '';
            if ($success) {
                $message = _l('task_assignee_removed');
            }
            echo json_encode([
                'success'  => $success,
                'message'  => $message,
                'taskHtml' => $this->get_task_data($taskid, true),
            ]);
        }
    }

    /* Remove task follower / ajax */
    public function remove_follower($id, $taskid)
    {
        $task = $this->tasks_model->get($taskid);

        if (staff_can('edit', 'tasks') ||
                ($task->current_user_is_creator && staff_can('create', 'tasks'))) {
            $success = $this->tasks_model->remove_follower($id, $taskid);
            $message = '';
            if ($success) {
                $message = _l('task_follower_removed');
            }
            echo json_encode([
                'success'  => $success,
                'message'  => $message,
                'taskHtml' => $this->get_task_data($taskid, true),
            ]);
        }
    }

    /* Unmark task as complete / ajax*/
    public function unmark_complete($id)
    {
        if (
            $this->tasks_model->is_task_assignee(get_staff_user_id(), $id)
            || $this->tasks_model->is_task_creator(get_staff_user_id(), $id)
            || staff_can('edit',  'tasks')
        ) {
            $success = $this->tasks_model->unmark_complete($id);

            // Don't do this query if the action is not performed via task single
            $taskHtml = $this->input->get('single_task') === 'true' ? $this->get_task_data($id, true) : '';

            $message = '';
            if ($success) {
                $message = _l('task_unmarked_as_complete');
            }
            echo json_encode([
                'success'  => $success,
                'message'  => $message,
                'taskHtml' => $taskHtml,
            ]);
        } else {
            echo json_encode([
                'success'  => false,
                'message'  => '',
                'taskHtml' => '',
            ]);
        }
    }

    public function mark_as($status, $id)
    {
        if (
            $this->tasks_model->is_task_assignee(get_staff_user_id(), $id)
            || $this->tasks_model->is_task_creator(get_staff_user_id(), $id)
            || staff_can('edit',  'tasks')
        ) {
            $success = $this->tasks_model->mark_as($status, $id);

            // Don't do this query if the action is not performed via task single
            $taskHtml = $this->input->get('single_task') === 'true' ? $this->get_task_data($id, true) : '';

            $message = '';

            if ($success) {
                $message = _l('task_marked_as_success', format_task_status($status, true, true));
            }

            echo json_encode([
                'success'  => $success,
                'message'  => $message,
                'taskHtml' => $taskHtml,
            ]);
        } else {
            echo json_encode([
                'success'  => false,
                'message'  => '',
                'taskHtml' => '',
            ]);
        }
    }

    public function change_priority($priority_id, $id)
    {
        if (staff_can('edit',  'tasks')) {
            $data = hooks()->apply_filters('before_update_task', ['priority' => $priority_id], $id);

            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'tasks', $data);

            $success = $this->db->affected_rows() > 0 ? true : false;

            hooks()->do_action('after_update_task', $id);

            // Don't do this query if the action is not performed via task single
            $taskHtml = $this->input->get('single_task') === 'true' ? $this->get_task_data($id, true) : '';
            echo json_encode([
                'success'  => $success,
                'taskHtml' => $taskHtml,
            ]);
        } else {
            echo json_encode([
                'success'  => false,
                'taskHtml' => $taskHtml,
            ]);
        }
    }

    public function change_milestone($milestone_id, $id)
    {
        if (staff_can('edit',  'tasks')) {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'tasks', ['milestone' => $milestone_id]);

            $success = $this->db->affected_rows() > 0 ? true : false;
            // Don't do this query if the action is not performed via task single
            $taskHtml = $this->input->get('single_task') === 'true' ? $this->get_task_data($id, true) : '';
            echo json_encode([
                'success'  => $success,
                'taskHtml' => $taskHtml,
            ]);
        } else {
            echo json_encode([
                'success'  => false,
                'taskHtml' => $taskHtml,
            ]);
        }
    }

    public function task_single_inline_update($task_id)
    {
        if (staff_can('edit',  'tasks')) {
            $post_data = $this->input->post();
            foreach ($post_data as $key => $val) {
                $data = hooks()->apply_filters('before_update_task', [
                    $key => to_sql_date($val),
                ], $task_id);

                $this->db->where('id', $task_id);
                $this->db->update(db_prefix() . 'tasks', $data);

                hooks()->do_action('after_update_task', $task_id);
            }
        }
    }

    /* Delete task from database */
    public function delete_task($id)
    {
        if (staff_cant('delete', 'tasks')) {
            access_denied('tasks');
        }
        $success = $this->tasks_model->delete_task($id);
        $message = _l('problem_deleting', _l('task_lowercase'));
        if ($success) {
            $message = _l('deleted', _l('task'));
            set_alert('success', $message);
        } else {
            set_alert('warning', $message);
        }

        if (empty($_SERVER['HTTP_REFERER']) ||
            strpos($_SERVER['HTTP_REFERER'], 'tasks/index') !== false ||
            strpos($_SERVER['HTTP_REFERER'], 'tasks/view') !== false) {
            redirect(admin_url('tasks'));
        } else {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        }
    }

    /**
     * Remove task attachment
     * @since  Version 1.0.1
     * @param  mixed $id attachment it
     * @return json
     */
    public function remove_task_attachment($id)
    {
        if ($this->input->is_ajax_request()) {
            echo json_encode($this->tasks_model->remove_task_attachment($id));
        }
    }

    /**
     * Upload task attachment
     * @since  Version 1.0.1
     */
    public function upload_file()
    {
        if ($this->input->post()) {
            $taskid  = $this->input->post('taskid');
            $files   = handle_task_attachments_array($taskid, 'file');
            $success = false;

            if ($files) {
                $i   = 0;
                $len = count($files);
                foreach ($files as $file) {
                    $success = $this->tasks_model->add_attachment_to_database($taskid, [$file], false, ($i == $len - 1 ? true : false));
                    $i++;
                }
            }

            echo json_encode([
                'success'  => $success,
                'taskHtml' => $this->get_task_data($taskid, true),
            ]);
        }
    }

    public function timer_tracking()
    {
        $task_id   = $this->input->post('task_id');
        $adminStop = $this->input->get('admin_stop') && is_admin() ? true : false;

        if ($adminStop) {
            $this->session->set_flashdata('task_single_timesheets_open', true);
        }

        echo json_encode([
            'success' => $this->tasks_model->timer_tracking(
                $task_id,
                $this->input->post('timer_id'),
                nl2br($this->input->post('note')),
                $adminStop
            ),
            'taskHtml' => $this->input->get('single_task') === 'true' ? $this->get_task_data($task_id, true) : '',
            'timers'   => $this->get_staff_started_timers(true),
        ]);
    }

    public function delete_user_unfinished_timesheet($id)
    {
        $this->db->where('id', $id);
        $timesheet = $this->db->get(db_prefix() . 'taskstimers')->row();
        if ($timesheet && $timesheet->end_time == null && $timesheet->staff_id == get_staff_user_id()) {
            $this->db->where('id', $id);
            $this->db->delete(db_prefix() . 'taskstimers');
        }
        echo json_encode(['timers' => $this->get_staff_started_timers(true)]);
    }

    public function delete_timesheet($id)
    {
        if (staff_can('delete_timesheet', 'tasks') || staff_can('delete_own_timesheet', 'tasks') && total_rows(db_prefix() . 'taskstimers', ['staff_id' => get_staff_user_id(), 'id' => $id]) > 0) {
            $alert_type = 'warning';
            $success    = $this->tasks_model->delete_timesheet($id);
            if ($success) {
                $this->session->set_flashdata('task_single_timesheets_open', true);
                $message = _l('deleted', _l('project_timesheet'));
                set_alert('success', $message);
            }
            if (!$this->input->is_ajax_request()) {
                redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
            }
        }
    }

    public function update_timesheet()
    {
        if ($this->input->is_ajax_request()) {
            if (staff_can('edit_timesheet', 'tasks') || (staff_can('edit_own_timesheet', 'tasks') && total_rows(db_prefix() . 'taskstimers', ['staff_id' => get_staff_user_id(), 'id' => $this->input->post('timer_id')]) > 0)) {
                $success = $this->tasks_model->timesheet($this->input->post());
                if ($success === true) {
                    $this->session->set_flashdata('task_single_timesheets_open', true);
                    $message = _l('updated_successfully', _l('project_timesheet'));
                } else {
                    $message = _l('failed_to_update_timesheet');
                }

                echo json_encode([
                    'success' => $success,
                    'message' => $message,
                ]);
                die;
            }

            echo json_encode([
                'success' => false,
                'message' => _l('access_denied'),
            ]);
            die;
        }
    }

    public function log_time()
    {
        $success = $this->tasks_model->timesheet($this->input->post());
        if ($success === true) {
            $this->session->set_flashdata('task_single_timesheets_open', true);
            $message = _l('added_successfully', _l('project_timesheet'));
        } elseif (is_array($success) && isset($success['end_time_smaller'])) {
            $message = _l('failed_to_add_project_timesheet_end_time_smaller');
        } else {
            $message = _l('project_timesheet_not_updated');
        }

        echo json_encode([
            'success' => $success,
            'message' => $message,
        ]);
        die;
    }

    public function update_tags()
    {
        if (staff_can('create',  'tasks') || staff_can('edit',  'tasks')) {
            $id = $this->input->post('task_id');

            $data = hooks()->apply_filters('before_update_task', [
                'tags' => $this->input->post('tags'),
            ], $id);

            handle_tags_save($data['tags'], $id, 'task');

            hooks()->do_action('after_update_task', $id);
        }
    }

    public function bulk_action()
    {
        hooks()->do_action('before_do_bulk_action_for_tasks');
        $total_deleted = 0;
        if ($this->input->post()) {
            $status    = $this->input->post('status');
            $ids       = $this->input->post('ids');
            $tags      = $this->input->post('tags');
            $assignees = $this->input->post('assignees');
            $milestone = $this->input->post('milestone');
            $priority  = $this->input->post('priority');
            $billable  = $this->input->post('billable');
            $is_admin  = is_admin();
            if (is_array($ids)) {
                foreach ($ids as $id) {
                    if ($this->input->post('mass_delete')) {
                        if (staff_can('delete',  'tasks')) {
                            if ($this->tasks_model->delete_task($id)) {
                                $total_deleted++;
                            }
                        }
                    } else {
                        if ($status) {
                            if (
                                $this->tasks_model->is_task_creator(get_staff_user_id(), $id)
                                || $is_admin
                                || $this->tasks_model->is_task_assignee(get_staff_user_id(), $id)
                            ) {
                                $this->tasks_model->mark_as($status, $id);
                            }
                        }
                        if ($priority || $milestone || ($billable === 'billable' || $billable === 'not_billable')) {
                            $update = [];

                            if ($priority) {
                                $update['priority'] = $priority;
                            }

                            if ($milestone) {
                                $update['milestone'] = $milestone;
                            }

                            if ($billable) {
                                $update['billable'] = $billable === 'billable' ? 1 : 0;
                            }

                            $this->db->where('id', $id);
                            $this->db->update(db_prefix() . 'tasks', $update);
                        }
                        if ($tags) {
                            handle_tags_save($tags, $id, 'task');
                        }
                        if ($assignees) {
                            $notifiedUsers = [];
                            foreach ($assignees as $user_id) {
                                if (!$this->tasks_model->is_task_assignee($user_id, $id)) {
                                    $this->db->select('rel_type,rel_id');
                                    $this->db->where('id', $id);
                                    $task = $this->db->get(db_prefix() . 'tasks')->row();
                                    if ($task->rel_type == 'project') {
                                        // User is we are trying to assign the task is not project member
                                        if (total_rows(db_prefix() . 'project_members', ['project_id' => $task->rel_id, 'staff_id' => $user_id]) == 0) {
                                            $this->db->insert(db_prefix() . 'project_members', ['project_id' => $task->rel_id, 'staff_id' => $user_id]);
                                        }
                                    }
                                    $this->db->insert(db_prefix() . 'task_assigned', [
                                        'staffid'       => $user_id,
                                        'taskid'        => $id,
                                        'assigned_from' => get_staff_user_id(),
                                    ]);
                                    if ($user_id != get_staff_user_id()) {
                                        $notification_data = [
                                            'description' => 'not_task_assigned_to_you',
                                            'touserid'    => $user_id,
                                            'link'        => '#taskid=' . $id,
                                        ];

                                        $notification_data['additional_data'] = serialize([
                                            get_task_subject_by_id($id),
                                        ]);
                                        if (add_notification($notification_data)) {
                                            array_push($notifiedUsers, $user_id);
                                        }
                                    }
                                }
                            }
                            pusher_trigger_notification($notifiedUsers);
                        }
                    }
                }
            }
            if ($this->input->post('mass_delete')) {
                set_alert('success', _l('total_tasks_deleted', $total_deleted));
            }
        }
    }

    public function gantt_date_update($task_id)
    {
        if (staff_can('edit', 'tasks')) {
            $post_data = $this->input->post();
            foreach ($post_data as $key => $val) {
                $this->db->where('id', $task_id);
                $this->db->update(db_prefix() . 'tasks', [$key => $val]);
            }
        }
    }

    public function get_task_by_id($id)
    {
        if ($this->input->is_ajax_request()) {
            $tasks_where = [];
            if (staff_cant('view', 'tasks')) {
                $tasks_where = get_tasks_where_string(false);
            }
            $task = $this->tasks_model->get($id, $tasks_where);
            if (!$task) {
                header('HTTP/1.0 404 Not Found');
                echo 'Task not found';
                die();
            }
            echo json_encode($task);
        }
    }

    public function get_staff_names_for_mentions($taskid)
    {
        if ($this->input->is_ajax_request()) {
            $taskId = $this->db->escape_str($taskid);

            $members = $this->tasks_model->get_staff_members_that_can_access_task($taskId);
            $members = array_map(function ($member) {
                $_member['id'] = $member['staffid'];
                $_member['name'] = e($member['firstname'] . ' ' . $member['lastname']);

                return $_member;
            }, $members);

            echo json_encode($members);
        }
    }

    public function save_checklist_assigned_staff()
    {
        if ($this->input->post() && $this->input->is_ajax_request()) {
            $payload = $this->input->post();
            $item    = $this->tasks_model->get_checklist_item($payload['checklistId']);
            if ($item->addedfrom == get_staff_user_id()
                || is_admin() ||
                $this->tasks_model->is_task_creator(get_staff_user_id(), $payload['taskId'])) {
                $this->tasks_model->update_checklist_assigned_staff($payload);
                die;
            }

            ajax_access_denied();
        }
    }
}

<?php

use app\services\AbstractKanban;
use app\services\tasks\TasksKanban;

defined('BASEPATH') or exit('No direct script access allowed');

class Tasks_model extends App_Model
{
    const STATUS_NOT_STARTED = 1;

    const STATUS_AWAITING_FEEDBACK = 2;

    const STATUS_TESTING = 3;

    const STATUS_IN_PROGRESS = 4;

    const STATUS_COMPLETE = 5;

    public function __construct()
    {
        parent::__construct();
        $this->load->model('projects_model');
        $this->load->model('staff_model');
    }

    // Not used?
    public function get_user_tasks_assigned()
    {
        $this->db->where('id IN (SELECT taskid FROM ' . db_prefix() . 'task_assigned WHERE staffid = ' . get_staff_user_id() . ')');
        $this->db->where('status !=', 5);
        $this->db->order_by('duedate', 'asc');

        return $this->db->get(db_prefix() . 'tasks')->result_array();
    }

    public function get_statuses()
    {
        $statuses = hooks()->apply_filters('before_get_task_statuses', [
            [
                'id'             => static::STATUS_NOT_STARTED,
                'color'          => '#64748b',
                'name'           => _l('task_status_1'),
                'order'          => 1,
                'filter_default' => true,
            ],
            [
                'id'             => static::STATUS_IN_PROGRESS,
                'color'          => '#3b82f6',
                'name'           => _l('task_status_4'),
                'order'          => 2,
                'filter_default' => true,
            ],
            [
                'id'             => static::STATUS_TESTING,
                'color'          => '#0284c7',
                'name'           => _l('task_status_3'),
                'order'          => 3,
                'filter_default' => true,
            ],
            [
                'id'             => static::STATUS_AWAITING_FEEDBACK,
                'color'          => '#84cc16',
                'name'           => _l('task_status_2'),
                'order'          => 4,
                'filter_default' => true,
            ],
            [
                'id'             => static::STATUS_COMPLETE,
                'color'          => '#22c55e',
                'name'           => _l('task_status_5'),
                'order'          => 100,
                'filter_default' => false,
            ],
        ]);

        usort($statuses, function ($a, $b) {
            return $a['order'] - $b['order'];
        });

        return $statuses;
    }

    /**
     * Get task by id
     * @param  mixed $id task id
     * @return object
     */
    public function get($id, $where = [])
    {
        $is_admin = is_admin();
        $this->db->where('id', $id);
        $this->db->where($where);
        $task = $this->db->get(db_prefix() . 'tasks')->row();
        if ($task) {
            $task->comments      = $this->get_task_comments($id);
            $task->assignees     = $this->get_task_assignees($id);
            $task->assignees_ids = [];

            foreach ($task->assignees as $follower) {
                array_push($task->assignees_ids, $follower['assigneeid']);
            }

            $task->followers     = $this->get_task_followers($id);
            $task->followers_ids = [];
            foreach ($task->followers as $follower) {
                array_push($task->followers_ids, $follower['followerid']);
            }

            $task->attachments     = $this->get_task_attachments($id);
            $task->timesheets      = $this->get_timesheeets($id);
            $task->checklist_items = $this->get_checklist_items($id);

            if (is_staff_logged_in()) {
                $task->current_user_is_assigned = $this->is_task_assignee(get_staff_user_id(), $id);
                $task->current_user_is_creator  = $this->is_task_creator(get_staff_user_id(), $id);
            }

            $task->milestone_name = '';

            if ($task->rel_type == 'project') {
                $task->project_data = $this->projects_model->get($task->rel_id);
                if ($task->milestone != 0) {
                    $milestone = $this->get_milestone($task->milestone);
                    if ($milestone) {
                        $task->hide_milestone_from_customer = $milestone->hide_from_customer;
                        $task->milestone_name               = $milestone->name;
                    }
                }
            }
        }

        return hooks()->apply_filters('get_task', $task);
    }

    public function get_milestone($id)
    {
        $this->db->where('id', $id);

        return $this->db->get(db_prefix() . 'milestones')->row();
    }

    public function update_order($data)
    {
        AbstractKanban::updateOrder($data['order'], 'kanban_order', 'tasks', $data['status']);
    }

    public function get_distinct_tasks_years($get_from)
    {
        return $this->db->query('SELECT DISTINCT(YEAR(' . $this->db->escape_str($get_from) . ')) as year FROM ' . db_prefix() . 'tasks WHERE ' . $this->db->escape_str($get_from) . ' IS NOT NULL ORDER BY year DESC')->result_array();
    }

    public function is_task_billed($id)
    {
        return (total_rows(db_prefix() . 'tasks', [
            'id'     => $id,
            'billed' => 1,
        ]) > 0 ? true : false);
    }

    public function copy($data, $overwrites = [])
    {
        $task           = $this->get($data['copy_from']);
        $fields_tasks   = $this->db->list_fields(db_prefix() . 'tasks');
        $_new_task_data = [];

        foreach ($fields_tasks as $field) {
            if (isset($task->$field)) {
                $_new_task_data[$field] = $task->$field;
            }
        }

        unset($_new_task_data['id']);

        if (isset($data['copy_task_status']) && is_numeric($data['copy_task_status'])) {
            $_new_task_data['status'] = $data['copy_task_status'];
        } else {
            // fallback in case no status is provided
            $_new_task_data['status'] = 1;
        }

        $_new_task_data['dateadded']         = date('Y-m-d H:i:s');
        $_new_task_data['startdate']         = date('Y-m-d');
        $_new_task_data['deadline_notified'] = 0;
        $_new_task_data['billed']            = 0;
        $_new_task_data['invoice_id']        = 0;
        $_new_task_data['total_cycles']      = 0;
        $_new_task_data['is_recurring_from'] = null;

        if (is_staff_logged_in()) {
            $_new_task_data['addedfrom'] = get_staff_user_id();
        }

        if (!empty($task->duedate)) {
            $dStart                    = new DateTime($task->startdate);
            $dEnd                      = new DateTime($task->duedate);
            $dDiff                     = $dStart->diff($dEnd);
            $_new_task_data['duedate'] = date('Y-m-d', strtotime(date('Y-m-d', strtotime('+' . $dDiff->days . 'DAY'))));
        }

        // Overwrite data options
        if (count($overwrites) > 0) {
            foreach ($overwrites as $key => $val) {
                $_new_task_data[$key] = $val;
            }
        }

        unset($_new_task_data['datefinished']);

        $_new_task_data = hooks()->apply_filters('before_add_task', $_new_task_data);

        $this->db->insert(db_prefix() . 'tasks', $_new_task_data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            $tags = get_tags_in($data['copy_from'], 'task');
            handle_tags_save($tags, $insert_id, 'task');

            if (isset($data['copy_task_assignees']) && $data['copy_task_assignees'] == 'true') {
                $this->copy_task_assignees($data['copy_from'], $insert_id);
            }

            if (isset($data['copy_task_followers']) && $data['copy_task_followers'] == 'true') {
                $this->copy_task_followers($data['copy_from'], $insert_id);
            }

            if (isset($data['copy_task_checklist_items']) && $data['copy_task_checklist_items'] == 'true') {
                $this->copy_task_checklist_items($data['copy_from'], $insert_id);
            }

            if (isset($data['copy_task_attachments']) && $data['copy_task_attachments'] == 'true') {
                $attachments = $this->get_task_attachments($data['copy_from']);

                if (is_dir(get_upload_path_by_type('task') . $data['copy_from'])) {
                    xcopy(get_upload_path_by_type('task') . $data['copy_from'], get_upload_path_by_type('task') . $insert_id);
                }

                foreach ($attachments as $at) {
                    $_at      = [];
                    $_at[]    = $at;
                    $external = false;
                    if (!empty($at['external'])) {
                        $external       = $at['external'];
                        $_at[0]['name'] = $at['file_name'];
                        $_at[0]['link'] = $at['external_link'];
                        if (!empty($at['thumbnail_link'])) {
                            $_at[0]['thumbnailLink'] = $at['thumbnail_link'];
                        }
                    }
                    $this->add_attachment_to_database($insert_id, $_at, $external, false);
                }
            }

            $this->copy_task_custom_fields($data['copy_from'], $insert_id);

            hooks()->do_action('after_add_task', $insert_id);

            return $insert_id;
        }

        return false;
    }

    public function copy_task_followers($from_task, $to_task)
    {
        $followers = $this->get_task_followers($from_task);
        foreach ($followers as $follower) {
            $this->db->insert(db_prefix() . 'task_followers', [
                'taskid'  => $to_task,
                'staffid' => $follower['followerid'],
            ]);
        }
    }

    public function copy_task_assignees($from_task, $to_task)
    {
        $assignees = $this->get_task_assignees($from_task);
        foreach ($assignees as $assignee) {
            $this->db->insert(db_prefix() . 'task_assigned', [
                'taskid'        => $to_task,
                'staffid'       => $assignee['assigneeid'],
                'assigned_from' => get_staff_user_id(),
            ]);
        }
    }

    public function copy_task_checklist_items($from_task, $to_task)
    {
        $checklists = $this->get_checklist_items($from_task);
        foreach ($checklists as $list) {
            $this->db->insert(db_prefix() . 'task_checklist_items', [
                'taskid'      => $to_task,
                'finished'    => 0,
                'description' => $list['description'],
                'dateadded'   => date('Y-m-d H:i:s'),
                'addedfrom'   => $list['addedfrom'],
                'list_order'  => $list['list_order'],
                'assigned'    => $list['assigned'],
            ]);
        }
    }

    public function copy_task_custom_fields($from_task, $to_task)
    {
        $custom_fields = get_custom_fields('tasks');
        foreach ($custom_fields as $field) {
            $value = get_custom_field_value($from_task, $field['id'], 'tasks', false);
            if ($value != '') {
                $this->db->insert(db_prefix() . 'customfieldsvalues', [
                    'relid'   => $to_task,
                    'fieldid' => $field['id'],
                    'fieldto' => 'tasks',
                    'value'   => $value,
                ]);
            }
        }
    }

    public function get_billable_tasks($customer_id = false, $project_id = '')
    {
        $has_permission_view = staff_can('view',  'tasks');
        $noPermissionsQuery  = get_tasks_where_string(false);

        $this->db->where('billable', 1);
        $this->db->where('billed', 0);

        if ($project_id == '') {
            $this->db->where('rel_type != "project"');
        } else {
            $this->db->where('rel_type', 'project');
            $this->db->where('rel_id', $project_id);
        }

        if ($customer_id != false && $project_id == '') {
            $this->db->where(
                '
                (
                (rel_id IN (SELECT id FROM ' . db_prefix() . 'invoices WHERE clientid=' . $this->db->escape_str($customer_id) . ') AND rel_type="invoice")
                OR
                (rel_id IN (SELECT id FROM ' . db_prefix() . 'estimates WHERE clientid=' . $this->db->escape_str($customer_id) . ') AND rel_type="estimate")
                OR
                (rel_id IN (SELECT id FROM ' . db_prefix() . 'contracts WHERE client=' . $this->db->escape_str($customer_id) . ') AND rel_type="contract")
                OR
                ( rel_id IN (SELECT ticketid FROM ' . db_prefix() . 'tickets WHERE userid=' . $this->db->escape_str($customer_id) . ') AND rel_type="ticket")
                OR
                (rel_id IN (SELECT id FROM ' . db_prefix() . 'expenses WHERE clientid=' . $this->db->escape_str($customer_id) . ') AND rel_type="expense")
                OR
                (rel_id IN (SELECT id FROM ' . db_prefix() . 'proposals WHERE rel_id=' . $this->db->escape_str($customer_id) . ' AND rel_type="customer") AND rel_type="proposal")
                OR
                (rel_id IN (SELECT userid FROM ' . db_prefix() . 'clients WHERE userid=' . $this->db->escape_str($customer_id) . ') AND rel_type="customer")
                )'
            );
        }

        if (!$has_permission_view) {
            $this->db->where($noPermissionsQuery);
        }

        $tasks = $this->db->get(db_prefix() . 'tasks')->result_array();

        $i = 0;
        foreach ($tasks as $task) {
            $task_rel_data         = get_relation_data($task['rel_type'], $task['rel_id']);
            $task_rel_value        = get_relation_values($task_rel_data, $task['rel_type']);
            $tasks[$i]['rel_name'] = $task_rel_value['name'];
            if (total_rows(db_prefix() . 'taskstimers', [
                'task_id' => $task['id'],
                'end_time' => null,
            ]) > 0) {
                $tasks[$i]['started_timers'] = true;
            } else {
                $tasks[$i]['started_timers'] = false;
            }
            $i++;
        }

        return $tasks;
    }

    public function get_billable_amount($taskId)
    {
        $data = $this->get_billable_task_data($taskId);

        return app_format_number($data->total_hours * $data->hourly_rate);
    }

    public function get_billable_task_data($task_id)
    {
        $this->db->where('id', $task_id);
        $data = $this->db->get(db_prefix() . 'tasks')->row();
        if ($data->rel_type == 'project') {
            $this->db->select('billing_type,project_rate_per_hour,name');
            $this->db->where('id', $data->rel_id);
            $project      = $this->db->get(db_prefix() . 'projects')->row();
            $billing_type = get_project_billing_type($data->rel_id);

            if ($project->billing_type == 2) {
                $data->hourly_rate = $project->project_rate_per_hour;
            }

            $data->name = $project->name . ' - ' . $data->name;
        }
        $total_seconds       = task_timer_round($this->calc_task_total_time($task_id));
        $data->total_hours   = sec2qty($total_seconds);
        $data->total_seconds = $total_seconds;

        return $data;
    }

    public function get_tasks_by_staff_id($id, $where = [])
    {
        $this->db->where($where);
        $this->db->where('(id IN (SELECT taskid FROM ' . db_prefix() . 'task_assigned WHERE staffid=' . $this->db->escape_str($id) . '))');

        return $this->db->get(db_prefix() . 'tasks')->result_array();
    }

    /**
     * Add new staff task
     * @param array $data task $_POST data
     * @return mixed
     */
    public function add($data, $clientRequest = false)
    {
        $fromTicketId = null;

        if (isset($data['ticket_to_task'])) {
            $fromTicketId = $data['ticket_to_task'];
            unset($data['ticket_to_task']);
        }

        $data['startdate']             = to_sql_date($data['startdate']);
        $data['duedate']               = to_sql_date($data['duedate']);
        $data['dateadded']             = date('Y-m-d H:i:s');
        $data['addedfrom']             = $clientRequest == false ? get_staff_user_id() : get_contact_user_id();
        $data['is_added_from_contact'] = $clientRequest == false ? 0 : 1;

        $checklistItems = [];
        if (isset($data['checklist_items']) && count($data['checklist_items']) > 0) {
            $checklistItems = $data['checklist_items'];
            unset($data['checklist_items']);
        }

        if ($clientRequest == false) {
            $defaultStatus = get_option('default_task_status');
            if ($defaultStatus == 'auto') {
                if (date('Y-m-d') >= $data['startdate']) {
                    $data['status'] = 4;
                } else {
                    $data['status'] = 1;
                }
            } else {
                $data['status'] = $defaultStatus;
            }
        } else {
            // When client create task the default status is NOT STARTED
            // After staff will get the task will change the status
            $data['status'] = 1;
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            unset($data['custom_fields']);
        }

        if (isset($data['is_public'])) {
            $data['is_public'] = 1;
        } else {
            $data['is_public'] = 0;
        }

        if (isset($data['repeat_every']) && $data['repeat_every'] != '') {
            $data['recurring'] = 1;
            if ($data['repeat_every'] == 'custom') {
                $data['repeat_every']     = $data['repeat_every_custom'];
                $data['recurring_type']   = $data['repeat_type_custom'];
                $data['custom_recurring'] = 1;
            } else {
                $_temp                    = explode('-', $data['repeat_every']);
                $data['recurring_type']   = $_temp[1];
                $data['repeat_every']     = $_temp[0];
                $data['custom_recurring'] = 0;
            }
        } else {
            $data['recurring']    = 0;
            $data['repeat_every'] = null;
        }

        if (isset($data['repeat_type_custom']) && isset($data['repeat_every_custom'])) {
            unset($data['repeat_type_custom']);
            unset($data['repeat_every_custom']);
        }

        if (is_client_logged_in() || $clientRequest) {
            $data['visible_to_client'] = 1;
        } else {
            if (isset($data['visible_to_client'])) {
                $data['visible_to_client'] = 1;
            } else {
                $data['visible_to_client'] = 0;
            }
        }

        if (isset($data['billable'])) {
            $data['billable'] = 1;
        } else {
            $data['billable'] = 0;
        }

        if ((!isset($data['milestone']) || $data['milestone'] == '') || (isset($data['milestone']) && $data['milestone'] == '')) {
            $data['milestone'] = 0;
        } else {
            if ($data['rel_type'] != 'project') {
                $data['milestone'] = 0;
            }
        }
        if (empty($data['rel_type'])) {
            unset($data['rel_type']);
            unset($data['rel_id']);
        } else {
            if (empty($data['rel_id'])) {
                unset($data['rel_type']);
                unset($data['rel_id']);
            }
        }

        $withDefaultAssignee = true;
        if (isset($data['withDefaultAssignee'])) {
            $withDefaultAssignee = $data['withDefaultAssignee'];
            unset($data['withDefaultAssignee']);
        }

        $data = hooks()->apply_filters('before_add_task', $data);

        $tags = '';
        if (isset($data['tags'])) {
            $tags = $data['tags'];
            unset($data['tags']);
        }

        if (isset($data['assignees'])) {
            $assignees = $data['assignees'];
            unset($data['assignees']);
        }

        if (isset($data['followers'])) {
            $followers = $data['followers'];
            unset($data['followers']);
        }

        $this->db->insert(db_prefix() . 'tasks', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            foreach ($checklistItems as $key => $chkID) {
                if ($chkID != '') {
                    $itemTemplate = $this->get_checklist_template($chkID);
                    $this->db->insert(db_prefix() . 'task_checklist_items', [
                        'description' => $itemTemplate->description,
                        'taskid'      => $insert_id,
                        'dateadded'   => date('Y-m-d H:i:s'),
                        'addedfrom'   => get_staff_user_id(),
                        'list_order'  => $key,
                    ]);
                }
            }
            handle_tags_save($tags, $insert_id, 'task');
            if (isset($custom_fields)) {
                handle_custom_fields_post($insert_id, $custom_fields);
            }

            if (isset($data['rel_type']) && $data['rel_type'] == 'lead') {
                $this->load->model('leads_model');
                $this->leads_model->log_lead_activity($data['rel_id'], 'not_activity_new_task_created', false, serialize([
                    '<a href="' . admin_url('tasks/view/' . $insert_id) . '" onclick="init_task_modal(' . $insert_id . ');return false;">' . $data['name'] . '</a>',
                ]));
            }

            if ($clientRequest == false) {
                if (isset($assignees)) {
                    foreach ($assignees as $staff_id) {
                        $this->add_task_assignees([
                            'taskid'   => $insert_id,
                            'assignee' => $staff_id,
                        ]);
                    }
                }
                // else {
                //     $new_task_auto_assign_creator = (get_option('new_task_auto_assign_current_member') == '1' ? true : false);

                //     if ( isset($data['rel_type'])
                //         && $data['rel_type'] == 'project'
                //         && !$this->projects_model->is_member($data['rel_id'])
                //         || !$withDefaultAssignee
                //         ) {
                //         $new_task_auto_assign_creator = false;
                //     }
                //     if ($new_task_auto_assign_creator == true) {
                //         $this->db->insert(db_prefix() . 'task_assigned', [
                //             'taskid'        => $insert_id,
                //             'staffid'       => get_staff_user_id(),
                //             'assigned_from' => get_staff_user_id(),
                //         ]);
                //     }
                // }

                if (isset($followers)) {
                    foreach ($followers as $staff_id) {
                        $this->add_task_followers([
                            'taskid'   => $insert_id,
                            'follower' => $staff_id,
                        ]);
                    }
                }
                //  else {
                //     if (get_option('new_task_auto_follower_current_member') == '1') {
                //         $this->db->insert(db_prefix() . 'task_followers', [
                //             'taskid'  => $insert_id,
                //             'staffid' => get_staff_user_id(),
                //         ]);
                //     }
                // }

                if ($fromTicketId !== null) {
                    $ticket_attachments = $this->db->query('SELECT * FROM ' . db_prefix() . 'ticket_attachments WHERE ticketid=' . $this->db->escape_str($fromTicketId) . ' OR (ticketid=' . $this->db->escape_str($fromTicketId) . ' AND replyid IN (SELECT id FROM ' . db_prefix() . 'ticket_replies WHERE ticketid=' . $this->db->escape_str($fromTicketId) . '))')->result_array();

                    if (count($ticket_attachments) > 0) {
                        $task_path = get_upload_path_by_type('task') . $insert_id . '/';
                        _maybe_create_upload_path($task_path);

                        foreach ($ticket_attachments as $ticket_attachment) {
                            $path = get_upload_path_by_type('ticket') . $fromTicketId . '/' . $ticket_attachment['file_name'];
                            if (file_exists($path)) {
                                $f = fopen($path, FOPEN_READ);
                                if ($f) {
                                    $filename = unique_filename($task_path, $ticket_attachment['file_name']);
                                    $fpt      = fopen($task_path . $filename, 'w');
                                    if ($fpt && fwrite($fpt, stream_get_contents($f))) {
                                        $this->db->insert(db_prefix() . 'files', [
                                            'rel_id'         => $insert_id,
                                            'rel_type'       => 'task',
                                            'file_name'      => $filename,
                                            'filetype'       => $ticket_attachment['filetype'],
                                            'staffid'        => get_staff_user_id(),
                                            'dateadded'      => date('Y-m-d H:i:s'),
                                            'attachment_key' => app_generate_hash(),
                                        ]);
                                    }
                                    if ($fpt) {
                                        fclose($fpt);
                                    }
                                    fclose($f);
                                }
                            }
                        }
                    }
                }
            }

            log_activity('New Task Added [ID:' . $insert_id . ', Name: ' . $data['name'] . ']');
            hooks()->do_action('after_add_task', $insert_id);

            return $insert_id;
        }

        return false;
    }

    /**
     * Update task data
     * @param  array $data task data $_POST
     * @param  mixed $id   task id
     * @return boolean
     */
    public function update($data, $id, $clientRequest = false)
    {
        $affectedRows      = 0;
        $data['startdate'] = to_sql_date($data['startdate']);
        $data['duedate']   = to_sql_date($data['duedate']);

        $checklistItems = [];
        if (isset($data['checklist_items']) && count($data['checklist_items']) > 0) {
            $checklistItems = $data['checklist_items'];
            unset($data['checklist_items']);
        }

        if (isset($data['datefinished'])) {
            $data['datefinished'] = to_sql_date($data['datefinished'], true);
        }

        if ($clientRequest == false) {
            $data['cycles'] = !isset($data['cycles']) ? 0 : $data['cycles'];

            $original_task = $this->get($id);

            // Recurring task set to NO, Cancelled
            if ($original_task->repeat_every != '' && $data['repeat_every'] == '') {
                $data['cycles']              = 0;
                $data['total_cycles']        = 0;
                $data['last_recurring_date'] = null;
            }

            if ($data['repeat_every'] != '') {
                $data['recurring'] = 1;
                if ($data['repeat_every'] == 'custom') {
                    $data['repeat_every']     = $data['repeat_every_custom'];
                    $data['recurring_type']   = $data['repeat_type_custom'];
                    $data['custom_recurring'] = 1;
                } else {
                    $_temp                    = explode('-', $data['repeat_every']);
                    $data['recurring_type']   = $_temp[1];
                    $data['repeat_every']     = $_temp[0];
                    $data['custom_recurring'] = 0;
                }
            } else {
                $data['recurring'] = 0;
            }

            if (isset($data['repeat_type_custom']) && isset($data['repeat_every_custom'])) {
                unset($data['repeat_type_custom']);
                unset($data['repeat_every_custom']);
            }

            if (isset($data['is_public'])) {
                $data['is_public'] = 1;
            } else {
                $data['is_public'] = 0;
            }
            if (isset($data['billable'])) {
                $data['billable'] = 1;
            } else {
                $data['billable'] = 0;
            }

            if (isset($data['visible_to_client'])) {
                $data['visible_to_client'] = 1;
            } else {
                $data['visible_to_client'] = 0;
            }
        }

        if ((!isset($data['milestone']) || $data['milestone'] == '') || (isset($data['milestone']) && $data['milestone'] == '')) {
            $data['milestone'] = 0;
        } else {
            if ($data['rel_type'] != 'project') {
                $data['milestone'] = 0;
            }
        }


        if (empty($data['rel_type'])) {
            $data['rel_id']   = null;
            $data['rel_type'] = null;
        } else {
            if (empty($data['rel_id'])) {
                $data['rel_id']   = null;
                $data['rel_type'] = null;
            }
        }

        $data = hooks()->apply_filters('before_update_task', $data, $id);

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            if (handle_custom_fields_post($id, $custom_fields)) {
                $affectedRows++;
            }
            unset($data['custom_fields']);
        }

        if (isset($data['tags'])) {
            if (handle_tags_save($data['tags'], $id, 'task')) {
                $affectedRows++;
            }
            unset($data['tags']);
        }

        foreach ($checklistItems as $key => $chkID) {
            $itemTemplate = $this->get_checklist_template($chkID);
            $this->db->insert(db_prefix() . 'task_checklist_items', [
                'description' => $itemTemplate->description,
                'taskid'      => $id,
                'dateadded'   => date('Y-m-d H:i:s'),
                'addedfrom'   => get_staff_user_id(),
                'list_order'  => $key,
            ]);
            $affectedRows++;
        }

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'tasks', $data);
        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
        }

        if ($affectedRows > 0) {
            hooks()->do_action('after_update_task', $id);
            log_activity('Task Updated [ID:' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    public function get_checklist_item($id)
    {
        $this->db->where('id', $id);

        return $this->db->get(db_prefix() . 'task_checklist_items')->row();
    }

    public function get_checklist_items($taskid)
    {
        $this->db->where('taskid', $taskid);
        $this->db->order_by('list_order', 'asc');

        return $this->db->get(db_prefix() . 'task_checklist_items')->result_array();
    }

    public function add_checklist_template($description)
    {
        $this->db->insert(db_prefix() . 'tasks_checklist_templates', [
            'description' => $description,
        ]);

        return $this->db->insert_id();
    }

    public function remove_checklist_item_template($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'tasks_checklist_templates');
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    public function get_checklist_templates()
    {
        $this->db->order_by('description', 'asc');

        return $this->db->get(db_prefix() . 'tasks_checklist_templates')->result_array();
    }

    public function get_checklist_template($id)
    {
        $this->db->where('id', $id);

        return $this->db->get(db_prefix() . 'tasks_checklist_templates')->row();
    }

    /**
     * Add task new blank check list item
     * @param mixed $data $_POST data with taxid
     */
    public function add_checklist_item($data)
    {
        $this->db->insert(db_prefix() . 'task_checklist_items', [
            'taskid'      => $data['taskid'],
            'description' => $data['description'],
            'dateadded'   => date('Y-m-d H:i:s'),
            'addedfrom'   => get_staff_user_id(),
            'list_order'  => $data['list_order'] ?? 0,
        ]);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            hooks()->do_action('task_checklist_item_created', ['task_id' => $data['taskid'], 'checklist_id' => $insert_id]);

            return true;
        }

        return false;
    }

    public function delete_checklist_item($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'task_checklist_items');
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    public function update_checklist_order($data)
    {
        foreach ($data['order'] as $order) {
            $this->db->where('id', $order[0]);
            $this->db->update(db_prefix() . 'task_checklist_items', [
                'list_order' => $order[1],
            ]);
        }
    }

    /**
     * Update checklist item
     * @param  mixed $id          check list id
     * @param  mixed $description checklist description
     * @return void
     */
    public function update_checklist_item($id, $description)
    {
        $description = strip_tags($description, '<br>,<br/>');
        if ($description === '') {
            $this->db->where('id', $id);
            $this->db->delete(db_prefix() . 'task_checklist_items');
        } else {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'task_checklist_items', [
                'description' => nl2br($description),
            ]);
        }
    }

    /**
     * Make task public
     * @param  mixed $task_id task id
     * @return boolean
     */
    public function make_public($task_id)
    {
        $this->db->where('id', $task_id);
        $this->db->update(db_prefix() . 'tasks', [
            'is_public' => 1,
        ]);
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    /**
     * Get task creator id
     * @param  mixed $taskid task id
     * @return mixed
     */
    public function get_task_creator_id($taskid)
    {
        $this->db->select('addedfrom');
        $this->db->where('id', $taskid);

        return $this->db->get(db_prefix() . 'tasks')->row()->addedfrom;
    }

    /**
     * Add new task comment
     * @param array $data comment $_POST data
     * @return boolean
     */
    public function add_task_comment($data)
    {
        if (is_client_logged_in()) {
            $data['staffid']    = 0;
            $data['contact_id'] = get_contact_user_id();
        } else {
            $data['staffid']    = get_staff_user_id();
            $data['contact_id'] = 0;
        }

        $this->db->insert(db_prefix() . 'task_comments', [
            'taskid'     => $data['taskid'],
            'content'    => is_client_logged_in() ? _strip_tags($data['content']) : $data['content'],
            'staffid'    => $data['staffid'],
            'contact_id' => $data['contact_id'],
            'dateadded'  => date('Y-m-d H:i:s'),
        ]);

        $insert_id = $this->db->insert_id();

        if ($insert_id) {
            $this->db->select('rel_type,rel_id,name,visible_to_client');
            $this->db->where('id', $data['taskid']);
            $task = $this->db->get(db_prefix() . 'tasks')->row();

            $description     = 'not_task_new_comment';
            $additional_data = serialize([
                $task->name,
            ]);

            if ($task->rel_type == 'project') {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_new_task_comment', $task->name, $task->visible_to_client);
            }

            $regex = "/data\-mention\-id\=\"(\d+)\"/";
            if (preg_match_all($regex, $data['content'], $mentionedStaff, PREG_PATTERN_ORDER)) {
                $this->_send_task_mentioned_users_notification(
                    $description,
                    $data['taskid'],
                    $mentionedStaff[1],
                    'task_new_comment_to_staff',
                    $additional_data,
                    $insert_id
                );
            } else {
                $this->_send_task_responsible_users_notification(
                    $description,
                    $data['taskid'],
                    false,
                    'task_new_comment_to_staff',
                    $additional_data,
                    $insert_id
                );

                $this->db->where('project_id', $task->rel_id);
                $this->db->where('name', 'view_task_comments');
                $project_settings = $this->db->get(db_prefix() . 'project_settings')->row();

                if ($project_settings && $project_settings->value == 1) {
                    $this->_send_customer_contacts_notification($data['taskid'], 'task_new_comment_to_customer');
                }
            }

            hooks()->do_action('task_comment_added', ['task_id' => $data['taskid'], 'comment_id' => $insert_id]);

            return $insert_id;
        }

        return false;
    }

    /**
     * Add task followers
     * @param array $data followers $_POST data
     * @return boolean
     */
    public function add_task_followers($data)
    {
        $this->db->insert(db_prefix() . 'task_followers', [
            'taskid'  => $data['taskid'],
            'staffid' => $data['follower'],
        ]);
        if ($this->db->affected_rows() > 0) {
            $taskName = get_task_subject_by_id($data['taskid']);

            if (get_staff_user_id() != $data['follower']) {
                $notified = add_notification([
                    'description'     => 'not_task_added_you_as_follower',
                    'touserid'        => $data['follower'],
                    'link'            => '#taskid=' . $data['taskid'],
                    'additional_data' => serialize([
                        $taskName,
                    ]),
                ]);

                if ($notified) {
                    pusher_trigger_notification([$data['follower']]);
                }

                $member = $this->staff_model->get($data['follower']);

                send_mail_template('task_added_as_follower_to_staff', $member->email, $data['follower'], $data['taskid']);
            }

            $description = 'not_task_added_someone_as_follower';

            $additional_notification_data = serialize([
                get_staff_full_name($data['follower']),
                $taskName,
            ]);

            if ($data['follower'] == get_staff_user_id()) {
                $additional_notification_data = serialize([
                    $taskName,
                ]);
                $description = 'not_task_added_himself_as_follower';
            }

            $this->_send_task_responsible_users_notification($description, $data['taskid'], $data['follower'], '', $additional_notification_data);

            hooks()->do_action('task_follower_added', [
                'staff_id' => $data['follower'],
                'task_id'  => $data['taskid'],
            ]);

            return true;
        }

        return false;
    }

    /**
     * Assign task to staff
     * @param array $data task assignee $_POST data
     * @return boolean
     */
    public function add_task_assignees($data, $cronOrIntegration = false, $clientRequest = false)
    {
        $assignData = [
            'taskid'  => $data['taskid'],
            'staffid' => $data['assignee'],
        ];
        if ($cronOrIntegration) {
            $assignData['assigned_from'] = $data['assignee'];
        } elseif ($clientRequest) {
            $assignData['is_assigned_from_contact'] = 1;
            $assignData['assigned_from']            = get_contact_user_id();
        } else {
            $assignData['assigned_from'] = get_staff_user_id();
        }
        $this->db->insert(db_prefix() . 'task_assigned', $assignData);

        $assigneeId = $this->db->insert_id();

        if ($assigneeId) {
            $this->db->select('name,visible_to_client,rel_id,rel_type');
            $this->db->where('id', $data['taskid']);
            $task = $this->db->get(db_prefix() . 'tasks')->row();

            if (get_staff_user_id() != $data['assignee'] || $clientRequest) {
                $notification_data = [
                    'description' => ($cronOrIntegration == false ? 'not_task_assigned_to_you' : 'new_task_assigned_non_user'),
                    'touserid'    => $data['assignee'],
                    'link'        => '#taskid=' . $data['taskid'],
                ];

                $notification_data['additional_data'] = serialize([
                    $task->name,
                ]);

                if ($cronOrIntegration) {
                    $notification_data['fromcompany'] = 1;
                }

                if ($clientRequest) {
                    $notification_data['fromclientid'] = get_contact_user_id();
                }

                if (add_notification($notification_data)) {
                    pusher_trigger_notification([$data['assignee']]);
                }

                $member = $this->staff_model->get($data['assignee']);

                send_mail_template('task_assigned_to_staff', $member->email, $data['assignee'], $data['taskid']);
            }

            $description                  = 'not_task_assigned_someone';
            $additional_notification_data = serialize([
                get_staff_full_name($data['assignee']),
                $task->name,
            ]);
            if ($data['assignee'] == get_staff_user_id()) {
                $description                  = 'not_task_will_do_user';
                $additional_notification_data = serialize([
                    $task->name,
                ]);
            }

            if ($task->rel_type == 'project') {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_new_task_assignee', $task->name . ' - ' . get_staff_full_name($data['assignee']), $task->visible_to_client);
            }

            $this->_send_task_responsible_users_notification($description, $data['taskid'], $data['assignee'], '', $additional_notification_data);

            hooks()->do_action('task_assignee_added', [
                'staff_id' => $assigneeId,
                'task_id'  => $data['taskid'],
            ]);

            return $assigneeId;
        }

        return false;
    }

    /**
     * Get all task attachments
     * @param  mixed $taskid taskid
     * @return array
     */
    public function get_task_attachments($taskid, $where = [])
    {
        $this->db->select(implode(', ', prefixed_table_fields_array(db_prefix() . 'files')) . ', ' . db_prefix() . 'task_comments.id as comment_file_id');
        $this->db->where(db_prefix() . 'files.rel_id', $taskid);
        $this->db->where(db_prefix() . 'files.rel_type', 'task');

        if ((is_array($where) && count($where) > 0) || (is_string($where) && $where != '')) {
            $this->db->where($where);
        }

        $this->db->join(db_prefix() . 'task_comments', db_prefix() . 'task_comments.file_id = ' . db_prefix() . 'files.id', 'left');
        $this->db->join(db_prefix() . 'tasks', db_prefix() . 'tasks.id = ' . db_prefix() . 'files.rel_id');
        $this->db->order_by(db_prefix() . 'files.dateadded', 'desc');

        return $this->db->get(db_prefix() . 'files')->result_array();
    }

    /**
     * Remove task attachment from server and database
     * @param  mixed $id attachmentid
     * @return boolean
     */
    public function remove_task_attachment($id)
    {
        $comment_removed = false;
        $deleted         = false;
        // Get the attachment
        $this->db->where('id', $id);
        $attachment = $this->db->get(db_prefix() . 'files')->row();

        if ($attachment) {
            if (empty($attachment->external)) {
                $relPath  = get_upload_path_by_type('task') . $attachment->rel_id . '/';
                $fullPath = $relPath . $attachment->file_name;
                if(file_exists($fullPath)) {
                    unlink($fullPath);
                    $fname     = pathinfo($fullPath, PATHINFO_FILENAME);
                    $fext      = pathinfo($fullPath, PATHINFO_EXTENSION);
                    $thumbPath = $relPath . $fname . '_thumb.' . $fext;
                    if (file_exists($thumbPath)) {
                        unlink($thumbPath);
                    }
                }
            }

            $this->db->where('id', $attachment->id);
            $this->db->delete(db_prefix() . 'files');
            if ($this->db->affected_rows() > 0) {
                $deleted = true;
                log_activity('Task Attachment Deleted [TaskID: ' . $attachment->rel_id . ']');
            }

            if (is_dir(get_upload_path_by_type('task') . $attachment->rel_id)) {
                // Check if no attachments left, so we can delete the folder also
                $other_attachments = list_files(get_upload_path_by_type('task') . $attachment->rel_id);
                if (count($other_attachments) == 0) {
                    // okey only index.html so we can delete the folder also
                    delete_dir(get_upload_path_by_type('task') . $attachment->rel_id);
                }
            }
        }

        if ($deleted) {
            if ($attachment->task_comment_id != 0) {
                $total_comment_files = total_rows(db_prefix() . 'files', ['task_comment_id' => $attachment->task_comment_id]);
                if ($total_comment_files == 0) {
                    $this->db->where('id', $attachment->task_comment_id);
                    $comment = $this->db->get(db_prefix() . 'task_comments')->row();

                    if ($comment) {
                        // Comment is empty and uploaded only with attachments
                        // Now all attachments are deleted, we need to delete the comment too
                        if (empty($comment->content) || $comment->content === '[task_attachment]') {
                            $this->db->where('id', $attachment->task_comment_id);
                            $this->db->delete(db_prefix() . 'task_comments');
                            $comment_removed = $comment->id;
                        } else {
                            $this->db->query('UPDATE ' . db_prefix() . "task_comments
                            SET content = REPLACE(content, '[task_attachment]', '')
                            WHERE id = " . $attachment->task_comment_id);
                        }
                    }
                }
            }

            $this->db->where('file_id', $id);
            $comment_attachment = $this->db->get(db_prefix() . 'task_comments')->row();

            if ($comment_attachment) {
                $this->remove_comment($comment_attachment->id);
            }
        }

        return ['success' => $deleted, 'comment_removed' => $comment_removed];
    }

    /**
     * Add uploaded attachments to database
     * @since  Version 1.0.1
     * @param mixed $taskid     task id
     * @param array $attachment attachment data
     */
    public function add_attachment_to_database($rel_id, $attachment, $external = false, $notification = true)
    {
        $file_id = $this->misc_model->add_attachment_to_database($rel_id, 'task', $attachment, $external);
        if ($file_id) {
            $this->db->select('rel_type,rel_id,name,visible_to_client');
            $this->db->where('id', $rel_id);
            $task = $this->db->get(db_prefix() . 'tasks')->row();

            if ($task->rel_type == 'project') {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_new_task_attachment', $task->name, $task->visible_to_client);
            }

            if ($notification == true) {
                $description = 'not_task_new_attachment';
                $this->_send_task_responsible_users_notification($description, $rel_id, false, 'task_new_attachment_to_staff');
                $this->_send_customer_contacts_notification($rel_id, 'task_new_attachment_to_customer');
            }

            $task_attachment_as_comment = hooks()->apply_filters('add_task_attachment_as_comment', 'true');

            if ($task_attachment_as_comment == 'true') {
                $file = $this->misc_model->get_file($file_id);
                $this->db->insert(db_prefix() . 'task_comments', [
                    'content'    => '[task_attachment]',
                    'taskid'     => $rel_id,
                    'staffid'    => $file->staffid,
                    'contact_id' => $file->contact_id,
                    'file_id'    => $file_id,
                    'dateadded'  => date('Y-m-d H:i:s'),
                ]);
            }

            return true;
        }

        return false;
    }

    /**
     * Get all task followers
     * @param  mixed $id task id
     * @return array
     */
    public function get_task_followers($id)
    {
        $this->db->select('id,' . db_prefix() . 'task_followers.staffid as followerid, CONCAT(firstname, " ", lastname) as full_name');
        $this->db->from(db_prefix() . 'task_followers');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'staff.staffid = ' . db_prefix() . 'task_followers.staffid');
        $this->db->where('taskid', $id);

        return $this->db->get()->result_array();
    }

    /**
     * Get all task assigneed
     * @param  mixed $id task id
     * @return array
     */
    public function get_task_assignees($id)
    {
        $this->db->select('id,' . db_prefix() . 'task_assigned.staffid as assigneeid,assigned_from,firstname,lastname,CONCAT(firstname, " ", lastname) as full_name,is_assigned_from_contact');
        $this->db->from(db_prefix() . 'task_assigned');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'staff.staffid = ' . db_prefix() . 'task_assigned.staffid');
        $this->db->where('taskid', $id);
        $this->db->order_by('firstname', 'asc');

        return $this->db->get()->result_array();
    }

    /**
     * Get task comment
     * @param  mixed $id task id
     * @return array
     */
    public function get_task_comments($id)
    {
        $task_comments_order = hooks()->apply_filters('task_comments_order', 'DESC');

        $this->db->select('id,dateadded,content,' . db_prefix() . 'staff.firstname,' . db_prefix() . 'staff.lastname,' . db_prefix() . 'task_comments.staffid,' . db_prefix() . 'task_comments.contact_id as contact_id,file_id,CONCAT(firstname, " ", lastname) as staff_full_name');
        $this->db->from(db_prefix() . 'task_comments');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'staff.staffid = ' . db_prefix() . 'task_comments.staffid', 'left');
        $this->db->where('taskid', $id);
        $this->db->order_by('dateadded', $task_comments_order);

        $comments = $this->db->get()->result_array();

        $ids = [];
        foreach ($comments as $key => $comment) {
            array_push($ids, $comment['id']);
            $comments[$key]['attachments'] = [];
        }

        if (count($ids) > 0) {
            $allAttachments = $this->get_task_attachments($id, 'task_comment_id IN (' . implode(',', $ids) . ')');
            foreach ($comments as $key => $comment) {
                foreach ($allAttachments as $attachment) {
                    if ($comment['id'] == $attachment['task_comment_id']) {
                        $comments[$key]['attachments'][] = $attachment;
                    }
                }
            }
        }

        return $comments;
    }

    public function edit_comment($data)
    {
        // Check if user really creator
        $this->db->where('id', $data['id']);
        $comment = $this->db->get(db_prefix() . 'task_comments')->row();
        if ($comment->staffid == get_staff_user_id() || staff_can('edit',  'tasks') || $comment->contact_id == get_contact_user_id()) {
            $comment_added = strtotime($comment->dateadded);
            $minus_1_hour  = strtotime('-1 hours');
            if (get_option('client_staff_add_edit_delete_task_comments_first_hour') == 0 || (get_option('client_staff_add_edit_delete_task_comments_first_hour') == 1 && $comment_added >= $minus_1_hour) || is_admin()) {
                if (total_rows(db_prefix() . 'files', ['task_comment_id' => $comment->id]) > 0) {
                    $data['content'] .= '[task_attachment]';
                }

                $this->db->where('id', $data['id']);
                $this->db->update(db_prefix() . 'task_comments', [
                    'content' => $data['content'],
                ]);
                if ($this->db->affected_rows() > 0) {
                    hooks()->do_action('task_comment_updated', [
                        'comment_id' => $comment->id,
                        'task_id'    => $comment->taskid,
                    ]);

                    return true;
                }
            } else {
                return false;
            }

            return false;
        }
    }

    /**
     * Remove task comment from database
     * @param  mixed $id task id
     * @return boolean
     */
    public function remove_comment($id, $force = false)
    {
        // Check if user really creator
        $this->db->where('id', $id);
        $comment = $this->db->get(db_prefix() . 'task_comments')->row();

        if (!$comment) {
            return true;
        }

        if ($comment->staffid == get_staff_user_id() || staff_can('delete',  'tasks') || $comment->contact_id == get_contact_user_id() || $force === true) {
            $comment_added = strtotime($comment->dateadded);
            $minus_1_hour  = strtotime('-1 hours');
            if (
                get_option('client_staff_add_edit_delete_task_comments_first_hour') == 0 || (get_option('client_staff_add_edit_delete_task_comments_first_hour') == 1 && $comment_added >= $minus_1_hour)
                || (is_admin() || $force === true)
            ) {
                $this->db->where('id', $id);
                $this->db->delete(db_prefix() . 'task_comments');
                if ($this->db->affected_rows() > 0) {
                    if ($comment->file_id != 0) {
                        $this->remove_task_attachment($comment->file_id);
                    }

                    $commentAttachments = $this->get_task_attachments($comment->taskid, 'task_comment_id=' . $id);
                    foreach ($commentAttachments as $attachment) {
                        $this->remove_task_attachment($attachment['id']);
                    }

                    hooks()->do_action('task_comment_deleted', ['task_id' => $comment->taskid, 'comment_id' => $id]);

                    return true;
                }
            } else {
                return false;
            }
        }

        return false;
    }

    /**
     * Remove task assignee from database
     * @param  mixed $id     assignee id
     * @param  mixed $taskid task id
     * @return boolean
     */
    public function remove_assignee($id, $taskid)
    {
        $this->db->select('rel_type,rel_id,name,visible_to_client');
        $this->db->where('id', $taskid);
        $task = $this->db->get(db_prefix() . 'tasks')->row();

        $this->db->where('id', $id);
        $assignee_data = $this->db->get(db_prefix() . 'task_assigned')->row();

        // Delete timers
        //   $this->db->where('task_id', $taskid);
        ////   $this->db->where('staff_id', $assignee_data->staffid);
        ///   $this->db->delete(db_prefix().'taskstimers');

        // Stop all timers
        $this->db->where('task_id', $taskid);
        $this->db->where('staff_id', $assignee_data->staffid);
        $this->db->where('end_time IS NULL');
        $this->db->update(db_prefix() . 'taskstimers', ['end_time' => time()]);

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'task_assigned');
        if ($this->db->affected_rows() > 0) {
            if ($task->rel_type == 'project') {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_task_assignee_removed', $task->name . ' - ' . get_staff_full_name($assignee_data->staffid), $task->visible_to_client);
            }

            return true;
        }

        return false;
    }

    /**
     * Remove task follower from database
     * @param  mixed $id     followerid
     * @param  mixed $taskid task id
     * @return boolean
     */
    public function remove_follower($id, $taskid)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'task_followers');
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    /**
     * Change task status
     * @param  mixed $status  task status
     * @param  mixed $task_id task id
     * @return boolean
     */
    public function mark_as($status, $task_id)
    {
        $this->db->select('rel_type,rel_id,name,visible_to_client,status');
        $this->db->where('id', $task_id);
        $task = $this->db->get(db_prefix() . 'tasks')->row();

        if ($task->status == static::STATUS_COMPLETE) {
            return $this->unmark_complete($task_id, $status);
        }

        $update = ['status' => $status];

        if ($status == static::STATUS_COMPLETE) {
            $update['datefinished'] = date('Y-m-d H:i:s');
        }

        $this->db->where('id', $task_id);
        $this->db->update(db_prefix() . 'tasks', $update);
        if ($this->db->affected_rows() > 0) {
            $description = 'not_task_status_changed';

            $not_data = [
                $task->name,
                format_task_status($status, false, true),
            ];

            if ($status == static::STATUS_COMPLETE) {
                $description = 'not_task_marked_as_complete';
                unset($not_data[1]);

                $this->db->where('end_time IS NULL');
                $this->db->where('task_id', $task_id);
                $this->db->update(db_prefix() . 'taskstimers', [
                    'end_time' => time(),
                ]);
            }

            if ($task->rel_type == 'project') {
                $project_activity_log = $status == static::STATUS_COMPLETE ? 'project_activity_task_marked_complete' : 'not_project_activity_task_status_changed';

                $project_activity_desc = $task->name;

                if ($status != static::STATUS_COMPLETE) {
                    $project_activity_desc .= ' - ' . format_task_status($status);
                }

                $this->projects_model->log_activity($task->rel_id, $project_activity_log, $project_activity_desc, $task->visible_to_client);
            }

            $this->_send_task_responsible_users_notification($description, $task_id, false, 'task_status_changed_to_staff', serialize($not_data));

            $this->_send_customer_contacts_notification($task_id, 'task_status_changed_to_customer');
            hooks()->do_action('task_status_changed', ['status' => $status, 'task_id' => $task_id]);

            return true;
        }

        return false;
    }

    /**
     * Unmark task as complete
     * @param  mixed $id task id
     * @return boolean
     */
    public function unmark_complete($id, $force_to_status = false)
    {
        if ($force_to_status != false) {
            $status = $force_to_status;
        } else {
            $status = 1;
            $this->db->select('startdate');
            $this->db->where('id', $id);
            $_task = $this->db->get(db_prefix() . 'tasks')->row();
            if (date('Y-m-d') > date('Y-m-d', strtotime($_task->startdate))) {
                $status = 4;
            }
        }

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'tasks', [
            'datefinished' => null,
            'status'       => $status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->db->select('rel_type,rel_id,name,visible_to_client');
            $this->db->where('id', $id);
            $task = $this->db->get(db_prefix() . 'tasks')->row();

            if ($task->rel_type == 'project') {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_task_unmarked_complete', $task->name, $task->visible_to_client);
            }

            $description = 'not_task_unmarked_as_complete';

            $this->_send_task_responsible_users_notification('not_task_unmarked_as_complete', $id, false, 'task_status_changed_to_staff', serialize([
                $task->name,
            ]));

            hooks()->do_action('task_status_changed', ['status' => $status, 'task_id' => $id]);

            return true;
        }

        return false;
    }

    /**
     * Delete task and all connections
     * @param  mixed $id taskid
     * @return boolean
     */
    public function delete_task($id, $log_activity = true)
    {
        $this->db->select('rel_type,rel_id,name,visible_to_client');
        $this->db->where('id', $id);
        $task = $this->db->get(db_prefix() . 'tasks')->row();

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'tasks');
        if ($this->db->affected_rows() > 0) {

            // Log activity only if task is deleted indivudual not when deleting all projects
            if ($task->rel_type == 'project' && $log_activity == true) {
                $this->projects_model->log_activity($task->rel_id, 'project_activity_task_deleted', $task->name, $task->visible_to_client);
            }

            $this->db->where('taskid', $id);
            $this->db->delete(db_prefix() . 'task_followers');

            $this->db->where('taskid', $id);
            $this->db->delete(db_prefix() . 'task_assigned');

            $this->db->where('taskid', $id);
            $comments = $this->db->get(db_prefix() . 'task_comments')->result_array();
            foreach ($comments as $comment) {
                $this->remove_comment($comment['id'], true);
            }

            $this->db->where('taskid', $id);
            $this->db->delete(db_prefix() . 'task_checklist_items');

            // Delete the custom field values
            $this->db->where('relid', $id);
            $this->db->where('fieldto', 'tasks');
            $this->db->delete(db_prefix() . 'customfieldsvalues');

            $this->db->where('task_id', $id);
            $this->db->delete(db_prefix() . 'taskstimers');

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'task');
            $this->db->delete(db_prefix() . 'taggables');

            $this->db->where('rel_type', 'task');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'reminders');

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'task');
            $attachments = $this->db->get(db_prefix() . 'files')->result_array();
            foreach ($attachments as $at) {
                $this->remove_task_attachment($at['id']);
            }

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'task');
            $this->db->delete(db_prefix() . 'related_items');

            if (is_dir(get_upload_path_by_type('task') . $id)) {
                delete_dir(get_upload_path_by_type('task') . $id);
            }

            $this->db->where('meta_key', 'task-hide-completed-items-' . $id);
            $this->db->delete(db_prefix() . 'user_meta');

            hooks()->do_action('task_deleted', $id);

            return true;
        }

        return false;
    }

    /**
     * Send notification on task activity to creator,follower/s,assignee/s
     * @param  string  $description notification description
     * @param  mixed  $taskid      task id
     * @param  boolean $excludeid   excluded staff id to not send the notifications
     * @return boolean
     */
    private function _send_task_responsible_users_notification($description, $taskid, $excludeid = false, $email_template = '', $additional_notification_data = '', $comment_id = false)
    {
        $this->load->model('staff_model');

        $staff = $this->staff_model->get('', ['active' => 1]);

        $notifiedUsers = [];
        foreach ($staff as $member) {
            if (is_numeric($excludeid)) {
                if ($excludeid == $member['staffid']) {
                    continue;
                }
            }
            if (!is_client_logged_in()) {
                if ($member['staffid'] == get_staff_user_id()) {
                    continue;
                }
            }

            if ($this->should_staff_receive_notification($member['staffid'], $taskid)) {
                $link = '#taskid=' . $taskid;

                if ($comment_id) {
                    $link .= '#comment_' . $comment_id;
                }

                $notified = add_notification([
                    'description'     => $description,
                    'touserid'        => $member['staffid'],
                    'link'            => $link,
                    'additional_data' => $additional_notification_data,
                ]);

                if ($notified) {
                    array_push($notifiedUsers, $member['staffid']);
                }

                if ($email_template != '') {
                    send_mail_template($email_template, $member['email'], $member['staffid'], $taskid);
                }
            }
        }

        pusher_trigger_notification($notifiedUsers);
    }

    public function _send_customer_contacts_notification($taskid, $template_name)
    {
        $this->db->select('rel_id,visible_to_client,rel_type');
        $this->db->from(db_prefix() . 'tasks');
        $this->db->where('id', $taskid);
        $task = $this->db->get()->row();

        if ($task->rel_type == 'project') {
            $this->db->where('project_id', $task->rel_id);
            $this->db->where('name', 'view_tasks');
            $project_settings = $this->db->get(db_prefix() . 'project_settings')->row();
            if ($project_settings) {
                if ($project_settings->value == 1 && $task->visible_to_client == 1) {
                    $this->load->model('clients_model');
                    $contacts = $this->clients_model->get_contacts_for_project_notifications($project_settings->project_id, 'task_emails');
                    foreach ($contacts as $contact) {
                        if (is_client_logged_in() && get_contact_user_id() == $contact['id']) {
                            continue;
                        }
                        send_mail_template($template_name, $contact['email'], $contact['userid'], $contact['id'], $taskid);
                    }
                }
            }
        }
    }

    /**
     * Check if user has commented on task
     * @param  mixed $userid staff id to check
     * @param  mixed $taskid task id
     * @return boolean
     */
    public function staff_has_commented_on_task($userid, $taskid)
    {
        return total_rows(db_prefix() . 'task_comments', [
            'staffid' => $userid,
            'taskid'  => $taskid,
        ]) > 0;
    }

    /**
     * Check is user is task follower
     * @param  mixed  $userid staff id
     * @param  mixed  $taskid taskid
     * @return boolean
     */
    public function is_task_follower($userid, $taskid)
    {
        return total_rows(db_prefix() . 'task_followers', [
            'staffid' => $userid,
            'taskid'  => $taskid,
        ]) > 0;
    }

    /**
     * Check is user is task assignee
     * @param  mixed  $userid staff id
     * @param  mixed  $taskid taskid
     * @return boolean
     */
    public function is_task_assignee($userid, $taskid)
    {
        return total_rows(db_prefix() . 'task_assigned', [
            'staffid' => $userid,
            'taskid'  => $taskid,
        ]) > 0;
    }

    /**
     * Check is user is task creator
     * @param  mixed  $userid staff id
     * @param  mixed  $taskid taskid
     * @return boolean
     */
    public function is_task_creator($userid, $taskid)
    {
        return total_rows(db_prefix() . 'tasks', [
            'addedfrom'             => $userid,
            'id'                    => $taskid,
            'is_added_from_contact' => 0,
        ]) > 0;
    }

    /**
     * Timer action, START/STOP Timer
     * @param  mixed  $task_id   task id
     * @param  mixed  $timer_id  timer_id to stop the timer
     * @param  string  $note      note for timer
     * @param  boolean $adminStop is admin want to stop timer from another staff member
     * @return boolean
     */
    public function timer_tracking($task_id = '', $timer_id = '', $note = '', $adminStop = false)
    {
        if ($task_id == '' && $timer_id == '') {
            return false;
        }

        if ($task_id !== '0' && $adminStop == false) {
            if (!$this->is_task_assignee(get_staff_user_id(), $task_id)) {
                return false;
            } elseif ($this->is_task_billed($task_id)) {
                return false;
            }
        }

        $timer = $this->get_task_timer([
            'id' => $timer_id,
        ]);

        $newTimer = false;

        if ($timer == null) {
            $newTimer = true;
        }

        if ($newTimer) {
            $this->db->select('hourly_rate');
            $this->db->from(db_prefix() . 'staff');
            $this->db->where('staffid', get_staff_user_id());
            $hourly_rate = $this->db->get()->row()->hourly_rate;

            $this->db->insert(db_prefix() . 'taskstimers', [
                'start_time'  => time(),
                'staff_id'    => get_staff_user_id(),
                'task_id'     => $task_id,
                'hourly_rate' => $hourly_rate,
                'note'        => ($note != '' ? $note : null),
            ]);

            $_new_timer_id = $this->db->insert_id();

            if (get_option('auto_stop_tasks_timers_on_new_timer') == 1) {
                $this->db->where('id !=', $_new_timer_id);
                $this->db->where('end_time IS NULL');
                $this->db->where('task_id !=', '0');
                $this->db->where('staff_id', get_staff_user_id());
                $this->db->update(db_prefix() . 'taskstimers', [
                    'end_time' => time(),
                    'note'     => ($note != '' ? $note : null),
                ]);
            }

            if (
                $task_id != '0'
                && get_option('timer_started_change_status_in_progress') == '1'
                && total_rows(db_prefix() . 'tasks', ['id' => $task_id, 'status' => 1]) > 0
            ) {
                $this->mark_as(static::STATUS_IN_PROGRESS, $task_id);
            }

            hooks()->do_action('task_timer_started', ['task_id' => $task_id, 'timer_id' => $_new_timer_id]);

            return true;
        }

        if ($timer) {
            // time already ended
            if ($timer->end_time != null) {
                return false;
            }

            $end_time = hooks()->apply_filters('before_task_timer_stopped', time(), [
                'timer'   => $timer,
                'task_id' => $task_id,
                'note'    => $note,
            ]);

            $this->db->where('id', $timer_id);
            $this->db->update(db_prefix() . 'taskstimers', [
                'end_time' => $end_time,
                'task_id'  => $task_id,
                'note'     => ($note != '' ? $note : null),
            ]);
        }

        return true;
    }

    public function timesheet($data)
    {
        if (isset($data['timesheet_duration']) && $data['timesheet_duration'] != '') {
            $duration_array = explode(':', $data['timesheet_duration']);
            $hour           = $duration_array[0];
            $minutes        = $duration_array[1];
            $end_time       = time();
            $start_time     = strtotime('-' . $hour . ' hour -' . $minutes . ' minutes');
        } else {
            $start_time = to_sql_date($data['start_time'], true);
            $end_time   = to_sql_date($data['end_time'], true);
            $start_time = strtotime($start_time);
            $end_time   = strtotime($end_time);
        }

        if ($end_time < $start_time) {
            return [
                'end_time_smaller' => true,
            ];
        }

        $timesheet_staff_id = get_staff_user_id();
        if (isset($data['timesheet_staff_id']) && $data['timesheet_staff_id'] != '') {
            $timesheet_staff_id = $data['timesheet_staff_id'];
        }

        if (!isset($data['timer_id']) || (isset($data['timer_id']) && $data['timer_id'] == '')) {

            // Stop all other timesheets when adding new timesheet
            $this->db->where('task_id', $data['timesheet_task_id']);
            $this->db->where('staff_id', $timesheet_staff_id);
            $this->db->where('end_time IS NULL');
            $this->db->update(db_prefix() . 'taskstimers', [
                'end_time' => time(),
            ]);

            $this->db->select('hourly_rate');
            $this->db->from(db_prefix() . 'staff');
            $this->db->where('staffid', $timesheet_staff_id);
            $hourly_rate = $this->db->get()->row()->hourly_rate;

            $this->db->insert(db_prefix() . 'taskstimers', [
                'start_time'  => $start_time,
                'end_time'    => $end_time,
                'staff_id'    => $timesheet_staff_id,
                'task_id'     => $data['timesheet_task_id'],
                'hourly_rate' => $hourly_rate,
                'note'        => (isset($data['note']) && $data['note'] != '' ? nl2br($data['note']) : null),
            ]);

            $insert_id = $this->db->insert_id();
            $tags      = '';

            if (isset($data['tags'])) {
                $tags = $data['tags'];
            }

            handle_tags_save($tags, $insert_id, 'timesheet');

            if ($insert_id) {
                $this->db->select('rel_type,rel_id,name,visible_to_client');
                $this->db->where('id', $data['timesheet_task_id']);
                $task = $this->db->get(db_prefix() . 'tasks')->row();

                if ($task->rel_type == 'project') {
                    $total      = $end_time - $start_time;
                    $additional = '<seconds>' . $total . '</seconds>';
                    $additional .= '<br />';
                    $additional .= '<lang>project_activity_task_name</lang> ' . $task->name;
                    $this->projects_model->log_activity($task->rel_id, 'project_activity_recorded_timesheet', $additional, $task->visible_to_client);
                }

                return true;
            }

            return false;
        }
        $affectedRows = 0;
        $this->db->where('id', $data['timer_id']);
        $this->db->update(db_prefix() . 'taskstimers', [
            'start_time' => $start_time,
            'end_time'   => $end_time,
            'staff_id'   => $timesheet_staff_id,
            'task_id'    => $data['timesheet_task_id'],
            'note'       => (isset($data['note']) && $data['note'] != '' ? nl2br($data['note']) : null),
        ]);

        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
        }

        if (isset($data['tags'])) {
            if (handle_tags_save($data['tags'], $data['timer_id'], 'timesheet')) {
                $affectedRows++;
            }
        }

        return ($affectedRows > 0 ? true : false);
    }

    public function get_timers($task_id, $where = [])
    {
        $this->db->where($where);
        $this->db->where('task_id', $task_id);
        $this->db->order_by('start_time', 'DESC');

        return $this->db->get(db_prefix() . 'taskstimers')->result_array();
    }

    public function get_task_timer($where)
    {
        $this->db->where($where);

        return $this->db->get(db_prefix() . 'taskstimers')->row();
    }

    public function is_timer_started($task_id, $staff_id = '')
    {
        if ($staff_id == '') {
            $staff_id = get_staff_user_id();
        }

        $timer = $this->get_last_timer($task_id, $staff_id);

        if (!$timer || $timer->end_time != null) {
            return false;
        }

        return $timer;
    }

    public function is_timer_started_for_task($id, $where = [])
    {
        $this->db->where('task_id', $id);
        $this->db->where('end_time IS NULL');
        $this->db->where($where);
        $results = $this->db->count_all_results(db_prefix() . 'taskstimers');

        return $results > 0;
    }

    public function get_last_timer($task_id, $staff_id = '')
    {
        if ($staff_id == '') {
            $staff_id = get_staff_user_id();
        }
        $this->db->where('staff_id', $staff_id);
        $this->db->where('task_id', $task_id);
        $this->db->order_by('id', 'desc');
        $this->db->limit(1);
        $timer = $this->db->get(db_prefix() . 'taskstimers')->row();

        return $timer;
    }

    public function task_tracking_stats($id)
    {
        $loggers    = $this->db->query('SELECT DISTINCT(staff_id) FROM ' . db_prefix() . 'taskstimers WHERE task_id=' . $this->db->escape_str($id))->result_array();
        $labels     = [];
        $labels_ids = [];
        foreach ($loggers as $assignee) {
            array_push($labels, get_staff_full_name($assignee['staff_id']));
            array_push($labels_ids, $assignee['staff_id']);
        }
        $chart = [
            'labels'   => $labels,
            'datasets' => [
                [
                    'label' => _l('task_stats_logged_hours'),
                    'data'  => [],
                ],
            ],
        ];
        $i = 0;
        foreach ($labels_ids as $staffid) {
            $chart['datasets'][0]['data'][$i] = sec2qty($this->calc_task_total_time($id, ' AND staff_id=' . $staffid));
            $i++;
        }

        return $chart;
    }

    public function get_timesheeets($task_id)
    {
        $task_id = $this->db->escape_str($task_id);

        return $this->db->query("SELECT id,note,start_time,end_time,task_id,staff_id, CONCAT(firstname, ' ', lastname) as full_name,
        end_time - start_time time_spent FROM " . db_prefix() . 'taskstimers JOIN ' . db_prefix() . 'staff ON ' . db_prefix() . 'staff.staffid=' . db_prefix() . "taskstimers.staff_id WHERE task_id = '$task_id' ORDER BY start_time DESC")->result_array();
    }

    public function get_time_spent($seconds)
    {
        $minutes = $seconds / 60;
        $hours   = $minutes / 60;
        if ($minutes >= 60) {
            return round($hours, 2);
        } elseif ($seconds > 60) {
            return round($minutes, 2);
        }

        return $seconds;
    }

    public function calc_task_total_time($task_id, $where = '')
    {
        $sql = get_sql_calc_task_logged_time($task_id) . $where;

        $result = $this->db->query($sql)->row();

        if ($result) {
            return $result->total_logged_time;
        }

        return 0;
    }

    public function get_unique_member_logged_task_ids($staff_id, $where = '')
    {
        $sql = 'SELECT DISTINCT(task_id)
        FROM ' . db_prefix() . 'taskstimers WHERE staff_id =' . $staff_id . $where;

        return $this->db->query($sql)->result();
    }

    /**
     * @deprecated
     */
    private function _cal_total_logged_array_from_timers($timers)
    {
        $total = [];
        foreach ($timers as $key => $timer) {
            $_tspent = 0;
            if (is_null($timer->end_time)) {
                $_tspent = time() - $timer->start_time;
            } else {
                $_tspent = $timer->end_time - $timer->start_time;
            }
            $total[] = $_tspent;
        }

        return array_sum($total);
    }

    public function delete_timesheet($id)
    {
        $this->db->where('id', $id);
        $timesheet = $this->db->get(db_prefix() . 'taskstimers')->row();
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'taskstimers');
        if ($this->db->affected_rows() > 0) {
            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'timesheet');
            $this->db->delete(db_prefix() . 'taggables');

            $this->db->select('rel_type,rel_id,name,visible_to_client');
            $this->db->where('id', $timesheet->task_id);
            $task = $this->db->get(db_prefix() . 'tasks')->row();

            if ($task->rel_type == 'project') {
                $additional_data = $task->name;
                $total           = $timesheet->end_time - $timesheet->start_time;
                $additional_data .= '<br /><seconds>' . $total . '</seconds>';
                $this->projects_model->log_activity($task->rel_id, 'project_activity_task_timesheet_deleted', $additional_data, $task->visible_to_client);
            }

            hooks()->do_action('task_timer_deleted', $timesheet);

            log_activity('Timesheet Deleted [' . $id . ']');

            return true;
        }

        return false;
    }

    public function get_reminders($task_id)
    {
        $this->db->where('rel_id', $task_id);
        $this->db->where('rel_type', 'task');
        $this->db->order_by('isnotified,date', 'ASC');

        return $this->db->get(db_prefix() . 'reminders')->result_array();
    }

    /**
     * Check whether the given staff can access the given task
     *
     * @param  int $staff_id
     * @param  int $task_id
     *
     * @return boolean
     */
    public function can_staff_access_task($staff_id, $task_id)
    {
        $retVal              = false;
        $staffCanAccessTasks = $this->get_staff_members_that_can_access_task($task_id);

        foreach ($staffCanAccessTasks as $staff) {
            if ($staff['staffid'] == $staff_id) {
                $retVal = true;

                break;
            }
        }

        return $retVal;
    }

    /**
     * Get the staff members that can view the given task
     *
     * @param  int $taskId
     *
     * @return array
     */
    public function get_staff_members_that_can_access_task($taskId)
    {
        $taskId = $this->db->escape_str($taskId);
        $prefix = db_prefix();

        return $this->db->query("SELECT * FROM {$prefix}staff WHERE (
            admin=1
            OR staffid IN (SELECT staffid FROM {$prefix}task_assigned WHERE taskid=$taskId)
            OR staffid IN (SELECT staffid FROM {$prefix}task_followers WHERE taskid=$taskId)
            OR staffid IN (SELECT addedfrom FROM {$prefix}tasks WHERE id=$taskId AND is_added_from_contact=0)
            OR staffid IN(SELECT staff_id FROM {$prefix}staff_permissions WHERE feature = 'tasks' AND capability='view')
        ) AND active=1")->result_array();
    }

    /**
     * Check whether the given staff should receive notification for
     * the given task
     *
     * @param  int $staffid
     * @param  int $taskid  [description]
     *
     * @return boolean
     */
    private function should_staff_receive_notification($staffid, $taskid)
    {
        if (!$this->can_staff_access_task($staffid, $taskid)) {
            return false;
        }

        return hooks()->apply_filters('should_staff_receive_task_notification', ($this->is_task_assignee($staffid, $taskid)
            || $this->is_task_follower($staffid, $taskid)
            || $this->is_task_creator($staffid, $taskid)
            || $this->staff_has_commented_on_task($staffid, $taskid)), ['staff_id' => $staffid, 'task_id' => $taskid]);
    }

    /**
     * Send notifications on new task comment
     *
     * @param  string $description
     * @param  int $taskid
     * @param  array $staff
     * @param  string $email_template
     * @param  array $notification_data
     * @param  int $comment_id
     *
     * @return void
     */
    private function _send_task_mentioned_users_notification($description, $taskid, $staff, $email_template, $notification_data, $comment_id)
    {
        $staff = array_unique($staff, SORT_NUMERIC);

        $this->load->model('staff_model');
        $notifiedUsers = [];

        foreach ($staff as $staffId) {
            if (!is_client_logged_in()) {
                if ($staffId == get_staff_user_id()) {
                    continue;
                }
            }

            $member = $this->staff_model->get($staffId);

            $link = '#taskid=' . $taskid;

            if ($comment_id) {
                $link .= '#comment_' . $comment_id;
            }

            $notified = add_notification([
                'description'     => $description,
                'touserid'        => $member->staffid,
                'link'            => $link,
                'additional_data' => $notification_data,
            ]);

            if ($notified) {
                array_push($notifiedUsers, $member->staffid);
            }

            if ($email_template != '') {
                send_mail_template($email_template, $member->email, $member->staffid, $taskid);
            }
        }

        pusher_trigger_notification($notifiedUsers);
    }

    public function update_checklist_assigned_staff($data)
    {
        $assigned = $this->db->escape_str($data['assigned']);
        if (!is_numeric($assigned) || $assigned == 0) {
            $assigned = null;
        }
        $this->db->where('id', $data['checklistId']);
        $this->db->update(db_prefix() . 'task_checklist_items', [
            'assigned' => $assigned,
        ]);
    }

    public function do_kanban_query($status, $search = '', $page = 1, $count = false, $where = [])
    {
        _deprecated_function('Tasks_model::do_kanban_query', '2.9.2', 'TasksKanban class');

        $kanBan = (new TasksKanban($status))
            ->search($search)
            ->page($page)
            ->sortBy($sort['sort'] ?? null, $sort['sort_by'] ?? null);

        if ($where) {
            $kanBan->tapQuery(function ($status, $ci) use ($where) {
                $ci->db->where($where);
            });
        }

        if ($count) {
            return $kanBan->countAll();
        }

        return $kanBan->get();
    }
}


this are the controllers for the tasks and proposal i have to make controller functions in deals or use existing one directly in deals to use in deals i dont want to change the existing proposal and tasks

and also for remiders please check and update the data that way 




<?php

use app\services\AbstractKanban;

defined('BASEPATH') or exit('No direct script access allowed');

class Leads_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get lead
     * @param  string $id Optional - leadid
     * @return mixed
     */
    public function get($id = '', $where = [])
    {
        $this->db->select('*,' . db_prefix() . 'leads.name, ' . db_prefix() . 'leads.id,' . db_prefix() . 'leads_status.name as status_name,' . db_prefix() . 'leads_sources.name as source_name');
        $this->db->join(db_prefix() . 'leads_status', db_prefix() . 'leads_status.id=' . db_prefix() . 'leads.status', 'left');
        $this->db->join(db_prefix() . 'leads_sources', db_prefix() . 'leads_sources.id=' . db_prefix() . 'leads.source', 'left');

        $this->db->where($where);
        if (is_numeric($id)) {
            $this->db->where(db_prefix() . 'leads.id', $id);
            $lead = $this->db->get(db_prefix() . 'leads')->row();
            if ($lead) {
                if ($lead->from_form_id != 0) {
                    $lead->form_data = $this->get_form([
                        'id' => $lead->from_form_id,
                    ]);
                }
                $lead->attachments = $this->get_lead_attachments($id);
                $lead->public_url  = leads_public_url($id);
            }

            return $lead;
        }

        return $this->db->get(db_prefix() . 'leads')->result_array();
    }

    /**
     * Get lead by given email
     *
     * @since 2.8.0
     *
     * @param  string $email
     *
     * @return \strClass|null
     */
    public function get_lead_by_email($email)
    {
        $this->db->where('email', $email);
        $this->db->limit(1);

        return $this->db->get('leads')->row();
    }

    /**
     * Add new lead to database
     * @param mixed $data lead data
     * @return mixed false || leadid
     */
    public function add($data)
    {
        if (isset($data['custom_contact_date']) || isset($data['custom_contact_date'])) {
            if (isset($data['contacted_today'])) {
                $data['lastcontact'] = date('Y-m-d H:i:s');
                unset($data['contacted_today']);
            } else {
                $data['lastcontact'] = to_sql_date($data['custom_contact_date'], true);
            }
        }

        if (isset($data['is_public']) && ($data['is_public'] == 1 || $data['is_public'] === 'on')) {
            $data['is_public'] = 1;
        } else {
            $data['is_public'] = 0;
        }

        if (!isset($data['country']) || isset($data['country']) && $data['country'] == '') {
            $data['country'] = 0;
        }

        if (isset($data['custom_contact_date'])) {
            unset($data['custom_contact_date']);
        }

        $data['description'] = nl2br($data['description']);
        $data['dateadded']   = date('Y-m-d H:i:s');
        $data['addedfrom']   = get_staff_user_id();

        $data = hooks()->apply_filters('before_lead_added', $data);

        $tags = '';
        if (isset($data['tags'])) {
            $tags = $data['tags'];
            unset($data['tags']);
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            unset($data['custom_fields']);
        }

        $data['address'] = trim($data['address']);
        $data['address'] = nl2br($data['address']);

        $data['email'] = trim($data['email']);
        $this->db->insert(db_prefix() . 'leads', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Lead Added [ID: ' . $insert_id . ']');
            $this->log_lead_activity($insert_id, 'not_lead_activity_created');

            handle_tags_save($tags, $insert_id, 'lead');

            if (isset($custom_fields)) {
                handle_custom_fields_post($insert_id, $custom_fields);
            }

            $this->lead_assigned_member_notification($insert_id, $data['assigned']);
            hooks()->do_action('lead_created', $insert_id);

            return $insert_id;
        }

        return false;
    }

    public function lead_assigned_member_notification($lead_id, $assigned, $integration = false)
    {
        if (empty($assigned) || $assigned == 0) {
            return;
        }

        if ($integration == false) {
            if ($assigned == get_staff_user_id()) {
                return false;
            }
        }

        $name = $this->db->select('name')->from(db_prefix() . 'leads')->where('id', $lead_id)->get()->row()->name;

        $notification_data = [
            'description'     => ($integration == false) ? 'not_assigned_lead_to_you' : 'not_lead_assigned_from_form',
            'touserid'        => $assigned,
            'link'            => '#leadid=' . $lead_id,
            'additional_data' => ($integration == false ? serialize([
                $name,
            ]) : serialize([])),
        ];

        if ($integration != false) {
            $notification_data['fromcompany'] = 1;
        }

        if (add_notification($notification_data)) {
            pusher_trigger_notification([$assigned]);
        }

        $this->db->select('email');
        $this->db->where('staffid', $assigned);
        $email = $this->db->get(db_prefix() . 'staff')->row()->email;

        send_mail_template('lead_assigned', $lead_id, $email);

        $this->db->where('id', $lead_id);
        $this->db->update(db_prefix() . 'leads', [
            'dateassigned' => date('Y-m-d'),
        ]);

        $not_additional_data = [
            e(get_staff_full_name()),
            '<a href="' . admin_url('profile/' . $assigned) . '" target="_blank">' . e(get_staff_full_name($assigned)) . '</a>',
        ];

        if ($integration == true) {
            unset($not_additional_data[0]);
            array_values(($not_additional_data));
        }

        $not_additional_data = serialize($not_additional_data);

        $not_desc = ($integration == false ? 'not_lead_activity_assigned_to' : 'not_lead_activity_assigned_from_form');
        $this->log_lead_activity($lead_id, $not_desc, $integration, $not_additional_data);

        hooks()->do_action('after_lead_assigned_member_notification_sent', $lead_id);
    }

    /**
     * Update lead
     * @param  array $data lead data
     * @param  mixed $id   leadid
     * @return boolean
     */
    public function update($data, $id)
    {
        $current_lead_data = $this->get($id);
        $current_status    = $this->get_status($current_lead_data->status);
        if ($current_status) {
            $current_status_id = $current_status->id;
            $current_status    = $current_status->name;
        } else {
            if ($current_lead_data->junk == 1) {
                $current_status = _l('lead_junk');
            } elseif ($current_lead_data->lost == 1) {
                $current_status = _l('lead_lost');
            } else {
                $current_status = '';
            }
            $current_status_id = 0;
        }

        $affectedRows = 0;
        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            if (handle_custom_fields_post($id, $custom_fields)) {
                $affectedRows++;
            }
            unset($data['custom_fields']);
        }
        if (!defined('API')) {
            if (isset($data['is_public'])) {
                $data['is_public'] = 1;
            } else {
                $data['is_public'] = 0;
            }

            if (!isset($data['country']) || isset($data['country']) && $data['country'] == '') {
                $data['country'] = 0;
            }

            if (isset($data['description'])) {
                $data['description'] = nl2br($data['description']);
            }
        }

        if (isset($data['lastcontact']) && $data['lastcontact'] == '' || isset($data['lastcontact']) && $data['lastcontact'] == null) {
            $data['lastcontact'] = null;
        } elseif (isset($data['lastcontact'])) {
            $data['lastcontact'] = to_sql_date($data['lastcontact'], true);
        }

        if (isset($data['tags'])) {
            if (handle_tags_save($data['tags'], $id, 'lead')) {
                $affectedRows++;
            }
            unset($data['tags']);
        }

        if (isset($data['remove_attachments'])) {
            foreach ($data['remove_attachments'] as $key => $val) {
                $attachment = $this->get_lead_attachments($id, $key);
                if ($attachment) {
                    $this->delete_lead_attachment($attachment->id);
                }
            }
            unset($data['remove_attachments']);
        }

        $data['address'] = trim($data['address']);
        $data['address'] = nl2br($data['address']);

        $data['email'] = trim($data['email']);

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', $data);
        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
            if (isset($data['status']) && $current_status_id != $data['status']) {
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'leads', [
                    'last_status_change' => date('Y-m-d H:i:s'),
                ]);
                $new_status_name = $this->get_status($data['status'])->name;
                $this->log_lead_activity($id, 'not_lead_activity_status_updated', false, serialize([
                    get_staff_full_name(),
                    $current_status,
                    $new_status_name,
                ]));

                hooks()->do_action('lead_status_changed', [
                    'lead_id'    => $id,
                    'old_status' => $current_status_id,
                    'new_status' => $data['status'],
                ]);
            }

            if (($current_lead_data->junk == 1 || $current_lead_data->lost == 1) && $data['status'] != 0) {
                $this->db->where('id', $id);
                $this->db->update(db_prefix() . 'leads', [
                    'junk' => 0,
                    'lost' => 0,
                ]);
            }

            if (isset($data['assigned'])) {
                if ($current_lead_data->assigned != $data['assigned'] && (!empty($data['assigned']) && $data['assigned'] != 0)) {
                    $this->lead_assigned_member_notification($id, $data['assigned']);
                }
            }
            log_activity('Lead Updated [ID: ' . $id . ']');

            hooks()->do_action('after_lead_updated', $id);

            return true;
        }
        if ($affectedRows > 0) {
            hooks()->do_action('after_lead_updated', $id);
            return true;
        }

        return false;
    }

    /**
     * Delete lead from database and all connections
     * @param  mixed $id leadid
     * @return boolean
     */
    public function delete($id)
    {
        $affectedRows = 0;

        hooks()->do_action('before_lead_deleted', $id);

        $lead = $this->get($id);

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads');
        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Deleted [Deleted by: ' . get_staff_full_name() . ', ID: ' . $id . ']');

            $attachments = $this->get_lead_attachments($id);
            foreach ($attachments as $attachment) {
                $this->delete_lead_attachment($attachment['id']);
            }

            // Delete the custom field values
            $this->db->where('relid', $id);
            $this->db->where('fieldto', 'leads');
            $this->db->delete(db_prefix() . 'customfieldsvalues');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_activity_log');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_integration_emails');

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $this->db->delete(db_prefix() . 'notes');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'reminders');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'taggables');

            $this->load->model('proposals_model');
            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $proposals = $this->db->get(db_prefix() . 'proposals')->result_array();

            foreach ($proposals as $proposal) {
                $this->proposals_model->delete($proposal['id']);
            }

            // Get related tasks
            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $tasks = $this->db->get(db_prefix() . 'tasks')->result_array();
            foreach ($tasks as $task) {
                $this->tasks_model->delete_task($task['id']);
            }

            if (is_gdpr()) {
                $this->db->where('(description LIKE "%' . $lead->email . '%" OR description LIKE "%' . $lead->name . '%" OR description LIKE "%' . $lead->phonenumber . '%")');
                $this->db->delete(db_prefix() . 'activity_log');
            }

            $affectedRows++;
        }
        if ($affectedRows > 0) {
            hooks()->do_action('after_lead_deleted', $id);
            return true;
        }

        return false;
    }

    /**
     * Mark lead as lost
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_lost($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_lost');

            log_activity('Lead Marked as Lost [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_lost', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as lost
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_lost($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_lost');

            log_activity('Lead Unmarked as Lost [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Mark lead as junk
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_junk($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_junk');

            log_activity('Lead Marked as Junk [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_junk', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as junk
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_junk($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_junk');
            log_activity('Lead Unmarked as Junk [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Get lead attachments
     * @since Version 1.0.4
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_attachments($id = '', $attachment_id = '', $where = [])
    {
        $this->db->where($where);
        $idIsHash = !is_numeric($attachment_id) && strlen($attachment_id) == 32;
        if (is_numeric($attachment_id) || $idIsHash) {
            $this->db->where($idIsHash ? 'attachment_key' : 'id', $attachment_id);

            return $this->db->get(db_prefix() . 'files')->row();
        }
        $this->db->where('rel_id', $id);
        $this->db->where('rel_type', 'lead');
        $this->db->order_by('dateadded', 'DESC');

        return $this->db->get(db_prefix() . 'files')->result_array();
    }

    public function add_attachment_to_database($lead_id, $attachment, $external = false, $form_activity = false)
    {
        $this->misc_model->add_attachment_to_database($lead_id, 'lead', $attachment, $external);

        if ($form_activity == false) {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_added_attachment');
        } else {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_log_attachment', true, serialize([
                $form_activity,
            ]));
        }

        // No notification when attachment is imported from web to lead form
        if ($form_activity == false) {
            $lead         = $this->get($lead_id);
            $not_user_ids = [];
            if ($lead->addedfrom != get_staff_user_id()) {
                array_push($not_user_ids, $lead->addedfrom);
            }
            if ($lead->assigned != get_staff_user_id() && $lead->assigned != 0) {
                array_push($not_user_ids, $lead->assigned);
            }
            $notifiedUsers = [];
            foreach ($not_user_ids as $uid) {
                $notified = add_notification([
                    'description'     => 'not_lead_added_attachment',
                    'touserid'        => $uid,
                    'link'            => '#leadid=' . $lead_id,
                    'additional_data' => serialize([
                        $lead->name,
                    ]),
                ]);
                if ($notified) {
                    array_push($notifiedUsers, $uid);
                }
            }
            pusher_trigger_notification($notifiedUsers);
        }
    }

    /**
     * Delete lead attachment
     * @param  mixed $id attachment id
     * @return boolean
     */
    public function delete_lead_attachment($id)
    {
        $attachment = $this->get_lead_attachments('', $id);
        $deleted    = false;

        if ($attachment) {
            if (empty($attachment->external)) {
                unlink(get_upload_path_by_type('lead') . $attachment->rel_id . '/' . $attachment->file_name);
            }
            $this->db->where('id', $attachment->id);
            $this->db->delete(db_prefix() . 'files');
            if ($this->db->affected_rows() > 0) {
                $deleted = true;
                log_activity('Lead Attachment Deleted [ID: ' . $attachment->rel_id . ']');
            }

            if (is_dir(get_upload_path_by_type('lead') . $attachment->rel_id)) {
                // Check if no attachments left, so we can delete the folder also
                $other_attachments = list_files(get_upload_path_by_type('lead') . $attachment->rel_id);
                if (count($other_attachments) == 0) {
                    // okey only index.html so we can delete the folder also
                    delete_dir(get_upload_path_by_type('lead') . $attachment->rel_id);
                }
            }
        }

        return $deleted;
    }

    // Sources

    /**
     * Get leads sources
     * @param  mixed $id Optional - Source ID
     * @return mixed object if id passed else array
     */
    public function get_source($id = false)
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);

            return $this->db->get(db_prefix() . 'leads_sources')->row();
        }

        $this->db->order_by('name', 'asc');

        return $this->db->get(db_prefix() . 'leads_sources')->result_array();
    }

    /**
     * Add new lead source
     * @param mixed $data source data
     */
    public function add_source($data)
    {
        $this->db->insert(db_prefix() . 'leads_sources', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Source Added [SourceID: ' . $insert_id . ', Name: ' . $data['name'] . ']');
        }

        return $insert_id;
    }

    /**
     * Update lead source
     * @param  mixed $data source data
     * @param  mixed $id   source id
     * @return boolean
     */
    public function update_source($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_sources', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Source Updated [SourceID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead source from database
     * @param  mixed $id source id
     * @return mixed
     */
    public function delete_source($id)
    {
        $current = $this->get_source($id);
        // Check if is already using in table
        if (is_reference_in_table('source', db_prefix() . 'leads', $id) || is_reference_in_table('lead_source', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_sources');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_source') == $id) {
                update_option('leads_default_source', '');
            }
            log_activity('Leads Source Deleted [SourceID: ' . $id . ']');

            return true;
        }

        return false;
    }

    // Statuses

    /**
     * Get lead statuses
     * @param  mixed $id status id
     * @return mixed      object if id passed else array
     */
    public function get_status($id = '', $where = [])
    {
        if (is_numeric($id)) {
            $this->db->where($where);
            $this->db->where('id', $id);

            return $this->db->get(db_prefix() . 'leads_status')->row();
        }

        $whereKey = md5(serialize($where));
      
        $statuses = $this->app_object_cache->get('leads-all-statuses-'.$whereKey);

        if (!$statuses) {
            $this->db->where($where);
            $this->db->order_by('statusorder', 'asc');

            $statuses = $this->db->get(db_prefix() . 'leads_status')->result_array();
            $this->app_object_cache->add('leads-all-statuses-'.$whereKey, $statuses);
        }

        return $statuses;
    }

    /**
     * Add new lead status
     * @param array $data lead status data
     */
    public function add_status($data)
    {
        if (isset($data['color']) && $data['color'] == '') {
            $data['color'] = hooks()->apply_filters('default_lead_status_color', '#757575');
        }

        if (!isset($data['statusorder'])) {
            $data['statusorder'] = total_rows(db_prefix() . 'leads_status') + 1;
        }

        $this->db->insert(db_prefix() . 'leads_status', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Status Added [StatusID: ' . $insert_id . ', Name: ' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_status($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_status', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Status Updated [StatusID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead status from database
     * @param  mixed $id status id
     * @return boolean
     */
    public function delete_status($id)
    {
        $current = $this->get_status($id);
        // Check if is already using in table
        if (is_reference_in_table('status', db_prefix() . 'leads', $id) || is_reference_in_table('lead_status', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_status');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_status') == $id) {
                update_option('leads_default_status', '');
            }
            log_activity('Leads Status Deleted [StatusID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Update canban lead status when drag and drop
     * @param  array $data lead data
     * @return boolean
     */
    public function update_lead_status($data)
    {
        $this->db->select('status');
        $this->db->where('id', $data['leadid']);
        $_old = $this->db->get(db_prefix() . 'leads')->row();

        $old_status = '';

        if ($_old) {
            $old_status = $this->get_status($_old->status);
            if ($old_status) {
                $old_status = $old_status->name;
            }
        }

        $affectedRows   = 0;
        $current_status = $this->get_status($data['status'])->name;

        $this->db->where('id', $data['leadid']);
        $this->db->update(db_prefix() . 'leads', [
            'status' => $data['status'],
        ]);

        $_log_message = '';

        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
            if ($current_status != $old_status && $old_status != '') {
                $_log_message    = 'not_lead_activity_status_updated';
                $additional_data = serialize([
                    get_staff_full_name(),
                    $old_status,
                    $current_status,
                ]);

                hooks()->do_action('lead_status_changed', [
                    'lead_id'    => $data['leadid'],
                    'old_status' => $old_status,
                    'new_status' => $current_status,
                ]);
            }
            $this->db->where('id', $data['leadid']);
            $this->db->update(db_prefix() . 'leads', [
                'last_status_change' => date('Y-m-d H:i:s'),
            ]);
        }

        if (isset($data['order'])) {
            AbstractKanban::updateOrder($data['order'], 'leadorder', 'leads', $data['status']);
        }

        if ($affectedRows > 0) {
            if ($_log_message == '') {
                return true;
            }

            $this->log_lead_activity($data['leadid'], $_log_message, false, $additional_data);

            return true;
        }

        return false;
    }

    /* Ajax */

    /**
     * All lead activity by staff
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_activity_log($id)
    {
        $sorting = hooks()->apply_filters('lead_activity_log_default_sort', 'ASC');

        $this->db->where('leadid', $id);
        $this->db->order_by('date', $sorting);

        return $this->db->get(db_prefix() . 'lead_activity_log')->result_array();
    }

    public function staff_can_access_lead($id, $staff_id = '')
    {
        $staff_id = $staff_id == '' ? get_staff_user_id() : $staff_id;

        if (has_permission('leads', $staff_id, 'view')) {
            return true;
        }

        $CI = &get_instance();

        if (total_rows(db_prefix() . 'leads', 'id="' . $CI->db->escape_str($id) . '" AND (assigned=' . $CI->db->escape_str($staff_id) . ' OR is_public=1 OR addedfrom=' . $CI->db->escape_str($staff_id) . ')') > 0) {
            return true;
        }

        return false;
    }

    /**
     * Add lead activity from staff
     * @param  mixed  $id          lead id
     * @param  string  $description activity description
     */
    public function log_lead_activity($id, $description, $integration = false, $additional_data = '')
    {
        $log = [
            'date'            => date('Y-m-d H:i:s'),
            'description'     => $description,
            'leadid'          => $id,
            'staffid'         => get_staff_user_id(),
            'additional_data' => $additional_data,
            'full_name'       => get_staff_full_name(get_staff_user_id()),
        ];
        if ($integration == true) {
            $log['staffid']   = 0;
            $log['full_name'] = '[CRON]';
        }

        $this->db->insert(db_prefix() . 'lead_activity_log', $log);

        return $this->db->insert_id();
    }

    /**
     * Get email integration config
     * @return object
     */
    public function get_email_integration()
    {
        $this->db->where('id', 1);

        return $this->db->get(db_prefix() . 'leads_email_integration')->row();
    }

    /**
     * Get lead imported email activity
     * @param  mixed $id leadid
     * @return array
     */
    public function get_mail_activity($id)
    {
        $this->db->where('leadid', $id);
        $this->db->order_by('dateadded', 'asc');

        return $this->db->get(db_prefix() . 'lead_integration_emails')->result_array();
    }

    /**
     * Update email integration config
     * @param  mixed $data All $_POST data
     * @return boolean
     */
    public function update_email_integration($data)
    {
        $this->db->where('id', 1);
        $original_settings = $this->db->get(db_prefix() . 'leads_email_integration')->row();

        $data['create_task_if_customer']        = isset($data['create_task_if_customer']) ? 1 : 0;
        $data['active']                         = isset($data['active']) ? 1 : 0;
        $data['delete_after_import']            = isset($data['delete_after_import']) ? 1 : 0;
        $data['notify_lead_imported']           = isset($data['notify_lead_imported']) ? 1 : 0;
        $data['only_loop_on_unseen_emails']     = isset($data['only_loop_on_unseen_emails']) ? 1 : 0;
        $data['notify_lead_contact_more_times'] = isset($data['notify_lead_contact_more_times']) ? 1 : 0;
        $data['mark_public']                    = isset($data['mark_public']) ? 1 : 0;
        $data['responsible']                    = !isset($data['responsible']) ? 0 : $data['responsible'];

        if ($data['notify_lead_contact_more_times'] != 0 || $data['notify_lead_imported'] != 0) {
            if (isset($data['notify_type']) && $data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        // Check if not empty $data['password']
        // Get original
        // Decrypt original
        // Compare with $data['password']
        // If equal unset
        // If not encrypt and save
        if (!empty($data['password'])) {
            $or_decrypted = $this->encryption->decrypt($original_settings->password);
            if ($or_decrypted == $data['password']) {
                unset($data['password']);
            } else {
                $data['password'] = $this->encryption->encrypt($data['password']);
            }
        }

        $this->db->where('id', 1);
        $this->db->update(db_prefix() . 'leads_email_integration', $data);
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    public function change_status_color($data)
    {
        $this->db->where('id', $data['status_id']);
        $this->db->update(db_prefix() . 'leads_status', [
            'color' => $data['color'],
        ]);
    }

    public function update_status_order($data)
    {
        foreach ($data['order'] as $status) {
            $this->db->where('id', $status[0]);
            $this->db->update(db_prefix() . 'leads_status', [
                'statusorder' => $status[1],
            ]);
        }
    }

    public function get_form($where)
    {
        $this->db->where($where);

        return $this->db->get(db_prefix() . 'web_to_lead')->row();
    }

    public function add_form($data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);
        $data['form_key']           = app_generate_hash();

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $data['dateadded'] = date('Y-m-d H:i:s');

        $this->db->insert(db_prefix() . 'web_to_lead', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Web to Lead Form Added [' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_form($id, $data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'web_to_lead', $data);

        return ($this->db->affected_rows() > 0 ? true : false);
    }

    public function delete_form($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'web_to_lead');

        $this->db->where('from_form_id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'from_form_id' => 0,
        ]);

        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Form Deleted [' . $id . ']');

            return true;
        }

        return false;
    }

    private function _do_lead_web_to_form_responsibles($data)
    {
        if (isset($data['notify_lead_imported'])) {
            $data['notify_lead_imported'] = 1;
        } else {
            $data['notify_lead_imported'] = 0;
        }

        if ($data['responsible'] == '') {
            $data['responsible'] = 0;
        }
        if ($data['notify_lead_imported'] != 0) {
            if ($data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        return $data;
    }

    public function do_kanban_query($status, $search = '', $page = 1, $sort = [], $count = false)
    {
        _deprecated_function('Leads_model::do_kanban_query', '2.9.2', 'LeadsKanban class');

        $kanBan = (new LeadsKanban($status))
            ->search($search)
            ->page($page)
            ->sortBy($sort['sort'] ?? null, $sort['sort_by'] ?? null);

        if ($count) {
            return $kanBan->countAll();
        }

        return $kanBan->get();
    }
}
