<?php

defined('BASEPATH') or exit('No direct script access allowed');

hooks()->add_action('app_admin_head', 'campaigns_app_admin_head_data');

function campaigns_app_admin_head_data()
{
    ?>
    <script>
        var campaignUniqueValidationFields = <?php echo json_decode(json_encode(get_option('campaign_unique_validation'))); ?>;
        var campaignAttachmentsDropzone;
    </script>
    <?php
}

/**
 * Check if the user is campaign creator
 * @since  Version 1.0.4
 * @param  mixed  $campaignid campaignid
 * @param  mixed  $staff_id staff id (Optional)
 * @return boolean
 */

function is_campaign_creator($campaign_id, $staff_id = '')
{
    if (!is_numeric($staff_id)) {
        $staff_id = get_staff_user_id();
    }

    return total_rows(db_prefix() . 'campaigns', [
        'addedfrom' => $staff_id,
        'id'        => $campaign_id,
    ]) > 0;
}

/**
 * Campaign consent URL
 * @param  mixed $id campaign id
 * @return string
 */
function campaign_consent_url($id)
{
    return site_url('consent/l/' . get_campaign_hash($id));
}

/**
 * Campaign public form URL
 * @param  mixed $id campaign id
 * @return string
 */
function campaigns_public_url($id)
{
    return site_url('forms/l/' . get_campaign_hash($id));
}

/**
 * Get and generate campaign hash if don't exists.
 * @param  mixed $id  campaign id
 * @return string
 */
function get_campaign_hash($id)
{
    $CI   = &get_instance();
    $hash = '';

    $CI->db->select('hash');
    $CI->db->where('id', $id);
    $campaign = $CI->db->get(db_prefix() . 'campaigns')->row();
    if ($campaign) {
        $hash = $campaign->hash;
        if (empty($hash)) {
            $hash = app_generate_hash() . '-' . app_generate_hash();
            $CI->db->where('id', $id);
            $CI->db->update(db_prefix() . 'campaigns', ['hash' => $hash]);
        }
    }

    return $hash;
}

/**
 * Get campaigns summary
 * @return array
 */
function get_campaigns_summary()
{
    $CI = &get_instance();
    if (!class_exists('campaigns_model')) {
        $CI->load->model('campaigns_model');
    }
    $statuses = $CI->campaigns_model->get_status();

    $totalStatuses         = count($statuses);
    $has_permission_view   = staff_can('view',  'campaigns');
    $sql                   = '';
    $whereNoViewPermission = '(addedfrom = ' . get_staff_user_id() . ' OR assigned=' . get_staff_user_id() . ' OR is_public = 1)';

    $statuses[] = [
        'lost'  => true,
        'name'  => _l('lost_campaigns'),
        'color' => '#fc2d42',
    ];

/*    $statuses[] = [
        'junk'  => true,
        'name'  => _l('junk_campaigns'),
        'color' => '',
    ];*/

    foreach ($statuses as $status) {
        $sql .= ' SELECT COUNT(*) as total';
        $sql .= ',SUM(campaign_value) as value';
        $sql .= ' FROM ' . db_prefix() . 'campaigns';

        if (isset($status['lost'])) {
            $sql .= ' WHERE lost=1';
        } elseif (isset($status['junk'])) {
            $sql .= ' WHERE junk=1';
        } else {
            $sql .= ' WHERE status=' . $status['id'];
        }
        if (!$has_permission_view) {
            $sql .= ' AND ' . $whereNoViewPermission;
        }
        $sql .= ' UNION ALL ';
        $sql = trim($sql);
    }

    $result = [];

    // Remove the last UNION ALL
    $sql    = substr($sql, 0, -10);
    $result = $CI->db->query($sql)->result();

    if (!$has_permission_view) {
        $CI->db->where($whereNoViewPermission);
    }

    $total_campaigns = $CI->db->count_all_results(db_prefix() . 'campaigns');

    foreach ($statuses as $key => $status) {
        if (isset($status['lost']) || isset($status['junk'])) {
            $statuses[$key]['percent'] = ($total_campaigns > 0 ? number_format(($result[$key]->total * 100) / $total_campaigns, 2) : 0);
        }

        $statuses[$key]['total'] = $result[$key]->total;
        $statuses[$key]['value'] = $result[$key]->value;
    }

    return $statuses;
}

/**
 * Render campaign status select field with ability to create inline statuses with + sign
 * @param  array  $statuses         current statuses
 * @param  string  $selected        selected status
 * @param  string  $lang_key        the label of the select
 * @param  string  $name            the name of the select
 * @param  array   $select_attrs    additional select attributes
 * @param  boolean $exclude_default whether to exclude default Client status
 * @return string
 */
function render_campaigns_status_select($statuses, $selected = '', $lang_key = '', $name = 'status', $select_attrs = [], $exclude_default = false)
{
    foreach ($statuses as $key => $status) {
        if ($status['isdefault'] == 1) {
            if ($exclude_default == false) {
                $statuses[$key]['option_attributes'] = ['data-subtext' => _l('campaigns_converted_to_client')];
            } else {
                unset($statuses[$key]);
            }

            break;
        }
    }

    if (is_admin() || get_option('staff_members_create_inline_campaign_status') == '1') {
        return render_select_with_input_group($name, $statuses, ['id', 'name'], $lang_key, $selected, '<div class="input-group-btn"><a href="#" class="btn btn-default" onclick="new_campaign_status_inline();return false;" class="inline-field-new"><i class="fa fa-plus"></i></a></div>', $select_attrs);
    }

    return render_select($name, $statuses, ['id', 'name'], $lang_key, $selected, $select_attrs);
}

/**
 * Render campaign source select field with ability to create inline source with + sign
 * @param  array   $sources         current sourcees
 * @param  string  $selected        selected source
 * @param  string  $lang_key        the label of the select
 * @param  string  $name            the name of the select
 * @param  array   $select_attrs    additional select attributes
 * @return string
 */
function render_campaigns_source_select($sources, $selected = '', $lang_key = '', $name = 'source', $select_attrs = [])
{
    if (is_admin() || get_option('staff_members_create_inline_campaign_source') == '1') {
        echo render_select_with_input_group($name, $sources, ['id', 'name'], $lang_key, $selected, '<div class="input-group-btn"><a href="#" class="btn btn-default" onclick="new_campaign_source_inline();return false;" class="inline-field-new"><i class="fa fa-plus"></i></a></div>', $select_attrs);
    } else {
        echo render_select($name, $sources, ['id', 'name'], $lang_key, $selected, $select_attrs);
    }
}

/**
 * Load campaign language
 * Used in public GDPR form
 * @param  string $campaign_id
 * @return string return loaded language
 */
function load_campaign_language($campaign_id)
{
    $CI = & get_instance();
    $CI->db->where('id', $campaign_id);
    $campaign = $CI->db->get(db_prefix() . 'campaigns')->row();

    // Campaign not found or default language already loaded
    if (!$campaign || empty($campaign->default_language)) {
        return false;
    }

    $language = $campaign->default_language;

    if (!file_exists(APPPATH . 'language/' . $language)) {
        return false;
    }

    $CI->lang->is_loaded = [];
    $CI->lang->language  = [];

    $CI->lang->load($language . '_lang', $language);
    load_custom_lang_file($language);
    $CI->lang->set_last_loaded_language($language);

    return true;
}
