jQuery(document).ready(function () {
    console.log('Deals Filter: Script loaded, jQuery version:', jQuery.fn.jquery);

    if (typeof Vue === 'undefined') {
        console.error('Vue.js is not loaded. Please ensure vue.min.js is included.');
        return;
    }

    // Helper: robust JSON parse (like deals_table_order.js)
    function safeParseJSON(response) {
        if (typeof response === 'object' && response !== null) return response;
        if (typeof response !== 'string') return null;
        let trimmed = response.trim();
        try {
            let parsed = JSON.parse(trimmed);
            if (parsed && parsed.final_output) {
                return JSON.parse(parsed.final_output);
            }
            return parsed;
        } catch (e) {
            // fallback: try to extract first JSON object
            let match = trimmed.match(/\{[\s\S]*\}/);
            if (match) {
                try {
                    return JSON.parse(match[0]);
                } catch (e2) {}
            }
        }
        return null;
    }

    // Vue instance for deals filter
    new Vue({
        el: '#vueApp',
        data: {
            selectedPipelines: [],
            activeFilters: [],
            pipelineStages: <?php echo json_encode(array_map(function($pipeline) {
                return [
                    'id' => $pipeline['id'],
                    'name' => $pipeline['name'],
                    'stages' => $this->deals_model->get_stages($pipeline['id']) ?? []
                ];
            }, $pipelines ?? [])) ?: '[]'; ?>,
            defaultDealsRules: <?php echo App_table::find('deals')->rulesJs() ?: '{}'; ?>,
            statuses: <?php echo json_encode($statuses ?? []) ?: '[]'; ?>,
            businessCategories: <?php echo json_encode($business_categories ?? []) ?: '[]'; ?>,
            sources: <?php echo json_encode($sources ?? []) ?: '[]'; ?>,
            ratings: <?php echo json_encode($ratings ?? []) ?: '[]'; ?>
        },
        methods: {
            setDealsPipelineFilter(pipelineId) {
                console.log('Deals Filter: Pipeline filter clicked:', pipelineId);
                const pipeline = this.pipelineStages.find(p => p && p.id == pipelineId);
                if (!pipeline) {
                    console.error('Deals Filter: Pipeline not found:', pipelineId);
                    return;
                }
                if (!this.selectedPipelines.some(p => p.id == pipelineId)) {
                    this.selectedPipelines.push(pipeline);
                    this.addFilter('pipeline_' + pipelineId, pipeline.name, 'pipeline', pipelineId);
                    console.log('Deals Filter: Added pipeline filter:', pipeline.name);
                }
                this.applyFilters();
            },
            setDealsStageFilter(pipelineId, stageId) {
                console.log('Deals Filter: Stage filter clicked:', { pipelineId, stageId });
                const pipeline = this.pipelineStages.find(p => p && p.id == pipelineId);
                if (!pipeline) {
                    console.error('Deals Filter: Pipeline not found for stage:', pipelineId);
                    return;
                }
                const stage = pipeline.stages.find(s => s && s.id == stageId);
                if (!stage) {
                    console.error('Deals Filter: Stage not found:', stageId);
                    return;
                }
                this.addFilter('stage_' + pipelineId + '_' + stageId, stage.name, 'stage', stageId, pipelineId);
                console.log('Deals Filter: Added stage filter:', stage.name);
                this.applyFilters();
            },
            setDealsStatusFilter(statusId) {
                console.log('Deals Filter: Status filter clicked:', statusId);
                const status = this.statuses.find(s => s && s.id == statusId);
                if (!status) {
                    console.error('Deals Filter: Status not found:', statusId);
                    return;
                }
                this.addFilter('status_' + statusId, status.name, 'status', statusId);
                console.log('Deals Filter: Added status filter:', status.name);
                this.applyFilters();
            },
            setDealsCategoryFilter(categoryId) {
                console.log('Deals Filter: Category filter clicked:', categoryId);
                const category = this.businessCategories.find(c => c && c.id == categoryId);
                if (!category) {
                    console.error('Deals Filter: Category not found:', categoryId);
                    return;
                }
                this.addFilter('category_' + categoryId, category.name, 'business_category', categoryId);
                console.log('Deals Filter: Added category filter:', category.name);
                this.applyFilters();
            },
            setDealsSourceFilter(sourceId) {
                console.log('Deals Filter: Source filter clicked:', sourceId);
                const source = this.sources.find(s => s && s.id == sourceId);
                if (!source) {
                    console.error('Deals Filter: Source not found:', sourceId);
                    return;
                }
                this.addFilter('source_' + sourceId, source.name, 'source', sourceId);
                console.log('Deals Filter: Added source filter:', source.name);
                this.applyFilters();
            },
            setDealsRatingFilter(ratingId) {
                console.log('Deals Filter: Rating filter clicked:', ratingId);
                const rating = this.ratings.find(r => r && r.id == ratingId);
                if (!rating) {
                    console.error('Deals Filter: Rating not found:', ratingId);
                    return;
                }
                this.addFilter('rating_' + ratingId, rating.name + ' (' + rating.score + ')', 'rating', ratingId);
                console.log('Deals Filter: Added rating filter:', rating.name);
                this.applyFilters();
            },
            addFilter(id, name, type, value, pipelineId = null) {
                if (!this.activeFilters.some(f => f.id === id)) {
                    this.activeFilters.push({ id, name, type, value, pipelineId });
                    console.log('Deals Filter: Filter added:', { id, name, type, value, pipelineId });
                }
            },
            removeFilter(filterId) {
                console.log('Deals Filter: Removing filter:', filterId);
                this.activeFilters = this.activeFilters.filter(f => f.id !== filterId);
                if (filterId.startsWith('stage_')) {
                    const pipelineId = filterId.split('_')[1];
                    if (!this.activeFilters.some(f => f.pipelineId == pipelineId && f.type === 'stage')) {
                        this.selectedPipelines = this.selectedPipelines.filter(p => p.id != pipelineId);
                        console.log('Deals Filter: Removed pipeline:', pipelineId);
                    }
                }
                this.applyFilters();
            },
            clearFilters() {
                console.log('Deals Filter: Clearing all filters');
                this.activeFilters = [];
                this.selectedPipelines = [];
                this.applyFilters();
            },
            applyFilters() {
                console.log('Deals Filter: Applying filters:', this.activeFilters);
                const table = window.dealsTable;
                if (!table) {
                    console.error('Deals Filter: DataTable not initialized');
                    return;
                }

                // Remove previous dealsFilters search
                $.fn.dataTable.ext.search = $.fn.dataTable.ext.search.filter(f => f.name !== 'dealsFilters');
                // Add new dealsFilters search
                const dealsFilters = function(settings, data, dataIndex) {
                    if (settings.nTable.id !== 'table-deals') return true;

                    const colMap = {
                        pipeline: 4,
                        stage: 5,
                        status: 9,
                        business_category: -1,
                        source: -1,
                        rating: -1,
                        pipeline_relations: 6
                    };

                    for (let filter of this.activeFilters) {
                        if (filter.type === 'pipeline' && data[colMap.pipeline_relations]) {
                            const pipelineName = this.pipelineStages.find(p => p && p.id == filter.value)?.name;
                            if (!pipelineName || !data[colMap.pipeline_relations].includes(pipelineName)) {
                                console.log('Deals Filter: Pipeline filter failed:', { filter, pipelineName, data: data[colMap.pipeline_relations] });
                                return false;
                            }
                        } else if (filter.type === 'stage' && data[colMap.pipeline_relations]) {
                            const stage = this.pipelineStages.find(p => p && p.id == filter.pipelineId)?.stages.find(s => s && s.id == filter.value);
                            if (!stage || !data[colMap.pipeline_relations].includes(stage.name)) {
                                console.log('Deals Filter: Stage filter failed:', { filter, stageName: stage?.name, data: data[colMap.pipeline_relations] });
                                return false;
                            }
                        } else if (filter.type === 'status' && data[colMap.status]) {
                            const status = this.statuses.find(s => s && s.id == filter.value);
                            if (!status || data[colMap.status].replace(/(<([^>]+)>)/gi, "").trim().toLowerCase() !== status.name.toLowerCase()) {
                                console.log('Deals Filter: Status filter failed:', { filter, statusName: status?.name, data: data[colMap.status] });
                                return false;
                            }
                        }
                    }
                    return true;
                }.bind(this);
                dealsFilters.name = 'dealsFilters';
                $.fn.dataTable.ext.search.push(dealsFilters);

                console.log('Deals Filter: Triggering table redraw');
                table.draw(false);
            }
        },
        mounted() {
            console.log('Deals Filter: Vue instance mounted');
            // Initialize DataTable
            if (!$.fn.DataTable.isDataTable('#table-deals')) {
                console.log('Deals Filter: Initializing DataTable');
                window.dealsTable = $('#table-deals').DataTable({
                    order: [[1, 'desc']],
                    pageLength: 25,
                    responsive: true,
                    columnDefs: [{ orderable: false, targets: 0 }],
                    serverSide: true,
                    ajax: {
                        url: '<?php echo admin_url('deals/table'); ?>',
                        type: 'POST',
                        data: function(d) {
                            d.filters = JSON.stringify(this.activeFilters);
                            console.log('Deals Filter: Sending AJAX filters:', d.filters);
                            return d;
                        }.bind(this),
                        error: function(xhr, error, thrown) {
                            console.error('Deals Filter: DataTable AJAX error:', { xhr, error, thrown });
                        }
                    },
                    initComplete: function() {
                        console.log('Deals Filter: DataTable initialized');
                    }
                });
            } else {
                window.dealsTable = $('#table-deals').DataTable();
                console.log('Deals Filter: DataTable already initialized');
            }

            // Initialize selectpickers for bulk actions
            if (typeof $.fn.selectpicker === 'function') {
                $('.selectpicker').selectpicker();
                console.log('Deals Filter: Bootstrap Select initialized');
            } else {
                console.error('Deals Filter: Bootstrap Select not available');
            }

            // Handle mass select all
            $('#mass_select_all').on('change', function() {
                console.log('Deals Filter: Mass select all changed:', $(this).prop('checked'));
                var checked = $(this).prop('checked');
                $('.individual').prop('checked', checked);
                toggleBulkActions();
            });

            // Handle individual checkboxes
            $('.individual').on('change', function() {
                console.log('Deals Filter: Individual checkbox changed:', $(this).val());
                toggleBulkActions();
                var allChecked = $('.individual:checked').length === $('.individual').length;
                $('#mass_select_all').prop('checked', allChecked);
            });

            function toggleBulkActions() {
                var checkedBoxes = $('.individual:checked').length;
                console.log('Deals Filter: Toggling bulk actions:', checkedBoxes);
                if (checkedBoxes > 0) {
                    $('.bulk-actions-btn').removeClass('hide');
                } else {
                    $('.bulk-actions-btn').addClass('hide');
                }
            }
        }
    });
});