/* JavaScript for Deals Table Order module - Robust error handling and JSON parsing for malformed responses */

jQuery(document).ready(function () {
    console.log('Deals Table Order: Script loaded, jQuery version:', jQuery.fn.jquery);
    console.log('Deals Table Order: DataTables version:', (jQuery.fn.DataTable && jQuery.fn.DataTable.version) ? jQuery.fn.DataTable.version : 'Not loaded');

    if (jQuery('.table-deals').length) {
        console.log('Deals Table Order: Deals table detected');
        waitForUtilities(function () {
            injectConfigureColumnsButton();
            waitForDataTable(function () {
                applyCustomColumnOrder();
            });
        }, 10000);
    } else {
        console.log('Deals Table Order: Deals table not found');
    }

    jQuery(document).on('click', '.deals-table-order-btn', function (e) {
        e.preventDefault();
        console.log('Deals Table Order: Configure Columns button clicked');
        loadColumnOrderModal();
    });
});

function waitForUtilities(callback, timeoutMs) {
    let startTime = Date.now();
    function checkUtilities() {
        if (typeof _l !== 'undefined' && typeof admin_url !== 'undefined' && typeof alert_float !== 'undefined') {
            console.log('Deals Table Order: Utilities defined');
            callback();
        } else if (Date.now() - startTime > timeoutMs) {
            console.warn('Deals Table Order: Timeout waiting for utilities');
            callback();
        } else {
            setTimeout(checkUtilities, 100);
        }
    }
    checkUtilities();
}

function waitForDataTable(callback) {
    let attempts = 0;
    const maxAttempts = 200;
    function checkDataTable() {
        const $table = jQuery('.table-deals');
        if ($table.length && jQuery.fn.DataTable && jQuery.fn.DataTable.isDataTable($table)) {
            console.log('Deals Table Order: DataTable initialized');
            callback();
        } else if (attempts >= maxAttempts) {
            console.warn('Deals Table Order: Timeout waiting for DataTable');
            callback();
        } else {
            attempts++;
            setTimeout(checkDataTable, 100);
        }
    }
    checkDataTable();
}

function injectConfigureColumnsButton() {
    if (jQuery('.deals-table-order-btn').length) {
        console.log('Deals Table Order: Button already exists');
        return;
    }

    let $buttonsContainer = jQuery('._buttons');
    if ($buttonsContainer.length) {
        var configureText = typeof _l !== 'undefined' ? _l('deals_table_order_configure') : 'Configure Columns';
        var buttonHtml = `
            <div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
                <a href="#" class="btn btn-default deals-table-order-btn">
                    <i class="fa fa-columns tw-mr-1"></i> ${configureText}
                </a>
            </div>
        `;
        $buttonsContainer.find('.pull-right.deals-search').before(buttonHtml);
        console.log('Deals Table Order: Button injected');
    } else {
        console.error('Deals Table Order: Could not find _buttons container');
    }
}

/**
 * Attempts to extract the first valid JSON object from a string, even if the string contains
 * leading/trailing non-JSON data (such as "null" or XSSI/XSS prefixes).
 * Returns the parsed object, or null if parsing fails.
 */
function safeParseJSON(response) {
    let parsed = null;
    if (typeof response === 'object' && response !== null) {
        return response;
    }

    if (typeof response !== 'string') return null;
    try {
        parsed = JSON.parse(trimmed);
        // If the response has final_output, decode it
        if (parsed && parsed.final_output) {
            return JSON.parse(parsed.final_output);
        }
        return parsed;
    } catch (e) {
        // existing fallbacks
    }

}

function applyCustomColumnOrder() {
    console.log('Deals Table Order: Applying custom column order');
    jQuery.get(admin_url + 'deals/get_column_order')
        .done(function (response) {
            // Parse the response safely
            let parsedResponse;
            try {
                parsedResponse = typeof response === 'string' ? JSON.parse(response) : response;
            } catch (e) {
                console.error('Deals Table Order: Failed to parse JSON:', response, e);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            if (!parsedResponse || !parsedResponse.success || !parsedResponse.column_order) {
                console.error('Deals Table Order: Invalid or missing JSON response:', response);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            const $table = jQuery('.table-deals');
            if (!$table.length || !$table.DataTable) {
                console.warn('Deals Table Order: .table-deals not found or DataTable not initialized');
                return;
            }
            const table = $table.DataTable();

            // Map th-xxx class to column id
            const headers = table.columns().header().toArray().map(function (th, index) {
                const $th = jQuery(th);
                let match = $th.attr('class') ? $th.attr('class').match(/th-([^\s]+)/) : null;
                let colId = match ? match[1] : 'column_' + index;
                // Checkbox column detection
                if ($th.find('input[type="checkbox"]').length) {
                    colId = 'checkbox';
                }
                return { id: colId, index: index };
            });

            // Build id->index and index->id maps
            const columnIdToIndex = {};
            const indexToColumnId = {};
            headers.forEach(function (header) {
                columnIdToIndex[header.id] = header.index;
                indexToColumnId[header.index] = header.id;
            });

            // Build new order array (DataTables column indexes)
            let columnOrder = [];
            // Always ensure checkbox column is first if present
            if (columnIdToIndex['checkbox'] !== undefined) {
                columnOrder.push(columnIdToIndex['checkbox']);
            }
            parsedResponse.column_order.forEach(function (colId) {
                if (columnIdToIndex[colId] !== undefined && colId !== 'checkbox') {
                    columnOrder.push(columnIdToIndex[colId]);
                }
            });

            // Set column visibility: show only those in column_order (plus checkbox)
            table.columns().every(function (index) {
                const colId = indexToColumnId[index];
                // Always show checkbox, otherwise only if in column_order
                this.visible(colId === 'checkbox' || parsedResponse.column_order.includes(colId));
            });

            // Reorder columns if possible
            if (table.colReorder && typeof table.colReorder.order === 'function') {
                table.colReorder.order(columnOrder, true); // true = reset fixed columns
            } else {
                console.warn('Deals Table Order: colReorder not available on DataTable');
            }

            table.draw(false);

            console.log('Deals Table Order: Columns reordered and visibility applied', columnOrder);
        })
        .fail(function (xhr, status, error) {
            let msg = 'Failed to load column order.';
            if (xhr && xhr.responseText) {
                msg += ' ' + xhr.responseText;
            }
            console.error('Deals Table Order: Failed to fetch column order:', status, error, xhr && xhr.responseText);
            alert_float('danger', msg);
        });
}

function loadColumnOrderModal() {
    console.log('Deals Table Order: Loading column order modal');
    jQuery.get(admin_url + 'deals/get_column_order')
        .done(function (response) {
            let parsedResponse = safeParseJSON(response);
            if (!parsedResponse) {
                console.error('Deals Table Order: Invalid JSON response:', response);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            const $table = jQuery('.table-deals');
            const headers = $table.find('thead th').map(function (index) {
                const $th = jQuery(this);
                let match = $th.attr('class') ? $th.attr('class').match(/th-([^\s]+)/) : null;
                let colId = match ? match[1] : 'column_' + index;
                let colText = $th.text().trim() || 'Column ' + index;
                if (colId === 'checkbox' || $th.find('input[type="checkbox"]').length) {
                    colId = 'checkbox';
                    colText = (typeof _l !== 'undefined' ? _l('select_all') : 'Select All');
                }
                return { id: colId, text: colText };
            }).get();

            // Remove any existing modal to prevent duplicates
            jQuery('#deals-table-order-modal').remove();

            let modalHtml = `
                <div class="modal fade" id="deals-table-order-modal" tabindex="-1" role="dialog">
                    <div class="modal-dialog" role="document">
                        <div class="modal-content">
                            <div class="modal-header">
                                <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
                                <h4 class="modal-title">${typeof _l !== 'undefined' ? _l('deals_table_order_configure') : 'Configure Columns'}</h4>
                            </div>
                            <div class="modal-body">
                                <ul class="list-group deals-column-order-list">
                                    ${headers.map(function (header) {
                const isChecked = (parsedResponse.column_order && parsedResponse.column_order.includes(header.id)) || header.id === 'checkbox' ? 'checked' : '';
                const isDisabled = header.id === 'checkbox' ? 'disabled' : '';
                return `
                                            <li class="list-group-item" data-column-id="${header.id}">
                                                <input type="checkbox" id="col_${header.id}" ${isChecked} ${isDisabled}>
                                                <label for="col_${header.id}">${header.text}</label>
                                            </li>
                                        `;
            }).join('')}
                                </ul>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-default" data-dismiss="modal">${typeof _l !== 'undefined' ? _l('close') : 'Close'}</button>
                                <button type="button" class="btn btn-primary" onclick="saveDealsColumnOrder()">${typeof _l !== 'undefined' ? _l('save') : 'Save'}</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            jQuery('body').append(modalHtml);
            jQuery('#deals-table-order-modal .deals-column-order-list').sortable({
                placeholder: 'ui-state-highlight',
                items: 'li:not([data-column-id="checkbox"])',
                update: function () {
                    console.log('Deals Table Order: Columns reordered in modal');
                }
            });
            jQuery('#deals-table-order-modal').modal('show');
        })
        .fail(function (xhr, status, error) {
            let msg = 'Failed to load column configuration.';
            if (xhr && xhr.responseText) {
                msg += ' ' + xhr.responseText;
            }
            console.error('Deals Table Order: Failed to load modal data:', status, error, xhr && xhr.responseText);
            alert_float('danger', msg);
        });
}

function saveDealsColumnOrder() {
    console.log('Deals Table Order: Saving column order');
    const $modal = jQuery('#deals-table-order-modal');
    const columnOrder = [];
    $modal.find('.deals-column-order-list li').each(function () {
        const $li = jQuery(this);
        const colId = $li.data('column-id');
        const isVisible = $li.find('input[type="checkbox"]').is(':checked');
        if (isVisible && colId !== 'checkbox') {
            columnOrder.push(colId);
        }
    });

    jQuery.post(admin_url + 'deals/save_column_order', {
        column_order: JSON.stringify(columnOrder)
    }).done(function (response) {
        let parsedResponse = safeParseJSON(response);
        if (!parsedResponse) {
            console.error('Deals Table Order: Invalid JSON response:', response);
            alert_float('danger', 'Error parsing server response.');
            return;
        }

        if (parsedResponse.success) {
            console.log('Deals Table Order: Column order saved successfully');
            alert_float('success', parsedResponse.message || 'Column order saved.');
            $modal.modal('hide');
            applyCustomColumnOrder();
        } else {
            console.error('Deals Table Order: Failed to save column order:', parsedResponse.message);
            alert_float('danger', parsedResponse.message || 'Failed to save column order.');
        }
    }).fail(function (xhr, status, error) {
        let msg = 'Failed to save column order.';
        if (xhr && xhr.responseText) {
            msg += ' ' + xhr.responseText;
        }
        console.error('Deals Table Order: Failed to save column order:', status, error, xhr && xhr.responseText);
        alert_float('danger', msg);
    });
}