<?php

defined('BASEPATH') || exit('No direct script access allowed');

/*
    Module Name: Deals
    Description: Deals management with list, kanban, and status (open/won/lost)
    Version: 1.0.0
    
    Requires at least: 2.9.*
*/

define('DEALS_MODULE', 'deals');

// Load installer
require_once __DIR__ . '/install.php';

$CI = &get_instance();

// Activation hook
register_activation_hook(DEALS_MODULE, 'deals_module_activation_hook');
function deals_module_activation_hook()
{
    $CI = &get_instance();
    require_once __DIR__ . '/install.php';
}



// Register language files
register_language_files(DEALS_MODULE, [DEALS_MODULE]);

// Include assets and menu
require_once __DIR__ . '/includes/assets.php';
require_once __DIR__ . '/includes/sidebar_menu_links.php';

// Register settings section and pipelines manager tab
hooks()->add_action('admin_init', function () {
    $CI = &get_instance();
    $CI->app->add_settings_section('deals', [
        'title'    => _l('deals'),
        'position' => 18,
        'children' => [
            [
                'name' => _l('pipelines'),
                'view' => 'deals/settings',
                'position' => 10,
                'icon' => 'fa-solid fa-diagram-project',
            ],
            [
                'name' => _l('lost_reasons'),
                'view' => 'deals/lost_reasons',
                'position' => 15,
                'icon' => 'fa-regular fa-face-frown',
            ],
            [
                'name' => _l('deals_sources'),
                'view' => 'deals/manage_sources',
                'position' => 20,
                'icon' => 'fa-regular fa-tag',
            ],
            [
                'name' => _l('deals_statuses'),
                'view' => 'deals/manage_statuses',
                'position' => 25,
                'icon' => 'fa-regular fa-circle-check',
            ],
			[
                'name' => _l('deals_interested_in'),
                'view' => 'deals/interested_in',
                'position' => 25,
                'icon' => 'fa-regular fa-address-book',
            ],
            [
                'name' => _l('deals_business_categories'),
                'view' => 'deals/manage_business_categories',
                'position' => 30,
                'icon' => 'fa-regular fa-briefcase',
            ],
            // [
            //     'name' => _l('deals_ratings'),
            //     'view' => 'deals/ratings',
            //     'position' => 35,
            //     'icon' => 'fa-regular fa-star',
            // ],
            [
                'name' => _l('email_templates'),
                'view' => 'deals/settings_email_templates',
                'position' => 40,
                'icon' => 'fa-regular fa-envelope',
            ],
        ],
    ]);

    // // Add parent menu under Setup using Perfex's app_menu API (like extended_email)
    // $CI->app_menu->add_setup_menu_item('deals-setup', [
    //     'name'     => _l('deals'),
    //     // 'icon'     => 'fa fa-handshake',
    //     'position' => 18,
    // ]);

    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-pipelines',
    //     'name'     => _l('pipelines'),
    //     'href'     => admin_url('deals/settings'),
    //     // 'icon'     => 'fa-solid fa-diagram-project',
    //     'position' => 10,
    // ]);
    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-lost-reasons',
    //     'name'     => _l('lost_reasons'),
    //     'href'     => admin_url('deals/lost_reasons'),
    //     // 'icon'     => 'fa-regular fa-face-frown',
    //     'position' => 15,
    // ]);
    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-sources',
    //     'name'     => _l('deals_sources'),
    //     'href'     => admin_url('deals/manage_sources'),
    //     // 'icon'     => 'fa-regular fa-tag',
    //     'position' => 20,
    // ]);
    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-statuses',
    //     'name'     => _l('deals_statuses'),
    //     'href'     => admin_url('deals/manage_statuses'),
    //     // 'icon'     => 'fa-regular fa-circle-check',
    //     'position' => 25,
    // ]);
    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-business-categories',
    //     'name'     => _l('deals_business_categories'),
    //     'href'     => admin_url('deals/manage_business_categories'),
    //     // 'icon'     => 'fa-regular fa-briefcase',
    //     'position' => 30,
    // ]);
    // $CI->app_menu->add_setup_children_item('deals-setup', [
    //     'slug'     => 'deals-ratings',
    //     'name'     => _l('deals_ratings'),
    //     'href'     => admin_url('deals/ratings'),
    //     // 'icon'     => 'fa-regular fa-star',
    //     'position' => 35,
    // ]);

    // Register deals App_table (filters/rules like leads)
    if (class_exists('App_table')) {
        try {
            App_table::register(
                App_table::new('deals')->customfieldable('deals')
            );

            // Define filters/rules similar to leads
            $CI->load->model('deals/deals_model');
            $CI->load->model('staff_model');

            $rules = [
                App_table_filter::new('name', 'TextRule')->label(_l('deals_dt_name')),
                App_table_filter::new('company', 'TextRule')->label(_l('company')),
                App_table_filter::new('deal_value', 'NumberRule')->label(_l('deal_value')),
                App_table_filter::new('dateadded', 'DateRule')->label(_l('date_created')),
                App_table_filter::new('status_final', 'MultiSelectRule')->label(_l('status'))->options([
                    ['value' => 'open', 'label' => _l('open')],
                    ['value' => 'won', 'label' => _l('won')],
                    ['value' => 'lost', 'label' => _l('lost')],
                ]),
                App_table_filter::new('pipeline_id', 'SelectRule')->label(_l('pipeline'))->options(function () use ($CI) {
                    $pipelines = $CI->db->get(db_prefix() . 'deals_pipelines')->result_array();
                    return collect($pipelines)->map(fn($p) => ['value' => $p['id'], 'label' => $p['name']])->all();
                }),
                App_table_filter::new('stage_id', 'SelectRule')->label(_l('stage'))->options(function () use ($CI) {
                    $stages = $CI->db->get(db_prefix() . 'deals_stages')->result_array();
                    return collect($stages)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
                App_table_filter::new('status', 'SelectRule')->label(_l('leads_dt_status'))->options(function () use ($CI) {
                    $statuses = $CI->deals_model->get_status();
                    return collect($statuses)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
                App_table_filter::new('source', 'SelectRule')->label(_l('leads_source'))->options(function () use ($CI) {
                    $sources = $CI->deals_model->get_source();
                    return collect($sources)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
            ];

            $rules[] = App_table_filter::new('assigned', 'SelectRule')->label(_l('assigned_to'))
                ->withEmptyOperators()->emptyOperatorValue(0)
                ->isVisible(fn () => staff_can('view', 'deals'))
                ->options(function () use ($CI) {
                    $staff = $CI->staff_model->get('', ['active' => 1]);
                    return collect($staff)->map(fn ($m) => ['value' => $m['staffid'], 'label' => $m['firstname'] . ' ' . $m['lastname']])->all();
                });

            App_table::find('deals')->setRules($rules)
                ->outputUsing(function ($params) {
                    extract($params);
                    $aColumns = [
                        '1',
                        db_prefix() . 'deals.id as id',
                        db_prefix() . 'deals.name as name',
                        'company',
                        // Combined pipelines / stage / score column
                        '(SELECT GROUP_CONCAT(CONCAT(p2.name, " - ", s2.name, IF(r2.score IS NOT NULL, CONCAT(" (", r2.score, ")"), "")) SEPARATOR "; ") FROM ' . db_prefix() . 'deals_pipeline_relations r2 JOIN ' . db_prefix() . 'deals_pipelines p2 ON p2.id = r2.pipeline_id JOIN ' . db_prefix() . 'deals_stages s2 ON s2.id = r2.stage_id LEFT JOIN ' . db_prefix() . 'deals_stage_statuses ss2 ON ss2.id = r2.stage_status_id WHERE r2.deal_id = ' . db_prefix() . 'deals.id) as pipeline_relations',
                        db_prefix() . 'deals.deal_value as deal_value',
                        'status_final',
                        '(SELECT GROUP_CONCAT(name SEPARATOR ",") FROM ' . db_prefix() . 'taggables JOIN ' . db_prefix() . 'tags ON ' . db_prefix() . 'taggables.tag_id = ' . db_prefix() . 'tags.id WHERE rel_id = ' . db_prefix() . 'deals.id and rel_type="deal" ORDER by tag_order ASC LIMIT 1) as tags',
                        'dateadded',
                    ];

                    $sIndexColumn = 'id';
                    $sTable       = db_prefix() . 'deals';
                    $join = [
                        'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = ' . db_prefix() . 'deals.pipeline_id',
                        'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = ' . db_prefix() . 'deals.stage_id',
                    ];

                    $where = [];
                    if ($filtersWhere = $this->getWhereFromRules()) {
                        $where[] = $filtersWhere;
                    }

                    if (staff_cant('view', 'deals')) {
                        $where[] = 'AND (' . db_prefix() . 'deals.assigned = ' . get_staff_user_id() . ' OR ' . db_prefix() . 'deals.addedfrom=' . get_staff_user_id() . ' OR ' . db_prefix() . 'deals.is_public=1)';
                    }

                    $additionalColumns = [
                        db_prefix() . 'deals.addedfrom as addedfrom',
                    ];

                    $result = data_tables_init($aColumns, $sIndexColumn, $sTable, $join, $where, $additionalColumns);
                    $output  = $result['output'];
                    $rResult = $result['rResult'];

                    foreach ($rResult as $aRow) {
                        $row = [];
                        $row[] = '<div class="checkbox"><input type="checkbox" value="' . $aRow['id'] . '"><label></label></div>';
                        $hrefAttr = 'href="' . admin_url('deals/view/' . $aRow['id']) . '"';
                        $row[] = '<a ' . $hrefAttr . ' class="tw-font-medium">' . $aRow['id'] . '</a>';
                        $nameRow = '<a ' . $hrefAttr . ' class="tw-font-medium">' . e($aRow['name']) . '</a>';
                        $nameRow .= '<div class="row-options">';
                        $nameRow .= '<a ' . $hrefAttr . '>' . _l('view') . '</a>';
                        $nameRow .= ' | <a href="' . admin_url('deals/edit/' . $aRow['id']) . '">' . _l('edit') . '</a>';
                        $nameRow .= ' | <a href="' . admin_url('deals/delete/' . $aRow['id']) . '" class="_delete text-danger">' . _l('delete') . '</a>';
                        $nameRow .= '</div>';
                        $row[] = $nameRow;
                        $row[] = e($aRow['company']);
                        // Render combined pipelines / stage / score as list
                        if (!empty($aRow['pipeline_relations'])) {
                            $parts = explode('; ', $aRow['pipeline_relations']);
                            $listHtml = '<ul class="deals-pipeline-relations-list" style="list-style:none;padding:0;margin:0;">';
                            foreach ($parts as $p) {
                                $p = trim($p);
                                if ($p === '') continue;
                                $segments = explode(' - ', $p, 3);
                                $pipelineName = isset($segments[0]) ? htmlspecialchars($segments[0], ENT_QUOTES, 'UTF-8') : '';
                                $stageAndScore = isset($segments[1]) ? htmlspecialchars($segments[1], ENT_QUOTES, 'UTF-8') : '';
                                if (isset($segments[2])) {
                                    $stageAndScore .= ' - ' . htmlspecialchars($segments[2], ENT_QUOTES, 'UTF-8');
                                }
                                $listHtml .= '<li style="margin-bottom:4px;"><span class="label label-default deals-pipeline-name" style="background:#f0f6ff;color:#09326a;margin-right:6px;padding:3px 6px;border-radius:3px;">' . $pipelineName . '</span>' . $stageAndScore . '</li>';
                            }
                            $listHtml .= '</ul>';
                            $row[] = $listHtml;
                        } else {
                            $row[] = '-';
                        }
                        $row[] = '<span class="tw-font-medium">' . app_format_money((float)$aRow['deal_value'], get_base_currency()->id) . '</span>';
                        $row[] = ucfirst($aRow['status_final']);
                        $row[] = render_tags($aRow['tags']);
                        $row[] = '<span data-toggle="tooltip" data-title="' . e(_dt($aRow['dateadded'])) . '" class="text-has-action is-date">' . e(time_ago($aRow['dateadded'])) . '</span>';
                        $row['DT_RowId'] = 'deal_' . $aRow['id'];
                        $output['aaData'][] = $row;
                    }

                    return $output;
                });
        } catch (Exception $e) {
            // Already registered or not available; ignore
        }
    }
});

// Hooks to add 'deals' as rel_type in proposals, tasks, etc.
hooks()->add_filter('proposals_relation_types', 'add_deals_rel_type');
hooks()->add_filter('tasks_rel_types', 'add_deals_rel_type');
hooks()->add_filter('attachments_rel_types', 'add_deals_rel_type');
hooks()->add_filter('reminders_rel_types', 'add_deals_rel_type');
hooks()->add_filter('notes_rel_types', 'add_deals_rel_type');
hooks()->add_filter('activity_log_rel_types', 'add_deals_rel_type');
hooks()->add_filter('emails_rel_types', 'add_deals_rel_type');
hooks()->add_filter('calls_rel_types', 'add_deals_rel_type');

// Add default options for deals module if not present
function deals_add_default_options() {
    $CI = &get_instance();
    $defaults = [
        'default_deals_kanban_sort' => 'dateadded',
        'default_deals_kanban_sort_type' => 'desc',
        'deals_kanban_limit' => 20,
        'deal_lock_after_convert_to_customer' => 1,
    ];
    foreach ($defaults as $key => $value) {
        if (get_option($key) === false) {
            add_option($key, $value);
        }
    }
}
hooks()->add_action('admin_init', 'deals_add_default_options');

function add_deals_rel_type($types)
{
    $types[] = ['key' => 'deal', 'name' => _l('deal')];
    return $types;
}

// Dashboard: Inject Deals tabs into user_data widget without modifying core
hooks()->add_action('after_user_data_widget_tabs', function () {
    if (!is_staff_member()) { return; }
    echo '<li role="presentation"><a href="#home_deals_tab" aria-controls="home_deals_tab" role="tab" data-toggle="tab"><i class="fa fa-handshake menu-icon"></i> ' . _l('deals') . '</a></li>';
    echo '<li role="presentation"><a href="#home_deals_activity_tab" aria-controls="home_deals_activity_tab" role="tab" data-toggle="tab"><i class="fa fa-stream menu-icon"></i> ' . _l('activity_log') . '</a></li>';
});

hooks()->add_action('after_user_data_widge_tabs_content', function () {
    if (!is_staff_member()) { return; }
    $CI = &get_instance();
    $CI->load->model('deals/deals_model');
    $deals = $CI->deals_model->get_dashboard_deals(10);
    $activities = $CI->deals_model->get_recent_deal_activities(20);
    $data = [
        'dashboard_deals' => $deals,
        'dashboard_deal_activities' => $activities,
        'base_currency' => function_exists('get_base_currency') ? get_base_currency() : null,
    ];
    $CI->load->view('deals/dashboard/deals_widget', $data);
    $CI->load->view('deals/dashboard/deals_activity_widget', $data);
});

