<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('deals/deals_model');
    }

    public function index()
    {
        if ($this->input->get('view') === 'kanban') {
            return $this->kanban();
        }

        $data['title'] = _l('deals');
        $data['deals'] = $this->deals_model->get_all();
        // dd($data);
        $this->load->view('list', $data);
    }

public function kanban()
{
        $data['title'] = _l('deals_kanban');
        $data['stages'] = $this->deals_model->get_stages_with_deals();
        $this->load->view('kanban', $data);
    }

    public function create()
    {
        //dd(admin_url('deals'));
        if ($this->input->post()) {
            $post = $this->input->post();
            if (empty($post['pipeline_id'])) {
                $post['pipeline_id'] = $this->deals_model->get_default_pipeline_id();
            }
            if (empty($post['stage_id'])) {
                $post['stage_id'] = $this->deals_model->get_default_open_stage_id();
            }
            $this->deals_model->create($post);
            set_alert('success', _l('added_successfully', _l('deal')));
            
            return redirect(admin_url('deals'));
        }
        $data['title'] = _l('new_deal');
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $this->load->view('create', $data);
    }

    public function edit($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($this->input->post()) {
            $post = $this->input->post();
            $this->deals_model->update_deal($id, $post);
            set_alert('success', _l('updated_successfully', _l('deal')));
            return redirect(admin_url('deals/view/'.$id));
        }
        $data['title'] = _l('edit');
        $data['deal'] = $deal;
        $data['pipelines'] = $this->deals_model->get_pipelines();
        $data['stages'] = $this->deals_model->get_stages($deal['pipeline_id']);
        $this->load->view('create', $data);
    }


    /**
     * Update the deal value (amount) via AJAX.
     * Expects POST 'deal_value'.
     * Returns JSON: ['success' => bool]
     */
    public function update_amount($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $amount = $this->input->post('deal_value');
        if (!is_numeric($amount) || $amount < 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid amount']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['deal_value' => $amount]);
        echo json_encode(['success' => (bool)$updated]);
    }

    /**
     * Update the expected close date via AJAX.
     * Expects POST 'expected_close_date'.
     * Returns JSON: ['success' => bool]
     */
    public function update_close_date($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $date = $this->input->post('expected_close_date');
        if (!$date || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            echo json_encode(['success' => false, 'message' => 'Invalid date']);
            return;
        }
        $updated = $this->deals_model->update_deal($id, ['expected_close_date' => $date]);
        echo json_encode(['success' => (bool)$updated]);
    }

    public function update_details($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($this->input->post()) {
            $post = $this->input->post();
            $this->deals_model->update_deal($id, $post);
            set_alert('success', _l('updated_successfully', _l('deal')));
        }
        redirect(admin_url('deals/view/'.$id));
    }

    public function upload_attachment($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        if ($this->input->post()) {
            $config['upload_path'] = './uploads/deals/';
            $config['allowed_types'] = 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx|txt';
            $config['max_size'] = 2048;
            
            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0755, true);
            }
            
            $this->load->library('upload', $config);
            
            if ($this->upload->do_upload('attachment')) {
                $upload_data = $this->upload->data();
                $attachment_data = [
                    'deal_id' => $id,
                    'file_name' => $upload_data['file_name'],
                    'description' => $this->input->post('description'),
                    'dateadded' => date('Y-m-d H:i:s')
                ];
                $this->deals_model->add_attachment($attachment_data);
                set_alert('success', _l('file_uploaded_successfully'));
            } else {
                set_alert('danger', $this->upload->display_errors());
            }
        }
        redirect(admin_url('deals/view/'.$id));
    }

    public function delete($id)
    {
        if ($this->deals_model->delete($id)) {
            set_alert('success', _l('deleted', _l('deal')));
        }
        redirect(admin_url('deals'));
    }

    public function table()
    {
        $this->app->get_table_data('deals', [
            'select' => [
                'd.id',
                'd.name',
                'd.company',
                'd.deal_value',
                'd.dateadded',
                'd.status_final',
                'dp.name as pipeline_name',
                'ds.name as stage_name',
                'CONCAT(s.firstname, " ", s.lastname) as assigned_staff'
            ],
            'from' => db_prefix() . 'deals d',
            'join' => [
                'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = d.pipeline_id',
                'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = d.stage_id',
                'LEFT JOIN ' . db_prefix() . 'staff s ON s.staffid = d.addedfrom'
            ]
        ]);
    }

    public function get_stages($pipeline_id)
    {
        $stages = $this->deals_model->get_stages($pipeline_id);
        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => true,
            'stages' => $stages
        ]));
    }

    public function bulk_action()
    {
        $action = $this->input->post('bulk_action');
        $ids = $this->input->post('ids');
        
        if (!$ids || !$action) {
            $this->output->set_content_type('application/json')->set_output(json_encode([
                'success' => false,
                'message' => _l('no_action_selected')
            ]));
            return;
        }

        $success = false;
        $message = '';

        switch ($action) {
            case 'delete':
                foreach ($ids as $id) {
                    $this->deals_model->delete($id);
                }
                $success = true;
                $message = _l('deals_deleted_successfully');
                break;
                
            case 'move_pipeline':
                $pipeline_id = $this->input->post('pipeline_id');
                $stage_id = $this->input->post('stage_id');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, [
                        'pipeline_id' => $pipeline_id,
                        'stage_id' => $stage_id
                    ]);
                }
                $success = true;
                $message = _l('deals_moved_successfully');
                break;
                
            case 'assign':
                $assigned_to = $this->input->post('assigned_to');
                foreach ($ids as $id) {
                    $this->deals_model->update_deal($id, ['addedfrom' => $assigned_to]);
                }
                $success = true;
                $message = _l('deals_assigned_successfully');
                break;
        }

        $this->output->set_content_type('application/json')->set_output(json_encode([
            'success' => $success,
            'message' => $message
        ]));
    }

    public function view($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        $data['deal'] = $deal;
        $pipelineId = $deal['pipeline_id'] ?: $this->deals_model->get_default_pipeline_id();
        $data['stages'] = $this->deals_model->get_stages($pipelineId);
        $data['lost_reasons'] = $this->deals_model->get_lost_reasons();
        $data['title'] = $deal['name'];
        $this->load->view('view', $data);
    }

    public function set_stage($id)
    {
        $stageId = (int)$this->input->post('stage_id');
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        if ($deal['status_final'] === 'won' || $deal['status_final'] === 'lost') {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Cannot change stage for won/lost deals']);
                return;
            }
            return redirect(admin_url('deals/view/'.$id));
        }
        
        $old_stage = $this->deals_model->get_stage_name($deal['stage_id']);
        $new_stage = $this->deals_model->get_stage_name($stageId);
        
        $this->deals_model->update_deal($id, ['stage_id'=>$stageId]);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Stage changed from ' . $old_stage . ' to ' . $new_stage, null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            redirect(admin_url('deals/view/'.$id));
        }
    }

    public function update_status($id)
    {
        $status = $this->input->post('status'); // open|won|lost
        $old_status = $this->deals_model->get_one($id)['status_final'];
        $this->deals_model->update_status($id, $status);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Status changed from ' . $old_status . ' to ' . $status, null, 'activities');
        
        set_alert('success', _l('updated_successfully', _l('deal')));
        redirect($_SERVER['HTTP_REFERER']);
    }

    public function mark_won($id)
    {
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        $this->deals_model->mark_won($id);
        
        // Log activity
        $this->deals_model->add_deal_activity($id, 'Deal marked as won', null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    public function mark_lost($id)
    {
        $reasonId = $this->input->post('lost_reason_id') ? (int)$this->input->post('lost_reason_id') : null;
        $deal = $this->deals_model->get_one($id);
        if (!$deal) { show_404(); }
        
        $reason_name = '';
        if ($reasonId) {
            $reason = $this->deals_model->get_lost_reason($reasonId);
            $reason_name = $reason ? $reason['name'] : '';
        }
        
        $this->deals_model->mark_lost($id, $reasonId);
        
        // Log activity
        $activity_message = 'Deal marked as lost';
        if ($reason_name) {
            $activity_message .= ' - Reason: ' . $reason_name;
        }
        $this->deals_model->add_deal_activity($id, $activity_message, null, 'activities');
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true]);
        } else {
            set_alert('success', _l('updated_successfully', _l('deal')));
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    // Deal Contacts Methods
    public function add_contact_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        

        $contact_id = (int)$this->input->post('contact_id');
        if ($contact_id) {
            // Add existing contact to deal
            $this->deals_model->add_contact_to_deal($deal_id, $contact_id);
            $this->deals_model->add_deal_activity($deal_id, 'Contact added to deal', null, 'activities');
            echo json_encode(['success' => true, 'message' => 'Contact added']);
        } else {
            // Create new contact and add to deal
            $contact_data = [
                'firstname' => $this->input->post('firstname'),
                'lastname' => $this->input->post('lastname'),
                'email' => $this->input->post('email'),
                'phonenumber' => $this->input->post('phonenumber'),
            ];

            $new_contact_id = $this->deals_model->create_module_contact($contact_data);
            
            if ($new_contact_id) {
                $this->deals_model->add_contact_to_deal($deal_id, $new_contact_id);
                $this->deals_model->add_deal_activity($deal_id, 'New contact created and added to deal', null, 'activities');
                echo json_encode(['success' => true, 'message' => 'Contact created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
            }
        }
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_contact_from_deal($deal_id, $contact_id);
        $this->deals_model->add_deal_activity($deal_id, 'Contact removed from deal', null, 'activities');
        echo json_encode(['success' => true, 'message' => 'Contact removed']);
    }

    public function get_deal_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_deal_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    public function get_available_contacts($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contacts = $this->deals_model->get_available_contacts($deal_id);
        echo json_encode(['success' => true, 'contacts' => $contacts]);
    }

    // Deal Companies Methods
    public function add_company_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $company_id = (int)$this->input->post('company_id');
        if ($company_id) {
            // Add existing company to deal
            $this->deals_model->add_company_to_deal($deal_id, $company_id);
            $this->deals_model->add_deal_activity($deal_id, 'Company added to deal', null, 'activities');
            echo json_encode(['success' => true, 'message' => 'Company added']);
        } else {
            // Create new company and add to deal
            $company_data = [
                'name' => $this->input->post('company_name'),
                'email' => $this->input->post('company_email'),
                'phone' => $this->input->post('company_phone'),
                'website' => $this->input->post('company_website'),
                'address' => $this->input->post('company_address')
            ];

            $new_company_id = $this->deals_model->create_company($company_data);
            
            if ($new_company_id) {
                $this->deals_model->add_company_to_deal($deal_id, $new_company_id);
                $this->deals_model->add_deal_activity($deal_id, 'New company created and added to deal', null, 'activities');
                echo json_encode(['success' => true, 'message' => 'Company created and added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to create company']);
            }
        }
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_company_from_deal($deal_id, $company_id);
        $this->deals_model->add_deal_activity($deal_id, 'Company removed from deal', null, 'activities');
        echo json_encode(['success' => true, 'message' => 'Company removed']);
    }

    public function get_deal_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_deal_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function get_available_companies($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $companies = $this->deals_model->get_available_companies($deal_id);
        echo json_encode(['success' => true, 'companies' => $companies]);
    }

    public function add_activity($deal_id)
{
    if (!$this->input->is_ajax_request()) {
        show_404();
    }
    $deal = $this->deals_model->get_one($deal_id);
    if (!$deal) {
        echo json_encode(['success' => false, 'message' => 'Deal not found']);
        return;
    }
    $payload = $this->input->post();
    $id = $this->deals_model->add_deal_activity($deal_id, $payload);
    echo json_encode(['success' => (bool)$id, 'id' => $id]);
}

    // Deal Attachments Methods
    public function add_attachment_to_deal($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Upload new attachment and add to deal
        $config['upload_path'] = './uploads/deals/';
        $config['allowed_types'] = 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx|txt';
        $config['max_size'] = 2048;
        
        if (!is_dir($config['upload_path'])) {
            mkdir($config['upload_path'], 0755, true);
        }
        
        $this->load->library('upload', $config);
        
        if ($this->upload->do_upload('attachment')) {
            $upload_data = $this->upload->data();
            $attachment_data = [
                'file_name' => $upload_data['file_name'],
                'filetype' => $upload_data['file_type'],
                'description' => $this->input->post('description'),
                'staffid' => get_staff_user_id(),
                'dateadded' => date('Y-m-d H:i:s')
            ];
            
            $new_attachment_id = $this->deals_model->add_attachment($attachment_data);
            
            if ($new_attachment_id) {
                $this->deals_model->add_attachment_to_deal($deal_id, $new_attachment_id);
                $this->deals_model->add_deal_activity($deal_id, 'New attachment uploaded and added to deal', null, 'documents');
                echo json_encode(['success' => true, 'message' => 'Attachment added']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to save attachment']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => $this->upload->display_errors()]);
        }
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            $this->output->set_status_header(404);
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        $this->deals_model->remove_attachment_from_deal($deal_id, $attachment_id);
        $this->deals_model->add_deal_activity($deal_id, 'Attachment removed from deal', null, 'documents');
        echo json_encode(['success' => true, 'message' => 'Attachment removed']);
    }

    public function get_deal_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $attachments = $this->deals_model->get_deal_attachments($deal_id);
        echo json_encode(['success' => true, 'attachments' => $attachments]);
    }

    public function get_available_attachments($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $attachments = $this->deals_model->get_available_attachments($deal_id);
        echo json_encode(['success' => true, 'attachments' => $attachments]);
    }

    public function download_attachment($attachment_id)
    {
        $attachment = $this->deals_model->get_attachment($attachment_id);
        if (!$attachment) {
            show_404();
        }
        $file_path = FCPATH . 'uploads/deals/' . $attachment['file_name'];
        if (!file_exists($file_path)) {
            show_404();
        }
        $this->load->helper('download');
        force_download($file_path, null);
    }

    // Deal Activities Methods
    public function get_deal_activities($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $activities = $this->deals_model->get_deal_activities($deal_id);
        echo json_encode(['success' => true, 'activities' => $activities]);
    }
    public function update_activity_status($id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $status = $this->input->post('status');
        $success = $this->deals_model->update_activity_status($id, $status);
        echo json_encode(['success' => $success]);
    }

    public function get_staff_ajax()
{
    if (!$this->input->is_ajax_request()) {
        show_404();
    }
    echo json_encode(['success' => true, 'staff' => $this->deals_model->get_staff()]);
}
    public function add_note($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $description = trim((string)$this->input->post('description'));
        if ($description === '') {
            echo json_encode(['success' => false, 'message' => 'Description required']);
            return;
        }
        $id = $this->deals_model->add_note($deal_id, $description);
        echo json_encode(['success' => (bool)$id, 'id' => $id]);
    }

    public function get_notes($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $notes = $this->deals_model->get_notes($deal_id);
        echo json_encode(['success' => true, 'notes' => $notes]);
    }

    public function add_call($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }
        $payload = $this->input->post();
        $id = $this->deals_model->add_call($deal_id, $payload);
        echo json_encode(['success' => (bool)$id, 'id' => $id]);
    }
    public function get_calls($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        $calls = $this->deals_model->get_calls($deal_id);
        echo json_encode(['success' => true, 'calls' => $calls]);
    }

    public function test_db($deal_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $deal = $this->deals_model->get_one($deal_id);
        if (!$deal) {
            echo json_encode(['success' => false, 'message' => 'Deal not found']);
            return;
        }

        // Test creating a contact
        $contact_data = [
            'firstname' => 'Test',
            'lastname' => 'Contact',
            'email' => 'test' . time() . '@example.com',
            'phonenumber' => '+1234567890'
        ];

        $contact_id = $this->deals_model->create_module_contact($contact_data);
        
        if ($contact_id) {
            // Test linking contact to deal
            $link_id = $this->deals_model->add_contact_to_deal($deal_id, $contact_id);
            
            if ($link_id) {
                // Test activity logging
                $activity_id = $this->deals_model->add_deal_activity($deal_id, 'Test contact added', null, 'activities');
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Database test successful',
                    'contact_id' => $contact_id,
                    'link_id' => $link_id,
                    'activity_id' => $activity_id
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to link contact to deal']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to create contact']);
        }
    }
}



<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Settings extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('deals/deals_model');
    }

    public function index()
    {
        // This controller is used only for AJAX actions from the settings view
        show_404();
    }

    public function pipelines()
    {
        // Re-order via drag & drop
        if ($this->input->post('action') === 'reorder_stages') {
            $ok = $this->deals_model->reorder_stages((int)$this->input->post('pipeline_id'), $this->input->post('order') ?? []);
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'create') {
            $id = $this->deals_model->create_pipeline($this->input->post('name'));
            return $this->json(['success' => (bool)$id, 'id' => $id]);
        }
        if ($this->input->post('action') === 'update') {
            $ok = $this->deals_model->update_pipeline((int)$this->input->post('id'), ['name'=>$this->input->post('name')]);
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'delete') {
            $ok = $this->deals_model->delete_pipeline((int)$this->input->post('id'));
            return $this->json(['success' => $ok]);
        }

        // Manage stages order and CRUD
        if ($this->input->post('action') === 'create_stage') {
            $id = $this->deals_model->create_stage((int)$this->input->post('pipeline_id'), [
                'name' => $this->input->post('name'),
                'position' => (int)$this->input->post('position') ?? 0,
                'is_won' => (int)$this->input->post('is_won') ?? 0,
                'is_lost' => (int)$this->input->post('is_lost') ?? 0,
            ]);
            return $this->json(['success' => (bool)$id, 'id' => $id]);
        }
        if ($this->input->post('action') === 'update_stage') {
            $id = (int)$this->input->post('id');
            $data = [];
            if ($this->input->post('name') !== null) { $data['name'] = $this->input->post('name'); }
            if ($this->input->post('position') !== null) { $data['position'] = (int)$this->input->post('position'); }
            if ($this->input->post('win_probability') !== null) { $data['win_probability'] = (int)$this->input->post('win_probability'); }
            if ($this->input->post('is_won') !== null) { $data['is_won'] = (int)$this->input->post('is_won'); }
            if ($this->input->post('is_lost') !== null) { $data['is_lost'] = (int)$this->input->post('is_lost'); }
            $ok = !empty($data) ? $this->deals_model->update_stage($id, $data) : true;
            return $this->json(['success' => $ok]);
        }
        if ($this->input->post('action') === 'delete_stage') {
            $ok = $this->deals_model->delete_stage((int)$this->input->post('id'));
            return $this->json(['success' => $ok]);
        }

        // Fetch pipelines + stages
        $pipelines = $this->deals_model->get_pipelines();
        foreach ($pipelines as &$p) {
            $p['stages'] = $this->deals_model->get_stages($p['id']);
        }

        $data['pipelines'] = $pipelines;
        $this->load->view('settings', $data);
    }

    public function lost_reasons()
    {
        if ($this->input->post('action') === 'create') {
            $id = $this->deals_model->create_lost_reason($this->input->post('name'));
            return $this->json(['success'=> (bool)$id, 'id'=>$id]);
        }
        if ($this->input->post('action') === 'update') {
            $ok = $this->deals_model->update_lost_reason((int)$this->input->post('id'), $this->input->post('name'));
            return $this->json(['success'=>$ok]);
        }
        if ($this->input->post('action') === 'delete') {
            $ok = $this->deals_model->delete_lost_reason((int)$this->input->post('id'));
            return $this->json(['success'=>$ok]);
        }

        $data['reasons'] = $this->deals_model->get_lost_reasons();
        $this->load->view('lost_reasons', $data);
    }

    private function json($payload)
    {
        $this->output->set_content_type('application/json')->set_output(json_encode($payload));
    }
}






<?php

/*
 * Inject sidebar menu and links for deals module
 */

hooks()->add_action('admin_init', function () {
    $CI = &get_instance();

    $CI->app_menu->add_sidebar_menu_item('deals', [
        'slug'     => 'deals',
        'name'     => _l('deals'),
        'icon'     => 'fa-solid fa-handshake',
        'href'     => admin_url('deals'),
        'position' => 16,
    ]);
});


<?php

hooks()->add_action('app_admin_head', function () {
    echo '<style>.deal-status-badge{padding:3px 6px;border-radius:4px;font-size:11px}</style>';
});

hooks()->add_action('app_admin_footer', function () {
    echo '<script>window.dealsModuleLoaded=true;</script>';
});




<?php
$lang['deals'] = 'Deals';
$lang['deals_kanban'] = 'Deals Kanban';
$lang['new_deal'] = 'New Deal';
$lang['edit_deal'] = 'Edit Deal';
$lang['deal'] = 'Deal';
$lang['mark_as_open'] = 'Open';
$lang['mark_as_won'] = 'Won';
$lang['mark_as_lost'] = 'Lost';
$lang['switch_to_list_view'] = 'List View';
$lang['deal_value'] = 'Amount';
$lang['expected_close_date'] = 'Expected Close Date';
$lang['pipeline'] = 'Pipeline';
$lang['stage'] = 'Stage';
$lang['pipelines'] = 'Pipelines';
$lang['stages'] = 'Stages';
$lang['lost_reasons'] = 'Lost Reasons';
$lang['add_contact'] = 'Add Contact';
$lang['add_company'] = 'Add Company';
$lang['add_attachment'] = 'Add Attachment';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['no_activities_found'] = 'No activities found';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['view'] = 'View';
$lang['add'] = 'Add';
$lang['edit'] = 'Edit';
$lang['update'] = 'Update';
$lang['save'] = 'Save';
$lang['close'] = 'Close';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['file_uploaded_successfully'] = 'File uploaded successfully';
$lang['select_pipeline'] = 'Select Pipeline';
$lang['select_stage'] = 'Select Stage';
$lang['pipeline_name'] = 'Pipeline Name';
$lang['enter_pipeline_name'] = 'Enter pipeline name';
$lang['enter_stage_name'] = 'Enter stage name';
$lang['delete_pipeline_confirm'] = 'Are you sure you want to delete this pipeline?';
$lang['delete_stage_confirm'] = 'Are you sure you want to delete this stage?';
$lang['pipeline_name_required'] = 'Pipeline name is required';
$lang['deal_name'] = 'Deal Name';
$lang['assigned_to'] = 'Assigned To';
$lang['total_deals'] = 'Total Deals';
$lang['total_value'] = 'Total Value';
$lang['deals_summary'] = 'Deals Summary';
$lang['deals_table_order_configure'] = 'Configure Table Columns';
$lang['move_to_pipeline'] = 'Move to Pipeline';
$lang['move_to_stage'] = 'Move to Stage';
$lang['assign_to'] = 'Assign To';
$lang['select_staff'] = 'Select Staff';
$lang['no_action_selected'] = 'No action selected';
$lang['deals_deleted_successfully'] = 'Deals deleted successfully';
$lang['deals_moved_successfully'] = 'Deals moved successfully';
$lang['deals_assigned_successfully'] = 'Deals assigned successfully';
$lang['company_name'] = 'Company Name';
$lang['company_email'] = 'Company Email';
$lang['company_phone'] = 'Company Phone';
$lang['firstname'] = 'First Name';
$lang['lastname'] = 'Last Name';
$lang['email'] = 'Email';
$lang['phonenumber'] = 'Phone Number';
$lang['please_select_lost_reason'] = 'Please select a lost reason';
$lang['won'] = 'Won';
$lang['lost'] = 'Lost';
$lang['lost_reason'] = 'Lost Reason';
$lang['select_lost_reason'] = 'Select Lost Reason';
$lang['cancel'] = 'Cancel';
$lang['submit'] = 'Submit';
$lang['dropdown_non_selected_tex'] = 'Nothing selected';
$lang['updated_successfully'] = 'Updated successfully';
$lang['deleted'] = 'Deleted successfully';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['no_activities_found'] = 'No activities found';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['view'] = 'View';
$lang['add'] = 'Add';
$lang['edit'] = 'Edit';
$lang['update'] = 'Update';
$lang['save'] = 'Save';
$lang['close'] = 'Close';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['file_uploaded_successfully'] = 'File uploaded successfully';
$lang['select_pipeline'] = 'Select Pipeline';
$lang['select_stage'] = 'Select Stage';
$lang['products'] = 'Products';
$lang['tasks'] = 'Tasks';
$lang['reminders'] = 'Reminders';
$lang['emails'] = 'Emails';
$lang['calls'] = 'Calls';
$lang['meetings'] = 'Meetings';
$lang['documents'] = 'Documents';
$lang['expenses'] = 'Expenses';
$lang['invoices'] = 'Invoices';
$lang['estimates'] = 'Estimates';
$lang['proposals'] = 'Proposals';
$lang['contracts'] = 'Contracts';
$lang['credit_notes'] = 'Credit Notes';
$lang['templates'] = 'Templates';
$lang['automation'] = 'Automation';
$lang['webhooks'] = 'Webhooks';
$lang['custom_fields'] = 'Custom Fields';
$lang['probability'] = 'Probability';
$lang['currency'] = 'Currency';
$lang['tags'] = 'Tags';
$lang['archived'] = 'Archived';
$lang['archived_date'] = 'Archived Date';
$lang['archived_by'] = 'Archived By';
$lang['is_primary'] = 'Is Primary';
$lang['dateadded'] = 'Date Added';
$lang['staffid'] = 'Staff ID';
$lang['external'] = 'External';
$lang['external_link'] = 'External Link';
$lang['filetype'] = 'File Type';
$lang['subject'] = 'Subject';
$lang['message'] = 'Message';
$lang['from_email'] = 'From Email';
$lang['from_name'] = 'From Name';
$lang['to_email'] = 'To Email';
$lang['to_name'] = 'To Name';
$lang['cc'] = 'CC';
$lang['bcc'] = 'BCC';
$lang['sent'] = 'Sent';
$lang['sent_date'] = 'Sent Date';
$lang['call_type'] = 'Call Type';
$lang['call_status'] = 'Call Status';
$lang['call_duration'] = 'Call Duration';
$lang['call_recording'] = 'Call Recording';
$lang['call_notes'] = 'Call Notes';
$lang['call_date'] = 'Call Date';
$lang['meeting_subject'] = 'Meeting Subject';
$lang['meeting_description'] = 'Meeting Description';
$lang['meeting_date'] = 'Meeting Date';
$lang['meeting_duration'] = 'Meeting Duration';
$lang['meeting_location'] = 'Meeting Location';
$lang['meeting_type'] = 'Meeting Type';
$lang['meeting_status'] = 'Meeting Status';
$lang['meeting_notes'] = 'Meeting Notes';
$lang['document_name'] = 'Document Name';
$lang['document_type'] = 'Document Type';
$lang['document_file'] = 'Document File';
$lang['document_description'] = 'Document Description';
$lang['document_version'] = 'Document Version';
$lang['document_status'] = 'Document Status';
$lang['template_name'] = 'Template Name';
$lang['template_content'] = 'Template Content';
$lang['template_type'] = 'Template Type';
$lang['template_category'] = 'Template Category';
$lang['is_active'] = 'Is Active';
$lang['created_by'] = 'Created By';
$lang['automation_name'] = 'Automation Name';
$lang['automation_type'] = 'Automation Type';
$lang['trigger_condition'] = 'Trigger Condition';
$lang['action_type'] = 'Action Type';
$lang['action_data'] = 'Action Data';
$lang['webhook_name'] = 'Webhook Name';
$lang['webhook_url'] = 'Webhook URL';
$lang['webhook_events'] = 'Webhook Events';
$lang['webhook_secret'] = 'Webhook Secret';
$lang['event_type'] = 'Event Type';
$lang['payload'] = 'Payload';
$lang['response_code'] = 'Response Code';
$lang['response_body'] = 'Response Body';
$lang['status'] = 'Status';
$lang['field_name'] = 'Field Name';
$lang['field_type'] = 'Field Type';
$lang['field_label'] = 'Field Label';
$lang['field_placeholder'] = 'Field Placeholder';
$lang['field_options'] = 'Field Options';
$lang['field_required'] = 'Field Required';
$lang['field_order'] = 'Field Order';
$lang['field_visible'] = 'Field Visible';
$lang['field_value'] = 'Field Value';
$lang['qty'] = 'Quantity';
$lang['rate'] = 'Rate';
$lang['discount'] = 'Discount';
$lang['tax'] = 'Tax';
$lang['total'] = 'Total';
$lang['product_id'] = 'Product ID';
$lang['task_id'] = 'Task ID';
$lang['expense_id'] = 'Expense ID';
$lang['invoice_id'] = 'Invoice ID';
$lang['estimate_id'] = 'Estimate ID';
$lang['proposal_id'] = 'Proposal ID';
$lang['contract_id'] = 'Contract ID';
$lang['credit_note_id'] = 'Credit Note ID';
$lang['contact_id'] = 'Contact ID';
$lang['company_id'] = 'Company ID';
$lang['field_id'] = 'Field ID';
$lang['webhook_id'] = 'Webhook ID';
$lang['filter_name'] = 'Filter Name';
$lang['field'] = 'Field';
$lang['operator'] = 'Operator';
$lang['value'] = 'Value';
$lang['share_with_team'] = 'Share with Team';
$lang['is_default'] = 'Is Default';
$lang['created_at'] = 'Created At';
$lang['filter_name'] = 'Filter Name';
$lang['field'] = 'Field';
$lang['operator'] = 'Operator';
$lang['value'] = 'Value';
$lang['share_with_team'] = 'Share with Team';
$lang['is_default'] = 'Is Default';
$lang['created_at'] = 'Created At';

// Additional language strings for the updated deals module
$lang['all'] = 'All';
$lang['filter_by_all'] = 'Filter by All';
$lang['select_existing'] = 'Select Existing';
$lang['create_new'] = 'Create New';
$lang['select_contact'] = 'Select Contact';
$lang['select_company'] = 'Select Company';
$lang['select_attachment'] = 'Select Attachment';
$lang['upload_new'] = 'Upload New';
$lang['confirm_action_prompt'] = 'Are you sure you want to perform this action?';
$lang['pin_on_top'] = 'Pin on top';
$lang['notes'] = 'Notes';
$lang['calls'] = 'Calls';
$lang['documents'] = 'Documents';
$lang['emails'] = 'Emails';
$lang['activities'] = 'Activities';
$lang['details'] = 'Details';
$lang['contacts'] = 'Contacts';
$lang['companies'] = 'Companies';
$lang['attachments'] = 'Attachments';
$lang['no_contacts_found'] = 'No contacts found';
$lang['no_company_associated'] = 'No company associated';
$lang['no_attachments_uploaded'] = 'No attachments uploaded';
$lang['add_contact'] = 'Add Contact';
$lang['add_company'] = 'Add Company';
$lang['add_attachment'] = 'Add Attachment';
$lang['company_name'] = 'Company Name';
$lang['company_email'] = 'Company Email';
$lang['company_phone'] = 'Company Phone';
$lang['company_website'] = 'Company Website';
$lang['company_address'] = 'Company Address';
$lang['file'] = 'File';
$lang['description'] = 'Description';
$lang['close'] = 'Close';
$lang['save'] = 'Save';
$lang['firstname'] = 'First Name';
$lang['lastname'] = 'Last Name';
$lang['email'] = 'Email';
$lang['phonenumber'] = 'Phone Number';




model



<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals_model extends App_Model
{
    public function get_all()
    {
        return $this->db->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
    }

    public function get_stages_with_deals($pipelineId = null)
    {
        if (!$pipelineId) {
            $pipelineId = $this->get_default_pipeline_id();
        }
        $stages = $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
        foreach ($stages as &$stage) {
            $stage['deals'] = $this->db->where('stage_id', (int) $stage['id'])->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
        }
        return $stages;
    }

    public function get_one($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals')->row_array();
    }

    public function update_deal($id, array $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals');
        return $this->db->affected_rows() > 0;
    }

    public function add_attachment($data)
    {
        $this->db->insert(db_prefix() . 'deals_attachments', $data);
        return $this->db->insert_id();
    }

    public function get_attachments($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)->get(db_prefix() . 'deals_attachments')->result_array();
    }



    public function create($data)
    {
        $insert = [
            'name' => $data['name'] ?? null,
            'company' => $data['company'] ?? null,
            'description' => $data['description'] ?? null,
            'status' => $data['status'] ?? null,
            'pipeline_id' => $data['pipeline_id'] ?? $this->get_default_pipeline_id(),
            'stage_id' => $data['stage_id'] ?? $this->get_default_open_stage_id(),
            'deal_value' => $data['deal_value'] ?? 0,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'status_final' => 'open',
        ];
        $this->db->insert(db_prefix() . 'deals', $insert);
        return $this->db->insert_id();
    }

    public function update_status($id, $status)
    {
        $statusId = $this->map_status_to_stage($status);
        $payload = [
            'last_status_change' => date('Y-m-d H:i:s'),
            'status' => $statusId,
        ];
        if ($status === 'won' || $status === 'lost') {
            $payload['status_final'] = $status;
        }
        $this->db->where('id', $id)->update(db_prefix() . 'deals', $payload);
        return $this->db->affected_rows() > 0;
    }

    public function mark_won($id)
    {
        return $this->update_status($id, 'won');
    }

    public function mark_lost($id, $reasonId = null)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'lost',
            'lost_reason_id' => $reasonId,
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }

    private function map_status_to_stage($status)
    {
        $statusName = null;
        if ($status === 'won') {
            $statusName = 'Won';
        } elseif ($status === 'lost') {
            $statusName = 'Lost';
        } else {
            $statusName = 'Open';
        }

        $row = $this->db->where('name', $statusName)->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : $this->get_default_open_stage_id();
    }

    // Pipelines helpers
    public function get_default_pipeline_id()
    {
        $row = $this->db->order_by('`order`', 'ASC')->limit(1)->get(db_prefix() . 'deals_pipelines')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_default_open_stage_id()
    {
        $pipelineId = $this->get_default_pipeline_id();
        if (!$pipelineId) {
            return null;
        }
        $row = $this->db->where(['pipeline_id' => $pipelineId, 'is_won' => 0, 'is_lost' => 0])->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_pipelines()
    {
        return $this->db->order_by('`order`', 'ASC')->get(db_prefix() . 'deals_pipelines')->result_array();
    }

    public function create_pipeline($name)
    {
        $this->db->insert(db_prefix() . 'deals_pipelines', ['name' => $name, 'order' => (int) ($this->db->count_all(db_prefix() . 'deals_pipelines') + 1)]);
        return $this->db->insert_id();
    }

    public function update_pipeline($id, $data)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals_pipelines', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_pipeline($id)
    {
        $this->db->where('id', $id)->delete(db_prefix() . 'deals_pipelines');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages($pipelineId)
    {
        return $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
    }

    public function create_stage($pipelineId, $data)
    {
        $data['pipeline_id'] = (int) $pipelineId;
        if (!isset($data['position'])) {
            $data['position'] = (int) $this->db->where('pipeline_id', (int) $pipelineId)->count_all_results(db_prefix() . 'deals_stages') + 1;
        }
        $this->db->insert(db_prefix() . 'deals_stages', $data);
        return $this->db->insert_id();
    }

    public function update_stage($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_stages', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_stage($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_stages');
        return $this->db->affected_rows() > 0;
    }

    public function reorder_stages($pipelineId, $orderedIds)
    {
        $position = 1;
        foreach ((array) $orderedIds as $stageId) {
            $this->db->where('id', (int) $stageId)->where('pipeline_id', (int) $pipelineId)->update(db_prefix() . 'deals_stages', ['position' => $position]);
            $position++;
        }
        return true;
    }

    // Lost reasons CRUD
    public function get_lost_reasons()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_lost_reasons')->result_array();
    }

    public function get_lost_reason($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_lost_reasons')->row_array();
    }

    public function get_staff()
    {
        return $this->db->select('staffid, firstname, lastname')
            ->where('active', 1)
            ->order_by('firstname', 'ASC')
            ->get(db_prefix() . 'staff')
            ->result_array();
    }

    public function get_stage_name($stage_id)
    {
        $stage = $this->db->where('id', (int) $stage_id)->get(db_prefix() . 'deals_stages')->row_array();
        return $stage ? $stage['name'] : 'Unknown Stage';
    }

    public function create_lost_reason($name)
    {
        $this->db->insert(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->insert_id();
    }
    public function update_lost_reason($id, $name)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->affected_rows() > 0;
    }
    public function delete_lost_reason($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_lost_reasons');
        return $this->db->affected_rows() > 0;
    }

    // Deal Contacts Methods
    public function get_deal_contacts($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_contact_links l')
            ->join(db_prefix() . 'deals_contacts_book cb', 'cb.id = l.contact_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_contact_to_deal($deal_id, $contact_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'contact_id' => (int) $contact_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_contact_links', $data);
        return $this->db->insert_id();
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('contact_id', (int) $contact_id)
            ->delete(db_prefix() . 'deal_contact_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_contacts($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_contacts_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT contact_id FROM ' . db_prefix() . 'deal_contact_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.firstname', 'ASC')->get()->result_array();
    }

    public function create_module_contact($data)
    {
        $insert = [
            'firstname' => $data['firstname'] ?? '',
            'lastname' => $data['lastname'] ?? '',
            'email' => $data['email'] ?? '',
            'phonenumber' => $data['phonenumber'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1,
        ];
        $this->db->insert(db_prefix() . 'deals_contacts_book', $insert);
        return $this->db->insert_id();
    }

    // Deal Companies Methods
    public function get_deal_companies($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_company_links l')
            ->join(db_prefix() . 'deals_companies_book cb', 'cb.id = l.company_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_company_to_deal($deal_id, $company_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'company_id' => (int) $company_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_company_links', $data);
        return $this->db->insert_id();
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('company_id', (int) $company_id)
            ->delete(db_prefix() . 'deal_company_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_companies($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_companies_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT company_id FROM ' . db_prefix() . 'deal_company_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.name', 'ASC')->get()->result_array();
    }

    // Company CRUD Methods (module-specific)
    public function create_company($data)
    {
        // Check if email or phone already exists
        if (!empty($data['email'])) {
            $existing = $this->db->where('email', $data['email'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Email already exists', 'existing_id' => $existing['id']];
            }
        }

        if (!empty($data['phone'])) {
            $existing = $this->db->where('phone', $data['phone'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Phone number already exists', 'existing_id' => $existing['id']];
            }
        }

        $insert = [
            'name' => $data['name'] ?? '',
            'email' => $data['email'] ?? '',
            'phone' => $data['phone'] ?? '',
            'website' => $data['website'] ?? '',
            'address' => $data['address'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1
        ];

        $this->db->insert(db_prefix() . 'deals_companies_book', $insert);
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function get_company($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_companies_book')->row_array();
    }

    public function update_company($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_companies_book', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_company($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_companies_book');
        return $this->db->affected_rows() > 0;
    }

    // Deal Attachments Relations Methods
    public function get_deal_attachments($deal_id)
    {
        $this->db->select('da.*, dar.dateadded as linked_date')
            ->from(db_prefix() . 'deal_attachments_relations dar')
            ->join(db_prefix() . 'deals_attachments da', 'da.id = dar.attachment_id', 'left')
            ->where('dar.deal_id', (int) $deal_id)
            ->order_by('dar.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function get_attachment($attachment_id)
    {
        return $this->db->where('id', (int) $attachment_id)->get(db_prefix() . 'deals_attachments')->row_array();
    }

    public function add_attachment_to_deal($deal_id, $attachment_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'attachment_id' => (int) $attachment_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];

        $this->db->insert(db_prefix() . 'deal_attachments_relations', $data);
        return $this->db->insert_id();
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('attachment_id', (int) $attachment_id)
            ->delete(db_prefix() . 'deal_attachments_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_attachments($deal_id = null)
    {
        $this->db->select('da.*')
            ->from(db_prefix() . 'deals_attachments da');

        if ($deal_id) {
            $this->db->where('da.id NOT IN (SELECT attachment_id FROM ' . db_prefix() . 'deal_attachments_relations WHERE deal_id = ' . (int) $deal_id . ')');
        }

        return $this->db->order_by('da.dateadded', 'DESC')->get()->result_array();
    }

    // Deal Activities Methods with type
    public function get_deal_activities($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('date', 'DESC')
            ->get(db_prefix() . 'deals_activity')
            ->result_array();
    }

    public function add_deal_activity($deal_id, $payload)
    {
        $data = [
            'title' => $payload['title'] ?? '',
            'rel_id' => (int) $deal_id,
            'description' => $payload['description'] ?? '',
            'additional_data' => $payload['additional_data'] ?? null,
            'staffid' => get_staff_user_id(),
            'full_name' => get_staff_full_name(get_staff_user_id()),
            'date' => date('Y-m-d H:i:s'),
            'type' => $payload['type'] ?? 'activity',
            'activity_date' => $payload['activity_date'] ?? null,
            'reminder' => $payload['reminder'] ?? 0,
            'owner_id' => $payload['owner_id'] ?? get_staff_user_id(),
            'guests' => $payload['guests'] ?? null,
            'status' => $payload['status'] ?? 'pending',
        ];
        $this->db->insert(db_prefix() . 'deals_activity', $data);
        return $this->db->insert_id();
    }

    // Notes
    public function add_note($deal_id, $description)
    {
        $noteData = [
            'rel_id' => (int) $deal_id,
            'description' => (string) $description,
            'addedfrom' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        $this->db->insert(db_prefix() . 'deals_notes', $noteData);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Note added',
                'additional_data' => $description,
                'type' => 'notes'
            ]);
        }
        return $insertId;
    }
    public function update_activity_status($id, $status)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_activity', ['status' => $status]);
        return $this->db->affected_rows() > 0;
    }

    public function get_notes($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('dateadded', 'DESC')
            ->get(db_prefix() . 'deals_notes')
            ->result_array();
    }

    // Calls
    public function add_call($deal_id, $payload)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'outcome' => $payload['outcome'] ?? '',
            'call_notes' => $payload['call_notes'] ?? null,
            'call_date' => !empty($payload['call_date']) ? $payload['call_date'] : date('Y-m-d H:i:s'),
            'staffid' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        $this->db->insert(db_prefix() . 'deals_calls', $data);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Call logged',
                'additional_data' => json_encode([
                    'outcome' => $data['outcome'],
                    'date' => $data['call_date'],
                    'notes' => $data['call_notes']
                ]),
                'type' => 'calls'
            ]);
        }
        return $insertId;
    }

    public function get_calls($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)
            ->order_by('call_date', 'DESC')
            ->get(db_prefix() . 'deals_calls')
            ->result_array();
    }
}



└──views
    ├── _kan_ban_card.php
    ├── create.php
    ├── deals.css
    ├── kanban.php
    ├── list.php
    ├── lost_reasons.php
    ├── settings.php
    └── view.phpthis are view file and in last install and deals file

<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
 
<li data-deal-id="<?= e($deal['id']); ?>"
    <?= isset($deal['is_deal_client']) && $deal['is_deal_client'] !== '0' ? 'data-toggle="tooltip" title="' . _l('deal_have_client_profile') . '"' : ''; ?>
    class="deal-kan-ban<?= $deal['assigned'] == get_staff_user_id() ? ' current-user-deal' : ''; ?><?= isset($deal['is_deal_client']) && $deal['is_deal_client'] !== '0' && get_option('deal_lock_after_convert_to_customer') == 1 && !is_admin() ? ' not-sortable' : ''; ?>">
    <div class="panel-body deal-body">
        <div class="tw-flex deal-name">
            <?php if ($deal['assigned'] != 0) { ?>
                <a href="<?= admin_url('profile/' . $deal['assigned']); ?>" data-placement="right" data-toggle="tooltip"
                    title="<?= e(get_staff_full_name($deal['assigned'])); ?>" class="mtop8 tw-mr-1.5">
                    <?= staff_profile_image($deal['assigned'], ['staff-profile-image-xs']); ?>
                </a>
            <?php } ?>
            <a href="javascript:void(0)" title="#<?= e($deal['id']) . ' - ' . e($deal['name'] ?? 'Unnamed Deal'); ?>"
                onclick="openViewDealModal(<?= e($deal['id']); ?>); return false;" class="tw-block tw-min-w-0 tw-font-medium">
                <span class="mtop10 mbot10 tw-truncate tw-block">
                    #<?= e($deal['id']) . ' - ' . e($deal['name'] ?? 'Unnamed Deal'); ?>
                </span>
            </a>
        </div>
        <div class="row">
            <div class="col-md-12">
                <div class="tw-flex">
                    <div class="tw-grow tw-mr-2">
                        <p class="tw-text-sm tw-mb-0">
                            <?= 'Source: ' . ($deal['source_name'] ?? 'Unknown'); ?>
                        </p>
                        <?php
                        $deal_value = isset($deal['deal_value']) && $deal['deal_value'] != 0
                            ? app_format_money($deal['deal_value'], $base_currency->symbol)
                            : '--';
                        ?>
                        <p class="tw-text-sm tw-mb-0">
                            <?= 'Deal Value: ' . $deal_value; ?>
                        </p>
 
                    </div>
                    <div class="text-right">
                        <?php if (isset($deal['lastcontact']) && is_date($deal['lastcontact']) && $deal['lastcontact'] != '0000-00-00 00:00:00') { ?>
                            <small class="text-dark tw-text-sm">
                                <?= _l('deals_dt_last_contact'); ?>
                                <span class="bold">
                                    <span class="text-has-action" data-toggle="tooltip" data-title="<?= e(_dt($deal['lastcontact'])); ?>">
                                        <?= e(time_ago($deal['lastcontact'])); ?>
                                    </span>
                                </span>
                            </small><br />
                        <?php } ?>
                        <small class="text-dark">
                            <?= _l('deal_created'); ?>:
                            <span class="bold">
                                <span class="text-has-action" data-toggle="tooltip" data-title="<?= e(_dt($deal['dateadded'])); ?>">
 
                                    <?= e(time_ago($deal['dateadded'])); ?>
                                </span>
                            </span>
                        </small><br />
                        <?php hooks()->do_action('before_deals_kanban_card_icons', $deal); ?>
                        <span class="mright5 mtop5 inline-block text-muted" data-toggle="tooltip" data-placement="left"
                            data-title="<?= _l('deals_canban_notes', $deal['total_notes'] ?? 0); ?>">
                            <i class="fa-regular fa-note-sticky"></i> <?= e($deal['total_notes'] ?? 0); ?>
                        </span>
                        <span class="mtop5 inline-block text-muted" data-placement="left" data-toggle="tooltip"
                            data-title="<?= _l('deal_kan_ban_attachments', $deal['total_files'] ?? 0); ?>">
                            <i class="fa fa-paperclip"></i> <?= e($deal['total_files'] ?? 0); ?>
                        </span>
                        <?php hooks()->do_action('after_deals_kanban_card_icons', $deal); ?>
                    </div>
                </div>
            </div>
            <?php if (!empty($deal['tags'] ?? [])) { ?>
                <div class="col-md-12">
                    <div class="kanban-tags tw-text-sm tw-inline-flex">
                        <?= render_tags($deal['tags'] ?? []); ?>
                    </div>
                </div>
            <?php } ?>
            <a href="javascript:void(0)" class="pull-right text-muted kan-ban-expand-top"
                onclick="$('#kan-ban-expand-<?= e($deal['id']); ?>').slideToggle(); return false;">
                <i class="fa fa-expand" aria-hidden="true"></i>
            </a>
 
            <div class="clearfix no-margin"></div>
            <div id="kan-ban-expand-<?= e($deal['id']); ?>" class="padding-10" style="display:none;">
                <div class="clearfix"></div>
                <hr class="hr-10" />
                <p class="text-muted deal-field-heading"><?= _l('deal_title'); ?></p>
                <p class="bold tw-text-sm"><?= e($deal['name'] ?? '-'); ?></p>
                <p class="text-muted deal-field-heading"><?= _l('deal_add_edit_email'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['email']) ? '<a href="mailto:' . e($deal['email']) . '">' . e($deal['email']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_website'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['website']) ? '<a href="' . e(maybe_add_http($deal['website'])) . '" target="_blank">' . e($deal['website']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_add_edit_phonenumber'); ?></p>
                <p class="bold tw-text-sm">
                    <?= !empty($deal['phonenumber']) ? '<a href="tel:' . e($deal['phonenumber']) . '">' . e($deal['phonenumber']) . '</a>' : '-'; ?>
                </p>
                <p class="text-muted deal-field-heading"><?= _l('deal_company'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['company']) ? $deal['company'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_address'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['address']) ? $deal['address'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_city'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['city']) ? $deal['city'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_state'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['state']) ? $deal['state'] : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_country'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['country']) && $deal['country'] != 0 ? get_country($deal['country'])->short_name : '-'); ?></p>
 
                <p class="text-muted deal-field-heading"><?= _l('deal_zip'); ?></p>
                <p class="bold tw-text-sm"><?= e(!empty($deal['zip']) ? $deal['zip'] : '-'); ?></p>
 
            </div>
        </div>
</li>






<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-8 col-md-offset-2">
        <div class="panel_s">
          <div class="panel-body">
            <h4 class="tw-mt-0"><?php echo isset($deal) ? _l('edit_deal') : _l('new_deal'); ?></h4>
            <hr class="hr-panel-heading" />
            <?php echo form_open(admin_url(isset($deal) ? 'deals/edit/'.$deal['id'] : 'deals/create')); ?>
              <div class="row">
                <div class="col-md-6">
                  <?php echo render_input('name', 'name', isset($deal) ? $deal['name'] : ''); ?>
                </div>
                <div class="col-md-6">
                  <?php echo render_input('company', 'company', isset($deal) ? $deal['company'] : ''); ?>
                </div>
                <div class="col-md-6">
                  <div class="form-group">
                    <label for="pipeline_id" class="control-label"><?php echo _l('pipeline'); ?></label>
                    <select name="pipeline_id" id="pipeline_id" class="selectpicker" data-width="100%" data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                      <?php foreach(($pipelines ?? []) as $pipeline){ ?>
                        <option value="<?php echo (int)$pipeline['id']; ?>" <?php echo (isset($deal) && $deal['pipeline_id'] == $pipeline['id']) ? 'selected' : ''; ?>>
                          <?php echo html_entity_decode($pipeline['name']); ?>
                        </option>
                      <?php } ?>
                    </select>
                  </div>
                </div>
                <?php if (isset($deal) && isset($stages)): ?>
                <div class="col-md-6">
                  <div class="form-group">
                    <label for="stage_id" class="control-label"><?php echo _l('stage'); ?></label>
                    <select name="stage_id" id="stage_id" class="selectpicker" data-width="100%" data-none-selected-text="<?php echo _l('dropdown_non_selected_tex'); ?>">
                      <?php foreach($stages as $stage){ ?>
                        <option value="<?php echo (int)$stage['id']; ?>" <?php echo ($deal['stage_id'] == $stage['id']) ? 'selected' : ''; ?>>
                          <?php echo html_entity_decode($stage['name']); ?>
                        </option>
                      <?php } ?>
                    </select>
                  </div>
                </div>
                <?php endif; ?>
                <div class="col-md-12">
                  <?php echo render_textarea('description', 'description', isset($deal) ? $deal['description'] : ''); ?>
                </div>
                <div class="col-md-4">
                  <?php echo render_input('deal_value', 'deal_value', isset($deal) ? $deal['deal_value'] : '', 'number'); ?>
                </div>
                <div class="col-md-4">
                  <?php echo render_date_input('expected_close_date', 'expected_close_date', isset($deal) ? $deal['expected_close_date'] : ''); ?>
                </div>
              </div>
              <div class="text-right">
                <button type="submit" class="btn btn-primary"><?php echo _l('submit'); ?></button>
              </div>
            <?php echo form_close(); ?>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>
<?php init_tail(); ?>






/* Deals Module Custom Styles */

/* Pipeline Stages */
.pipeline-stage {
    display: inline-flex;
    align-items: center;
    margin: 0 10px;
}

.pipeline-stage .stage-button {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    border: none;
    color: white;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
}

.pipeline-stage .stage-button:hover {
    transform: scale(1.1);
}

.pipeline-stage .stage-button:disabled {
    cursor: not-allowed;
    opacity: 0.7;
}

.pipeline-stage .stage-name {
    margin: 0 10px;
    font-weight: 500;
    color: #333;
}

.pipeline-stage .stage-arrow {
    margin: 0 10px;
    color: #ccc;
}

/* Activity Cards */
.activity-card {
    border-left: 4px solid #007cba;
    padding-left: 15px;
    margin-bottom: 20px;
    background: #f9f9f9;
    padding: 15px;
    border-radius: 5px;
}

.activity-card .activity-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 10px;
}

.activity-card .activity-description {
    font-weight: 500;
    color: #333;
    margin-bottom: 5px;
}

.activity-card .activity-user {
    font-size: 12px;
    color: #666;
}

.activity-card .activity-time {
    font-size: 12px;
    color: #999;
    text-align: right;
}

.activity-card .activity-actions {
    text-align: right;
}

/* Contact, Company, Attachment Items */
.item-list {
    margin-bottom: 10px;
}

.item-list .item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid #eee;
}

.item-list .item:last-child {
    border-bottom: none;
}

.item-list .item-info {
    display: flex;
    align-items: center;
}

.item-list .item-icon {
    margin-right: 8px;
    color: #007cba;
}

.item-list .item-name {
    font-weight: 500;
    color: #333;
}

.item-list .item-remove {
    background: #dc3545;
    color: white;
    border: none;
    border-radius: 3px;
    padding: 2px 6px;
    font-size: 10px;
    cursor: pointer;
}

.item-list .item-remove:hover {
    background: #c82333;
}

/* Modal Tabs */
.modal-tabs {
    margin-bottom: 20px;
}

.modal-tabs .nav-tabs {
    border-bottom: 2px solid #dee2e6;
}

.modal-tabs .nav-tabs > li > a {
    border: none;
    color: #666;
    padding: 10px 15px;
}

.modal-tabs .nav-tabs > li.active > a {
    border: none;
    border-bottom: 2px solid #007cba;
    color: #007cba;
    background: transparent;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 40px 20px;
    color: #999;
}

.empty-state .empty-icon {
    font-size: 48px;
    margin-bottom: 15px;
    color: #ccc;
}

.empty-state .empty-text {
    font-size: 14px;
    color: #999;
}

/* Responsive Design */
@media (max-width: 768px) {
    .pipeline-stage {
        margin: 5px;
    }
    
    .pipeline-stage .stage-name {
        display: none;
    }
    
    .activity-card .activity-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .activity-card .activity-time {
        text-align: left;
        margin-top: 5px;
    }
}

/* Custom Colors for Stages */
.stage-open { background-color: #28B8DA !important; }
.stage-won { background-color: #84c529 !important; }
.stage-lost { background-color: #fc2d42 !important; }
.stage-completed { background-color: #84c529 !important; }
.stage-pending { background-color: #e3e3e3 !important; }

/* Badge Styles */
.badge {
    background-color: #007cba;
    color: white;
    padding: 2px 6px;
    border-radius: 10px;
    font-size: 10px;
    margin-left: 5px;
}

/* Form Enhancements */
.form-group label {
    font-weight: 500;
    color: #333;
    margin-bottom: 5px;
}

.form-control {
    border-radius: 4px;
    border: 1px solid #ddd;
    padding: 8px 12px;
}

.form-control:focus {
    border-color: #007cba;
    box-shadow: 0 0 0 0.2rem rgba(0, 124, 186, 0.25);
}

/* Button Enhancements */
.btn-primary {
    background-color: #007cba;
    border-color: #007cba;
}

.btn-primary:hover {
    background-color: #005a8b;
    border-color: #005a8b;
}

.btn-success {
    background-color: #84c529;
    border-color: #84c529;
}

.btn-danger {
    background-color: #fc2d42;
    border-color: #fc2d42;
}






<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-12">
        <div class="panel_s">
          <div class="panel-body">
            <div class="tw-flex tw-justify-between tw-items-center">
              <h4 class="tw-m-0"><?php echo _l('deals_kanban'); ?></h4>
              <a href="<?php echo admin_url('deals'); ?>" class="btn btn-default"><i class="fa fa-list"></i> <?php echo _l('switch_to_list_view'); ?></a>
            </div>
            <hr class="hr-panel-heading" />
            <div class="kanban-wrapper">
              <div class="row">
                <?php foreach ($stages as $stage) { ?>
                  <div class="col-md-3">
                    <div class="kanban-col">
                      <div class="kanban-col-header">
                        <strong><?php echo html_entity_decode($stage['name']); ?></strong>
                      </div>
                      <div class="kanban-items">
                        <?php foreach ($stage['deals'] as $deal) { ?>
                          <div class="kanban-card panel_s">
                            <div class="panel-body">
                              <div class="tw-flex tw-justify-between tw-items-center">
                                <div>
                                  <strong><?php echo html_entity_decode($deal['name']); ?></strong>
                                  <div class="text-muted">#<?php echo (int)$deal['id']; ?> · <?php echo html_entity_decode($deal['company']); ?></div>
                                </div>
                                <div class="text-right">
                                  <div><?php echo app_format_money((float)$deal['deal_value'], get_base_currency()); ?></div>
                                </div>
                              </div>
                              <div class="tw-mt-2">
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-flex tw-gap-2">
                                  <input type="hidden" name="status" value="open">
                                  <button class="btn btn-xs btn-default" type="submit"><?php echo _l('mark_as_open'); ?></button>
                                </form>
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-inline-block tw-ml-2">
                                  <input type="hidden" name="status" value="won">
                                  <button class="btn btn-xs btn-success" type="submit"><?php echo _l('mark_as_won'); ?></button>
                                </form>
                                <form method="post" action="<?php echo admin_url('deals/update_status/' . (int)$deal['id']); ?>" class="tw-inline-block tw-ml-2">
                                  <input type="hidden" name="status" value="lost">
                                  <button class="btn btn-xs btn-danger" type="submit"><?php echo _l('mark_as_lost'); ?></button>
                                </form>
                              </div>
                            </div>
                          </div>
                        <?php } ?>
                      </div>
                    </div>
                  </div>
                <?php } ?>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>
<?php init_tail(); ?>








<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="_buttons tw-mb-2 sm:tw-mb-4">
                    <a href="<?php echo admin_url('deals/create'); ?>" class="btn btn-primary mright5 pull-left display-block">
                        <i class="fa-regular fa-plus tw-mr-1"></i>
                        <?php echo _l('new_deal'); ?>
                    </a>
                    <a href="<?php echo admin_url('deals?view=kanban'); ?>" class="btn btn-default pull-left display-block mleft5">
                        <i class="fa-solid fa-grip-vertical tw-mr-1"></i>
                        <?php echo _l('deals_kanban'); ?>
                    </a>
                    <div class="row">
                        <div class="col-sm-5">
                            <a href="#" class="btn btn-default btn-with-tooltip" data-toggle="tooltip"
                                data-title="<?php echo _l('deals_summary'); ?>" data-placement="top"
                                onclick="slideToggle('.deals-overview'); return false;">
                                <i class="fa fa-bar-chart"></i>
                            </a>
                        </div>
                        <div class="col-sm-4 col-xs-12 pull-right deals-search">
                            <div class="tw-inline pull-right">
                                <div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
                                    <a href="#" class="btn btn-default deals-table-order-btn" data-toggle="modal" data-target="#deals-table-order-modal" title="<?php echo _l('deals_table_order_configure'); ?>">
                                        <i class="fa fa-columns tw-mr-1"></i> <?php echo _l('deals_table_order_configure'); ?>
                                    </a>
                                </div>
                            </div>
                            <?php echo form_hidden('sort_type'); ?>
                            <?php echo form_hidden('sort', ''); ?>
                        </div>
                    </div>
                    <div class="clearfix"></div>
                    
                    <!-- Deals Summary Overview -->
                    <div class="hide deals-overview tw-mt-2 sm:tw-mt-4 tw-mb-4 sm:tw-mb-0">
                        <h4 class="tw-mt-0 tw-font-semibold tw-text-lg">
                            <?php echo _l('deals_summary'); ?>
                        </h4>
                        <div class="tw-flex tw-flex-wrap tw-flex-col lg:tw-flex-row tw-w-full tw-gap-3 lg:tw-gap-6">
                            <div class="lg:tw-border-r lg:tw-border-solid lg:tw-border-neutral-300 tw-flex-1 tw-flex tw-items-center last:tw-border-r-0">
                                <span class="tw-font-semibold tw-mr-3 rtl:tw-ml-3 tw-text-lg">
                                    <?php echo count($deals); ?>
                                </span>
                                <span class="tw-text-neutral-600"><?php echo _l('total_deals'); ?></span>
                            </div>
                            <div class="lg:tw-border-r lg:tw-border-solid lg:tw-border-neutral-300 tw-flex-1 tw-flex tw-items-center last:tw-border-r-0">
                                <span class="tw-font-semibold tw-mr-3 rtl:tw-ml-3 tw-text-lg">
                                    <?php 
                                    $total_value = 0;
                                    foreach($deals as $deal) {
                                        $total_value += (float)$deal['deal_value'];
                                    }
                                    echo app_format_money($total_value, get_base_currency());
                                    ?>
                                </span>
                                <span class="tw-text-neutral-600"><?php echo _l('total_value'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="panel_s">
                    <div class="panel-body">
                        <div class="row" id="deals-table">
                            <div class="col-md-12">
                                <!-- Bulk Actions Modal -->
                                <div class="modal fade bulk_actions" id="deals_bulk_actions" tabindex="-1" role="dialog">
                                    <div class="modal-dialog" role="document">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                                    <span aria-hidden="true">&times;</span>
                                                </button>
                                                <h4 class="modal-title"><?php echo _l('bulk_actions'); ?></h4>
                                            </div>
                                            <div class="modal-body">
                                                <div class="checkbox checkbox-danger">
                                                    <input type="checkbox" name="mass_delete" id="mass_delete">
                                                    <label for="mass_delete"><?php echo _l('mass_delete'); ?></label>
                                                </div>
                                                <hr class="mass_delete_separator" />
                                                <div id="bulk_change">
                                                    <div class="form-group">
                                                        <label for="move_to_pipeline_bulk"><?php echo _l('move_to_pipeline'); ?></label>
                                                        <select name="move_to_pipeline_bulk" id="move_to_pipeline_bulk" class="selectpicker" data-width="100%">
                                                            <option value=""><?php echo _l('select_pipeline'); ?></option>
                                                            <?php foreach($pipelines ?? [] as $pipeline) { ?>
                                                                <option value="<?php echo $pipeline['id']; ?>"><?php echo $pipeline['name']; ?></option>
                                                            <?php } ?>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
                                                <a href="#" class="btn btn-primary" onclick="deals_bulk_action(this); return false;"><?php echo _l('confirm'); ?></a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Deals Table -->
                                <div class="panel-table-full">
                                    <div class="table-responsive">
                                        <table class="table dt-table table-deals" data-order-col="0" data-order-type="desc">
                                            <thead>
                                                <tr>
                                                    <th><input type="checkbox" id="mass_select_all" data-to-table="deals"></th>
                                                    <th class="th-deal-number"><?php echo _l('the_number_sign'); ?></th>
                                                    <th class="th-deal-name"><?php echo _l('deal_name'); ?></th>
                                                    <th class="th-deal-company"><?php echo _l('company'); ?></th>
                                                    <th class="th-deal-pipeline"><?php echo _l('pipeline'); ?></th>
                                                    <th class="th-deal-stage"><?php echo _l('stage'); ?></th>
                                                    <th class="th-deal-value"><?php echo _l('deal_value'); ?></th>
                                                    <th class="th-deal-assigned"><?php echo _l('assigned_to'); ?></th>
                                                    <th class="th-deal-status"><?php echo _l('status'); ?></th>
                                                    <th class="th-deal-created"><?php echo _l('dateadded'); ?></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($deals as $deal) { ?>
                                                    <tr>
                                                        <td><input type="checkbox" class="individual" data-to-table="deals" value="<?php echo $deal['id']; ?>"></td>
                                                        <td><?php echo '#' . $deal['id']; ?></td>
                                                        <td>
                                                            <a href="<?php echo admin_url('deals/view/' . $deal['id']); ?>" class="tw-font-medium">
                                                                <?php echo html_entity_decode($deal['name']); ?>
                                                            </a>
                                                            <div class="row-options">
                                                                <a href="<?php echo admin_url('deals/view/' . $deal['id']); ?>"><?php echo _l('view'); ?></a>
                                                                <span> | </span>
                                                                <a href="<?php echo admin_url('deals/edit/' . $deal['id']); ?>"><?php echo _l('edit'); ?></a>
                                                                <span> | </span>
                                                                <a href="<?php echo admin_url('deals/delete/' . $deal['id']); ?>" class="text-danger _delete"><?php echo _l('delete'); ?></a>
                                                            </div>
                                                        </td>
                                                        <td><?php echo html_entity_decode($deal['company']); ?></td>
                                                        <td><?php echo isset($deal['pipeline_name']) ? $deal['pipeline_name'] : '-'; ?></td>
                                                        <td><?php echo isset($deal['stage_name']) ? $deal['stage_name'] : '-'; ?></td>
                                                        <td><?php echo app_format_money((float)$deal['deal_value'], get_base_currency()); ?></td>
                                                        <td><?php echo isset($deal['assigned_staff']) ? $deal['assigned_staff'] : '-'; ?></td>
                                                        <td>
                                                            <span class="label label-<?php echo ($deal['status_final'] == 'won' ? 'success' : ($deal['status_final'] == 'lost' ? 'danger' : 'default')); ?>">
                                                                <?php echo ucfirst($deal['status_final']); ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo _dt($deal['dateadded']); ?></td>
                                                    </tr>
                                                <?php } ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php init_tail(); ?>

<script>
$(function() {
    // Initialize DataTable
    $('.table-deals').DataTable({
        "order": [[ 1, "desc" ]],
        "pageLength": 25,
        "responsive": true,
        "columnDefs": [
            { "orderable": false, "targets": 0 }
        ]
    });
    
    // Handle mass select all
    $('#mass_select_all').on('change', function() {
        var checked = $(this).prop('checked');
        $('.individual').prop('checked', checked);
        toggleBulkActions();
    });
    
    // Handle individual checkboxes
    $('.individual').on('change', function() {
        toggleBulkActions();
        var allChecked = $('.individual:checked').length === $('.individual').length;
        $('#mass_select_all').prop('checked', allChecked);
    });
    
    function toggleBulkActions() {
        var checkedBoxes = $('.individual:checked').length;
        if (checkedBoxes > 0) {
            $('.bulk-actions-btn').removeClass('hide');
        } else {
            $('.bulk-actions-btn').addClass('hide');
        }
    }
});

function deals_bulk_action(button) {
    var checkedIds = [];
    $('.individual:checked').each(function() {
        checkedIds.push($(this).val());
    });
    
    if (checkedIds.length === 0) {
        alert('<?php echo _l('no_action_selected'); ?>');
        return;
    }
    
    var massDelete = $('#mass_delete').prop('checked');
    var pipelineId = $('#move_to_pipeline_bulk').val();
    
    if (massDelete) {
        if (confirm('<?php echo _l('confirm_bulk_delete'); ?>')) {
            // Perform delete action
            $.post(admin_url + 'deals/bulk_delete', {ids: checkedIds}, function(response) {
                if (response.success) {
                    location.reload();
                }
            });
        }
    } else if (pipelineId) {
        // Move to pipeline
        $.post(admin_url + 'deals/bulk_move', {ids: checkedIds, pipeline_id: pipelineId}, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    }
}
</script>







<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php
$CI = &get_instance();
if (!isset($reasons)) {
  $CI->load->model('deals/deals_model');
  $reasons = $CI->deals_model->get_lost_reasons();
}
?>

<div class="panel_s">
  <div class="panel-body">
    <div class="row">
      <div class="col-md-6">
        <div class="input-group">
          <input type="text" id="lost_reason_name" class="form-control" placeholder="<?php echo _l('lost_reason'); ?>">
          <span class="input-group-btn">
            <button class="btn btn-primary" onclick="lrCreate()"><?php echo _l('add'); ?></button>
          </span>
        </div>
        <ul class="list-group tw-mt-3">
          <?php foreach (($reasons ?? []) as $r) { ?>
            <li class="list-group-item tw-flex tw-justify-between tw-items-center">
              <span><?php echo html_entity_decode($r['name']); ?></span>
              <span>
                <button class="btn btn-xs btn-default" onclick="lrEdit(<?php echo (int)$r['id']; ?>,'<?php echo html_escape($r['name']); ?>')"><i class="fa fa-pencil"></i></button>
                <button class="btn btn-xs btn-danger" onclick="lrDelete(<?php echo (int)$r['id']; ?>)"><i class="fa fa-trash"></i></button>
              </span>
            </li>
          <?php } ?>
        </ul>
      </div>
    </div>
  </div>
</div>

<script>
var CSRF = { name: '<?php echo $this->security->get_csrf_token_name(); ?>', value: '<?php echo $this->security->get_csrf_hash(); ?>' };
function lrCreate(){
  var name = document.getElementById('lost_reason_name').value.trim();
  if(!name) return;
  var f = new FormData(); f.append('action','create'); f.append('name', name); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
function lrEdit(id, prev){
  var name = prompt('Lost reason', prev || '');
  if(name===null) return;
  var f = new FormData(); f.append('action','update'); f.append('id', id); f.append('name', name); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
function lrDelete(id){
  if(!confirm('Delete?')) return;
  var f = new FormData(); f.append('action','delete'); f.append('id', id); f.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/lost_reasons'); ?>',{method:'POST',body:f,credentials:'same-origin'}).then(()=>location.reload());
}
</script>








<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php
$CI = &get_instance();
if (!isset($pipelines)) {
    $CI->load->model('deals/deals_model');
    $pipelines = $CI->deals_model->get_pipelines();
}
// Ensure stages are present for each pipeline
if (is_array($pipelines)) {
    foreach ($pipelines as $idx => $p) {
        if (!isset($p['stages'])) {
            $pipelines[$idx]['stages'] = $CI->deals_model->get_stages($p['id']);
        }
    }
}
$selectedPipelineId = isset($pipelines[0]['id']) ? (int)$pipelines[0]['id'] : 0;
?>

<h4 class="tw-mb-4"><?php echo _l('pipelines'); ?></h4>

<div class="panel_s">
  <div class="panel-body">
    <div class="row">
      <div class="col-md-12">
        <h5><?php echo _l('deals'); ?> <?php echo _l('pipelines'); ?></h5>
        
        <!-- Pipeline Management Section -->
        <div class="form-group">
          <label class="control-label"><?php echo _l('pipelines'); ?></label>
          <select id="pipeline_select" class="selectpicker" data-width="100%" onchange="onPipelineChange()">
            <option value=""><?php echo _l('select_pipeline'); ?></option>
            <?php foreach (($pipelines ?? []) as $p) { ?>
              <option value="<?php echo (int)$p['id']; ?>" <?php echo ((int)$p['id']===$selectedPipelineId?'selected':''); ?> data-name="<?php echo html_escape($p['name']); ?>"><?php echo html_entity_decode($p['name']); ?></option>
            <?php } ?>
          </select>
        </div>
        
        <!-- Add/Edit Pipeline Form -->
        <div class="form-group">
          <div class="input-group">
            <input type="text" id="pipeline_name" class="form-control" placeholder="<?php echo _l('pipeline_name'); ?>">
            <span class="input-group-btn">
              <button id="pipeline_action_btn" class="btn btn-primary" type="button" onclick="createOrUpdatePipeline()"><?php echo _l('add'); ?></button>
            </span>
          </div>
        </div>
        
        <!-- Pipeline List -->
        <ul class="list-group" id="pipelines-list" style="max-height:60vh; overflow:auto;">
          <?php foreach (($pipelines ?? []) as $p) { ?>
            <li class="list-group-item pipeline-item" data-id="<?php echo (int)$p['id']; ?>" style="<?php echo ((int)$p['id']!==$selectedPipelineId?'display:none;':''); ?>">
              <div class="tw-flex tw-justify-between tw-items-center">
                <strong><?php echo html_entity_decode($p['name']); ?></strong>
                <div>
                  <button class="btn btn-xs btn-default" onclick="editPipeline(<?php echo (int)$p['id']; ?>)"><i class="fa fa-pencil"></i></button>
                  <button class="btn btn-xs btn-danger" onclick="deletePipeline(<?php echo (int)$p['id']; ?>)"><i class="fa fa-trash"></i></button>
                </div>
              </div>
              
              <!-- Stages Section -->
              <div class="tw-mt-2">
                <small><?php echo _l('stages'); ?></small>
                <ul class="list-unstyled sortable-stages" data-pipeline="<?php echo (int)$p['id']; ?>">
                  <?php foreach (($p['stages'] ?? []) as $s) { ?>
                    <li class="stage-item tw-py-2 tw-flex tw-items-center tw-justify-between" data-id="<?php echo (int)$s['id']; ?>">
                      <div class="tw-flex tw-items-center tw-gap-2">
                        <i class="fa fa-grip-vertical text-muted handle" style="cursor: move;"></i>
                        <input class="form-control input-sm" style="width:260px" value="<?php echo html_entity_decode($s['name']); ?>" onchange="updateStage(<?php echo (int)$s['id']; ?>,{name:this.value})">
                      </div>
                      <div class="tw-flex tw-items-center tw-gap-2">
                        <div class="tw-flex tw-items-center" style="width:260px">
                          <input type="range" min="0" max="100" value="<?php echo (int)($s['win_probability'] ?? 100); ?>" class="form-control" oninput="document.getElementById('prob_<?php echo (int)$s['id']; ?>').innerText=this.value;" onchange="updateStage(<?php echo (int)$s['id']; ?>,{win_probability:this.value})">
                          <span id="prob_<?php echo (int)$s['id']; ?>" class="tw-ml-2"><?php echo (int)($s['win_probability'] ?? 100); ?></span>
                        </div>
                        <button class="btn btn-xs btn-danger" onclick="deleteStage(<?php echo (int)$s['id']; ?>)"><i class="fa fa-trash"></i></button>
                      </div>
                    </li>
                  <?php } ?>
                </ul>
                <button class="btn btn-xs btn-success" onclick="addStage(<?php echo (int)$p['id']; ?>)"><i class="fa fa-plus"></i> <?php echo _l('add'); ?> <?php echo _l('stage'); ?></button>
              </div>
            </li>
          <?php } ?>
        </ul>
      </div>
    </div>
  </div>
</div>

<!-- Include jQuery UI for drag and drop -->
<link rel="stylesheet" href="//code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
<script src="https://code.jquery.com/ui/1.13.2/jquery-ui.js"></script>

<style>
.sortable-stages {
    min-height: 20px;
}

.stage-item {
    background: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    margin-bottom: 5px;
    padding: 8px 12px;
    transition: all 0.2s ease;
}

.stage-item:hover {
    background: #e9ecef;
    border-color: #adb5bd;
}

.stage-item.ui-sortable-helper {
    background: #fff;
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    transform: rotate(2deg);
}

.stage-item.ui-sortable-placeholder {
    background: #f8f9fa;
    border: 2px dashed #dee2e6;
    visibility: visible !important;
    height: 40px;
}

.handle {
    cursor: move;
    color: #6c757d;
}

.handle:hover {
    color: #495057;
}

.pipeline-item {
    margin-bottom: 20px;
    border: 1px solid #dee2e6;
    border-radius: 6px;
}

.pipeline-item .list-group-item {
    border: none;
    border-radius: 6px;
}

.form-group {
    margin-bottom: 20px;
}

.btn-xs {
    padding: 2px 8px;
    font-size: 12px;
}
</style>

<script>
var CSRF = { name: '<?php echo $this->security->get_csrf_token_name(); ?>', value: '<?php echo $this->security->get_csrf_hash(); ?>' };

function createOrUpdatePipeline(){
  var select = document.getElementById('pipeline_select');
  var id = select.value;
  var name = document.getElementById('pipeline_name').value.trim();
  if (!name) { 
    alert('<?php echo _l('pipeline_name_required'); ?>');
    return; 
  }
  
  if (id) { 
    request('update', {id:id, name:name}); 
  } else { 
    request('create', {name:name}); 
  }
}

function onPipelineChange(){
  var select = document.getElementById('pipeline_select');
  var id = select.value;
  var name = select.options[select.selectedIndex] ? select.options[select.selectedIndex].getAttribute('data-name') : '';
  
  // populate input and button state
  document.getElementById('pipeline_name').value = name || '';
  document.getElementById('pipeline_action_btn').innerText = name ? '<?php echo _l('edit'); ?>' : '<?php echo _l('add'); ?>';
  
  // toggle pipeline blocks
  Array.prototype.forEach.call(document.querySelectorAll('.pipeline-item'), function(el){
    el.style.display = (String(el.getAttribute('data-id')) === String(id)) ? '' : 'none';
  });
}

function editPipeline(id){
  var name = prompt('<?php echo _l('enter_pipeline_name'); ?>');
  if(name && name.trim()){ 
    request('update', {id:id, name:name.trim()}); 
  }
}

function deletePipeline(id){
  if(confirm('<?php echo _l('delete_pipeline_confirm'); ?>')){ 
    request('delete', {id:id}); 
  }
}

function addStage(pipeline_id){
  var name = prompt('<?php echo _l('enter_stage_name'); ?>');
  if(name && name.trim()){ 
    request('create_stage', {pipeline_id:pipeline_id, name:name.trim()}); 
  }
}

function updateStage(id, payload){
  payload = payload || {};
  payload.id = id;
  payload.action = 'update_stage';
  var form = new FormData();
  Object.keys(payload).forEach(function(k){ form.append(k, payload[k]); });
  form.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { method:'POST', body: form, credentials: 'same-origin' })
    .then(r=>r.json()).then(()=>location.reload());
}

function deleteStage(id){
  if(confirm('<?php echo _l('delete_stage_confirm'); ?>')){ 
    request('delete_stage', {id:id}); 
  }
}

function request(action, payload){
  var form = new FormData();
  form.append('action', action);
  Object.keys(payload||{}).forEach(function(k){ form.append(k, payload[k]); });
  form.append(CSRF.name, CSRF.value);
  fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { method:'POST', body: form, credentials: 'same-origin' })
    .then(r=>r.json())
    .then(()=>location.reload());
}

// Initialize drag and drop functionality
$(document).ready(function() {
  // Make stages sortable within each pipeline
  $('.sortable-stages').sortable({
    handle: '.handle',
    axis: 'y',
    cursor: 'move',
    opacity: 0.8,
    update: function(event, ui) {
      var pipeline = $(this).data('pipeline');
      var order = $(this).children('li').map(function(){ 
        return $(this).data('id'); 
      }).get();
      
      // Send reorder request to server
      var form = new FormData();
      form.append('action', 'reorder_stages');
      form.append('pipeline_id', pipeline);
      order.forEach(function(id){ 
        form.append('order[]', id); 
      });
      form.append(CSRF.name, CSRF.value);
      
      fetch('<?php echo admin_url('deals/settings/pipelines'); ?>', { 
        method: 'POST', 
        body: form, 
        credentials: 'same-origin' 
      }).then(r => r.json()).then(data => {
        if (data.success) {
          // Show success message
          console.log('Stages reordered successfully');
        }
      });
    }
  });
  
  // Initialize default view
  onPipelineChange();
});
</script>









<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>

<?php init_head(); ?>
<link rel="stylesheet" href="<?php echo base_url('modules/deals/views/deals.css'); ?>">
<div id="wrapper">
  <div class="content">
    <div class="row">
      <div class="col-md-12">
        <div class="panel_s">
          <div class="panel-body">
            <!-- Header Section -->
            <div class="row">
              <div class="col-md-8">
                <h4 class="tw-m-0"><?php echo html_entity_decode($deal['name']); ?></h4>
                <p class="text-muted tw-mt-1">
                  <?php echo _l('deal_value'); ?>:
                  <?php echo app_format_money((float) $deal['deal_value'], get_base_currency()); ?> |
                  <?php echo _l('expected_close_date'); ?>:
                  <?php echo isset($deal['expected_close_date']) ? _d($deal['expected_close_date']) : '-'; ?>
                </p>
                <p class="text-muted tw-mt-1">
                  <?php echo _l('created_at'); ?>: <?php echo _dt($deal['dateadded']); ?>
                </p>
              </div>
              <div class="col-md-4 text-right">
                <div class="tw-flex tw-gap-2 tw-justify-end">
                  <button type="button" class="btn btn-info btn-xs" onclick="testDatabase()">
                    <i class="fa fa-database"></i> Test DB
                  </button>
                  <?php if ($deal['status_final'] === 'won') { ?>
                    <span class="btn btn-success" style="cursor: default;"><?php echo _l('won'); ?></span>
                  <?php } elseif ($deal['status_final'] === 'lost') { ?>
                    <span class="btn btn-danger" style="cursor: default;"><?php echo _l('lost'); ?></span>
                    <?php if (isset($deal['lost_reason_name']) && $deal['lost_reason_name']) { ?>
                      <span class="btn btn-default"
                        style="cursor: default;"><?php echo html_entity_decode($deal['lost_reason_name']); ?></span>
                    <?php } ?>
                  <?php } else { ?>
                    <button type="button" class="btn btn-success"
                      onclick="markDealWon(<?php echo (int) $deal['id']; ?>)"><?php echo _l('mark_as_won'); ?></button>
                    <button type="button" class="btn btn-danger"
                      onclick="showLostReasonModal()"><?php echo _l('mark_as_lost'); ?></button>
                  <?php } ?>
                </div>
              </div>
            </div>

            <hr class="hr-panel-heading" />

            <!-- Pipeline Stages -->
            <div class="tw-flex tw-gap-2 tw-flex-wrap tw-mb-4 tw-justify-center">
              <?php
              $final = $deal['status_final'];
              $currentStage = (int) $deal['stage_id'];
              $stageCount = count($stages);

              foreach ($stages as $index => $stage) {
                $stageId = (int) $stage['id'];
                if ($final === 'won') {
                  $bg = '#84c529';
                } elseif ($final === 'lost') {
                  $bg = '#fc2d42';
                } else {
                  $currentIndex = -1;
                  foreach ($stages as $idx => $s) {
                    if ((int) $s['id'] === $currentStage) {
                      $currentIndex = $idx;
                      break;
                    }
                  }
                  if ($index < $currentIndex) {
                    $bg = '#84c529';
                  } elseif ($index === $currentIndex) {
                    $bg = '#FFD600';
                  } else {
                    $bg = '#e3e3e3';
                  }
                }
                ?>
                <div class="tw-flex tw-items-center">
                  <button type="button" class="btn"
                    style="background: <?php echo $bg; ?>; color:#fff; border-radius: 50%; width: 40px; height: 40px; padding: 0; font-size: 16px;"
                    onclick="setDealStage(<?php echo (int) $deal['id']; ?>, <?php echo (int) $stage['id']; ?>)" <?php echo ($final === 'won' || $final === 'lost') ? 'disabled' : ''; ?>>
                    <?php
                    if ($final === 'won' || $final === 'lost') {
                      // Always show check icon for all stages if deal is won/lost
                      echo '<i class="fa fa-check"></i>';
                    } else {
                      $currentIndex = -1;
                      foreach ($stages as $idx => $s) {
                        if ((int) $s['id'] === $currentStage) {
                          $currentIndex = $idx;
                          break;
                        }
                      }
                      if ($index < $currentIndex) {
                        // Completed stage: check icon
                        echo '<i class="fa fa-check"></i>';
                      } elseif ($index === $currentIndex) {
                        // Current stage: dot icon
                        echo '<i class="fa fa-exclamation"></i>';
                      } else {
                        // Future stage: show number
                        echo ($index + 1);
                      }
                    }
                    ?>
                  </button>
                  <span class="tw-mx-2 tw-text-sm tw-font-medium"><?php echo html_entity_decode($stage['name']); ?></span>
                  <?php if ($index < $stageCount - 1) { ?>
                    <i class="fa fa-arrow-right tw-mx-2 tw-text-gray-400"></i>
                  <?php } ?>
                </div>
              <?php } ?>
            </div>

            <!-- Main Content: Left Sidebar (33%) and Right Content (66%) -->
            <div class="row">
              <!-- Left Sidebar - Details, Contacts, Companies, Attachments -->
              <div class="col-md-4">
                <!-- Details Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title"><?php echo _l('details'); ?></h5>
                  </div>
                  <div class="modern-card-body">
                    <div class="form-group deal-value-group" style="position:relative;">
                      <label class="tw-font-medium"><?php echo _l('deal_value'); ?></label>
                      <p class="tw-text-lg tw-font-bold tw-text-green-600 deal-value-display"
                        style="margin-bottom:0; display:inline-block;">
                        <?php echo app_format_money((float) $deal['deal_value'], get_base_currency()); ?>
                        <button class="btn btn-xs btn-default tw-ml-2 edit-amount-btn"
                          style="display:none; position:relative; top:-2px;" onclick="editDealAmount()">
                          <i class="fa fa-edit"></i>
                        </button>
                      </p>
                      <form id="edit-amount-form" style="display:none; margin-top:8px;">
                        <div class="input-group input-group-sm" style="max-width:180px;">
                          <input type="number" step="0.01" min="0" class="form-control" id="edit-amount-input"
                            value="<?php echo (float) $deal['deal_value']; ?>">
                          <span class="input-group-btn">
                            <button class="btn btn-success" type="submit"><i class="fa fa-check"></i></button>
                            <button class="btn btn-default" type="button" onclick="cancelEditAmount()"><i
                                class="fa fa-times"></i></button>
                          </span>
                        </div>
                      </form>
                    </div>
                    <div class="form-group close-date-group" style="position:relative;">
                      <label class="tw-font-medium"><?php echo _l('expected_close_date'); ?></label>
                      <p class="tw-text-gray-700 close-date-display" style="margin-bottom:0; display:inline-block;">
                        <?php echo isset($deal['expected_close_date']) ? _d($deal['expected_close_date']) : '-'; ?>
                        <button class="btn btn-xs btn-default tw-ml-2 edit-close-date-btn"
                          style="display:none; position:relative; top:-2px;" onclick="editCloseDate()">
                          <i class="fa fa-edit"></i>
                        </button>
                      </p>
                      <form id="edit-close-date-form" style="display:none; margin-top:8px;">
                        <div class="input-group input-group-sm" style="max-width:180px;">
                          <input type="date" class="form-control" id="edit-close-date-input"
                            value="<?php echo isset($deal['expected_close_date']) ? date('Y-m-d', strtotime($deal['expected_close_date'])) : ''; ?>">
                          <span class="input-group-btn">
                            <button class="btn btn-success" type="submit"><i class="fa fa-check"></i></button>
                            <button class="btn btn-default" type="button" onclick="cancelEditCloseDate()"><i
                                class="fa fa-times"></i></button>
                          </span>
                        </div>
                      </form>
                    </div>
                  </div>
                </div>

                <!-- Contacts Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('contacts'); ?>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addContact()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="contacts-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-users fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_contacts_found'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>

                <!-- Companies Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('companies'); ?> <span id="companies-count">(0)</span>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addCompany()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="companies-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-building fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_company_associated'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>

                <!-- Attachments Card -->
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title">
                      <?php echo _l('attachments'); ?>
                      <button class="btn btn-xs btn-primary pull-right" onclick="addAttachment()">
                        <i class="fa fa-plus"></i>
                      </button>
                    </h5>
                  </div>
                  <div class="modern-card-body">
                    <div id="attachments-list">
                      <div class="text-muted text-center tw-py-4">
                        <i class="fa fa-paperclip fa-2x tw-mb-2"></i>
                        <p><?php echo _l('no_attachments_uploaded'); ?></p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              <!-- Right Side - Activities -->
              <div class="col-md-8">
                <div class="modern-card">
                  <div class="modern-card-header">
                    <h5 class="modern-card-title"><?php echo _l('activities'); ?></h5>
                  </div>
                  <div class="modern-card-body">
                    <!-- Activity Tabs -->
                    <ul class="nav nav-tabs" role="tablist">
                      <li role="presentation" class="active">
                        <a href="#all-activities" aria-controls="all-activities" role="tab"
                          data-toggle="tab"><?php echo _l('all'); ?></a>
                      </li>
                      <li role="presentation">
                        <a href="#activities-tab" aria-controls="activities-tab" role="tab"
                          data-toggle="tab"><?php echo _l('activities'); ?> <span class="badge"
                            id="badge-activities">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#emails-tab" aria-controls="emails-tab" role="tab"
                          data-toggle="tab"><?php echo _l('emails'); ?> <span class="badge"
                            id="badge-emails">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#documents-tab" aria-controls="documents-tab" role="tab"
                          data-toggle="tab"><?php echo _l('documents'); ?> <span class="badge"
                            id="badge-documents">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#calls-tab" aria-controls="calls-tab" role="tab"
                          data-toggle="tab"><?php echo _l('calls'); ?> <span class="badge" id="badge-calls">0</span></a>
                      </li>
                      <li role="presentation">
                        <a href="#notes-tab" aria-controls="notes-tab" role="tab"
                          data-toggle="tab"><?php echo _l('notes'); ?> <span class="badge" id="badge-notes">0</span></a>
                      </li>
                    </ul>

                    <!-- Tab Content -->
                    <div class="tab-content tw-mt-3">
                      <div role="tabpanel" class="tab-pane active" id="all-activities">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <select class="form-control tw-w-auto" id="activity-filter">
                            <option value="all"><?php echo _l('filter_by_all'); ?></option>
                            <option value="activities"><?php echo _l('activities'); ?></option>
                            <option value="emails"><?php echo _l('emails'); ?></option>
                            <option value="documents"><?php echo _l('documents'); ?></option>
                            <option value="calls"><?php echo _l('calls'); ?></option>
                            <option value="notes"><?php echo _l('notes'); ?></option>
                          </select>
                        </div>
                        <div class="timeline-container" id="activities-list">
                          <!-- Activities will be loaded here -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="activities-tab">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="activities-search"
                            placeholder="Search...">
                          <button class="btn btn-primary btn-sm"
                            onclick="toggleAddActivityForm()"><?php echo _l('add_activity'); ?></button>
                        </div>
                        <div id="add-activity-form" style="display: none;" class="tw-mb-3">
                          <div class="form-group">
                            <label>* Title</label>
                            <input type="text" id="activity-title" class="form-control" required>
                          </div>
                          <div class="tw-flex tw-gap-2">
                            <div class="form-group tw-flex-1">
                              <label>Date</label>
                              <input type="date" id="activity-date" class="form-control"
                                value="<?php echo date('Y-m-d'); ?>">
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>Time</label>
                              <input type="time" id="activity-time" class="form-control"
                                value="<?php echo date('H:i'); ?>">
                            </div>
                          </div>
                          <div class="tw-flex tw-gap-2">
                            <div class="form-group tw-flex-1">
                              <label>Reminder</label>
                              <input type="number" id="reminder-num" class="form-control" min="0" value="30">
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>&nbsp;</label>
                              <select id="reminder-unit" class="form-control">
                                <option value="minutes">minutes</option>
                                <option value="hours">hours</option>
                                <option value="days">days</option>
                              </select>
                            </div>
                            <div class="form-group tw-flex-1">
                              <label>&nbsp;</label>
                              <select id="reminder-when" class="form-control">
                                <option value="before due">before due</option>
                                <option value="after due">after due</option>
                              </select>
                            </div>
                          </div>
                          <div class="form-group">
                            <label>* Owner</label>
                            <select id="activity-owner" class="form-control">
                              <option value="">Admin</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label>+ Guests</label>
                            <select id="activity-guests" class="form-control" multiple>
                            </select>
                          </div>
                          <div class="form-group">
                            <label>+ Description</label>
                            <textarea id="activity-description" class="form-control" rows="4"></textarea>
                          </div>
                          <div class="form-group">
                            <label>Note</label>
                            <textarea id="activity-note" class="form-control"
                              placeholder="Notes are private and visible only for the sales reps." rows="4"></textarea>
                          </div>
                          <p>Associated with 1 record</p>
                          <label><input type="checkbox" id="activity-completed"> Mark as completed</label>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" onclick="toggleAddActivityForm()">Cancel</button>
                            <button class="btn btn-primary" onclick="saveActivity()">Add Activity</button>
                          </div>
                        </div>
                        <div class="tw-flex tw-overflow-x-auto tw-gap-2 tw-mb-3" id="activity-periods">
                          <button class="active" data-period="all">All (1)</button>
                          <button data-period="today">Today (0)</button>
                          <button data-period="tomorrow">Tomorrow (0)</button>
                          <button data-period="this_week">This Week (0)</button>
                          <button data-period="next_week">Next Week (0)</button>
                          <button data-period="done">Done (0)</button>
                        </div>
                        <div class="list-container" id="activities-only-list">
                          <!-- Activities only -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="emails-tab">
                        <div class="timeline-container" id="emails-list">
                          <!-- Emails -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="documents-tab">
                        <div class="timeline-container" id="documents-list">
                          <!-- Documents -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="calls-tab">
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="calls-search"
                            placeholder="Search calls...">
                          <button class="btn btn-primary btn-sm" onclick="toggleLogCallForm()">+ Log Call</button>
                        </div>
                        <div id="log-call-form" style="display: none;" class="tw-mb-3 tw-bg-gray-50 tw-p-4 tw-rounded">
                          <div class="form-group">
                            <label for="call-type"><span class="text-danger">*</span> Call Type</label>
                            <select id="call-type" class="form-control" required>
                              <option value="">Select Type</option>
                              <option value="inbound">Inbound</option>
                              <option value="outbound">Outbound</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label for="call-status"><span class="text-danger">*</span> Call Status</label>
                            <select id="call-status" class="form-control" required>
                              <option value="">Select Status</option>
                              <option value="completed">Completed</option>
                              <option value="missed">Missed</option>
                              <option value="scheduled">Scheduled</option>
                            </select>
                          </div>
                          <div class="form-group">
                            <label for="call-duration">Call Duration (minutes)</label>
                            <input type="number" min="0" id="call-duration" class="form-control"
                              placeholder="Duration in minutes">
                          </div>
                          <div class="form-group">
                            <label for="call-notes">Call Notes</label>
                            <textarea id="call-notes" class="form-control" rows="3"
                              placeholder="E.g. Spoke with client, discussed next steps..."></textarea>
                          </div>
                          <div class="form-group">
                            <label for="call-date"><span class="text-danger">*</span> Call Date &amp; Time</label>
                            <input type="datetime-local" id="call-date" class="form-control"
                              value="<?php echo date('Y-m-d\TH:i'); ?>">
                          </div>
                          <p class="tw-text-xs tw-text-gray-500">Associated with 1 record</p>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" type="button" onclick="toggleLogCallForm()">Cancel</button>
                            <button class="btn btn-primary" type="button" onclick="saveCall()">Log Call</button>
                          </div>
                        </div>
                        <div class="list-container" id="calls-list">
                          <!-- Calls -->
                        </div>
                      </div>

                      <div role="tabpanel" class="tab-pane" id="notes-tab">
                        <div class="tw-mb-3">
                          <h5>Manage Notes</h5>
                          <p>You can create notes for you and your team and keep track of important info here.</p>
                        </div>
                        <div class="tw-flex tw-justify-between tw-mb-3">
                          <input type="text" class="form-control tw-w-1/2" id="notes-search" placeholder="Search...">
                          <button class="btn btn-primary btn-sm" onclick="toggleAddNoteForm()">+ Add Note</button>
                        </div>
                        <div id="add-note-form" style="display: none;" class="tw-mb-3">
                          <div class="form-group">
                            <textarea id="note-description" class="form-control" rows="4"
                              placeholder="Paragraph"></textarea>
                          </div>
                          <p>Notes are private and visible only for the sales reps.</p>
                          <p>Associated with 1 record</p>
                          <label><input type="checkbox" id="create-followup-note"> Create follow up task</label>
                          <div class="tw-flex tw-justify-end tw-gap-2">
                            <button class="btn btn-default" onclick="toggleAddNoteForm()">Cancel</button>
                            <button class="btn btn-primary" onclick="saveNote()">Add Note</button>
                          </div>
                        </div>
                        <div class="list-container" id="notes-list">
                          <!-- Notes -->
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Lost Reason Modal -->
<div class="modal fade" id="lostReasonModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title"><?php echo _l('mark_as_lost'); ?></h4>
      </div>
      <div class="modal-body">
        <div class="form-group">
          <label><?php echo _l('lost_reason'); ?></label>
          <select id="lost-reason-select" class="form-control" required>
            <option value="">-- <?php echo _l('select_lost_reason'); ?> --</option>
            <?php foreach (($lost_reasons ?? []) as $r) { ?>
              <option value="<?php echo (int) $r['id']; ?>"><?php echo html_entity_decode($r['name']); ?></option>
            <?php } ?>
          </select>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('cancel'); ?></button>
        <button type="button" class="btn btn-danger"
          onclick="markDealLost(<?php echo (int) $deal['id']; ?>)"><?php echo _l('mark_as_lost'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Contact Modal -->
<div class="modal fade" id="contactModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title" id="contactModalTitle"><?php echo _l('add_contact'); ?></h4>
      </div>
      <div class="modal-body">
        <ul class="nav nav-tabs" role="tablist">
          <li role="presentation" class="active">
            <a href="#select-contact" aria-controls="select-contact" role="tab"
              data-toggle="tab"><?php echo _l('select_existing'); ?></a>
          </li>
          <li role="presentation">
            <a href="#create-contact" aria-controls="create-contact" role="tab"
              data-toggle="tab"><?php echo _l('create_new'); ?></a>
          </li>
        </ul>

        <div class="tab-content tw-mt-3">
          <div role="tabpanel" class="tab-pane active" id="select-contact">
            <div class="form-group">
              <label><?php echo _l('select_contact'); ?></label>
              <select class="form-control" id="existing-contact-select">
                <option value=""><?php echo _l('select_contact'); ?></option>
              </select>
            </div>
          </div>

          <div role="tabpanel" class="tab-pane" id="create-contact">
            <div class="form-group">
              <label><?php echo _l('firstname'); ?></label>
              <input type="text" id="contact-firstname" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('lastname'); ?></label>
              <input type="text" id="contact-lastname" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('email'); ?></label>
              <input type="email" id="contact-email" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('phonenumber'); ?></label>
              <input type="text" id="contact-phonenumber" class="form-control">
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveContact()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Company Modal -->
<div class="modal fade" id="companyModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title" id="companyModalTitle"><?php echo _l('add_company'); ?></h4>
      </div>
      <div class="modal-body">
        <ul class="nav nav-tabs" role="tablist">
          <li role="presentation" class="active">
            <a href="#select-company" aria-controls="select-company" role="tab"
              data-toggle="tab"><?php echo _l('select_existing'); ?></a>
          </li>
          <li role="presentation">
            <a href="#create-company" aria-controls="create-company" role="tab"
              data-toggle="tab"><?php echo _l('create_new'); ?></a>
          </li>
        </ul>

        <div class="tab-content tw-mt-3">
          <div role="tabpanel" class="tab-pane active" id="select-company">
            <div class="form-group">
              <label><?php echo _l('select_company'); ?></label>
              <select class="form-control" id="existing-company-select">
                <option value=""><?php echo _l('select_company'); ?></option>
              </select>
            </div>
          </div>

          <div role="tabpanel" class="tab-pane" id="create-company">
            <div class="form-group">
              <label><?php echo _l('company_name'); ?></label>
              <input type="text" id="company-name" class="form-control" required>
            </div>
            <div class="form-group">
              <label><?php echo _l('company_email'); ?></label>
              <input type="email" id="company-email" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_phone'); ?></label>
              <input type="text" id="company-phone" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_website'); ?></label>
              <input type="url" id="company-website" class="form-control">
            </div>
            <div class="form-group">
              <label><?php echo _l('company_address'); ?></label>
              <textarea id="company-address" class="form-control" rows="3"></textarea>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveCompany()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Attachment Modal -->
<div class="modal fade" id="attachmentModal" tabindex="-1" role="dialog">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title"><?php echo _l('add_attachment'); ?></h4>
      </div>
      <div class="modal-body">
        <div class="form-group">
          <label><?php echo _l('file'); ?></label>
          <input type="file" id="attachment-file" class="form-control" required>
        </div>
        <div class="form-group">
          <label><?php echo _l('description'); ?></label>
          <textarea id="attachment-description" class="form-control" rows="3"></textarea>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal"><?php echo _l('close'); ?></button>
        <button type="button" class="btn btn-primary" onclick="saveAttachment()"><?php echo _l('save'); ?></button>
      </div>
    </div>
  </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1" role="dialog">
  <div class="modal-dialog modal-lg" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
        <h4 class="modal-title">Preview</h4>
      </div>
      <div class="modal-body">
        <div id="preview-content"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<?php init_tail(); ?>



<script>
  var dealId = <?php echo (int) $deal['id']; ?>;
  var CSRF = {
    name: '<?php echo $this->security->get_csrf_token_name(); ?>',
    value: '<?php echo $this->security->get_csrf_hash(); ?>'
  };

  var staffMap = {};

  $(function () {
    loadContacts(); loadCompanies(); loadAttachments(); loadActivities();
    loadAvailableContacts(); loadAvailableCompanies(); loadStaff();

    $('#activities-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#activities-only-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });
    $('#calls-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#calls-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });
    $('#notes-search').on('keyup', function () {
      var val = $(this).val().toLowerCase();
      $('#notes-list .timeline-item').filter(function () {
        $(this).toggle($(this).text().toLowerCase().indexOf(val) > -1);
      });
    });

    $('#activity-filter').on('change', filterActivities);
    $('a[data-toggle="tab"]').on('shown.bs.tab', placeTimelineDots);
  });

  function parseIfString(response) {
    if (typeof response === 'string') {
      try {
        return JSON.parse(response);
      } catch (e) {
        return { __parseError: true, error: e, raw: response };
      }
    }
    return response;
  }

  function getActivityIconData(type) {
    // Use only one definition, prefer the one with more types and consistent with your payload
    switch (type) {
      case 'calls':
      case 'call':
        return { icon: 'fa-phone', bg: '#10b981', color: '#fff' };
      case 'emails':
        return { icon: 'fa-envelope', bg: '#6366f1', color: '#fff' };
      case 'notes':
      case 'note':
        return { icon: 'fa-sticky-note-o', bg: '#f59e42', color: '#fff' };
      case 'documents':
        return { icon: 'fa-file-text-o', bg: '#fbbf24', color: '#fff' };
      case 'stage':
        return { icon: 'fa-arrow-right', bg: '#3b82f6', color: '#fff' };
      case 'status':
        return { icon: 'fa-check-circle', bg: '#84c529', color: '#fff' };
      case 'contact':
        return { icon: 'fa-user-plus', bg: '#10b981', color: '#fff' };
      case 'company':
        return { icon: 'fa-building', bg: '#6366f1', color: '#fff' };
      case 'attachment':
        return { icon: 'fa-paperclip', bg: '#f59e42', color: '#fff' };
      default:
        return { icon: 'fa-clock-o', bg: '#6b7280', color: '#fff' };
    }
  }

  function getActivityBadge(type) {
    switch (type) {
      case 'calls': return 'call';
      case 'emails': return 'email';
      case 'notes': return 'note';
      case 'documents': return 'document';
      case 'stage': return 'stage';
      case 'status': return 'status';
      default: return 'activity';
    }
  }
  function isRowType(type) { return ['stage', 'status', 'contact', 'company', 'attachment', 'emails', 'documents'].indexOf(type || '') !== -1; }
  function isCardType(type) { return ['activities', 'activity', 'notes', 'note', 'calls', 'call'].indexOf(type || '') !== -1; }

  function loadStaff() {
    $.get(admin_url + 'deals/get_staff_ajax', function (res) {
      res = parseIfString(res);
      if (res.success && res.staff) {
        res.staff.forEach(function (s) {
          $('#activity-owner').append('<option value="' + s.staffid + '">' + s.firstname + ' ' + s.lastname + '</option>');
          $('#activity-guests').append('<option value="' + s.staffid + '">' + s.firstname + ' ' + s.lastname + '</option>');
        });
      }
    });
  }

  function cancelActivity() {
    $('#activities-add-form').hide();
    $('#activities-controls').show();
  }

  function cancelCall() {
    $('#calls-add-form').hide();
    $('#calls-controls').show();
  }

  function cancelNote() {
    $('#notes-add-form').hide();
    $('#notes-controls').show();
    $('#add-note-btn').show();
  }

  // Only one saveActivity, saveCall, saveNote function, matching your backend payload
  function saveActivity() {
    var title = $('#activity-title').val() ? $('#activity-title').val().trim() : '';
    if (!title) {
      alert('Title is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('title', title);
    formData.append('activity_date', $('#activity-date').val());
    formData.append('description', $('#activity-description').val());
    formData.append('status', $('#activity-completed').prop('checked') ? 'completed' : 'pending');
    formData.append('owner_id', $('#activity-owner').val());
    formData.append('guests', JSON.stringify($('#activity-guests').val() || []));
    // Reminder calculation
    var num = parseInt($('#reminder-num').val() || 0);
    var unit = $('#reminder-unit').val();
    var when = $('#reminder-when').val();
    var factor = unit === 'minutes' ? 60 : unit === 'hours' ? 3600 : 86400;
    var seconds = num * factor;
    if (when === 'after due') seconds = -seconds;
    formData.append('reminder', seconds);

    $.ajax({
      url: admin_url + 'deals/add_activity/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelActivity();
          loadActivities();
        } else {
          alert('Error adding activity');
        }
      }
    });
  }

  function saveCall() {
    var outcome = $('#call-outcome').val();
    if (!outcome) {
      alert('Outcome is required');
      return;
    }
    var callNotes = $('#call-description').val();
    var callDate = $('#call-date').val();

    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('outcome', outcome);
    // If callDate is empty, let backend default to now
    if (callDate) {
      formData.append('call_date', callDate);
    }
    if (callNotes) {
      formData.append('call_notes', callNotes);
    }

    $.ajax({
      url: admin_url + 'deals/add_call/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelCall();
          loadActivities();
        } else {
          alert('Error adding call: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding call');
      }
    });
  }

  function saveNote() {
    var description = $('#note-description').val() ? $('#note-description').val().trim() : '';
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);

    $.ajax({
      url: admin_url + 'deals/add_note/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          cancelNote();
          loadActivities();
        } else {
          alert('Error adding note');
        }
      }
    });
  }

  function toggleActivityStatus(id, checked) {
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('status', checked ? 'completed' : 'pending');

    $.ajax({
      url: admin_url + 'deals/update_activity_status/' + id,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          loadActivities();
        }
      }
    });
  }

  function filterActivitiesByPeriod(period) {
    var now = moment();
    $('#activities-only-list .activity-card').show();
    if (period !== 'all') {
      $('#activities-only-list .activity-card').hide();
      $('#activities-only-list .activity-card').filter(function () {
        var date = $(this).data('date');
        var status = $(this).data('status');
        if (period === 'done') return status === 'completed';
        if (!date) return false;
        var mDate = moment(date);
        if (period === 'today') return mDate.isSame(now, 'day');
        if (period === 'tomorrow') return mDate.isSame(now.clone().add(1, 'day'), 'day');
        if (period === 'this_week') return mDate.isSame(now, 'week');
        if (period === 'next_week') return mDate.isSame(now.clone().add(1, 'week'), 'week');
      }).show();
    }
  }

  function loadActivities() {
    $.get(admin_url + 'deals/get_deal_activities/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        var errorHtml = '<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing activities response</p></div>';
        $('#activities-list,#activities-only-list,#emails-list,#documents-list,#calls-list,#notes-list').html(errorHtml);
        return;
      }
      if (!(response.success && response.activities && response.activities.length)) {
        var noHtml = '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_activity_found"); ?></p></div>';
        $('#activities-list,#activities-only-list').html(noHtml);
        $('#emails-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_emails_found"); ?></p></div>');
        $('#documents-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_documents_found"); ?></p></div>');
        $('#calls-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_calls_found"); ?></p></div>');
        $('#notes-list').html('<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_notes_found"); ?></p></div>');
        $('#badge-activities,#badge-emails,#badge-documents,#badge-calls,#badge-notes').text('0');
        return;
      }

      var htmlAll = '', htmlActivitiesOnly = '', htmlEmails = '', htmlDocuments = '', htmlCalls = '', htmlNotes = '';
      var countActivities = 0, countEmails = 0, countDocuments = 0, countCalls = 0, countNotes = 0;

      response.activities.forEach(function (a) {
        var t = a.type || 'activity';
        var icon = getActivityIconData(t);
        var badge = getActivityBadge(t);

        function renderRow() {
          var row = '';
          row += '<div class="timeline-item timeline-row" data-type="' + t + '" data-icon="' + icon.icon + '" data-icon-bg="' + icon.bg + '" data-icon-color="' + icon.color + '">';
          row += '<span class="dot"></span>';
          row += '<p class="timeline-row-text">' + (a.description || '—') + ' <span class="timeline-row-meta">-  ' + (a.full_name || '—') + ' -  ' + (a.date ? moment(a.date).format('MMM DD, YYYY HH:mm') : '—') + '</span></p>';
          row += '</div>';
          return row;
        }
        function renderCard() {
          var card = '';
          card += '<div class="timeline-item timeline-card" data-type="' + t + '" data-icon="' + icon.icon + '" data-icon-bg="' + icon.bg + '" data-icon-color="' + icon.color + '">';
          card += '<span class="dot"></span>';
          card += '<div class="timeline-content">';
          card += '<div class="timeline-header">';
          card += '<div class="timeline-info">';
          card += '<h6 class="timeline-title">' + (a.title || a.description || 'No description') + '</h6>';
          card += '<p class="timeline-meta"><i class="fa fa-user"></i> ' + (a.full_name || 'Unknown user') + ' <span class="meta-sep">&bull;</span> <i class="fa fa-calendar"></i> ' + (a.date ? moment(a.date).format('MMM DD, YYYY HH:mm') : 'Unknown date') + '</p>';
          card += '</div>';
          card += '<span class="activity-badge ' + badge + '">' + (t || 'activity') + '</span>';
          card += '</div>';
          if (a.additional_data) {
            try {
              var addData = typeof a.additional_data === 'string' ? JSON.parse(a.additional_data) : a.additional_data;
              var desc = '';
              if (addData.outcome !== undefined) {
                desc += '<div><strong>Outcome:</strong> ' + (addData.outcome ? addData.outcome : '-') + '</div>';
              }
              if (addData.date !== undefined) {
                // Format date to DD-MM-YYYY (Indian format)
                var dateVal = addData.date;
                var formattedDate = '-';
                if (dateVal) {
                  var d = new Date(dateVal);
                  if (!isNaN(d.getTime())) {
                    var day = ('0' + d.getDate()).slice(-2);
                    var month = ('0' + (d.getMonth() + 1)).slice(-2);
                    var year = d.getFullYear();
                    formattedDate = day + '-' + month + '-' + year;
                  } else if (typeof dateVal === 'string' && dateVal.length >= 10) {
                    // fallback: try to parse as YYYY-MM-DD or YYYYMMDD
                    var m = dateVal.match(/^(\d{4})-?(\d{2})-?(\d{2})/);
                    if (m) {
                      formattedDate = m[3] + '-' + m[2] + '-' + m[1];
                    } else {
                      formattedDate = dateVal;
                    }
                  }
                }
                desc += '<div><strong>Date:</strong> ' + formattedDate + '</div>';
              }
              if (addData.notes !== undefined) {
                desc += '<div><strong>Notes:</strong> ' + (addData.notes ? addData.notes : '-') + '</div>';
              }
              card += '<div class="timeline-description">' + desc + '</div>';
            } catch (e) {
              card += '<div class="timeline-description">' + a.additional_data + '</div>';
            }
          }
          if (t === 'activity' || t === 'activities') {
            // card += '<div class="tw-mt-2"><label class="tw-text-xs"><input type="checkbox" ' + ((a.status === 'completed') ? 'checked' : '') + ' onchange="toggleActivityStatus(' + a.id + ', this.checked)"> Completed</label></div>';
          }
          card += '<div class="timeline-actions">';
          card += '<button class="btn btn-pin" onclick="pinActivity(' + a.id + ')" title="Pin to Top"><i class="fa fa-thumb-tack"></i></button>';
          card += '<button class="btn btn-edit" onclick="editActivity(' + a.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          card += '<button class="btn btn-delete" onclick="deleteActivity(' + a.id + ')" title="Delete"><i class="fa fa-trash"></i></button>';
          card += '</div>';
          card += '</div>';
          card += '</div>';
          return card;
        }

        var block = isRowType(t) ? renderRow() : renderCard();
        htmlAll += block;

        switch (t) {
          case 'emails': countEmails++; htmlEmails += block; break;
          case 'documents': countDocuments++; htmlDocuments += block; break;
          case 'calls': case 'call': countCalls++; htmlCalls += block; break;
          case 'notes': case 'note': countNotes++; htmlNotes += block; break;
          default: countActivities++; htmlActivitiesOnly += block; break;
        }
      });

      $('#activities-list').html(htmlAll);
      $('#activities-only-list').html(htmlActivitiesOnly || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_activity_found"); ?></p></div>');
      $('#emails-list').html(htmlEmails || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_emails_found"); ?></p></div>');
      $('#documents-list').html(htmlDocuments || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_documents_found"); ?></p></div>');
      $('#calls-list').html(htmlCalls || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_calls_found"); ?></p></div>');
      $('#notes-list').html(htmlNotes || '<div class="text-muted text-center tw-py-4"><p><?php echo _l("no_notes_found"); ?></p></div>');

      $('#badge-activities').text(countActivities);
      $('#badge-emails').text(countEmails);
      $('#badge-documents').text(countDocuments);
      $('#badge-calls').text(countCalls);
      $('#badge-notes').text(countNotes);

      placeTimelineDots();
    }).fail(function () {
      var errorHtml = '<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading activities</p></div>';
      $('#activities-list,#activities-only-list,#emails-list,#documents-list,#calls-list,#notes-list').html(errorHtml);
    });
  }


  function getActivityIconData(type) {
    switch (type) {
      case 'calls':
      case 'call':
        return { icon: 'fa-phone', bg: '#ff69b4', color: '#fff' }; // Pink as in screenshot for calls
      case 'emails':
        return { icon: 'fa-envelope', bg: '#6366f1', color: '#fff' };
      case 'notes':
      case 'note':
        return { icon: 'fa-sticky-note', bg: '#ff69b4', color: '#fff' }; // Pink for notes
      case 'documents':
        return { icon: 'fa-file-text', bg: '#fbbf24', color: '#fff' };
      case 'activity':
        return { icon: 'fa-check', bg: '#84c529', color: '#fff' }; // Green check for activities
      case 'stage':
        return { icon: 'fa-arrow-right', bg: '#add8e6', color: '#000' }; // Light blue arrow for stage changes
      case 'status':
        return { icon: 'fa-trophy', bg: '#add8e6', color: '#000' }; // Light blue for won/lost
      case 'contact':
      case 'company':
      case 'attachment':
        return { icon: 'fa-link', bg: '#add8e6', color: '#000' }; // Light blue for associations
      default:
        return { icon: 'fa-info-circle', bg: '#add8e6', color: '#fff' }; // Default light blue
    }
  }

  function loadContacts() {
    console.log('Loading contacts for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_contacts/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse contacts response:', response.error, response.raw);
        $('#contacts-list').html('<div class="error-state"><i class="fa fa-exclamation-triangle"></i><h4>Error parsing contacts response</h4></div>');
        return;
      }
      console.log('Contacts response:', response.success, response.contacts, (response.success && response.contacts && response.contacts.length > 0));
      if (response.success && response.contacts && response.contacts.length > 0) {
        var html = '';
        response.contacts.forEach(function (contact) {
          html += '<div class="contact-item">';
          html += '<div class="contact-header">';
          html += '<div class="contact-avatar">';
          html += '<i class="fa fa-user"></i>';
          html += '</div>';
          html += '<div class="contact-info">';
          html += '<h6 class="contact-name">' + (contact.firstname || '') + ' ' + (contact.lastname || '') + '</h6>';
          if (contact.email) {
            html += '<div class="contact-detail"><i class="fa fa-envelope"></i> ' + contact.email + '</div>';
          }
          if (contact.phonenumber) {
            html += '<div class="contact-detail"><i class="fa fa-phone"></i> ' + contact.phonenumber + '</div>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="contact-actions">';
          html += '<button class="btn btn-sm btn-outline-primary" onclick="editContact(' + contact.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeContact(' + contact.id + ')" title="Remove"><i class="fa fa-times"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#contacts-list').html(html);
        console.log('Contacts HTML updated:', html);
      } else {
        console.log('No contacts found or error in response');
        $('#contacts-list').html('<div class="empty-state"><i class="fa fa-users"></i><h4><?php echo _l("no_contacts_found"); ?></h4><p><?php echo _l("add_contacts_to_get_started"); ?></p></div>');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading contacts:', error);
      console.error('Response:', xhr.responseText);
      $('#contacts-list').html('<div class="error-state"><i class="fa fa-exclamation-triangle"></i><h4>Error loading contacts</h4></div>');
    });
  }

  function loadCompanies() {
    console.log('Loading companies for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_companies/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse companies response:', response.error, response.raw);
        $('#companies-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing companies response</p></div>');
        $('#companies-count').text('(0)');
        return;
      }
      console.log('Companies response:', response);
      if (response.success && response.companies && response.companies.length > 0) {
        var html = '';
        response.companies.forEach(function (company) {
          html += '<div class="company-item">';
          html += '<div class="company-header">';
          html += '<div class="company-avatar"><i class="fa fa-building"></i></div>';
          html += '<div class="company-info">';
          html += '<h6 class="company-name">' + (company.name || '') + '</h6>';
          if (company.email) {
            html += '<div class="company-detail"><i class="fa fa-envelope"></i> ' + company.email + '</div>';
          }
          if (company.phone) {
            html += '<div class="company-detail"><i class="fa fa-phone"></i> ' + company.phone + '</div>';
          }
          if (company.website) {
            html += '<div class="company-detail"><i class="fa fa-globe"></i> <a href="' + company.website + '" target="_blank">' + company.website + '</a></div>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="company-actions">';
          html += '<button class="btn btn-sm btn-outline-primary" onclick="editCompany(' + company.id + ')" title="Edit"><i class="fa fa-edit"></i></button>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeCompany(' + company.id + ')" title="Remove"><i class="fa fa-times"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#companies-list').html(html);
        $('#companies-count').text('(' + response.companies.length + ')');
        console.log('Companies HTML updated:', html);
      } else {
        console.log('No companies found or error in response');
        $('#companies-list').html('<div class="text-muted text-center tw-py-4"><i class="fa fa-building fa-2x tw-mb-2"></i><p><?php echo _l("no_company_associated"); ?></p></div>');
        $('#companies-count').text('(0)');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading companies:', error);
      console.error('Response:', xhr.responseText);
      $('#companies-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading companies</p></div>');
      $('#companies-count').text('(0)');
    });
  }

  function isImage(filename) {
    return /\.(gif|jpe?g|tiff?|png|webp|bmp)$/i.test(filename);
  }

  function previewAttachment(id, filename) {
    var url = admin_url + 'deals/download_attachment/' + id;
    var content = '';
    if (isImage(filename)) {
      content = '<img src="' + url + '" alt="' + filename + '" style="max-width:100%; height:auto;">';
    } else if (filename.toLowerCase().endsWith('.pdf')) {
      content = '<iframe src="' + url + '" width="100%" height="600px" frameborder="0"></iframe>';
    } else {
      content = '<p>Preview not available for this file type. <a href="' + url + '">Download</a> to view.</p>';
    }
    $('#preview-content').html(content);
    $('#previewModal').modal('show');
  }

  function loadAttachments() {
    console.log('Loading attachments for deal:', dealId);
    $.get(admin_url + 'deals/get_deal_attachments/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse attachments response:', response.error, response.raw);
        $('#attachments-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error parsing attachments response</p></div>');
        return;
      }
      console.log('Attachments response:', response);
      if (response.success && response.attachments && response.attachments.length > 0) {
        var html = '';
        response.attachments.forEach(function (attachment) {
          html += '<div class="attachment-item tw-flex tw-items-center tw-justify-between tw-mb-3">';
          html += '<div class="tw-flex tw-items-center">';
          var url = admin_url + 'deals/download_attachment/' + attachment.id;
          if (isImage(attachment.file_name)) {
            html += '<img src="' + url + '" style="max-width:300px; max-height:300px; width:auto; height:auto; object-fit:contain;" class="tw-rounded tw-mr-3" alt="' + attachment.file_name + '">';
          } else {
            html += '<i class="fa fa-file tw-text-4xl tw-mr-3 tw-text-gray-400"></i>';
          }
          html += '<div>';
          html += '<a href="#" onclick="previewAttachment(' + attachment.id + ', \'' + attachment.file_name + '\')" class="tw-font-medium">' + (attachment.file_name || 'Unknown file') + '</a>';
          if (attachment.description) {
            html += '<p class="tw-text-sm tw-text-gray-600">' + attachment.description + '</p>';
          }
          html += '</div>';
          html += '</div>';
          html += '<div class="tw-flex tw-gap-2">';
          html += '<a href="' + url + '" class="btn btn-sm btn-outline-primary"><i class="fa fa-download"></i></a>';
          html += '<button class="btn btn-sm btn-outline-danger" onclick="removeAttachment(' + attachment.id + ')"><i class="fa fa-trash"></i></button>';
          html += '</div>';
          html += '</div>';
        });
        $('#attachments-list').html(html);
        console.log('Attachments HTML updated:', html);
      } else {
        console.log('No attachments found or error in response');
        $('#attachments-list').html('<div class="text-muted text-center tw-py-4"><i class="fa fa-paperclip fa-2x tw-mb-2"></i><p><?php echo _l("no_attachments_uploaded"); ?></p></div>');
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading attachments:', error);
      console.error('Response:', xhr.responseText);
      $('#attachments-list').html('<div class="text-danger text-center tw-py-4"><i class="fa fa-exclamation-triangle fa-2x tw-mb-2"></i><p>Error loading attachments</p></div>');
    });
  }

  // --- TIMELINE ICONS ON LINE, NOT IN CARD ---


  function loadAvailableContacts() {
    $.get(admin_url + 'deals/get_available_contacts/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse available contacts response:', response.error, response.raw);
        $('#existing-contact-select').html('<option value=""><?php echo _l("select_contact"); ?></option>');
        return;
      }
      console.log('Contacts response:', response);
      if (response.success) {
        var html = '<option value=""><?php echo _l("select_contact"); ?></option>';
        if (response.contacts && response.contacts.length > 0) {
          response.contacts.forEach(function (contact) {
            html += '<option value="' + contact.id + '">' + contact.firstname + ' ' + contact.lastname + ' (' + contact.email + ')</option>';
          });
        }
        $('#existing-contact-select').html(html);
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading contacts:', error);
    });
  }

  function loadAvailableCompanies() {
    $.get(admin_url + 'deals/get_available_companies/' + dealId, function (response) {
      response = parseIfString(response);
      if (response.__parseError) {
        console.error('Failed to parse available companies response:', response.error, response.raw);
        $('#existing-company-select').html('<option value=""><?php echo _l("select_company"); ?></option>');
        return;
      }
      console.log('Companies response:', response);
      if (response.success) {
        var html = '<option value=""><?php echo _l("select_company"); ?></option>';
        if (response.companies && response.companies.length > 0) {
          response.companies.forEach(function (company) {
            html += '<option value="' + company.id + '">' + company.name + '</option>';
          });
        }
        $('#existing-company-select').html(html);
      }
    }).fail(function (xhr, status, error) {
      console.error('Error loading companies:', error);
    });
  }
  function toggleAddActivityForm() {
    $('#add-activity-form').toggle();
  }

  function toggleLogCallForm() {
    $('#log-call-form').toggle();
  }

  function toggleAddNoteForm() {
    $('#add-note-form').toggle();
  }
  function addComment(activityId) {
    var desc = prompt('Add comment');
    if (desc) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('description', desc);
      $.ajax({
        url: admin_url + 'deals/add_comment/' + activityId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.success) {
            loadActivities();
          }
        }
      });
    }
  }

  function addContact() {
    $('#contactModalTitle').text('<?php echo _l("add_contact"); ?>');
    // Clear form fields
    $('#contact-firstname').val('');
    $('#contact-lastname').val('');
    $('#contact-email').val('');
    $('#contact-phonenumber').val('');
    $('#existing-contact-select').val('');
    $('#contactModal').modal('show');
  }

  function addCompany() {
    $('#companyModalTitle').text('<?php echo _l("add_company"); ?>');
    // Clear form fields
    $('#company-name').val('');
    $('#company-email').val('');
    $('#company-phone').val('');
    $('#company-website').val('');
    $('#company-address').val('');
    $('#existing-company-select').val('');
    $('#companyModal').modal('show');
  }

  function addAttachment() {
    // Clear form fields
    $('#attachment-file').val('');
    $('#attachment-description').val('');
    $('#attachmentModal').modal('show');
  }

  function showLostReasonModal() {
    $('#lost-reason-select').val('');
    $('#lostReasonModal').modal('show');
  }

  function editContact(id) {
    alert('Edit contact functionality will be implemented for ID: ' + id);
  }

  function editCompany(id) {
    alert('Edit company functionality will be implemented for ID: ' + id);
  }

  function saveContact() {
    var contactId = $('#existing-contact-select').val();
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);

    if (contactId) {
      formData.append('contact_id', contactId);
    } else {
      formData.append('firstname', $('#contact-firstname').val());
      formData.append('lastname', $('#contact-lastname').val());
      formData.append('email', $('#contact-email').val());
      formData.append('phonenumber', $('#contact-phonenumber').val());
    }

    $.ajax({
      url: admin_url + 'deals/add_contact_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save contact response: ' + response.error);
          return;
        }
        console.log('Save contact response:', response);
        if (response.success) {
          $('#contactModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving contact:', error);
        alert('Error saving contact: ' + error);
      }
    });
  }

  function saveCompany() {
    var companyId = $('#existing-company-select').val();
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);

    if (companyId) {
      formData.append('company_id', companyId);
    } else {
      formData.append('company_name', $('#company-name').val());
      formData.append('company_email', $('#company-email').val());
      formData.append('company_phone', $('#company-phone').val());
      formData.append('company_website', $('#company-website').val());
      formData.append('company_address', $('#company-address').val());
    }

    $.ajax({
      url: admin_url + 'deals/add_company_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save company response: ' + response.error);
          return;
        }
        console.log('Save company response:', response);
        if (response.success) {
          $('#companyModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving company:', error);
        alert('Error saving company: ' + error);
      }
    });
  }

  function saveAttachment() {
    var formData = new FormData();

    // Add CSRF token
    formData.append(CSRF.name, CSRF.value);
    formData.append('attachment', $('#attachment-file')[0].files[0]);
    formData.append('description', $('#attachment-description').val());

    $.ajax({
      url: admin_url + 'deals/add_attachment_to_deal/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.__parseError) {
          alert('Error parsing save attachment response: ' + response.error);
          return;
        }
        console.log('Save attachment response:', response);
        if (response.success) {
          $('#attachmentModal').modal('hide');
          refreshAllData();
        } else {
          alert('Error: ' + (response.message || 'Unknown error occurred'));
        }
      },
      error: function (xhr, status, error) {
        console.error('Error saving attachment:', error);
        alert('Error saving attachment: ' + error);
      }
    });
  }

  function removeContact(contactId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_contact_from_deal/' + dealId + '/' + contactId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove contact response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function removeCompany(companyId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_company_from_deal/' + dealId + '/' + companyId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove company response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function removeAttachment(attachmentId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/remove_attachment_from_deal/' + dealId + '/' + attachmentId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing remove attachment response: ' + response.error);
            return;
          }
          if (response.success) {
            refreshAllData();
          }
        }
      });
    }
  }

  function setDealStage(dealId, stageId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('stage_id', stageId);

      $.ajax({
        url: admin_url + 'deals/set_stage/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing set stage response: ' + response.error);
            return;
          }
          window.location.reload();
        }
      });
    }
  }

  function markDealWon(dealId) {
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);

      $.ajax({
        url: admin_url + 'deals/mark_won/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing mark won response: ' + response.error);
            return;
          }
          window.location.reload();
        }
      });
    }
  }

  function markDealLost(dealId) {
    var lostReasonId = $('#lost-reason-select').val();
    if (lostReasonId === '') {
      alert('<?php echo _l("please_select_lost_reason"); ?>');
      return;
    }
    if (confirm('<?php echo _l("confirm_action_prompt"); ?>')) {
      var formData = new FormData();
      formData.append(CSRF.name, CSRF.value);
      formData.append('lost_reason_id', lostReasonId);

      $.ajax({
        url: admin_url + 'deals/mark_lost/' + dealId,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function (response) {
          response = parseIfString(response);
          if (response.__parseError) {
            alert('Error parsing mark lost response: ' + response.error);
            return;
          }
          $('#lostReasonModal').modal('hide');
          window.location.reload();
        }
      });
    }
  }



  function pinActivity(activityId) {
    // Implementation for pinning activity
    alert('Pin activity functionality will be implemented');
  }

  // Activity Management Functions
  function addActivity() {
    $('#activity-description').val('');
    $('#addActivityModal').modal('show');
  }

  function logCall() {
    $('#call-type').val('outbound');
    $('#call-status').val('completed');
    $('#call-duration').val('');
    $('#call-notes').val('');
    $('#call-date').val(new Date().toISOString().slice(0, 16));
    $('#logCallModal').modal('show');
  }

  function addNote() {
    $('#note-description').val('');
    $('#addNoteModal').modal('show');
  }

  function editActivity(activityId) {
    // Edit activity functionality
    console.log('Editing activity:', activityId);
    // Add your edit logic here
  }

  function deleteActivity(activityId) {
    // Delete activity functionality
    if (confirm('Are you sure you want to delete this activity?')) {
      console.log('Deleting activity:', activityId);
      // Add your delete logic here
    }
  }

  function filterActivities() {
    var filterValue = $('#activity-filter').val();
    console.log('Filtering activities by:', filterValue);

    if (filterValue === 'all') {
      $('.timeline-item').show();
    } else {
      $('.timeline-item').hide();
      $('.timeline-item[data-type="' + filterValue + '"]').show();
    }
  }

  function refreshAllData() {
    loadContacts();
    loadCompanies();
    loadAttachments();
    loadActivities();
    loadAvailableContacts();
    loadAvailableCompanies();
  }

  function saveActivity() {
    var description = $('#activity-description').val().trim();
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);
    $.ajax({
      url: admin_url + 'deals/add_activity/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#addActivityModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding activity: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding activity');
      }
    });
  }

  function saveCall() {
    var payload = {
      call_type: $('#call-type').val(),
      call_status: $('#call-status').val(),
      call_duration: $('#call-duration').val(),
      call_notes: $('#call-notes').val(),
      call_date: $('#call-date').val()
    };
    if (!payload.call_type || !payload.call_status) {
      alert('Call type and status are required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    for (var key in payload) {
      formData.append(key, payload[key] || '');
    }
    $.ajax({
      url: admin_url + 'deals/add_call/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#logCallModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding call: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding call');
      }
    });
  }

  function saveNote() {
    var description = $('#note-description').val().trim();
    if (!description) {
      alert('Description is required');
      return;
    }
    var formData = new FormData();
    formData.append(CSRF.name, CSRF.value);
    formData.append('description', description);
    $.ajax({
      url: admin_url + 'deals/add_note/' + dealId,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        response = parseIfString(response);
        if (response.success) {
          $('#addNoteModal').modal('hide');
          loadActivities();
        } else {
          alert('Error adding note: ' + (response.message || ''));
        }
      },
      error: function () {
        alert('Error adding note');
      }
    });
  }

  function placeTimelineDots() {
    $('.timeline-item').each(function () {
      var $it = $(this);
      var icon = $it.data('icon') || 'fa-clock-o';
      var bg = $it.data('icon-bg') || '#6b7280';
      var color = $it.data('icon-color') || '#fff';
      var $dot = $it.find('.dot');
      $dot.html('<i class="fa ' + icon + '"></i>').css({ background: bg, color: color });
    });
  }

  function filterActivities() {
    var v = $('#activity-filter').val();
    if (v === 'all') $('.timeline-item').show();
    else { $('.timeline-item').hide(); $('.timeline-item[data-type="' + v + '"]').show(); }
    placeTimelineDots();
  }
  function toggleAddActivityForm() { $('#add-activity-form').toggle(); }
  function toggleLogCallForm() { $('#log-call-form').toggle(); }
  function toggleAddNoteForm() { $('#add-note-form').toggle(); }

  function editActivity(id) { /* open modal, load item, submit to edit endpoints */ }
  function deleteActivity(id) { if (confirm('Delete this item?')) { /* call delete endpoint then reload */ } }
  function pinActivity(id) { /* optional */ }
  // Initialize activity filter
  $(document).ready(function () {
    $('#activity-filter').on('change', filterActivities);
  });

  // Remove duplicate/alternate saveActivity, saveCall, saveNote functions
  // Only keep the above versions that match your backend payload

  function placeTimelineDots() {
    $('.timeline-item').each(function () {
      var $it = $(this);
      var icon = $it.data('icon') || 'fa-clock-o';
      var bg = $it.data('icon-bg') || '#6b7280';
      var color = $it.data('icon-color') || '#fff';
      var $dot = $it.find('.dot');
      $dot.html('<i class="fa ' + icon + '"></i>').css({ background: bg, color: color });
    });
  }

  function filterActivities() {
    var v = $('#activity-filter').val();
    if (v === 'all') $('.timeline-item').show();
    else { $('.timeline-item').hide(); $('.timeline-item[data-type="' + v + '"]').show(); }
    placeTimelineDots();
  }

  // Remove duplicate toggleAddActivityForm, toggleLogCallForm, toggleAddNoteForm, etc.
  // Only keep one version of each function

  // Initialize activity filter
  $(document).ready(function () {
    $('#activity-filter').on('change', filterActivities);
  });











  $(function () {
    // Show edit icon on hover for amount
    $('.deal-value-group').hover(
      function () { $(this).find('.edit-amount-btn').show(); },
      function () { if (!$('#edit-amount-form').is(':visible')) $(this).find('.edit-amount-btn').hide(); }
    );
    // Show edit icon on hover for close date
    $('.close-date-group').hover(
      function () { $(this).find('.edit-close-date-btn').show(); },
      function () { if (!$('#edit-close-date-form').is(':visible')) $(this).find('.edit-close-date-btn').hide(); }
    );

    // Amount form submit
    $('#edit-amount-form').on('submit', function (e) {
      e.preventDefault();
      saveDealAmount();
    });

    // Close date form submit
    $('#edit-close-date-form').on('submit', function (e) {
      e.preventDefault();
      saveCloseDate();
    });
  });

  function editDealAmount() {
    $('.deal-value-display').hide();
    $('#edit-amount-form').show();
    $('.edit-amount-btn').hide();
    $('#edit-amount-input').focus();
  }
  function cancelEditAmount() {
    $('#edit-amount-form').hide();
    $('.deal-value-display').show();
    $('.edit-amount-btn').hide();
  }
  function saveDealAmount() {
    var amount = parseFloat($('#edit-amount-input').val());
    if (isNaN(amount) || amount < 0) {
      alert('Please enter a valid amount');
      return;
    }
    var formData = new FormData();
    formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', '<?php echo $this->security->get_csrf_hash(); ?>');
    formData.append('deal_value', amount);
    $.ajax({
      url: admin_url + 'deals/update_amount/<?php echo (int) $deal['id']; ?>',
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        try { response = typeof response === 'string' ? JSON.parse(response) : response; } catch (e) { }
        if (response && response.success) {
          location.reload();
        } else {
          alert('Failed to update amount');
        }
      },
      error: function () { alert('Failed to update amount'); }
    });
  }

  function editCloseDate() {
    $('.close-date-display').hide();
    $('#edit-close-date-form').show();
    $('.edit-close-date-btn').hide();
    $('#edit-close-date-input').focus();
  }
  function cancelEditCloseDate() {
    $('#edit-close-date-form').hide();
    $('.close-date-display').show();
    $('.edit-close-date-btn').hide();
  }
  function saveCloseDate() {
    var date = $('#edit-close-date-input').val();
    if (!date) {
      alert('Please select a date');
      return;
    }
    var formData = new FormData();
    formData.append('<?php echo $this->security->get_csrf_token_name(); ?>', '<?php echo $this->security->get_csrf_hash(); ?>');
    formData.append('expected_close_date', date);
    $.ajax({
      url: admin_url + 'deals/update_close_date/<?php echo (int) $deal['id']; ?>',
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        try { response = typeof response === 'string' ? JSON.parse(response) : response; } catch (e) { }
        if (response && response.success) {
          location.reload();
        } else {
          alert('Failed to update close date');
        }
      },
      error: function () { alert('Failed to update close date'); }
    });
  }
</script>


<style>
  /* Contact, Company, and Attachment Items - Simpler */
  .contact-item,
  .company-item,
  .attachment-item {
    background: #fff;
    /* White for clean look */
    border-radius: 4px;
    padding: 12px;
    margin-bottom: 8px;
    border: 1px solid #e5e7eb;
    transition: all 0.3s ease;
    position: relative;
  }

  .contact-item:hover,
  .company-item:hover,
  .attachment-item:hover {
    border-color: #d1d5db;
    /* Subtle hover */
  }

  .contact-actions,
  .company-actions {
    position: absolute;
    right: 12px;
    top: 50%;
    transform: translateY(-50%);
    opacity: 0;
    transition: all 0.3s ease;
    display: flex;
    gap: 6px;
  }

  .contact-item:hover .contact-actions,
  .company-item:hover .company-actions {
    opacity: 1;
  }

  .contact-header,
  .company-header {
    display: flex;
    align-items: center;
    margin-bottom: 6px;
  }

  .contact-avatar,
  .company-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: #3b82f6;
    color: #fff;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 12px;
  }

  .contact-name,
  .company-name {
    font-weight: 500;
    color: #1f2937;
  }

  .contact-detail,
  .company-detail {
    color: #6b7280;
    font-size: 13px;
    margin-bottom: 3px;
    padding-left: 44px;
  }



  /* Timeline spine */
  .timeline-container {
    position: relative;
    padding: 10px 0 10px 36px;
  }

  .timeline-container::before {
    content: '';
    position: absolute;
    left: 18px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: #e5e7eb;
    border-radius: 1px;
  }

  .timeline-item {
    position: relative;
    margin-bottom: 14px;
    min-height: 28px;
  }

  .timeline-item .dot {
    position: absolute;
    left: 0;
    top: 2px;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    box-shadow: 0 2px 8px rgba(0, 0, 0, .08);
    z-index: 2;
    font-size: 18px;
  }

  /* Single-line rows */
  .timeline-item.timeline-row .timeline-row-text {
    margin: 6px 0 0 50px;
    font-size: 13px;
    color: #334155;
  }

  .timeline-item.timeline-row .timeline-row-meta {
    color: #94a3b8;
    margin-left: 6px;
    font-size: 12px;
  }

  /* Cards */
  .timeline-item.timeline-card .timeline-content {
    margin-left: 50px;
    background: #fff;
    border: 1px solid #e5e7eb;
    border-radius: 6px;
    padding: 12px 14px;
  }

  .timeline-header {
    display: flex;
    align-items: center;
    margin-bottom: 6px;
  }

  .timeline-info {
    flex: 1;
  }

  .timeline-title {
    font-weight: 600;
    color: #1f2937;
    margin: 0;
    font-size: 14px;
  }

  .timeline-meta {
    color: #6b7280;
    font-size: 12px;
    margin: 0;
  }

  .meta-sep {
    margin: 0 6px;
    color: #cbd5e1;
  }

  .timeline-description {
    color: #475569;
    line-height: 1.5;
    margin: 8px 0 2px;
    font-size: 13px;
  }

  .timeline-actions {
    position: absolute;
    right: 8px;
    top: 6px;
    opacity: 0;
    transform: translateX(8px);
    transition: all .2s ease;
    display: flex;
    gap: 6px;
  }

  .timeline-item:hover .timeline-actions {
    opacity: 1;
    transform: translateX(0);
  }

  .timeline-actions .btn {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    font-size: 10px;
  }

  .timeline-actions .btn-edit {
    background: #3b82f6;
    color: #fff;
  }

  .timeline-actions .btn-delete {
    background: #ef4444;
    color: #fff;
  }

  .timeline-actions .btn-pin {
    background: #8b5cf6;
    color: #fff;
  }

  /* Badges */
  .activity-badge {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
    margin-left: 8px;
    background: #f3f4f6;
    color: #6b7280;
  }

  .activity-badge.call {
    background: #d1fae5;
    color: #10b981;
  }

  .activity-badge.email {
    background: #e0e7ff;
    color: #6366f1;
  }

  .activity-badge.note {
    background: #fef3c7;
    color: #f59e42;
  }

  .activity-badge.document {
    background: #fef9c3;
    color: #fbbf24;
  }

  .activity-badge.stage {
    background: #dbeafe;
    color: #3b82f6;
  }

  .activity-badge.status {
    background: #e9fbe5;
    color: #84c529;
  }

  .activity-badge.activity {
    background: #f3f4f6;
    color: #6b7280;
  }

  /* Cards in list-only tabs */
  #activities-only-list,
  #calls-list,
  #notes-list {
    padding: 0;
  }

  #activities-only-list .timeline-item.timeline-card .timeline-content,
  #calls-list .timeline-item.timeline-card .timeline-content,
  #notes-list .timeline-item.timeline-card .timeline-content {
    border-left: none;
    background: #fff;
  }

  /* General theme */
  .btn-primary {
    background: #6366f1;
    border-color: #6366f1;
  }

  .nav-tabs {
    background: #eff6ff;
    border: none;
    border-radius: 4px;
    padding: 6px;
  }

  .nav-tabs>li>a {
    border: none;
    border-radius: 4px;
    padding: 10px 16px;
    color: #333;
    font-weight: 500;
  }

  .nav-tabs>li.active>a {
    border-bottom: 2px solid #6366f1;
    background: transparent;
    color: #6366f1;
  }

  .badge {
    background: #ff0000;
    color: #fff;
    border-radius: 10px;
    padding: 3px 6px;
    font-size: 10px;
  }

  .form-control {
    border-radius: 8px;
    border: 1px solid #ddd;
  }

  .modern-card {
    background: #fff;
    border-radius: 4px;
    padding: 16px;
    border: 1px solid #e5e7eb;
    margin-bottom: 16px;
  }

  .modern-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 12px;
    padding-bottom: 12px;
    border-bottom: 1px solid #f3f4f6;
  }

  @media (max-width: 768px) {
    .timeline-container {
      padding-left: 34px;
    }

    .timeline-container::before {
      left: 16px;
    }

    .timeline-item .dot {
      width: 30px;
      height: 30px;
      font-size: 16px;
    }

    .timeline-item.timeline-card .timeline-content,
    .timeline-item.timeline-row .timeline-row-text {
      margin-left: 44px;
    }

    .timeline-actions {
      position: static;
      opacity: 1;
      transform: none;
      margin-top: 6px;
      justify-content: flex-end;
    }
  }
</style>















<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Install
{
    private $CI;

    public function __construct()
    {
        $this->CI = &get_instance();
    }

    public function run()
    {
        $this->CI->load->database();

        try {

            // Create deals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals` (
                 `id` INT NOT NULL AUTO_INCREMENT,
                    `hash` VARCHAR(65) NULL,
                    `name` VARCHAR(255) NULL,
                    `title` VARCHAR(100) NULL,
                    `company` VARCHAR(255) NULL,
                    `description` TEXT NULL,
                    `country` INT NULL,
                    `zip` VARCHAR(50) NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `address` TEXT NULL,
                    `assigned` INT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `from_form_id` INT NULL,
                    `status` INT NULL,
                    `source` INT NULL,
                    `lastcontact` DATETIME NULL,
                    `dateassigned` DATE NULL,
                    `last_status_change` DATETIME NULL,
                    `addedfrom` INT NULL,
                    `email` VARCHAR(100) NULL,
                    `website` VARCHAR(255) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `date_converted` DATETIME NULL,
                    `lost` TINYINT(1) DEFAULT 0,
                    `junk` TINYINT(1) DEFAULT 0,
                    `last_deal_status` INT NULL,
                    `is_imported_from_email_integration` TINYINT NULL,
                    `email_integration_uid` VARCHAR(30) NULL,
                    `is_public` VARCHAR(255) NULL,
                    `default_language` VARCHAR(50) NULL,
                    `client_id` INT NULL,
                    `deal_value` DECIMAL(15,2) DEFAULT 0.00,
                    `expected_close_date` DATE NULL,
                    `pipeline_id` INT NULL,
                    `stage_id` INT NULL,
                    `status_final` VARCHAR(10) NOT NULL DEFAULT "open",
                    `lost_reason_id` INT NULL,
                    `probability` TINYINT(3) DEFAULT 100,
                    `currency` INT NULL,
                    `tags` TEXT NULL,
                    `is_archived` TINYINT(1) DEFAULT 0,
                    `archived_date` DATETIME NULL,
                    `archived_by` INT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_pipeline_id` (`pipeline_id`),
                    KEY `idx_stage_id` (`stage_id`),
                    KEY `idx_status_final` (`status_final`),
                    KEY `idx_assigned` (`assigned`),
                    KEY `idx_dateadded` (`dateadded`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            if (!$this->CI->db->table_exists(db_prefix() . 'tags_deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'tags_deals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            //create web to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'web_to_deal')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'web_to_deal` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `name` VARCHAR(255) NOT NULL,
                `form_key` VARCHAR(32) NOT NULL UNIQUE,
                `form_data` TEXT,
                `recaptcha` TINYINT(1) DEFAULT 0,
                `language` VARCHAR(50),
                `lead_name_prefix` VARCHAR(50),
                `deal_source` INT,
                `deal_status` INT,
                `responsible` INT DEFAULT 0,
                `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                `submit_action` TINYINT(1) DEFAULT 0,
                `success_submit_msg` TEXT,
                `submit_redirect_url` VARCHAR(255),
                `mark_public` TINYINT(1) DEFAULT 0,
                `allow_duplicate` TINYINT(1) DEFAULT 1,
                `track_duplicate_field` VARCHAR(100),
                `track_duplicate_field_and` VARCHAR(100),
                `create_task_on_duplicate` TINYINT(1) DEFAULT 0,
                `notify_deal_imported` TINYINT(1) DEFAULT 1,
                `notify_type` ENUM("specific_staff", "roles", "assigned") DEFAULT NULL,
                `notify_ids` TEXT,
                `dateadded` DATETIME NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                $this->CI->db->query('CREATE INDEX `idx_form_key` ON `' . db_prefix() . 'web_to_deal` (`form_key`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_source` ON `' . db_prefix() . 'web_to_deal` (`deal_source`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_status` ON `' . db_prefix() . 'web_to_deal` (`deal_status`);');
                $this->CI->db->query('CREATE INDEX `idx_responsible` ON `' . db_prefix() . 'web_to_deal` (`responsible`);');
            }

            // Create deals_statuses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_statuses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_statuses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `statusorder` INT NOT NULL,
                    `isdefault` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages Open/Won/Lost
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'statusorder' => 1,
                    'isdefault' => 1,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Won',
                    'color' => '#84c529',
                    'statusorder' => 2,
                    'isdefault' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'statusorder' => 3,
                    'isdefault' => 0,
                ]);
            }

            // Pipelines table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_pipelines')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_pipelines` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(150) NOT NULL,
                    `order` INT DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
                // Seed default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_pipelines', ['name' => 'Sales Pipeline', 'order' => 1]);
                $defaultPipelineId = $this->CI->db->insert_id();

                // Stages table linked to pipeline
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_stages` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `pipeline_id` INT NOT NULL,
                    `name` VARCHAR(150) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `position` INT DEFAULT 0,
                    `win_probability` TINYINT(3) DEFAULT 100,
                    `is_won` TINYINT(1) DEFAULT 0,
                    `is_lost` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`),
                    KEY `pipeline_id` (`pipeline_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages into default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'position' => 1,
                    'win_probability' => 100,
                    'is_won' => 0,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Won',
                    'color' => '#84c529',
                    'position' => 2,
                    'win_probability' => 100,
                    'is_won' => 1,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'position' => 3,
                    'win_probability' => 0,
                    'is_won' => 0,
                    'is_lost' => 1,
                ]);
            }

            // Create deals attachments table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_attachments')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_attachments` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `file_name` VARCHAR(255) NOT NULL,
                    `filetype` VARCHAR(100) NULL,
                    `description` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `external` VARCHAR(100) NULL,
                    `external_link` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add win_probability to deals_stages if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals_stages')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_stages');
                if (!in_array('win_probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_stages` ADD `win_probability` TINYINT(3) DEFAULT 100 AFTER `position`');
                }
            }

            // Lost reasons table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_lost_reasons')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_lost_reasons` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(191) NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add pipeline_id and stage_id to deals if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `client_id`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
            }

            // Create deal_sources table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_sources')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_sources` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }



            // Create deal_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `date_contacted` DATETIME NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_activity table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_activity` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` VARCHAR(255) NOT NULL,
                    `additional_data` TEXT NULL,
                    `staffid` INT NULL,
                    `full_name` VARCHAR(100) NULL,
                    `date` DATETIME NOT NULL,
                    `type` VARCHAR(30) DEFAULT "activity",
                    `custom_activity` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Ensure "type" column exists
            if ($this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');
                if (!in_array('type', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `type` VARCHAR(30) DEFAULT "activity" AFTER `date`');
                }
            }

            // Create deal_email_integration table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_email_integration')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_email_integration` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `email` VARCHAR(100) NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    `created_at` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific contacts master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `firstname` VARCHAR(100) NOT NULL,
                    `lastname` VARCHAR(100) NULL,
                    `email` VARCHAR(191) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `email` (`email`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module contacts
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_contact_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_contact_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact_unique` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific companies master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(191) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unique_email` (`email`),
                    UNIQUE KEY `unique_phone` (`phone`),
                    KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module companies
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_company_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_company_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company_unique` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create companies table for storing company information
            if (!$this->CI->db->table_exists(db_prefix() . 'companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(100) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `zip` VARCHAR(50) NULL,
                    `country` INT NULL,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `name` (`name`),
                    KEY `active` (`active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_attachments_relations table for linking attachments to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_attachments_relations')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_attachments_relations` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `attachment_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_attachment_unique` (`deal_id`, `attachment_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `attachment_id` (`attachment_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_forms table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_forms')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_forms` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `form_key` VARCHAR(100) NOT NULL,
                    `lead_source` INT NULL,
                    `lead_status` INT NULL,
                    `responsible` INT NULL,
                    `language` VARCHAR(50) NULL,
                    `recaptcha` TINYINT(1) DEFAULT 0,
                    `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                    `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                    `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                    `success_submit_msg` TEXT NULL,
                    `submit_action` TINYINT(1) DEFAULT 0,
                    `submit_redirect_url` VARCHAR(255) NULL,
                    `mark_public` TINYINT(1) DEFAULT 0,
                    `allow_duplicate` TINYINT(1) DEFAULT 1,
                    `track_duplicate_field` VARCHAR(100) NULL,
                    `track_duplicate_field_and` VARCHAR(100) NULL,
                    `create_task_on_duplicate` TINYINT(1) DEFAULT 1,
                    `notify_lead_imported` TINYINT(1) DEFAULT 1,
                    `notify_type` VARCHAR(50) NULL,
                    `notify_ids` TEXT NULL,
                    `lead_name_prefix` VARCHAR(50) NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_saved_filters table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_saved_filters')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_saved_filters` (
                    `id` INT AUTO_INCREMENT PRIMARY KEY,
                    `filter_name` VARCHAR(255) NOT NULL,
                    `field` VARCHAR(100) NOT NULL,
                    `operator` VARCHAR(50) NOT NULL,
                    `value` VARCHAR(255) NOT NULL,
                    `share_with_team` TINYINT(1) DEFAULT 0,
                    `is_default` TINYINT(1) DEFAULT 0,
                    `created_by` INT NOT NULL,
                    `created_at` DATETIME NOT NULL,
                    UNIQUE KEY `filter_name` (`filter_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contacts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_companies table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_products table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_products')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_products` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `product_id` INT NOT NULL,
                    `qty` DECIMAL(15,2) NOT NULL DEFAULT 1.00,
                    `rate` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `discount` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `tax` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `total` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `product_id` (`product_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_tasks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_tasks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_tasks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `task_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_task` (`deal_id`, `task_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `task_id` (`task_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_fields table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_fields')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_fields` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `field_name` VARCHAR(100) NOT NULL,
                    `field_type` VARCHAR(50) NOT NULL,
                    `field_label` VARCHAR(255) NOT NULL,
                    `field_placeholder` VARCHAR(255) NULL,
                    `field_options` TEXT NULL,
                    `field_required` TINYINT(1) DEFAULT 0,
                    `field_order` INT DEFAULT 0,
                    `field_visible` TINYINT(1) DEFAULT 1,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_field_values table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_field_values')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_field_values` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `field_id` INT NOT NULL,
                    `field_value` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_field` (`deal_id`, `field_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `field_id` (`field_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_reminders table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_reminders')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_reminders` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `date` DATETIME NOT NULL,
                    `isnotified` TINYINT(1) DEFAULT 0,
                    `notify_by_email` TINYINT(1) DEFAULT 1,
                    `notify_by_sms` TINYINT(1) DEFAULT 0,
                    `creator` INT NOT NULL,
                    `datecreated` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `date` (`date`),
                    KEY `creator` (`creator`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_emails table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_emails')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_emails` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `subject` VARCHAR(255) NOT NULL,
                    `message` TEXT NOT NULL,
                    `from_email` VARCHAR(100) NOT NULL,
                    `from_name` VARCHAR(100) NOT NULL,
                    `to_email` VARCHAR(100) NOT NULL,
                    `to_name` VARCHAR(100) NOT NULL,
                    `cc` TEXT NULL,
                    `bcc` TEXT NULL,
                    `attachments` TEXT NULL,
                    `sent` TINYINT(1) DEFAULT 0,
                    `sent_date` DATETIME NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `sent` (`sent`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_calls table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_calls')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_calls` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `call_type` VARCHAR(50) NOT NULL,
                    `call_status` VARCHAR(50) NOT NULL,
                    `call_duration` INT NULL,
                    `call_recording` VARCHAR(255) NULL,
                    `call_notes` TEXT NULL,
                    `call_date` DATETIME NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `call_date` (`call_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_meetings table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_meetings')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_meetings` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `meeting_subject` VARCHAR(255) NOT NULL,
                    `meeting_description` TEXT NULL,
                    `meeting_date` DATETIME NOT NULL,
                    `meeting_duration` INT NULL,
                    `meeting_location` VARCHAR(255) NULL,
                    `meeting_type` VARCHAR(50) NULL,
                    `meeting_status` VARCHAR(50) NOT NULL,
                    `meeting_notes` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `meeting_date` (`meeting_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_documents table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_documents')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_documents` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `document_name` VARCHAR(255) NOT NULL,
                    `document_type` VARCHAR(100) NOT NULL,
                    `document_file` VARCHAR(255) NOT NULL,
                    `document_description` TEXT NULL,
                    `document_version` VARCHAR(20) NULL,
                    `document_status` VARCHAR(50) NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `document_status` (`document_status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_expenses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_expenses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_expenses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `expense_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_expense` (`deal_id`, `expense_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `expense_id` (`expense_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_invoices table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_invoices')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_invoices` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `invoice_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_invoice` (`deal_id`, `invoice_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `invoice_id` (`invoice_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_estimates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_estimates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_estimates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `estimate_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_estimate` (`deal_id`, `estimate_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `estimate_id` (`estimate_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_proposals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_proposals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_proposals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `proposal_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_proposal` (`deal_id`, `proposal_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `proposal_id` (`proposal_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contracts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contracts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contracts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contract_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contract` (`deal_id`, `contract_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contract_id` (`contract_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_credit_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_credit_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_credit_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `credit_note_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_credit_note` (`deal_id`, `credit_note_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `credit_note_id` (`credit_note_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_templates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_templates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_templates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `template_name` VARCHAR(255) NOT NULL,
                    `template_content` TEXT NOT NULL,
                    `template_type` VARCHAR(50) NOT NULL,
                    `template_category` VARCHAR(100) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `template_type` (`template_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_automation table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_automation')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_automation` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `automation_name` VARCHAR(255) NOT NULL,
                    `automation_type` VARCHAR(50) NOT NULL,
                    `trigger_condition` TEXT NOT NULL,
                    `action_type` VARCHAR(50) NOT NULL,
                    `action_data` TEXT NOT NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `automation_type` (`automation_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhooks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhooks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhooks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_name` VARCHAR(255) NOT NULL,
                    `webhook_url` VARCHAR(500) NOT NULL,
                    `webhook_events` TEXT NOT NULL,
                    `webhook_secret` VARCHAR(255) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhook_logs table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhook_logs')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhook_logs` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_id` INT NOT NULL,
                    `event_type` VARCHAR(100) NOT NULL,
                    `payload` TEXT NOT NULL,
                    `response_code` INT NULL,
                    `response_body` TEXT NULL,
                    `status` VARCHAR(50) NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `webhook_id` (`webhook_id`),
                    KEY `event_type` (`event_type`),
                    KEY `status` (`status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');

            if (!in_array('title', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `title` VARCHAR(255) NULL AFTER `id`');
            }
            if (!in_array('activity_date', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `activity_date` DATETIME NULL AFTER `title`');
            }
            if (!in_array('reminder', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `reminder` INT DEFAULT 0 AFTER `activity_date`');
            }
            if (!in_array('owner_id', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `owner_id` INT NULL AFTER `reminder`');
            }
            if (!in_array('guests', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `guests` TEXT NULL AFTER `owner_id`');
            }
            if (!in_array('status', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `status` VARCHAR(20) DEFAULT "pending" AFTER `guests`');
            }

            // For deals_calls, change call_status to outcome
            if (in_array('call_status', $fields = $this->CI->db->list_fields(db_prefix() . 'deals_calls'))) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` CHANGE `call_status` `outcome` VARCHAR(50) DEFAULT ""');
            }

            // Remove unnecessary columns if exist
            if (in_array('call_type', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_type`');
            }
            if (in_array('call_duration', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_duration`');
            }
            if (in_array('call_recording', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` DROP `call_recording`');
            }

            // Add missing fields to existing tables if they don't exist
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');

                if (!in_array('expected_close_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `expected_close_date` DATE NULL AFTER `deal_value`');
                }
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `expected_close_date`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
                if (!in_array('probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `probability` TINYINT(3) DEFAULT 100 AFTER `lost_reason_id`');
                }
                if (!in_array('currency', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `currency` INT NULL AFTER `probability`');
                }
                if (!in_array('tags', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `tags` TEXT NULL AFTER `currency`');
                }
                if (!in_array('is_archived', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `is_archived` TINYINT(1) DEFAULT 0 AFTER `tags`');
                }
                if (!in_array('archived_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_date` DATETIME NULL AFTER `is_archived`');
                }
                if (!in_array('archived_by', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_by` INT NULL AFTER `archived_date`');
                }
            }

            return true;
        } catch (Exception $e) {
            log_message('error', 'Deals module installation failed: ' . $e->getMessage());
            return false;
        }
    }

}

$installer = new Install();
$installer->run();












<?php

defined('BASEPATH') || exit('No direct script access allowed');

/*
    Module Name: Deals
    Description: Deals management with list, kanban, and status (open/won/lost)
    Version: 1.0.0
    Requires at least: 2.9.*
*/

define('DEALS_MODULE', 'deals');

// Load installer
require_once __DIR__ . '/install.php';

$CI = &get_instance();

// Activation hook
register_activation_hook(DEALS_MODULE, 'deals_module_activation_hook');
function deals_module_activation_hook()
{
    $CI = &get_instance();
    require_once __DIR__ . '/install.php';
}

// Register language files
register_language_files(DEALS_MODULE, [DEALS_MODULE]);

// Include assets and menu
require_once __DIR__ . '/includes/assets.php';
require_once __DIR__ . '/includes/sidebar_menu_links.php';

// Register settings section and pipelines manager tab
hooks()->add_action('admin_init', function () {
    $CI = &get_instance();
    $CI->app->add_settings_section('deals', [
        'title'    => _l('deals'),
        'position' => 18,
        'children' => [
            [
                'name' => _l('pipelines'),
                'view' => 'deals/settings',
                'position' => 10,
                'icon' => 'fa-solid fa-diagram-project',
            ],
            [
                'name' => _l('lost_reasons'),
                'view' => 'deals/lost_reasons',
                'position' => 15,
                'icon' => 'fa-regular fa-face-frown',
            ],
        ],
    ]);
});
