<?php

defined('BASEPATH') or exit('No direct script access allowed');

// Build deals filters and server-side output similar to leads
$CI = &get_instance();
$CI->load->model('deals/deals_model');
$CI->load->model('staff_model');

$rules = [
    App_table_filter::new('name', 'TextRule')->label(_l('deals_dt_name')),
    App_table_filter::new('company', 'TextRule')->label(_l('company')),
    App_table_filter::new('deal_value', 'NumberRule')->label(_l('deal_value')),
    App_table_filter::new('dateadded', 'DateRule')->label(_l('date_created')),
    App_table_filter::new('business_category', 'SelectRule')->label(_l('business_categories'))->options(function () use ($CI) {
        $cats = $CI->db->order_by('name', 'asc')->get(db_prefix() . 'deals_business_categories')->result_array();
        return collect($cats)->map(fn($c) => ['value' => $c['id'], 'label' => $c['name']])->all();
    }),
    App_table_filter::new('rating', 'SelectRule')->label(_l('ratings'))->options(function () use ($CI) {
        $ratings = $CI->deals_model->get_ratings();
        return collect($ratings)->map(fn($r) => ['value' => $r['id'], 'label' => trim($r['name'] . ' ' . ($r['score'] ? '(' . $r['score'] . ')' : ''))])->all();
    }),
    App_table_filter::new('status_final', 'MultiSelectRule')->label(_l('status'))->options([
        ['value' => 'open', 'label' => _l('open')],
        ['value' => 'won', 'label' => _l('won')],
        ['value' => 'lost', 'label' => _l('lost')],
    ]),
    App_table_filter::new('pipeline_id', 'SelectRule')->label(_l('pipeline'))->options(function () use ($CI) {
        $pipelines = $CI->db->get(db_prefix() . 'deals_pipelines')->result_array();
        return collect($pipelines)->map(fn($p) => ['value' => $p['id'], 'label' => $p['name']])->all();
    }),
    App_table_filter::new('stage_id', 'SelectRule')->label(_l('stage'))->options(function () use ($CI) {
        $stages = $CI->db->get(db_prefix() . 'deals_stages')->result_array();
        return collect($stages)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
    }),
    App_table_filter::new('status', 'SelectRule')->label(_l('leads_dt_status'))->options(function () use ($CI) {
        $statuses = $CI->deals_model->get_statuses();
        return collect($statuses)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
    }),
    App_table_filter::new('source', 'SelectRule')->label(_l('leads_source'))->options(function () use ($CI) {
        $sources = $CI->deals_model->get_sources();
        return collect($sources)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
    }),
];

$rules[] = App_table_filter::new('assigned', 'SelectRule')->label(_l('assigned_to'))
    ->withEmptyOperators()->emptyOperatorValue(0)
    ->isVisible(fn () => staff_can('view', 'deals'))
    ->options(function () use ($CI) {
        $staff = $CI->staff_model->get('', ['active' => 1]);
        return collect($staff)->map(fn ($m) => ['value' => $m['staffid'], 'label' => $m['firstname'] . ' ' . $m['lastname']])->all();
    });

return App_table::find('deals')
    ->setRules($rules)
    ->outputUsing(function ($params) {
        extract($params);
        $aColumns = [
            '1',
            db_prefix() . 'deals.id as id',
            db_prefix() . 'deals.name as name',
            'company',
            // Combined pipelines / stage / score column
            '(SELECT GROUP_CONCAT(CONCAT(p2.name, " - ", s2.name, IF(r2.score IS NOT NULL, CONCAT(" (", r2.score, ")"), "")) SEPARATOR "; ") FROM ' . db_prefix() . 'deals_pipeline_relations r2 JOIN ' . db_prefix() . 'deals_pipelines p2 ON p2.id = r2.pipeline_id JOIN ' . db_prefix() . 'deals_stages s2 ON s2.id = r2.stage_id LEFT JOIN ' . db_prefix() . 'deals_stage_statuses ss2 ON ss2.id = r2.stage_status_id WHERE r2.deal_id = ' . db_prefix() . 'deals.id) as pipeline_relations',
            db_prefix() . 'deals.deal_value as deal_value',
            'status_final',
            'dateadded',
        ];

        $sIndexColumn = 'id';
        $sTable       = db_prefix() . 'deals';
        $join = [
            'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = ' . db_prefix() . 'deals.pipeline_id',
            'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = ' . db_prefix() . 'deals.stage_id',
        ];

        $where = [];
        if ($filtersWhere = $this->getWhereFromRules()) {
            $where[] = $filtersWhere;
        }

        if (staff_cant('view', 'deals')) {
            $staffid = (string)get_staff_user_id();
            // Find all team group IDs where this staff is a manager or TL (JSON array fields)
            $team_group_ids = $this->db
                ->select('id')
                ->where("JSON_CONTAINS(manager_ids, '\"$staffid\"')", null, false)
                ->or_where("JSON_CONTAINS(tl_ids, '\"$staffid\"')", null, false)
                ->get(db_prefix() . 'team_groups')
                ->result_array();
            $team_group_ids = array_column($team_group_ids, 'id');

            // Find all sub group IDs where this staff is a TL or employee (JSON array fields)
            $sub_group_ids = $this->db
                ->select('id')
                ->where("JSON_CONTAINS(tl_ids, '\"$staffid\"')", null, false)
                ->or_where("JSON_CONTAINS(employee_ids, '\"$staffid\"')", null, false)
                ->get(db_prefix() . 'team_sub_groups')
                ->result_array();
            $sub_group_ids = array_column($sub_group_ids, 'id');
            // Build visibility SQL
            $visibility_sql = [];
            if (!empty($team_group_ids)) {
                $team_group_ids_escaped = array_map('strval', $team_group_ids);
                $visibility_sql[] = '(' . implode(' OR ', array_map(function($id) {
                    // Deals visibility_team_groups is a JSON array of strings, e.g. ["1","2"]
                    return "JSON_CONTAINS(" . db_prefix() . "deals.visibility_team_groups, '\"$id\"')";
                }, $team_group_ids_escaped)) . ')';
            }
            if (!empty($sub_group_ids)) {
                $sub_group_ids_escaped = array_map('strval', $sub_group_ids);
                $visibility_sql[] = '(' . implode(' OR ', array_map(function($id) {
                    // Deals visibility_sub_groups is a JSON array of strings, e.g. ["1","2"]
                    return "JSON_CONTAINS(" . db_prefix() . "deals.visibility_sub_groups, '\"$id\"')";
                }, $sub_group_ids_escaped)) . ')';
                
            }

            $visibilityCondition = '(' . db_prefix() . 'deals.assigned = ' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.is_public=1';

            // Add visibility SQL if any
            if (!empty($visibility_sql)) {
                $visibilityCondition .= ' OR ' . implode(' OR ', $visibility_sql);
            }

            $visibilityCondition .= ')';
            $where[] = $visibilityCondition;
        }
        $additionalColumns = [
            db_prefix() . 'deals.addedfrom as addedfrom',
        ];

        $result = data_tables_init($aColumns, $sIndexColumn, $sTable, $join, $where, $additionalColumns);
        $output  = $result['output'];
        $rResult = $result['rResult'];

        foreach ($rResult as $aRow) {
            $row = [];
            $row[] = '<div class="checkbox"><input type="checkbox" value="' . $aRow['id'] . '"><label></label></div>';
            $hrefAttr = 'href="' . admin_url('deals/view/' . $aRow['id']) . '"';
            $row[] = '<a ' . $hrefAttr . ' class="tw-font-medium">' . $aRow['id'] . '</a>';
            $nameRow = '<a ' . $hrefAttr . ' class="tw-font-medium">' . html_entity_decode($aRow['name']) . '</a>';
            $nameRow .= '<div class="row-options">';
            $nameRow .= '<a ' . $hrefAttr . '>' . _l('view') . '</a>';
            $nameRow .= ' | <a href="' . admin_url('deals/edit/' . $aRow['id']) . '">' . _l('edit') . '</a>';
            $nameRow .= ' | <a href="' . admin_url('deals/delete/' . $aRow['id']) . '" class="text-danger _delete">' . _l('delete') . '</a>';
            $nameRow .= '</div>';
            $row[] = $nameRow;
            $row[] = html_entity_decode($aRow['company']);
            // Render combined pipelines / stage / score as an HTML list
            if (!empty($aRow['pipeline_relations'])) {
                $parts = explode('; ', $aRow['pipeline_relations']);
                $listHtml = '<ul class="deals-pipeline-relations-list" style="list-style:none;padding:0;margin:0;">';
                foreach ($parts as $p) {
                    $p = trim($p);
                    if ($p === '') continue;
                    $segments = explode(' - ', $p, 3);
                    $pipelineName = isset($segments[0]) ? htmlspecialchars($segments[0], ENT_QUOTES, 'UTF-8') : '';
                    $stageAndScore = isset($segments[1]) ? htmlspecialchars($segments[1], ENT_QUOTES, 'UTF-8') : '';
                    if (isset($segments[2])) {
                        $stageAndScore .= ' - ' . htmlspecialchars($segments[2], ENT_QUOTES, 'UTF-8');
                    }
                    $listHtml .= '<li style="margin-bottom:4px;"><span class="label label-default deals-pipeline-name" style="background:#f0f6ff;color:#09326a;margin-right:6px;padding:3px 6px;border-radius:3px;">' . $pipelineName . '</span>' . $stageAndScore . '</li>';
                }
                $listHtml .= '</ul>';
                $row[] = $listHtml;
            } else {
                $row[] = '-';
            }
            $row[] = '<span class="tw-font-medium">' . app_format_money((float)$aRow['deal_value'], get_base_currency()->id) . '</span>';
            $row[] = ucfirst($aRow['status_final']);
            $row[] = '<span data-toggle="tooltip" data-title="' . e(_dt($aRow['dateadded'])) . '" class="text-has-action is-date">' . e(time_ago($aRow['dateadded'])) . '</span>';
            $row['DT_RowId'] = 'deal_' . $aRow['id'];
            $output['aaData'][] = $row;
        }

        return $output;
    });