<?php

error_reporting(E_ALL);

ini_set('display_errors', 1);

defined('BASEPATH') or exit('No direct script access allowed');

class Client extends ClientsController
{
    public function __construct()
    {
        parent::__construct();

        $this->load->model('restoration/projects_model');
        $this->load->model('clients_model');
        $this->load->model('currencies_model');
        $this->load->model('restoration/template_model');
        $this->load->model('restoration/template_table_model');
        $this->load->model('restoration/task_model');
        $this->load->model('restoration/task_template_table_model');
        $this->load->model('restoration/imagecategory_model');
        $this->load->model('restoration/filecategory_model');
        $this->load->helper('date');
    }

    public function index()
    {
        $client_id = get_client_user_id();
        $data['projects'] = $this->projects_model->get_projects_by_client($client_id);
        $data['title'] = _l('restoration_projects');
        $this->data($data);
        $this->view('restoration/client/index');
        $this->layout();
    }

    public function project($id)
    {
        $client_id = get_client_user_id();
        $project = $this->projects_model->get($id);

        if (!$project || !isset($project->clientid) || $project->clientid != $client_id) {
            show_404();
        }

        $data['project'] = $project;
        $data['project_status'] = get_project_status_by_id($project->status);

        // Fetch customer details
        $client = $this->clients_model->get($project->clientid);
        $data['customer'] = $client ? ($client->company ?? $client->firstname . ' ' . $client->lastname) : 'Unknown';

        // Fetch currency with fallback
        $data['currency'] = $this->projects_model->get_currency($id) ?? (object)[
            'symbol' => '$',
            'decimal_separator' => '.',
            'thousand_separator' => ',',
            'placement' => 'before'
        ];

        // Billing type
        $data['billing_type'] = isset($billing_types[$project->billing_type]) ? $billing_types[$project->billing_type] : 'Not Specified';

        // Rate per hour
        $data['rate_per_hour'] = isset($project->project_rate_per_hour) && $project->billing_type == 2 ? number_format($project->project_rate_per_hour, 2) : '0.00';

        // Date created with robust check
        $data['date_created'] = (isset($project->datecreated) && $project->datecreated) ? _d($project->datecreated) : 'N/A';

        // Total logged hours with fallback
        $data['project_total_logged_time'] = $this->projects_model->total_logged_time($id) ?: 0;

        // Vault entries (fallback to empty array if not implemented)
        $data['project']->shared_vault_entries = []; // Remove get_vault_entries call until implemented

        $group = $this->input->get('group') ?: 'project_overview';
        $group = str_replace('#', '', $group);

        $tabs = [
            'project_overview' => [
                'slug' => 'project_overview',
                'name' => _l('project_overview'),
                'icon' => 'fa fa-th',
                'view' => 'restoration/client/project_overview',
            ],
            'project_tasks' => [
                'slug' => 'project_tasks',
                'name' => _l('tasks'),
                'icon' => 'fa-regular fa-check-circle',
                'view' => 'restoration/client/project_tasks',
            ],
            'project_images' => [
                'slug' => 'project_images',
                'name' => _l('Photos'),
                'icon' => 'fa-solid fa-image',
                'view' => 'restoration/client/project_images',
            ],
            'project_files' => [
                'slug' => 'project_files',
                'name' => _l('Documents'),
                'icon' => 'fa-solid fa-file',
                'view' => 'restoration/client/project_files',
            ],
            'project_report' => [
                'slug' => 'project_report',
                'name' => _l('Reports'),
                'icon' => 'fa-regular fa-note-sticky',
                'view' => 'restoration/client/project_report',
            ],
            'project_notes' => [
                'slug' => 'project_notes',
                'name' => _l('Conclusion'),
                'icon' => 'fa-regular fa-note-sticky',
                'view' => 'restoration/client/project_notes',
            ],
        ];

        if (!isset($tabs[$group])) {
            show_404();
        }

        $data['tabs'] = $tabs;
        $data['tab'] = $tabs[$group];

        // Load tab-specific data
        switch ($group) {
            case 'project_overview':
                // Project progress
                $data['percent'] = $this->projects_model->calc_progress($id);

                // Task progress
                $__total_where_tasks = 'project_id = ' . $this->db->escape_str($id);
                $where = ($__total_where_tasks == '' ? '' : $__total_where_tasks . ' AND ') . 'task_status = 1';
                $data['tasks_completed'] = total_rows(db_prefix() . 'restoration_task', $where);
                $data['total_tasks'] = total_rows(db_prefix() . 'restoration_task', $__total_where_tasks);
                $where = ($__total_where_tasks == '' ? '' : $__total_where_tasks . ' AND ') . 'task_status = 1';
                $data['tasks_not_completed'] = total_rows(db_prefix() . 'restoration_task', $where);
                $data['tasks_not_completed_progress'] = ($data['total_tasks'] > 0 ? round(($data['tasks_completed'] * 100) / $data['total_tasks'], 2) : 0);

                // Client report progress
                $__total_where_client_report = 'project_id = ' . $this->db->escape_str($id);
                $where = ($__total_where_client_report == '' ? '' : $__total_where_client_report . ' AND ') . 'client_report = 1';
                $data['client_report_completed'] = total_rows(db_prefix() . 'restoration_task', $where);
                $data['total_client_report'] = total_rows(db_prefix() . 'restoration_task', $__total_where_client_report);
                $where = ($__total_where_client_report == '' ? '' : $__total_where_client_report . ' AND ') . 'client_report = 1';
                $data['client_report_not_completed'] = total_rows(db_prefix() . 'restoration_task', $where);
                $data['client_report_not_completed_progress'] = ($data['total_client_report'] > 0 ? round(($data['client_report_completed'] * 100) / $data['total_client_report'], 2) : 0);

                // Deadline progress
                if (property_exists($project, 'deadline') && $project->deadline) {
                    $data['project_total_days'] = round((human_to_unix($project->deadline . ' 00:00') - human_to_unix($project->start_date . ' 00:00')) / 3600 / 24);
                    $data['project_days_left'] = $data['project_total_days'];
                    $data['project_time_left_percent'] = 100;

                    if ($project->start_date < date('Y-m-d') && $project->deadline > date('Y-m-d')) {
                        $data['project_days_left'] = round((human_to_unix($project->deadline . ' 00:00') - time()) / 3600 / 24);
                        $data['project_time_left_percent'] = round(($data['project_days_left'] / $data['project_total_days']) * 100, 2);
                    }

                    if ($project->deadline < date('Y-m-d')) {
                        $data['project_days_left'] = 0;
                        $data['project_time_left_percent'] = 0;
                    }
                }

                // Optional chart data
                $data['project_overview_chart'] = [];
                break;

            case 'project_tasks':
                $data['project']->tasks = $this->task_model->all(['project_id' => $id]);
                break;

            case 'project_images':
                $data['category'] = $this->imagecategory_model->all();
                $data['project_images'] = $this->get_restoration_images($id);
                break;

            case 'project_files':
                $data['category'] = $this->filecategory_model->all();
                $data['files'] = $this->get_restoration_files($id);
                break;

            case 'project_report':
                $tasks = $this->task_model->all(['project_id' => $id, 'client_report' => 1]);
                foreach ($tasks as &$task) {
                    $tables = $this->task_template_table_model->all(['task_id' => $task['id']]);
                    foreach ($tables as &$table) {
                        if (is_null($table['column_data'])) {
                            log_message('error', "column_data is null for task_id {$task['id']} in task_template_table");
                        }
                        if (is_null($table['row_data'])) {
                            log_message('error', "row_data is null for task_id {$task['id']} in task_template_table");
                        }
                        if (is_null($table['value_data'])) {
                            log_message('error', "value_data is null for task_id {$task['id']} in task_template_table");
                        }
                        $table['column_data'] = is_string($table['column_data']) && !empty($table['column_data'])
                            ? json_decode($table['column_data'], true)
                            : [];
                        if (json_last_error() !== JSON_ERROR_NONE) {
                            log_message('error', "Invalid JSON in column_data for task_id {$task['id']}: " . json_last_error_msg());
                            $table['column_data'] = [];
                        }
                        $table['row_data'] = is_string($table['row_data']) && !empty($table['row_data'])
                            ? json_decode($table['row_data'], true)
                            : [];
                        if (json_last_error() !== JSON_ERROR_NONE) {
                            log_message('error', "Invalid JSON in row_data for task_id {$task['id']}: " . json_last_error_msg());
                            $table['row_data'] = [];
                        }
                        $table['value_data'] = is_string($table['value_data']) && !empty($table['value_data'])
                            ? json_decode($table['value_data'], true)
                            : [];
                        if (json_last_error() !== JSON_ERROR_NONE) {
                            log_message('error', "Invalid JSON in value_data for task_id {$task['id']}: " . json_last_error_msg());
                            $table['value_data'] = [];
                        }
                    }
                    $task['template_table'] = $tables;
                }
                $data['tasks'] = $tasks;
                break;

            case 'project_notes':
                $staff_notes_raw = $this->projects_model->get_staff_notes($id);
                $data['staff_notes'] = is_string($staff_notes_raw)
                    ? json_decode($staff_notes_raw, true)
                    : (is_array($staff_notes_raw) ? $staff_notes_raw : []);
                break;
        }

        $data['title'] = $project->name;
        $this->data($data);
        $this->view('restoration/client/view');
        $this->layout();
    }

    public function get_restoration_files($project_id)
    {
        $this->db->where('project_id', $project_id);
        $this->db->where('rel_type', 'restoration');
        $this->db->where('project_file', 'project_file');
        return $this->db->get(db_prefix() . 'project_files')->result_array();
    }

    public function get_restoration_images($project_id)
    {
        $this->db->where('project_id', $project_id);
        $this->db->where('project_image', 'project_image');
        $this->db->where('rel_type', 'restoration');
        return $this->db->get(db_prefix() . 'project_files')->result_array();
    }
   
}